# -*- coding: utf-8 -*-
# pylint: disable=too-many-lines

import abc
import enum

import attr
import six

from cryptoparser.common.algorithm import Authentication, BlockCipher, BlockCipherMode, KeyExchange, MAC
from cryptoparser.common.base import Serializable
from cryptoparser.common.base import TwoByteEnumComposer, TwoByteEnumParsable
from cryptoparser.common.base import ThreeByteEnumParsable, ThreeByteEnumComposer
from cryptoparser.tls.version import (
    SslProtocolVersion,
    TlsProtocolVersionBase,
    TlsProtocolVersionDraft,
    TlsProtocolVersionFinal,
    TlsVersion
)


class TlsCipherSuiteExtension(enum.IntEnum):
    FALLBACK_SCSV = 0x5600
    EMPTY_RENEGOTIATION_INFO_SCSV = 0x00ff


class TlsCipherSuiteFactory(TwoByteEnumParsable):
    @classmethod
    def get_enum_class(cls):
        return TlsCipherSuite

    @abc.abstractmethod
    def compose(self):
        raise NotImplementedError()


@attr.s
class CipherSuiteParams(Serializable):  # pylint: disable=too-many-instance-attributes
    code = attr.ib(validator=attr.validators.instance_of(int))
    iana_name = attr.ib(validator=attr.validators.optional(attr.validators.instance_of(str)))
    openssl_name = attr.ib(validator=attr.validators.optional(attr.validators.instance_of(str)))
    key_exchange = attr.ib(validator=attr.validators.optional(attr.validators.in_(KeyExchange)))
    authentication = attr.ib(validator=attr.validators.optional(attr.validators.in_(Authentication)))
    bulk_cipher = attr.ib(validator=attr.validators.optional(attr.validators.in_(BlockCipher)))
    block_cipher_mode = attr.ib(validator=attr.validators.optional(attr.validators.in_(BlockCipherMode)))
    mac = attr.ib(validator=attr.validators.optional(attr.validators.in_(MAC)))
    authenticated_encryption = attr.ib(validator=attr.validators.instance_of(bool))
    initial_version = attr.ib(validator=attr.validators.instance_of((TlsProtocolVersionBase, SslProtocolVersion)))
    last_version = attr.ib(init=False, validator=attr.validators.instance_of(TlsProtocolVersionBase))

    def __attrs_post_init__(self):
        if self.code > 0xffff:
            self.last_version = SslProtocolVersion()
        else:
            if (self.code & 0xff00) == 0x1300:
                self.last_version = TlsProtocolVersionFinal(TlsVersion.TLS1_3)
            else:
                self.last_version = TlsProtocolVersionFinal(TlsVersion.TLS1_2)

    def _as_markdown(self, level):
        if self.iana_name is None:
            result = six.next(
                cipher_suite
                for cipher_suite in TlsCipherSuite
                if cipher_suite.value.code == self.code
            ).name
        else:
            result = self.iana_name

        if self.openssl_name:
            result += ' (' + self.openssl_name + ')'

        return self._markdown_result(result, level)


class TlsCipherSuite(TwoByteEnumComposer):
    TLS_NULL_WITH_NULL_NULL = CipherSuiteParams(
        code=0x0000,
        iana_name='TLS_NULL_WITH_NULL_NULL',
        openssl_name=None,
        key_exchange=None,
        authentication=Authentication.anon,
        bulk_cipher=None,
        block_cipher_mode=None,
        mac=None,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.SSL3),
    )
    TLS_RSA_WITH_NULL_MD5 = CipherSuiteParams(
        code=0x0001,
        iana_name='TLS_RSA_WITH_NULL_MD5',
        openssl_name='NULL-MD5',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=None,
        block_cipher_mode=None,
        mac=MAC.MD5,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.SSL3),
    )
    TLS_RSA_WITH_NULL_SHA = CipherSuiteParams(
        code=0x0002,
        iana_name='TLS_RSA_WITH_NULL_SHA',
        openssl_name='NULL-SHA',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=None,
        block_cipher_mode=None,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.SSL3),
    )
    TLS_RSA_EXPORT_WITH_RC4_40_MD5 = CipherSuiteParams(
        code=0x0003,
        iana_name='TLS_RSA_EXPORT_WITH_RC4_40_MD5',
        openssl_name='EXP-RC4-MD5',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.RC4_40,
        block_cipher_mode=None,
        mac=MAC.MD5,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.SSL3),
    )
    TLS_RSA_WITH_RC4_128_MD5 = CipherSuiteParams(
        code=0x0004,
        iana_name='TLS_RSA_WITH_RC4_128_MD5',
        openssl_name='RC4-MD5',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.RC4_128,
        block_cipher_mode=None,
        mac=MAC.MD5,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.SSL3),
    )
    TLS_RSA_WITH_RC4_128_SHA = CipherSuiteParams(
        code=0x0005,
        iana_name='TLS_RSA_WITH_RC4_128_SHA',
        openssl_name='RC4-SHA',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.RC4_128,
        block_cipher_mode=None,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.SSL3),
    )
    TLS_RSA_EXPORT_WITH_RC2_CBC_40_MD5 = CipherSuiteParams(
        code=0x0006,
        iana_name='TLS_RSA_EXPORT_WITH_RC2_CBC_40_MD5',
        openssl_name='EXP-RC2-CBC-MD5',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.RC2_40,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.MD5,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.SSL3),
    )
    TLS_RSA_WITH_IDEA_CBC_SHA = CipherSuiteParams(
        code=0x0007,
        iana_name='TLS_RSA_WITH_IDEA_CBC_SHA',
        openssl_name='IDEA-CBC-SHA',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.IDEA,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.SSL3),
    )
    TLS_RSA_EXPORT_WITH_DES40_CBC_SHA = CipherSuiteParams(
        code=0x0008,
        iana_name='TLS_RSA_EXPORT_WITH_DES40_CBC_SHA',
        openssl_name='EXP-DES-CBC-SHA',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.DES40,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.SSL3),
    )
    TLS_RSA_WITH_DES_CBC_SHA = CipherSuiteParams(
        code=0x0009,
        iana_name='TLS_RSA_WITH_DES_CBC_SHA',
        openssl_name='DES-CBC-SHA',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.DES,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.SSL3),
    )
    TLS_RSA_WITH_3DES_EDE_CBC_SHA = CipherSuiteParams(
        code=0x000a,
        iana_name='TLS_RSA_WITH_3DES_EDE_CBC_SHA',
        openssl_name='DES-CBC3-SHA',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.TRIPLE_DES_EDE,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.SSL3),
    )
    TLS_DH_DSS_EXPORT_WITH_DES40_CBC_SHA = CipherSuiteParams(
        code=0x000b,
        iana_name='TLS_DH_DSS_EXPORT_WITH_DES40_CBC_SHA',
        openssl_name='EXP-DH-DSS-DES-CBC-SHA',
        key_exchange=KeyExchange.DH,
        authentication=Authentication.DSS,
        bulk_cipher=BlockCipher.DES40,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.SSL3),
    )
    TLS_DH_DSS_WITH_DES_CBC_SHA = CipherSuiteParams(
        code=0x000c,
        iana_name='TLS_DH_DSS_WITH_DES_CBC_SHA',
        openssl_name='DH-DSS-DES-CBC-SHA',
        key_exchange=KeyExchange.DH,
        authentication=Authentication.DSS,
        bulk_cipher=BlockCipher.DES,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.SSL3),
    )
    TLS_DH_DSS_WITH_3DES_EDE_CBC_SHA = CipherSuiteParams(
        code=0x000d,
        iana_name='TLS_DH_DSS_WITH_3DES_EDE_CBC_SHA',
        openssl_name='DH-DSS-DES-CBC3-SHA',
        key_exchange=KeyExchange.DH,
        authentication=Authentication.DSS,
        bulk_cipher=BlockCipher.TRIPLE_DES_EDE,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.SSL3),
    )
    TLS_DH_RSA_EXPORT_WITH_DES40_CBC_SHA = CipherSuiteParams(
        code=0x000e,
        iana_name='TLS_DH_RSA_EXPORT_WITH_DES40_CBC_SHA',
        openssl_name='EXP-DH-RSA-DES-CBC-SHA',
        key_exchange=KeyExchange.DH,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.DES40,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.SSL3),
    )
    TLS_DH_RSA_WITH_DES_CBC_SHA = CipherSuiteParams(
        code=0x000f,
        iana_name='TLS_DH_RSA_WITH_DES_CBC_SHA',
        openssl_name='DH-RSA-DES-CBC-SHA',
        key_exchange=KeyExchange.DH,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.DES,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.SSL3),
    )
    TLS_DH_RSA_WITH_3DES_EDE_CBC_SHA = CipherSuiteParams(
        code=0x0010,
        iana_name='TLS_DH_RSA_WITH_3DES_EDE_CBC_SHA',
        openssl_name='DH-RSA-DES-CBC3-SHA',
        key_exchange=KeyExchange.DH,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.TRIPLE_DES_EDE,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.SSL3),
    )
    TLS_DHE_DSS_EXPORT_WITH_DES40_CBC_SHA = CipherSuiteParams(
        code=0x0011,
        iana_name='TLS_DHE_DSS_EXPORT_WITH_DES40_CBC_SHA',
        openssl_name='EXP-EDH-DSS-DES-CBC-SHA',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.DSS,
        bulk_cipher=BlockCipher.DES40,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.SSL3),
    )
    TLS_DHE_DSS_WITH_DES_CBC_SHA = CipherSuiteParams(
        code=0x0012,
        iana_name='TLS_DHE_DSS_WITH_DES_CBC_SHA',
        openssl_name='EDH-DSS-DES-CBC-SHA',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.DSS,
        bulk_cipher=BlockCipher.DES,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.SSL3),
    )
    TLS_DHE_DSS_WITH_3DES_EDE_CBC_SHA = CipherSuiteParams(
        code=0x0013,
        iana_name='TLS_DHE_DSS_WITH_3DES_EDE_CBC_SHA',
        openssl_name='EDH-DSS-DES-CBC3-SHA',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.DSS,
        bulk_cipher=BlockCipher.TRIPLE_DES_EDE,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.SSL3),
    )
    TLS_DHE_RSA_EXPORT_WITH_DES40_CBC_SHA = CipherSuiteParams(
        code=0x0014,
        iana_name='TLS_DHE_RSA_EXPORT_WITH_DES40_CBC_SHA',
        openssl_name='EXP-EDH-RSA-DES-CBC-SHA',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.DES40,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.SSL3),
    )
    TLS_DHE_RSA_WITH_DES_CBC_SHA = CipherSuiteParams(
        code=0x0015,
        iana_name='TLS_DHE_RSA_WITH_DES_CBC_SHA',
        openssl_name='EDH-RSA-DES-CBC-SHA',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.DES,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.SSL3),
    )
    TLS_DHE_RSA_WITH_3DES_EDE_CBC_SHA = CipherSuiteParams(
        code=0x0016,
        iana_name='TLS_DHE_RSA_WITH_3DES_EDE_CBC_SHA',
        openssl_name='EDH-RSA-DES-CBC3-SHA',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.TRIPLE_DES_EDE,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.SSL3),
    )
    TLS_DH_anon_EXPORT_WITH_RC4_40_MD5 = CipherSuiteParams(  # pylint: disable=invalid-name
        code=0x0017,
        iana_name='TLS_DH_anon_EXPORT_WITH_RC4_40_MD5',
        openssl_name='EXP-ADH-RC4-MD5',
        key_exchange=KeyExchange.ADH,
        authentication=Authentication.anon,
        bulk_cipher=BlockCipher.RC4_40,
        block_cipher_mode=None,
        mac=MAC.MD5,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.SSL3),
    )
    TLS_DH_anon_WITH_RC4_128_MD5 = CipherSuiteParams(  # pylint: disable=invalid-name
        code=0x0018,
        iana_name='TLS_DH_anon_WITH_RC4_128_MD5',
        openssl_name='ADH-RC4-MD5',
        key_exchange=KeyExchange.ADH,
        authentication=Authentication.anon,
        bulk_cipher=BlockCipher.RC4_128,
        block_cipher_mode=None,
        mac=MAC.MD5,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.SSL3),
    )
    TLS_DH_anon_EXPORT_WITH_DES40_CBC_SHA = CipherSuiteParams(  # pylint: disable=invalid-name
        code=0x0019,
        iana_name='TLS_DH_anon_EXPORT_WITH_DES40_CBC_SHA',
        openssl_name='EXP-ADH-DES-CBC-SHA',
        key_exchange=KeyExchange.ADH,
        authentication=Authentication.anon,
        bulk_cipher=BlockCipher.DES40,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.SSL3),
    )
    TLS_DH_anon_WITH_DES_CBC_SHA = CipherSuiteParams(  # pylint: disable=invalid-name
        code=0x001a,
        iana_name='TLS_DH_anon_WITH_DES_CBC_SHA',
        openssl_name='ADH-DES-CBC-SHA',
        key_exchange=KeyExchange.ADH,
        authentication=Authentication.anon,
        bulk_cipher=BlockCipher.DES,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.SSL3),
    )
    TLS_DH_anon_WITH_3DES_EDE_CBC_SHA = CipherSuiteParams(  # pylint: disable=invalid-name
        code=0x001b,
        iana_name='TLS_DH_anon_WITH_3DES_EDE_CBC_SHA',
        openssl_name='ADH-DES-CBC3-SHA',
        key_exchange=KeyExchange.ADH,
        authentication=Authentication.anon,
        bulk_cipher=BlockCipher.TRIPLE_DES_EDE,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.SSL3),
    )
    TLS_FORTEZZA_KEA_WITH_NULL_SHA = CipherSuiteParams(
        code=0x001c,
        iana_name='TLS_FORTEZZA_KEA_WITH_NULL_SHA',
        openssl_name=None,
        key_exchange=KeyExchange.FORTEZZA_KEA,
        authentication=Authentication.FORTEZZA,
        bulk_cipher=None,
        block_cipher_mode=None,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.SSL3),
    )
    TLS_FORTEZZA_KEA_WITH_FORTEZZA_CBC_SHA = CipherSuiteParams(
        code=0x001d,
        iana_name='TLS_FORTEZZA_KEA_WITH_FORTEZZA_CBC_SHA',
        openssl_name=None,
        key_exchange=KeyExchange.FORTEZZA_KEA,
        authentication=Authentication.FORTEZZA,
        bulk_cipher=BlockCipher.FORTEZZA,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.SSL3),
    )
    TLS_KRB5_WITH_DES_CBC_SHA = CipherSuiteParams(
        code=0x001e,
        iana_name='TLS_KRB5_WITH_DES_CBC_SHA',
        openssl_name=None,
        key_exchange=KeyExchange.KRB5,
        authentication=Authentication.KRB5,
        bulk_cipher=BlockCipher.DES,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_KRB5_WITH_3DES_EDE_CBC_SHA = CipherSuiteParams(
        code=0x001f,
        iana_name='TLS_KRB5_WITH_3DES_EDE_CBC_SHA',
        openssl_name=None,
        key_exchange=KeyExchange.KRB5,
        authentication=Authentication.KRB5,
        bulk_cipher=BlockCipher.TRIPLE_DES_EDE,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_KRB5_WITH_RC4_128_SHA = CipherSuiteParams(
        code=0x0020,
        iana_name='TLS_KRB5_WITH_RC4_128_SHA',
        openssl_name=None,
        key_exchange=KeyExchange.KRB5,
        authentication=Authentication.KRB5,
        bulk_cipher=BlockCipher.RC4_128,
        block_cipher_mode=None,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_KRB5_WITH_IDEA_CBC_SHA = CipherSuiteParams(
        code=0x0021,
        iana_name='TLS_KRB5_WITH_IDEA_CBC_SHA',
        openssl_name=None,
        key_exchange=KeyExchange.KRB5,
        authentication=Authentication.KRB5,
        bulk_cipher=BlockCipher.IDEA,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_KRB5_WITH_DES_CBC_MD5 = CipherSuiteParams(
        code=0x0022,
        iana_name='TLS_KRB5_WITH_DES_CBC_MD5',
        openssl_name=None,
        key_exchange=KeyExchange.KRB5,
        authentication=Authentication.KRB5,
        bulk_cipher=BlockCipher.DES,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.MD5,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_KRB5_WITH_3DES_EDE_CBC_MD5 = CipherSuiteParams(
        code=0x0023,
        iana_name='TLS_KRB5_WITH_3DES_EDE_CBC_MD5',
        openssl_name=None,
        key_exchange=KeyExchange.KRB5,
        authentication=Authentication.KRB5,
        bulk_cipher=BlockCipher.TRIPLE_DES_EDE,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.MD5,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_KRB5_WITH_RC4_128_MD5 = CipherSuiteParams(
        code=0x0024,
        iana_name='TLS_KRB5_WITH_RC4_128_MD5',
        openssl_name=None,
        key_exchange=KeyExchange.KRB5,
        authentication=Authentication.KRB5,
        bulk_cipher=BlockCipher.RC4_128,
        block_cipher_mode=None,
        mac=MAC.MD5,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_KRB5_WITH_IDEA_CBC_MD5 = CipherSuiteParams(
        code=0x0025,
        iana_name='TLS_KRB5_WITH_IDEA_CBC_MD5',
        openssl_name=None,
        key_exchange=KeyExchange.KRB5,
        authentication=Authentication.KRB5,
        bulk_cipher=BlockCipher.IDEA,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.MD5,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_KRB5_EXPORT_WITH_DES_CBC_40_SHA = CipherSuiteParams(
        code=0x0026,
        iana_name='TLS_KRB5_EXPORT_WITH_DES_CBC_40_SHA',
        openssl_name=None,
        key_exchange=KeyExchange.KRB5,
        authentication=Authentication.KRB5,
        bulk_cipher=BlockCipher.DES40,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_KRB5_EXPORT_WITH_RC2_CBC_40_SHA = CipherSuiteParams(
        code=0x0027,
        iana_name='TLS_KRB5_EXPORT_WITH_RC2_CBC_40_SHA',
        openssl_name=None,
        key_exchange=KeyExchange.KRB5,
        authentication=Authentication.KRB5,
        bulk_cipher=BlockCipher.RC2_40,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_KRB5_EXPORT_WITH_RC4_40_SHA = CipherSuiteParams(
        code=0x0028,
        iana_name='TLS_KRB5_EXPORT_WITH_RC4_40_SHA',
        openssl_name=None,
        key_exchange=KeyExchange.KRB5,
        authentication=Authentication.KRB5,
        bulk_cipher=BlockCipher.RC4_40,
        block_cipher_mode=None,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_KRB5_EXPORT_WITH_DES_CBC_40_MD5 = CipherSuiteParams(
        code=0x0029,
        iana_name='TLS_KRB5_EXPORT_WITH_DES_CBC_40_MD5',
        openssl_name=None,
        key_exchange=KeyExchange.KRB5,
        authentication=Authentication.KRB5,
        bulk_cipher=BlockCipher.DES40,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.MD5,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_KRB5_EXPORT_WITH_RC2_CBC_40_MD5 = CipherSuiteParams(
        code=0x002a,
        iana_name='TLS_KRB5_EXPORT_WITH_RC2_CBC_40_MD5',
        openssl_name=None,
        key_exchange=KeyExchange.KRB5,
        authentication=Authentication.KRB5,
        bulk_cipher=BlockCipher.RC2_40,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.MD5,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_KRB5_EXPORT_WITH_RC4_40_MD5 = CipherSuiteParams(
        code=0x002b,
        iana_name='TLS_KRB5_EXPORT_WITH_RC4_40_MD5',
        openssl_name=None,
        key_exchange=KeyExchange.KRB5,
        authentication=Authentication.KRB5,
        bulk_cipher=BlockCipher.RC4_40,
        block_cipher_mode=None,
        mac=MAC.MD5,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_PSK_WITH_NULL_SHA = CipherSuiteParams(
        code=0x002c,
        iana_name='TLS_PSK_WITH_NULL_SHA',
        openssl_name='PSK-NULL-SHA',
        key_exchange=KeyExchange.PSK,
        authentication=Authentication.PSK,
        bulk_cipher=None,
        block_cipher_mode=None,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DHE_PSK_WITH_NULL_SHA = CipherSuiteParams(
        code=0x002d,
        iana_name='TLS_DHE_PSK_WITH_NULL_SHA',
        openssl_name='DHE-PSK-NULL-SHA',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.PSK,
        bulk_cipher=None,
        block_cipher_mode=None,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_RSA_PSK_WITH_NULL_SHA = CipherSuiteParams(
        code=0x002e,
        iana_name='TLS_RSA_PSK_WITH_NULL_SHA',
        openssl_name='RSA-PSK-NULL-SHA',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.PSK,
        bulk_cipher=None,
        block_cipher_mode=None,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_RSA_WITH_AES_128_CBC_SHA = CipherSuiteParams(
        code=0x002f,
        iana_name='TLS_RSA_WITH_AES_128_CBC_SHA',
        openssl_name='AES128-SHA',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DH_DSS_WITH_AES_128_CBC_SHA = CipherSuiteParams(
        code=0x0030,
        iana_name='TLS_DH_DSS_WITH_AES_128_CBC_SHA',
        openssl_name='DH-DSS-AES128-SHA',
        key_exchange=KeyExchange.DH,
        authentication=Authentication.DSS,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DH_RSA_WITH_AES_128_CBC_SHA = CipherSuiteParams(
        code=0x0031,
        iana_name='TLS_DH_RSA_WITH_AES_128_CBC_SHA',
        openssl_name='DH-RSA-AES128-SHA',
        key_exchange=KeyExchange.DH,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DHE_DSS_WITH_AES_128_CBC_SHA = CipherSuiteParams(
        code=0x0032,
        iana_name='TLS_DHE_DSS_WITH_AES_128_CBC_SHA',
        openssl_name='DHE-DSS-AES128-SHA',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.DSS,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DHE_RSA_WITH_AES_128_CBC_SHA = CipherSuiteParams(
        code=0x0033,
        iana_name='TLS_DHE_RSA_WITH_AES_128_CBC_SHA',
        openssl_name='DHE-RSA-AES128-SHA',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DH_anon_WITH_AES_128_CBC_SHA = CipherSuiteParams(  # pylint: disable=invalid-name
        code=0x0034,
        iana_name='TLS_DH_anon_WITH_AES_128_CBC_SHA',
        openssl_name='ADH-AES128-SHA',
        key_exchange=KeyExchange.ADH,
        authentication=Authentication.anon,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_RSA_WITH_AES_256_CBC_SHA = CipherSuiteParams(
        code=0x0035,
        iana_name='TLS_RSA_WITH_AES_256_CBC_SHA',
        openssl_name='AES256-SHA',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DH_DSS_WITH_AES_256_CBC_SHA = CipherSuiteParams(
        code=0x0036,
        iana_name='TLS_DH_DSS_WITH_AES_256_CBC_SHA',
        openssl_name='DH-DSS-AES256-SHA',
        key_exchange=KeyExchange.DH,
        authentication=Authentication.DSS,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DH_RSA_WITH_AES_256_CBC_SHA = CipherSuiteParams(
        code=0x0037,
        iana_name='TLS_DH_RSA_WITH_AES_256_CBC_SHA',
        openssl_name='DH-RSA-AES256-SHA',
        key_exchange=KeyExchange.DH,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DHE_DSS_WITH_AES_256_CBC_SHA = CipherSuiteParams(
        code=0x0038,
        iana_name='TLS_DHE_DSS_WITH_AES_256_CBC_SHA',
        openssl_name='DHE-DSS-AES256-SHA',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.DSS,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DHE_RSA_WITH_AES_256_CBC_SHA = CipherSuiteParams(
        code=0x0039,
        iana_name='TLS_DHE_RSA_WITH_AES_256_CBC_SHA',
        openssl_name='DHE-RSA-AES256-SHA',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DH_anon_WITH_AES_256_CBC_SHA = CipherSuiteParams(  # pylint: disable=invalid-name
        code=0x003a,
        iana_name='TLS_DH_anon_WITH_AES_256_CBC_SHA',
        openssl_name='ADH-AES256-SHA',
        key_exchange=KeyExchange.ADH,
        authentication=Authentication.anon,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_RSA_WITH_NULL_SHA256 = CipherSuiteParams(
        code=0x003b,
        iana_name='TLS_RSA_WITH_NULL_SHA256',
        openssl_name='NULL-SHA256',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=None,
        block_cipher_mode=None,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_RSA_WITH_AES_128_CBC_SHA256 = CipherSuiteParams(
        code=0x003c,
        iana_name='TLS_RSA_WITH_AES_128_CBC_SHA256',
        openssl_name='AES128-SHA256',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_RSA_WITH_AES_256_CBC_SHA256 = CipherSuiteParams(
        code=0x003d,
        iana_name='TLS_RSA_WITH_AES_256_CBC_SHA256',
        openssl_name='AES256-SHA256',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DH_DSS_WITH_AES_128_CBC_SHA256 = CipherSuiteParams(
        code=0x003e,
        iana_name='TLS_DH_DSS_WITH_AES_128_CBC_SHA256',
        openssl_name='DH-DSS-AES128-SHA256',
        key_exchange=KeyExchange.DH,
        authentication=Authentication.DSS,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DH_RSA_WITH_AES_128_CBC_SHA256 = CipherSuiteParams(
        code=0x003f,
        iana_name='TLS_DH_RSA_WITH_AES_128_CBC_SHA256',
        openssl_name='DH-RSA-AES128-SHA256',
        key_exchange=KeyExchange.DH,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DHE_DSS_WITH_AES_128_CBC_SHA256 = CipherSuiteParams(
        code=0x0040,
        iana_name='TLS_DHE_DSS_WITH_AES_128_CBC_SHA256',
        openssl_name='DHE-DSS-AES128-SHA256',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.DSS,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_RSA_WITH_CAMELLIA_128_CBC_SHA = CipherSuiteParams(
        code=0x0041,
        iana_name='TLS_RSA_WITH_CAMELLIA_128_CBC_SHA',
        openssl_name='CAMELLIA128-SHA',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.CAMELLIA_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DH_DSS_WITH_CAMELLIA_128_CBC_SHA = CipherSuiteParams(
        code=0x0042,
        iana_name='TLS_DH_DSS_WITH_CAMELLIA_128_CBC_SHA',
        openssl_name='DH-DSS-CAMELLIA128-SHA',
        key_exchange=KeyExchange.DH,
        authentication=Authentication.DSS,
        bulk_cipher=BlockCipher.CAMELLIA_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DH_RSA_WITH_CAMELLIA_128_CBC_SHA = CipherSuiteParams(
        code=0x0043,
        iana_name='TLS_DH_RSA_WITH_CAMELLIA_128_CBC_SHA',
        openssl_name='DH-RSA-CAMELLIA128-SHA',
        key_exchange=KeyExchange.DH,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.CAMELLIA_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DHE_DSS_WITH_CAMELLIA_128_CBC_SHA = CipherSuiteParams(
        code=0x0044,
        iana_name='TLS_DHE_DSS_WITH_CAMELLIA_128_CBC_SHA',
        openssl_name='DHE-DSS-CAMELLIA128-SHA',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.DSS,
        bulk_cipher=BlockCipher.CAMELLIA_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA = CipherSuiteParams(
        code=0x0045,
        iana_name='TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA',
        openssl_name='DHE-RSA-CAMELLIA128-SHA',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.CAMELLIA_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DH_anon_WITH_CAMELLIA_128_CBC_SHA = CipherSuiteParams(  # pylint: disable=invalid-name
        code=0x0046,
        iana_name='TLS_DH_anon_WITH_CAMELLIA_128_CBC_SHA',
        openssl_name='ADH-CAMELLIA128-SHA',
        key_exchange=KeyExchange.ADH,
        authentication=Authentication.anon,
        bulk_cipher=BlockCipher.CAMELLIA_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    OLD_TLS_ECDH_ECDSA_WITH_NULL_SHA = CipherSuiteParams(
        code=0x0047,
        iana_name=None,
        openssl_name=None,
        key_exchange=KeyExchange.ECDH,
        authentication=Authentication.ECDSA,
        bulk_cipher=None,
        block_cipher_mode=None,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    OLD_TLS_ECDH_ECDSA_WITH_RC4_128_SHA = CipherSuiteParams(
        code=0x0048,
        iana_name=None,
        openssl_name=None,
        key_exchange=KeyExchange.ECDH,
        authentication=Authentication.ECDSA,
        bulk_cipher=BlockCipher.RC4_128,
        block_cipher_mode=None,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_ECDH_ECDSA_WITH_DES_CBC_SHA = CipherSuiteParams(
        code=0x0049,
        iana_name='TLS_ECDH_ECDSA_WITH_DES_CBC_SHA',
        openssl_name=None,
        key_exchange=KeyExchange.ECDH,
        authentication=Authentication.ECDSA,
        bulk_cipher=BlockCipher.DES,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    OLD_TLS_ECDH_ECDSA_WITH_3DES_EDE_CBC_SHA = CipherSuiteParams(
        code=0x004a,
        iana_name=None,
        openssl_name=None,
        key_exchange=KeyExchange.ECDH,
        authentication=Authentication.ECDSA,
        bulk_cipher=BlockCipher.TRIPLE_DES_EDE,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    OLD_TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA = CipherSuiteParams(
        code=0x004b,
        iana_name=None,
        openssl_name=None,
        key_exchange=KeyExchange.ECDH,
        authentication=Authentication.ECDSA,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    OLD_TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA = CipherSuiteParams(
        code=0x004c,
        iana_name=None,
        openssl_name=None,
        key_exchange=KeyExchange.ECDH,
        authentication=Authentication.ECDSA,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_RSA_EXPORT1024_WITH_RC4_56_MD5 = CipherSuiteParams(
        code=0x0060,
        iana_name='TLS_RSA_EXPORT1024_WITH_RC4_56_MD5',
        openssl_name='EXP1024-RC4-MD5',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.RC4_56,
        block_cipher_mode=None,
        mac=MAC.MD5,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.SSL3),
    )
    TLS_RSA_EXPORT1024_WITH_RC2_CBC_56_MD5 = CipherSuiteParams(
        code=0x0061,
        iana_name='TLS_RSA_EXPORT1024_WITH_RC2_CBC_56_MD5',
        openssl_name='EXP1024-RC2-CBC-MD5',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.RC2_56,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.MD5,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.SSL3),
    )
    TLS_RSA_EXPORT1024_WITH_DES_CBC_SHA = CipherSuiteParams(
        code=0x0062,
        iana_name='TLS_RSA_EXPORT1024_WITH_DES_CBC_SHA',
        openssl_name='EXP1024-DES-CBC-SHA',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.DES,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.SSL3),
    )
    TLS_DHE_DSS_EXPORT1024_WITH_DES_CBC_SHA = CipherSuiteParams(
        code=0x0063,
        iana_name='TLS_DHE_DSS_EXPORT1024_WITH_DES_CBC_SHA',
        openssl_name='EXP1024-DHE-DSS-DES-CBC-SHA',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.DSS,
        bulk_cipher=BlockCipher.DES,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.SSL3),
    )
    TLS_RSA_EXPORT1024_WITH_RC4_56_SHA = CipherSuiteParams(
        code=0x0064,
        iana_name='TLS_RSA_EXPORT1024_WITH_RC4_56_SHA',
        openssl_name='EXP1024-RC4-SHA',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.RC4_56,
        block_cipher_mode=None,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.SSL3),
    )
    TLS_DHE_DSS_EXPORT1024_WITH_RC4_56_SHA = CipherSuiteParams(
        code=0x0065,
        iana_name='TLS_DHE_DSS_EXPORT1024_WITH_RC4_56_SHA',
        openssl_name='EXP1024-DHE-DSS-RC4-SHA',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.DSS,
        bulk_cipher=BlockCipher.RC4_56,
        block_cipher_mode=None,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.SSL3),
    )
    TLS_DHE_DSS_WITH_RC4_128_SHA = CipherSuiteParams(
        code=0x0066,
        iana_name='TLS_DHE_DSS_WITH_RC4_128_SHA',
        openssl_name='DHE-DSS-RC4-SHA',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.DSS,
        bulk_cipher=BlockCipher.RC4_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.SSL3),
    )
    TLS_DHE_RSA_WITH_AES_128_CBC_SHA256 = CipherSuiteParams(
        code=0x0067,
        iana_name='TLS_DHE_RSA_WITH_AES_128_CBC_SHA256',
        openssl_name='DHE-RSA-AES128-SHA256',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DH_DSS_WITH_AES_256_CBC_SHA256 = CipherSuiteParams(
        code=0x0068,
        iana_name='TLS_DH_DSS_WITH_AES_256_CBC_SHA256',
        openssl_name='DH-DSS-AES256-SHA256',
        key_exchange=KeyExchange.DH,
        authentication=Authentication.DSS,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DH_RSA_WITH_AES_256_CBC_SHA256 = CipherSuiteParams(
        code=0x0069,
        iana_name='TLS_DH_RSA_WITH_AES_256_CBC_SHA256',
        openssl_name='DH-RSA-AES256-SHA256',
        key_exchange=KeyExchange.DH,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DHE_DSS_WITH_AES_256_CBC_SHA256 = CipherSuiteParams(
        code=0x006a,
        iana_name='TLS_DHE_DSS_WITH_AES_256_CBC_SHA256',
        openssl_name='DHE-DSS-AES256-SHA256',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.DSS,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DHE_RSA_WITH_AES_256_CBC_SHA256 = CipherSuiteParams(
        code=0x006b,
        iana_name='TLS_DHE_RSA_WITH_AES_256_CBC_SHA256',
        openssl_name='DHE-RSA-AES256-SHA256',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DH_anon_WITH_AES_128_CBC_SHA256 = CipherSuiteParams(  # pylint: disable=invalid-name
        code=0x006c,
        iana_name='TLS_DH_anon_WITH_AES_128_CBC_SHA256',
        openssl_name='ADH-AES128-SHA256',
        key_exchange=KeyExchange.ADH,
        authentication=Authentication.anon,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DH_anon_WITH_AES_256_CBC_SHA256 = CipherSuiteParams(  # pylint: disable=invalid-name
        code=0x006d,
        iana_name='TLS_DH_anon_WITH_AES_256_CBC_SHA256',
        openssl_name='ADH-AES256-SHA256',
        key_exchange=KeyExchange.ADH,
        authentication=Authentication.anon,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_GOSTR341094_WITH_28147_CNT_IMIT = CipherSuiteParams(
        code=0x0080,
        iana_name='TLS_GOSTR341094_WITH_28147_CNT_IMIT',
        openssl_name='GOST94-GOST89-GOST89',
        key_exchange=KeyExchange.GOST_R3410_94,
        authentication=Authentication.GOST_R3410_94,
        bulk_cipher=BlockCipher.GOST2814789,
        block_cipher_mode=None,
        mac=MAC.IMIT_GOST28147,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_GOSTR341001_WITH_28147_CNT_IMIT = CipherSuiteParams(
        code=0x0081,
        iana_name='TLS_GOSTR341001_WITH_28147_CNT_IMIT',
        openssl_name='GOST2001-GOST89-GOST89',
        key_exchange=KeyExchange.GOST_R3410_01,
        authentication=Authentication.GOST_R3410_94,
        bulk_cipher=BlockCipher.GOST2814789,
        block_cipher_mode=None,
        mac=MAC.IMIT_GOST28147,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_GOSTR341094_WITH_NULL_GOSTR3411 = CipherSuiteParams(
        code=0x0082,
        iana_name='TLS_GOSTR341094_WITH_NULL_GOSTR3411',
        openssl_name='GOST94-NULL-GOST94',
        key_exchange=KeyExchange.GOST_R3410_94,
        authentication=Authentication.GOST_R3410_94,
        bulk_cipher=None,
        block_cipher_mode=None,
        mac=MAC.GOST_R3411_94,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_GOSTR341001_WITH_NULL_GOSTR3411 = CipherSuiteParams(
        code=0x0083,
        iana_name='TLS_GOSTR341001_WITH_NULL_GOSTR3411',
        openssl_name='GOST2001-NULL-GOST94',
        key_exchange=KeyExchange.GOST_R3410_01,
        authentication=Authentication.GOST_R3410_94,
        bulk_cipher=None,
        block_cipher_mode=None,
        mac=MAC.GOST_R3411_94,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_RSA_WITH_CAMELLIA_256_CBC_SHA = CipherSuiteParams(
        code=0x0084,
        iana_name='TLS_RSA_WITH_CAMELLIA_256_CBC_SHA',
        openssl_name='CAMELLIA256-SHA',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.CAMELLIA_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DH_DSS_WITH_CAMELLIA_256_CBC_SHA = CipherSuiteParams(
        code=0x0085,
        iana_name='TLS_DH_DSS_WITH_CAMELLIA_256_CBC_SHA',
        openssl_name='DH-DSS-CAMELLIA256-SHA',
        key_exchange=KeyExchange.DH,
        authentication=Authentication.DSS,
        bulk_cipher=BlockCipher.CAMELLIA_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DH_RSA_WITH_CAMELLIA_256_CBC_SHA = CipherSuiteParams(
        code=0x0086,
        iana_name='TLS_DH_RSA_WITH_CAMELLIA_256_CBC_SHA',
        openssl_name='DH-RSA-CAMELLIA256-SHA',
        key_exchange=KeyExchange.DH,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.CAMELLIA_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DHE_DSS_WITH_CAMELLIA_256_CBC_SHA = CipherSuiteParams(
        code=0x0087,
        iana_name='TLS_DHE_DSS_WITH_CAMELLIA_256_CBC_SHA',
        openssl_name='DHE-DSS-CAMELLIA256-SHA',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.DSS,
        bulk_cipher=BlockCipher.CAMELLIA_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA = CipherSuiteParams(
        code=0x0088,
        iana_name='TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA',
        openssl_name='DHE-RSA-CAMELLIA256-SHA',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.CAMELLIA_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DH_anon_WITH_CAMELLIA_256_CBC_SHA = CipherSuiteParams(  # pylint: disable=invalid-name
        code=0x0089,
        iana_name='TLS_DH_anon_WITH_CAMELLIA_256_CBC_SHA',
        openssl_name='ADH-CAMELLIA256-SHA',
        key_exchange=KeyExchange.ADH,
        authentication=Authentication.anon,
        bulk_cipher=BlockCipher.CAMELLIA_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_PSK_WITH_RC4_128_SHA = CipherSuiteParams(
        code=0x008a,
        iana_name='TLS_PSK_WITH_RC4_128_SHA',
        openssl_name='PSK-RC4-SHA',
        key_exchange=KeyExchange.PSK,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.RC4_128,
        block_cipher_mode=None,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_PSK_WITH_3DES_EDE_CBC_SHA = CipherSuiteParams(
        code=0x008b,
        iana_name='TLS_PSK_WITH_3DES_EDE_CBC_SHA',
        openssl_name='PSK-3DES-EDE-CBC-SHA',
        key_exchange=KeyExchange.PSK,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.TRIPLE_DES_EDE,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_PSK_WITH_AES_128_CBC_SHA = CipherSuiteParams(
        code=0x008c,
        iana_name='TLS_PSK_WITH_AES_128_CBC_SHA',
        openssl_name='PSK-AES128-CBC-SHA',
        key_exchange=KeyExchange.PSK,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_PSK_WITH_AES_256_CBC_SHA = CipherSuiteParams(
        code=0x008d,
        iana_name='TLS_PSK_WITH_AES_256_CBC_SHA',
        openssl_name='PSK-AES256-CBC-SHA',
        key_exchange=KeyExchange.PSK,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DHE_PSK_WITH_RC4_128_SHA = CipherSuiteParams(
        code=0x008e,
        iana_name='TLS_DHE_PSK_WITH_RC4_128_SHA',
        openssl_name='DHE-PSK-RC4-SHA',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.RC4_128,
        block_cipher_mode=None,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DHE_PSK_WITH_3DES_EDE_CBC_SHA = CipherSuiteParams(
        code=0x008f,
        iana_name='TLS_DHE_PSK_WITH_3DES_EDE_CBC_SHA',
        openssl_name='DHE-PSK-3DES-EDE-CBC-SHA',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.TRIPLE_DES_EDE,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DHE_PSK_WITH_AES_128_CBC_SHA = CipherSuiteParams(
        code=0x0090,
        iana_name='TLS_DHE_PSK_WITH_AES_128_CBC_SHA',
        openssl_name='DHE-PSK-AES128-CBC-SHA',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DHE_PSK_WITH_AES_256_CBC_SHA = CipherSuiteParams(
        code=0x0091,
        iana_name='TLS_DHE_PSK_WITH_AES_256_CBC_SHA',
        openssl_name='DHE-PSK-AES256-CBC-SHA',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_RSA_PSK_WITH_RC4_128_SHA = CipherSuiteParams(
        code=0x0092,
        iana_name='TLS_RSA_PSK_WITH_RC4_128_SHA',
        openssl_name='RSA-PSK-RC4-SHA',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.RC4_128,
        block_cipher_mode=None,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_RSA_PSK_WITH_3DES_EDE_CBC_SHA = CipherSuiteParams(
        code=0x0093,
        iana_name='TLS_RSA_PSK_WITH_3DES_EDE_CBC_SHA',
        openssl_name='RSA-PSK-3DES-EDE-CBC-SHA',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.TRIPLE_DES_EDE,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_RSA_PSK_WITH_AES_128_CBC_SHA = CipherSuiteParams(
        code=0x0094,
        iana_name='TLS_RSA_PSK_WITH_AES_128_CBC_SHA',
        openssl_name='RSA-PSK-AES128-CBC-SHA',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_RSA_PSK_WITH_AES_256_CBC_SHA = CipherSuiteParams(
        code=0x0095,
        iana_name='TLS_RSA_PSK_WITH_AES_256_CBC_SHA',
        openssl_name='RSA-PSK-AES256-CBC-SHA',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_RSA_WITH_SEED_CBC_SHA = CipherSuiteParams(
        code=0x0096,
        iana_name='TLS_RSA_WITH_SEED_CBC_SHA',
        openssl_name='SEED-SHA',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.SEED,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DH_DSS_WITH_SEED_CBC_SHA = CipherSuiteParams(
        code=0x0097,
        iana_name='TLS_DH_DSS_WITH_SEED_CBC_SHA',
        openssl_name='DH-DSS-SEED-SHA',
        key_exchange=KeyExchange.DH,
        authentication=Authentication.DSS,
        bulk_cipher=BlockCipher.SEED,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DH_RSA_WITH_SEED_CBC_SHA = CipherSuiteParams(
        code=0x0098,
        iana_name='TLS_DH_RSA_WITH_SEED_CBC_SHA',
        openssl_name='DH-RSA-SEED-SHA',
        key_exchange=KeyExchange.DH,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.SEED,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DHE_DSS_WITH_SEED_CBC_SHA = CipherSuiteParams(
        code=0x0099,
        iana_name='TLS_DHE_DSS_WITH_SEED_CBC_SHA',
        openssl_name='DHE-DSS-SEED-SHA',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.DSS,
        bulk_cipher=BlockCipher.SEED,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DHE_RSA_WITH_SEED_CBC_SHA = CipherSuiteParams(
        code=0x009a,
        iana_name='TLS_DHE_RSA_WITH_SEED_CBC_SHA',
        openssl_name='DHE-RSA-SEED-SHA',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.SEED,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DH_anon_WITH_SEED_CBC_SHA = CipherSuiteParams(  # pylint: disable=invalid-name
        code=0x009b,
        iana_name='TLS_DH_anon_WITH_SEED_CBC_SHA',
        openssl_name='ADH-SEED-SHA',
        key_exchange=KeyExchange.ADH,
        authentication=Authentication.anon,
        bulk_cipher=BlockCipher.SEED,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_RSA_WITH_AES_128_GCM_SHA256 = CipherSuiteParams(
        code=0x009c,
        iana_name='TLS_RSA_WITH_AES_128_GCM_SHA256',
        openssl_name='AES128-GCM-SHA256',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_256,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_RSA_WITH_AES_256_GCM_SHA384 = CipherSuiteParams(
        code=0x009d,
        iana_name='TLS_RSA_WITH_AES_256_GCM_SHA384',
        openssl_name='AES256-GCM-SHA384',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_384,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DHE_RSA_WITH_AES_128_GCM_SHA256 = CipherSuiteParams(
        code=0x009e,
        iana_name='TLS_DHE_RSA_WITH_AES_128_GCM_SHA256',
        openssl_name='DHE-RSA-AES128-GCM-SHA256',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_256,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DHE_RSA_WITH_AES_256_GCM_SHA384 = CipherSuiteParams(
        code=0x009f,
        iana_name='TLS_DHE_RSA_WITH_AES_256_GCM_SHA384',
        openssl_name='DHE-RSA-AES256-GCM-SHA384',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_384,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DH_RSA_WITH_AES_128_GCM_SHA256 = CipherSuiteParams(
        code=0x00a0,
        iana_name='TLS_DH_RSA_WITH_AES_128_GCM_SHA256',
        openssl_name='DH-RSA-AES128-GCM-SHA256',
        key_exchange=KeyExchange.DH,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_256,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DH_RSA_WITH_AES_256_GCM_SHA384 = CipherSuiteParams(
        code=0x00a1,
        iana_name='TLS_DH_RSA_WITH_AES_256_GCM_SHA384',
        openssl_name='DH-RSA-AES256-GCM-SHA384',
        key_exchange=KeyExchange.DH,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_384,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DHE_DSS_WITH_AES_128_GCM_SHA256 = CipherSuiteParams(
        code=0x00a2,
        iana_name='TLS_DHE_DSS_WITH_AES_128_GCM_SHA256',
        openssl_name='DHE-DSS-AES128-GCM-SHA256',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.DSS,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_256,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DHE_DSS_WITH_AES_256_GCM_SHA384 = CipherSuiteParams(
        code=0x00a3,
        iana_name='TLS_DHE_DSS_WITH_AES_256_GCM_SHA384',
        openssl_name='DHE-DSS-AES256-GCM-SHA384',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.DSS,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_384,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DH_DSS_WITH_AES_128_GCM_SHA256 = CipherSuiteParams(
        code=0x00a4,
        iana_name='TLS_DH_DSS_WITH_AES_128_GCM_SHA256',
        openssl_name='DH-DSS-AES128-GCM-SHA256',
        key_exchange=KeyExchange.DH,
        authentication=Authentication.DSS,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_256,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DH_DSS_WITH_AES_256_GCM_SHA384 = CipherSuiteParams(
        code=0x00a5,
        iana_name='TLS_DH_DSS_WITH_AES_256_GCM_SHA384',
        openssl_name='DH-DSS-AES256-GCM-SHA384',
        key_exchange=KeyExchange.DH,
        authentication=Authentication.DSS,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_384,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DH_anon_WITH_AES_128_GCM_SHA256 = CipherSuiteParams(  # pylint: disable=invalid-name
        code=0x00a6,
        iana_name='TLS_DH_anon_WITH_AES_128_GCM_SHA256',
        openssl_name='ADH-AES128-GCM-SHA256',
        key_exchange=KeyExchange.ADH,
        authentication=Authentication.anon,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_256,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DH_anon_WITH_AES_256_GCM_SHA384 = CipherSuiteParams(  # pylint: disable=invalid-name
        code=0x00a7,
        iana_name='TLS_DH_anon_WITH_AES_256_GCM_SHA384',
        openssl_name='ADH-AES256-GCM-SHA384',
        key_exchange=KeyExchange.ADH,
        authentication=Authentication.anon,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_384,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_PSK_WITH_AES_128_GCM_SHA256 = CipherSuiteParams(
        code=0x00a8,
        iana_name='TLS_PSK_WITH_AES_128_GCM_SHA256',
        openssl_name='PSK-AES128-GCM-SHA256',
        key_exchange=KeyExchange.PSK,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_256,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_PSK_WITH_AES_256_GCM_SHA384 = CipherSuiteParams(
        code=0x00a9,
        iana_name='TLS_PSK_WITH_AES_256_GCM_SHA384',
        openssl_name='PSK-AES256-GCM-SHA384',
        key_exchange=KeyExchange.PSK,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_384,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DHE_PSK_WITH_AES_128_GCM_SHA256 = CipherSuiteParams(
        code=0x00aa,
        iana_name='TLS_DHE_PSK_WITH_AES_128_GCM_SHA256',
        openssl_name='DHE-PSK-AES128-GCM-SHA256',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_256,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DHE_PSK_WITH_AES_256_GCM_SHA384 = CipherSuiteParams(
        code=0x00ab,
        iana_name='TLS_DHE_PSK_WITH_AES_256_GCM_SHA384',
        openssl_name='DHE-PSK-AES256-GCM-SHA384',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_384,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_RSA_PSK_WITH_AES_128_GCM_SHA256 = CipherSuiteParams(
        code=0x00ac,
        iana_name='TLS_RSA_PSK_WITH_AES_128_GCM_SHA256',
        openssl_name='RSA-PSK-AES128-GCM-SHA256',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_256,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_RSA_PSK_WITH_AES_256_GCM_SHA384 = CipherSuiteParams(
        code=0x00ad,
        iana_name='TLS_RSA_PSK_WITH_AES_256_GCM_SHA384',
        openssl_name='RSA-PSK-AES256-GCM-SHA384',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_384,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_PSK_WITH_AES_128_CBC_SHA256 = CipherSuiteParams(
        code=0x00ae,
        iana_name='TLS_PSK_WITH_AES_128_CBC_SHA256',
        openssl_name='PSK-AES128-CBC-SHA256',
        key_exchange=KeyExchange.PSK,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_PSK_WITH_AES_256_CBC_SHA384 = CipherSuiteParams(
        code=0x00af,
        iana_name='TLS_PSK_WITH_AES_256_CBC_SHA384',
        openssl_name='PSK-AES256-CBC-SHA384',
        key_exchange=KeyExchange.PSK,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_384,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_PSK_WITH_NULL_SHA256 = CipherSuiteParams(
        code=0x00b0,
        iana_name='TLS_PSK_WITH_NULL_SHA256',
        openssl_name='PSK-NULL-SHA256',
        key_exchange=KeyExchange.PSK,
        authentication=Authentication.PSK,
        bulk_cipher=None,
        block_cipher_mode=None,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_PSK_WITH_NULL_SHA384 = CipherSuiteParams(
        code=0x00b1,
        iana_name='TLS_PSK_WITH_NULL_SHA384',
        openssl_name='PSK-NULL-SHA384',
        key_exchange=KeyExchange.PSK,
        authentication=Authentication.PSK,
        bulk_cipher=None,
        block_cipher_mode=None,
        mac=MAC.SHA2_384,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DHE_PSK_WITH_AES_128_CBC_SHA256 = CipherSuiteParams(
        code=0x00b2,
        iana_name='TLS_DHE_PSK_WITH_AES_128_CBC_SHA256',
        openssl_name='DHE-PSK-AES128-CBC-SHA256',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DHE_PSK_WITH_AES_256_CBC_SHA384 = CipherSuiteParams(
        code=0x00b3,
        iana_name='TLS_DHE_PSK_WITH_AES_256_CBC_SHA384',
        openssl_name='DHE-PSK-AES256-CBC-SHA384',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_384,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DHE_PSK_WITH_NULL_SHA256 = CipherSuiteParams(
        code=0x00b4,
        iana_name='TLS_DHE_PSK_WITH_NULL_SHA256',
        openssl_name='DHE-PSK-NULL-SHA256',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.PSK,
        bulk_cipher=None,
        block_cipher_mode=None,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DHE_PSK_WITH_NULL_SHA384 = CipherSuiteParams(
        code=0x00b5,
        iana_name='TLS_DHE_PSK_WITH_NULL_SHA384',
        openssl_name='DHE-PSK-NULL-SHA384',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.PSK,
        bulk_cipher=None,
        block_cipher_mode=None,
        mac=MAC.SHA2_384,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_RSA_PSK_WITH_AES_128_CBC_SHA256 = CipherSuiteParams(
        code=0x00b6,
        iana_name='TLS_RSA_PSK_WITH_AES_128_CBC_SHA256',
        openssl_name='RSA-PSK-AES128-CBC-SHA256',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_RSA_PSK_WITH_AES_256_CBC_SHA384 = CipherSuiteParams(
        code=0x00b7,
        iana_name='TLS_RSA_PSK_WITH_AES_256_CBC_SHA384',
        openssl_name='RSA-PSK-AES256-CBC-SHA384',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_384,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_RSA_PSK_WITH_NULL_SHA256 = CipherSuiteParams(
        code=0x00b8,
        iana_name='TLS_RSA_PSK_WITH_NULL_SHA256',
        openssl_name='RSA-PSK-NULL-SHA256',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.PSK,
        bulk_cipher=None,
        block_cipher_mode=None,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_RSA_PSK_WITH_NULL_SHA384 = CipherSuiteParams(
        code=0x00b9,
        iana_name='TLS_RSA_PSK_WITH_NULL_SHA384',
        openssl_name='RSA-PSK-NULL-SHA384',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.PSK,
        bulk_cipher=None,
        block_cipher_mode=None,
        mac=MAC.SHA2_384,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_RSA_WITH_CAMELLIA_128_CBC_SHA256 = CipherSuiteParams(
        code=0x00ba,
        iana_name='TLS_RSA_WITH_CAMELLIA_128_CBC_SHA256',
        openssl_name='CAMELLIA128-SHA256',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.CAMELLIA_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DH_DSS_WITH_CAMELLIA_128_CBC_SHA256 = CipherSuiteParams(
        code=0x00bb,
        iana_name='TLS_DH_DSS_WITH_CAMELLIA_128_CBC_SHA256',
        openssl_name='DH-DSS-CAMELLIA128-SHA256',
        key_exchange=KeyExchange.DH,
        authentication=Authentication.DSS,
        bulk_cipher=BlockCipher.CAMELLIA_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DH_RSA_WITH_CAMELLIA_128_CBC_SHA256 = CipherSuiteParams(
        code=0x00bc,
        iana_name='TLS_DH_RSA_WITH_CAMELLIA_128_CBC_SHA256',
        openssl_name='DH-RSA-CAMELLIA128-SHA256',
        key_exchange=KeyExchange.DH,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.CAMELLIA_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DHE_DSS_WITH_CAMELLIA_128_CBC_SHA256 = CipherSuiteParams(
        code=0x00bd,
        iana_name='TLS_DHE_DSS_WITH_CAMELLIA_128_CBC_SHA256',
        openssl_name='DHE-DSS-CAMELLIA128-SHA256',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.DSS,
        bulk_cipher=BlockCipher.CAMELLIA_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA256 = CipherSuiteParams(
        code=0x00be,
        iana_name='TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA256',
        openssl_name='DHE-RSA-CAMELLIA128-SHA256',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.CAMELLIA_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DH_anon_WITH_CAMELLIA_128_CBC_SHA256 = CipherSuiteParams(  # pylint: disable=invalid-name
        code=0x00bf,
        iana_name='TLS_DH_anon_WITH_CAMELLIA_128_CBC_SHA256',
        openssl_name='ADH-CAMELLIA128-SHA256',
        key_exchange=KeyExchange.ADH,
        authentication=Authentication.anon,
        bulk_cipher=BlockCipher.CAMELLIA_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_RSA_WITH_CAMELLIA_256_CBC_SHA256 = CipherSuiteParams(
        code=0x00c0,
        iana_name='TLS_RSA_WITH_CAMELLIA_256_CBC_SHA256',
        openssl_name='CAMELLIA256-SHA256',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.CAMELLIA_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DH_DSS_WITH_CAMELLIA_256_CBC_SHA256 = CipherSuiteParams(
        code=0x00c1,
        iana_name='TLS_DH_DSS_WITH_CAMELLIA_256_CBC_SHA256',
        openssl_name='DH-DSS-CAMELLIA256-SHA256',
        key_exchange=KeyExchange.DH,
        authentication=Authentication.DSS,
        bulk_cipher=BlockCipher.CAMELLIA_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DH_RSA_WITH_CAMELLIA_256_CBC_SHA256 = CipherSuiteParams(
        code=0x00c2,
        iana_name='TLS_DH_RSA_WITH_CAMELLIA_256_CBC_SHA256',
        openssl_name='DH-RSA-CAMELLIA256-SHA256',
        key_exchange=KeyExchange.DH,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.CAMELLIA_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DHE_DSS_WITH_CAMELLIA_256_CBC_SHA256 = CipherSuiteParams(
        code=0x00c3,
        iana_name='TLS_DHE_DSS_WITH_CAMELLIA_256_CBC_SHA256',
        openssl_name='DHE-DSS-CAMELLIA256-SHA256',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.DSS,
        bulk_cipher=BlockCipher.CAMELLIA_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA256 = CipherSuiteParams(
        code=0x00c4,
        iana_name='TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA256',
        openssl_name='DHE-RSA-CAMELLIA256-SHA256',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.CAMELLIA_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DH_anon_WITH_CAMELLIA_256_CBC_SHA256 = CipherSuiteParams(  # pylint: disable=invalid-name
        code=0x00c5,
        iana_name='TLS_DH_anon_WITH_CAMELLIA_256_CBC_SHA256',
        openssl_name='ADH-CAMELLIA256-SHA256',
        key_exchange=KeyExchange.ADH,
        authentication=Authentication.anon,
        bulk_cipher=BlockCipher.CAMELLIA_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_CECPQ1_RSA_WITH_CHACHA20_POLY1305_SHA256 = CipherSuiteParams(
        code=0x16b7,
        iana_name='TLS_CECPQ1_RSA_WITH_CHACHA20_POLY1305_SHA256',
        openssl_name=None,
        key_exchange=KeyExchange.CECPQ1,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.CHACHA20,
        block_cipher_mode=None,
        mac=MAC.POLY1305,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_CECPQ1_ECDSA_WITH_CHACHA20_POLY1305_SHA256 = CipherSuiteParams(
        code=0x16b8,
        iana_name='TLS_CECPQ1_ECDSA_WITH_CHACHA20_POLY1305_SHA256',
        openssl_name=None,
        key_exchange=KeyExchange.CECPQ1,
        authentication=Authentication.ECDSA,
        bulk_cipher=BlockCipher.CHACHA20,
        block_cipher_mode=None,
        mac=MAC.POLY1305,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_CECPQ1_RSA_WITH_AES_256_GCM_SHA384 = CipherSuiteParams(
        code=0x16b9,
        iana_name='TLS_CECPQ1_RSA_WITH_AES_256_GCM_SHA384',
        openssl_name=None,
        key_exchange=KeyExchange.CECPQ1,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_384,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_CECPQ1_ECDSA_WITH_AES_256_GCM_SHA384 = CipherSuiteParams(
        code=0x16ba,
        iana_name='TLS_CECPQ1_ECDSA_WITH_AES_256_GCM_SHA384',
        openssl_name=None,
        key_exchange=KeyExchange.CECPQ1,
        authentication=Authentication.ECDSA,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_384,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_ECDH_ECDSA_WITH_NULL_SHA = CipherSuiteParams(
        code=0xc001,
        iana_name='TLS_ECDH_ECDSA_WITH_NULL_SHA',
        openssl_name='ECDH-ECDSA-NULL-SHA',
        key_exchange=KeyExchange.ECDH,
        authentication=Authentication.ECDSA,
        bulk_cipher=None,
        block_cipher_mode=None,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_ECDH_ECDSA_WITH_RC4_128_SHA = CipherSuiteParams(
        code=0xc002,
        iana_name='TLS_ECDH_ECDSA_WITH_RC4_128_SHA',
        openssl_name='ECDH-ECDSA-RC4-SHA',
        key_exchange=KeyExchange.ECDH,
        authentication=Authentication.ECDSA,
        bulk_cipher=BlockCipher.RC4_128,
        block_cipher_mode=None,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_ECDH_ECDSA_WITH_3DES_EDE_CBC_SHA = CipherSuiteParams(
        code=0xc003,
        iana_name='TLS_ECDH_ECDSA_WITH_3DES_EDE_CBC_SHA',
        openssl_name='ECDH-ECDSA-DES-CBC3-SHA',
        key_exchange=KeyExchange.ECDH,
        authentication=Authentication.ECDSA,
        bulk_cipher=BlockCipher.TRIPLE_DES_EDE,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA = CipherSuiteParams(
        code=0xc004,
        iana_name='TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA',
        openssl_name='ECDH-ECDSA-AES128-SHA',
        key_exchange=KeyExchange.ECDH,
        authentication=Authentication.ECDSA,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA = CipherSuiteParams(
        code=0xc005,
        iana_name='TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA',
        openssl_name='ECDH-ECDSA-AES256-SHA',
        key_exchange=KeyExchange.ECDH,
        authentication=Authentication.ECDSA,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_ECDHE_ECDSA_WITH_NULL_SHA = CipherSuiteParams(
        code=0xc006,
        iana_name='TLS_ECDHE_ECDSA_WITH_NULL_SHA',
        openssl_name='ECDHE-ECDSA-NULL-SHA',
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.ECDSA,
        bulk_cipher=None,
        block_cipher_mode=None,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_ECDHE_ECDSA_WITH_RC4_128_SHA = CipherSuiteParams(
        code=0xc007,
        iana_name='TLS_ECDHE_ECDSA_WITH_RC4_128_SHA',
        openssl_name='ECDHE-ECDSA-RC4-SHA',
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.ECDSA,
        bulk_cipher=BlockCipher.RC4_128,
        block_cipher_mode=None,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_ECDHE_ECDSA_WITH_3DES_EDE_CBC_SHA = CipherSuiteParams(
        code=0xc008,
        iana_name='TLS_ECDHE_ECDSA_WITH_3DES_EDE_CBC_SHA',
        openssl_name='ECDHE-ECDSA-DES-CBC3-SHA',
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.ECDSA,
        bulk_cipher=BlockCipher.TRIPLE_DES_EDE,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA = CipherSuiteParams(
        code=0xc009,
        iana_name='TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA',
        openssl_name='ECDHE-ECDSA-AES128-SHA',
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.ECDSA,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA = CipherSuiteParams(
        code=0xc00a,
        iana_name='TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA',
        openssl_name='ECDHE-ECDSA-AES256-SHA',
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.ECDSA,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_ECDH_RSA_WITH_NULL_SHA = CipherSuiteParams(
        code=0xc00b,
        iana_name='TLS_ECDH_RSA_WITH_NULL_SHA',
        openssl_name='ECDH-RSA-NULL-SHA',
        key_exchange=KeyExchange.ECDH,
        authentication=Authentication.RSA,
        bulk_cipher=None,
        block_cipher_mode=None,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_ECDH_RSA_WITH_RC4_128_SHA = CipherSuiteParams(
        code=0xc00c,
        iana_name='TLS_ECDH_RSA_WITH_RC4_128_SHA',
        openssl_name='ECDH-RSA-RC4-SHA',
        key_exchange=KeyExchange.ECDH,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.RC4_128,
        block_cipher_mode=None,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_ECDH_RSA_WITH_3DES_EDE_CBC_SHA = CipherSuiteParams(
        code=0xc00d,
        iana_name='TLS_ECDH_RSA_WITH_3DES_EDE_CBC_SHA',
        openssl_name='ECDH-RSA-DES-CBC3-SHA',
        key_exchange=KeyExchange.ECDH,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.TRIPLE_DES_EDE,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_ECDH_RSA_WITH_AES_128_CBC_SHA = CipherSuiteParams(
        code=0xc00e,
        iana_name='TLS_ECDH_RSA_WITH_AES_128_CBC_SHA',
        openssl_name='ECDH-RSA-AES128-SHA',
        key_exchange=KeyExchange.ECDH,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_ECDH_RSA_WITH_AES_256_CBC_SHA = CipherSuiteParams(
        code=0xc00f,
        iana_name='TLS_ECDH_RSA_WITH_AES_256_CBC_SHA',
        openssl_name='ECDH-RSA-AES256-SHA',
        key_exchange=KeyExchange.ECDH,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_ECDHE_RSA_WITH_NULL_SHA = CipherSuiteParams(
        code=0xc010,
        iana_name='TLS_ECDHE_RSA_WITH_NULL_SHA',
        openssl_name='ECDHE-RSA-NULL-SHA',
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.RSA,
        bulk_cipher=None,
        block_cipher_mode=None,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_ECDHE_RSA_WITH_RC4_128_SHA = CipherSuiteParams(
        code=0xc011,
        iana_name='TLS_ECDHE_RSA_WITH_RC4_128_SHA',
        openssl_name='ECDHE-RSA-RC4-SHA',
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.RC4_128,
        block_cipher_mode=None,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_ECDHE_RSA_WITH_3DES_EDE_CBC_SHA = CipherSuiteParams(
        code=0xc012,
        iana_name='TLS_ECDHE_RSA_WITH_3DES_EDE_CBC_SHA',
        openssl_name='ECDHE-RSA-DES-CBC3-SHA',
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.TRIPLE_DES_EDE,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA = CipherSuiteParams(
        code=0xc013,
        iana_name='TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA',
        openssl_name='ECDHE-RSA-AES128-SHA',
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA = CipherSuiteParams(
        code=0xc014,
        iana_name='TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA',
        openssl_name='ECDHE-RSA-AES256-SHA',
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_ECDH_anon_WITH_NULL_SHA = CipherSuiteParams(  # pylint: disable=invalid-name
        code=0xc015,
        iana_name='TLS_ECDH_anon_WITH_NULL_SHA',
        openssl_name='AECDH-NULL-SHA',
        key_exchange=KeyExchange.ECDH,
        authentication=Authentication.anon,
        bulk_cipher=None,
        block_cipher_mode=None,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_ECDH_anon_WITH_RC4_128_SHA = CipherSuiteParams(  # pylint: disable=invalid-name
        code=0xc016,
        iana_name='TLS_ECDH_anon_WITH_RC4_128_SHA',
        openssl_name='AECDH-RC4-SHA',
        key_exchange=KeyExchange.ECDH,
        authentication=Authentication.anon,
        bulk_cipher=BlockCipher.RC4_128,
        block_cipher_mode=None,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_ECDH_anon_WITH_3DES_EDE_CBC_SHA = CipherSuiteParams(  # pylint: disable=invalid-name
        code=0xc017,
        iana_name='TLS_ECDH_anon_WITH_3DES_EDE_CBC_SHA',
        openssl_name='AECDH-DES-CBC3-SHA',
        key_exchange=KeyExchange.ECDH,
        authentication=Authentication.anon,
        bulk_cipher=BlockCipher.TRIPLE_DES_EDE,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_ECDH_anon_WITH_AES_128_CBC_SHA = CipherSuiteParams(  # pylint: disable=invalid-name
        code=0xc018,
        iana_name='TLS_ECDH_anon_WITH_AES_128_CBC_SHA',
        openssl_name='AECDH-AES128-SHA',
        key_exchange=KeyExchange.ECDH,
        authentication=Authentication.anon,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_ECDH_anon_WITH_AES_256_CBC_SHA = CipherSuiteParams(  # pylint: disable=invalid-name
        code=0xc019,
        iana_name='TLS_ECDH_anon_WITH_AES_256_CBC_SHA',
        openssl_name='AECDH-AES256-SHA',
        key_exchange=KeyExchange.ECDH,
        authentication=Authentication.anon,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_SRP_SHA_WITH_3DES_EDE_CBC_SHA = CipherSuiteParams(
        code=0xc01a,
        iana_name='TLS_SRP_SHA_WITH_3DES_EDE_CBC_SHA',
        openssl_name='SRP-3DES-EDE-CBC-SHA',
        key_exchange=KeyExchange.SRP,
        authentication=Authentication.SRP,
        bulk_cipher=BlockCipher.TRIPLE_DES_EDE,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_SRP_SHA_RSA_WITH_3DES_EDE_CBC_SHA = CipherSuiteParams(
        code=0xc01b,
        iana_name='TLS_SRP_SHA_RSA_WITH_3DES_EDE_CBC_SHA',
        openssl_name='SRP-RSA-3DES-EDE-CBC-SHA',
        key_exchange=KeyExchange.SRP,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.TRIPLE_DES_EDE,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_SRP_SHA_DSS_WITH_3DES_EDE_CBC_SHA = CipherSuiteParams(
        code=0xc01c,
        iana_name='TLS_SRP_SHA_DSS_WITH_3DES_EDE_CBC_SHA',
        openssl_name='SRP-DSS-3DES-EDE-CBC-SHA',
        key_exchange=KeyExchange.SRP,
        authentication=Authentication.DSS,
        bulk_cipher=BlockCipher.TRIPLE_DES_EDE,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_SRP_SHA_WITH_AES_128_CBC_SHA = CipherSuiteParams(
        code=0xc01d,
        iana_name='TLS_SRP_SHA_WITH_AES_128_CBC_SHA',
        openssl_name='SRP-AES-128-CBC-SHA',
        key_exchange=KeyExchange.SRP,
        authentication=Authentication.SRP,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_SRP_SHA_RSA_WITH_AES_128_CBC_SHA = CipherSuiteParams(
        code=0xc01e,
        iana_name='TLS_SRP_SHA_RSA_WITH_AES_128_CBC_SHA',
        openssl_name='SRP-RSA-AES-128-CBC-SHA',
        key_exchange=KeyExchange.SRP,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_SRP_SHA_DSS_WITH_AES_128_CBC_SHA = CipherSuiteParams(
        code=0xc01f,
        iana_name='TLS_SRP_SHA_DSS_WITH_AES_128_CBC_SHA',
        openssl_name='SRP-DSS-AES-128-CBC-SHA',
        key_exchange=KeyExchange.SRP,
        authentication=Authentication.DSS,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_SRP_SHA_WITH_AES_256_CBC_SHA = CipherSuiteParams(
        code=0xc020,
        iana_name='TLS_SRP_SHA_WITH_AES_256_CBC_SHA',
        openssl_name='SRP-AES-256-CBC-SHA',
        key_exchange=KeyExchange.SRP,
        authentication=Authentication.SRP,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_SRP_SHA_RSA_WITH_AES_256_CBC_SHA = CipherSuiteParams(
        code=0xc021,
        iana_name='TLS_SRP_SHA_RSA_WITH_AES_256_CBC_SHA',
        openssl_name='SRP-RSA-AES-256-CBC-SHA',
        key_exchange=KeyExchange.SRP,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_SRP_SHA_DSS_WITH_AES_256_CBC_SHA = CipherSuiteParams(
        code=0xc022,
        iana_name='TLS_SRP_SHA_DSS_WITH_AES_256_CBC_SHA',
        openssl_name='SRP-DSS-AES-256-CBC-SHA',
        key_exchange=KeyExchange.SRP,
        authentication=Authentication.DSS,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA256 = CipherSuiteParams(
        code=0xc023,
        iana_name='TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA256',
        openssl_name='ECDHE-ECDSA-AES128-SHA256',
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.ECDSA,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA384 = CipherSuiteParams(
        code=0xc024,
        iana_name='TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA384',
        openssl_name='ECDHE-ECDSA-AES256-SHA384',
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.ECDSA,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_384,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA256 = CipherSuiteParams(
        code=0xc025,
        iana_name='TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA256',
        openssl_name='ECDH-ECDSA-AES128-SHA256',
        key_exchange=KeyExchange.ECDH,
        authentication=Authentication.ECDSA,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA384 = CipherSuiteParams(
        code=0xc026,
        iana_name='TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA384',
        openssl_name='ECDH-ECDSA-AES256-SHA384',
        key_exchange=KeyExchange.ECDH,
        authentication=Authentication.ECDSA,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_384,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA256 = CipherSuiteParams(
        code=0xc027,
        iana_name='TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA256',
        openssl_name='ECDHE-RSA-AES128-SHA256',
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA384 = CipherSuiteParams(
        code=0xc028,
        iana_name='TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA384',
        openssl_name='ECDHE-RSA-AES256-SHA384',
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_384,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_ECDH_RSA_WITH_AES_128_CBC_SHA256 = CipherSuiteParams(
        code=0xc029,
        iana_name='TLS_ECDH_RSA_WITH_AES_128_CBC_SHA256',
        openssl_name='ECDH-RSA-AES128-SHA256',
        key_exchange=KeyExchange.ECDH,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_ECDH_RSA_WITH_AES_256_CBC_SHA384 = CipherSuiteParams(
        code=0xc02a,
        iana_name='TLS_ECDH_RSA_WITH_AES_256_CBC_SHA384',
        openssl_name='ECDH-RSA-AES256-SHA384',
        key_exchange=KeyExchange.ECDH,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_384,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_ECDHE_ECDSA_WITH_AES_128_GCM_SHA256 = CipherSuiteParams(
        code=0xc02b,
        iana_name='TLS_ECDHE_ECDSA_WITH_AES_128_GCM_SHA256',
        openssl_name='ECDHE-ECDSA-AES128-GCM-SHA256',
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.ECDSA,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_256,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_ECDHE_ECDSA_WITH_AES_256_GCM_SHA384 = CipherSuiteParams(
        code=0xc02c,
        iana_name='TLS_ECDHE_ECDSA_WITH_AES_256_GCM_SHA384',
        openssl_name='ECDHE-ECDSA-AES256-GCM-SHA384',
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.ECDSA,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_384,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_ECDH_ECDSA_WITH_AES_128_GCM_SHA256 = CipherSuiteParams(
        code=0xc02d,
        iana_name='TLS_ECDH_ECDSA_WITH_AES_128_GCM_SHA256',
        openssl_name='ECDH-ECDSA-AES128-GCM-SHA256',
        key_exchange=KeyExchange.ECDH,
        authentication=Authentication.ECDSA,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_256,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_ECDH_ECDSA_WITH_AES_256_GCM_SHA384 = CipherSuiteParams(
        code=0xc02e,
        iana_name='TLS_ECDH_ECDSA_WITH_AES_256_GCM_SHA384',
        openssl_name='ECDH-ECDSA-AES256-GCM-SHA384',
        key_exchange=KeyExchange.ECDH,
        authentication=Authentication.ECDSA,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_384,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256 = CipherSuiteParams(
        code=0xc02f,
        iana_name='TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256',
        openssl_name='ECDHE-RSA-AES128-GCM-SHA256',
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_256,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384 = CipherSuiteParams(
        code=0xc030,
        iana_name='TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384',
        openssl_name='ECDHE-RSA-AES256-GCM-SHA384',
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_384,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_ECDH_RSA_WITH_AES_128_GCM_SHA256 = CipherSuiteParams(
        code=0xc031,
        iana_name='TLS_ECDH_RSA_WITH_AES_128_GCM_SHA256',
        openssl_name='ECDH-RSA-AES128-GCM-SHA256',
        key_exchange=KeyExchange.ECDH,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_256,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_ECDH_RSA_WITH_AES_256_GCM_SHA384 = CipherSuiteParams(
        code=0xc032,
        iana_name='TLS_ECDH_RSA_WITH_AES_256_GCM_SHA384',
        openssl_name='ECDH-RSA-AES256-GCM-SHA384',
        key_exchange=KeyExchange.ECDH,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_384,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_ECDHE_PSK_WITH_RC4_128_SHA = CipherSuiteParams(
        code=0xc033,
        iana_name='TLS_ECDHE_PSK_WITH_RC4_128_SHA',
        openssl_name='ECDHE-PSK-RC4-SHA',
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.RC4_128,
        block_cipher_mode=None,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_ECDHE_PSK_WITH_3DES_EDE_CBC_SHA = CipherSuiteParams(
        code=0xc034,
        iana_name='TLS_ECDHE_PSK_WITH_3DES_EDE_CBC_SHA',
        openssl_name='ECDHE-PSK-3DES-EDE-CBC-SHA',
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.TRIPLE_DES_EDE,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_ECDHE_PSK_WITH_AES_128_CBC_SHA = CipherSuiteParams(
        code=0xc035,
        iana_name='TLS_ECDHE_PSK_WITH_AES_128_CBC_SHA',
        openssl_name='ECDHE-PSK-AES128-CBC-SHA',
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_ECDHE_PSK_WITH_AES_256_CBC_SHA = CipherSuiteParams(
        code=0xc036,
        iana_name='TLS_ECDHE_PSK_WITH_AES_256_CBC_SHA',
        openssl_name='ECDHE-PSK-AES256-CBC-SHA',
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_ECDHE_PSK_WITH_AES_128_CBC_SHA256 = CipherSuiteParams(
        code=0xc037,
        iana_name='TLS_ECDHE_PSK_WITH_AES_128_CBC_SHA256',
        openssl_name='ECDHE-PSK-AES128-CBC-SHA256',
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_ECDHE_PSK_WITH_AES_256_CBC_SHA384 = CipherSuiteParams(
        code=0xc038,
        iana_name='TLS_ECDHE_PSK_WITH_AES_256_CBC_SHA384',
        openssl_name='ECDHE-PSK-AES256-CBC-SHA384',
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_384,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_ECDHE_PSK_WITH_NULL_SHA = CipherSuiteParams(
        code=0xc039,
        iana_name='TLS_ECDHE_PSK_WITH_NULL_SHA',
        openssl_name='ECDHE-PSK-NULL-SHA',
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.PSK,
        bulk_cipher=None,
        block_cipher_mode=None,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_ECDHE_PSK_WITH_NULL_SHA256 = CipherSuiteParams(
        code=0xc03a,
        iana_name='TLS_ECDHE_PSK_WITH_NULL_SHA256',
        openssl_name='ECDHE-PSK-NULL-SHA256',
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.PSK,
        bulk_cipher=None,
        block_cipher_mode=None,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_ECDHE_PSK_WITH_NULL_SHA384 = CipherSuiteParams(
        code=0xc03b,
        iana_name='TLS_ECDHE_PSK_WITH_NULL_SHA384',
        openssl_name='ECDHE-PSK-NULL-SHA384',
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.PSK,
        bulk_cipher=None,
        block_cipher_mode=None,
        mac=MAC.SHA2_384,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_RSA_WITH_ARIA_128_CBC_SHA256 = CipherSuiteParams(
        code=0xc03c,
        iana_name='TLS_RSA_WITH_ARIA_128_CBC_SHA256',
        openssl_name=None,
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.ARIA_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_RSA_WITH_ARIA_256_CBC_SHA384 = CipherSuiteParams(
        code=0xc03d,
        iana_name='TLS_RSA_WITH_ARIA_256_CBC_SHA384',
        openssl_name=None,
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.ARIA_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_384,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DH_DSS_WITH_ARIA_128_CBC_SHA256 = CipherSuiteParams(
        code=0xc03e,
        iana_name='TLS_DH_DSS_WITH_ARIA_128_CBC_SHA256',
        openssl_name=None,
        key_exchange=KeyExchange.DH,
        authentication=Authentication.DSS,
        bulk_cipher=BlockCipher.ARIA_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DH_DSS_WITH_ARIA_256_CBC_SHA384 = CipherSuiteParams(
        code=0xc03f,
        iana_name='TLS_DH_DSS_WITH_ARIA_256_CBC_SHA384',
        openssl_name=None,
        key_exchange=KeyExchange.DH,
        authentication=Authentication.DSS,
        bulk_cipher=BlockCipher.ARIA_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_384,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DH_RSA_WITH_ARIA_128_CBC_SHA256 = CipherSuiteParams(
        code=0xc040,
        iana_name='TLS_DH_RSA_WITH_ARIA_128_CBC_SHA256',
        openssl_name=None,
        key_exchange=KeyExchange.DH,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.ARIA_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DH_RSA_WITH_ARIA_256_CBC_SHA384 = CipherSuiteParams(
        code=0xc041,
        iana_name='TLS_DH_RSA_WITH_ARIA_256_CBC_SHA384',
        openssl_name=None,
        key_exchange=KeyExchange.DH,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.ARIA_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_384,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DHE_DSS_WITH_ARIA_128_CBC_SHA256 = CipherSuiteParams(
        code=0xc042,
        iana_name='TLS_DHE_DSS_WITH_ARIA_128_CBC_SHA256',
        openssl_name=None,
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.DSS,
        bulk_cipher=BlockCipher.ARIA_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DHE_DSS_WITH_ARIA_256_CBC_SHA384 = CipherSuiteParams(
        code=0xc043,
        iana_name='TLS_DHE_DSS_WITH_ARIA_256_CBC_SHA384',
        openssl_name=None,
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.DSS,
        bulk_cipher=BlockCipher.ARIA_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_384,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DHE_RSA_WITH_ARIA_128_CBC_SHA256 = CipherSuiteParams(
        code=0xc044,
        iana_name='TLS_DHE_RSA_WITH_ARIA_128_CBC_SHA256',
        openssl_name=None,
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.ARIA_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DHE_RSA_WITH_ARIA_256_CBC_SHA384 = CipherSuiteParams(
        code=0xc045,
        iana_name='TLS_DHE_RSA_WITH_ARIA_256_CBC_SHA384',
        openssl_name=None,
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.ARIA_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_384,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DH_anon_WITH_ARIA_128_CBC_SHA256 = CipherSuiteParams(  # pylint: disable=invalid-name
        code=0xc046,
        iana_name='TLS_DH_anon_WITH_ARIA_128_CBC_SHA256',
        openssl_name=None,
        key_exchange=KeyExchange.ADH,
        authentication=Authentication.anon,
        bulk_cipher=BlockCipher.ARIA_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DH_anon_WITH_ARIA_256_CBC_SHA384 = CipherSuiteParams(  # pylint: disable=invalid-name
        code=0xc047,
        iana_name='TLS_DH_anon_WITH_ARIA_256_CBC_SHA384',
        openssl_name=None,
        key_exchange=KeyExchange.ADH,
        authentication=Authentication.anon,
        bulk_cipher=BlockCipher.ARIA_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_384,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_ECDHE_ECDSA_WITH_ARIA_128_CBC_SHA256 = CipherSuiteParams(
        code=0xc048,
        iana_name='TLS_ECDHE_ECDSA_WITH_ARIA_128_CBC_SHA256',
        openssl_name=None,
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.ECDSA,
        bulk_cipher=BlockCipher.ARIA_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_ECDHE_ECDSA_WITH_ARIA_256_CBC_SHA384 = CipherSuiteParams(
        code=0xc049,
        iana_name='TLS_ECDHE_ECDSA_WITH_ARIA_256_CBC_SHA384',
        openssl_name=None,
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.ECDSA,
        bulk_cipher=BlockCipher.ARIA_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_384,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_ECDH_ECDSA_WITH_ARIA_128_CBC_SHA256 = CipherSuiteParams(
        code=0xc04a,
        iana_name='TLS_ECDH_ECDSA_WITH_ARIA_128_CBC_SHA256',
        openssl_name=None,
        key_exchange=KeyExchange.ECDH,
        authentication=Authentication.ECDSA,
        bulk_cipher=BlockCipher.ARIA_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_ECDH_ECDSA_WITH_ARIA_256_CBC_SHA384 = CipherSuiteParams(
        code=0xc04b,
        iana_name='TLS_ECDH_ECDSA_WITH_ARIA_256_CBC_SHA384',
        openssl_name=None,
        key_exchange=KeyExchange.ECDH,
        authentication=Authentication.ECDSA,
        bulk_cipher=BlockCipher.ARIA_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_384,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_ECDHE_RSA_WITH_ARIA_128_CBC_SHA256 = CipherSuiteParams(
        code=0xc04c,
        iana_name='TLS_ECDHE_RSA_WITH_ARIA_128_CBC_SHA256',
        openssl_name=None,
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.ARIA_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_ECDHE_RSA_WITH_ARIA_256_CBC_SHA384 = CipherSuiteParams(
        code=0xc04d,
        iana_name='TLS_ECDHE_RSA_WITH_ARIA_256_CBC_SHA384',
        openssl_name=None,
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.ARIA_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_384,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_ECDH_RSA_WITH_ARIA_128_CBC_SHA256 = CipherSuiteParams(
        code=0xc04e,
        iana_name='TLS_ECDH_RSA_WITH_ARIA_128_CBC_SHA256',
        openssl_name=None,
        key_exchange=KeyExchange.ECDH,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.ARIA_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_ECDH_RSA_WITH_ARIA_256_CBC_SHA384 = CipherSuiteParams(
        code=0xc04f,
        iana_name='TLS_ECDH_RSA_WITH_ARIA_256_CBC_SHA384',
        openssl_name=None,
        key_exchange=KeyExchange.ECDH,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.ARIA_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_384,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_RSA_WITH_ARIA_128_GCM_SHA256 = CipherSuiteParams(
        code=0xc050,
        iana_name='TLS_RSA_WITH_ARIA_128_GCM_SHA256',
        openssl_name='ARIA128-GCM-SHA256',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.ARIA_128,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_256,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_RSA_WITH_ARIA_256_GCM_SHA384 = CipherSuiteParams(
        code=0xc051,
        iana_name='TLS_RSA_WITH_ARIA_256_GCM_SHA384',
        openssl_name='ARIA256-GCM-SHA384',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.ARIA_256,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_384,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DHE_RSA_WITH_ARIA_128_GCM_SHA256 = CipherSuiteParams(
        code=0xc052,
        iana_name='TLS_DHE_RSA_WITH_ARIA_128_GCM_SHA256',
        openssl_name='DHE-RSA-ARIA128-GCM-SHA256',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.ARIA_128,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_256,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DHE_RSA_WITH_ARIA_256_GCM_SHA384 = CipherSuiteParams(
        code=0xc053,
        iana_name='TLS_DHE_RSA_WITH_ARIA_256_GCM_SHA384',
        openssl_name='DHE-RSA-ARIA256-GCM-SHA384',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.ARIA_256,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_384,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DH_RSA_WITH_ARIA_128_GCM_SHA256 = CipherSuiteParams(
        code=0xc054,
        iana_name='TLS_DH_RSA_WITH_ARIA_128_GCM_SHA256',
        openssl_name=None,
        key_exchange=KeyExchange.DH,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.ARIA_128,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_256,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DH_RSA_WITH_ARIA_256_GCM_SHA384 = CipherSuiteParams(
        code=0xc055,
        iana_name='TLS_DH_RSA_WITH_ARIA_256_GCM_SHA384',
        openssl_name=None,
        key_exchange=KeyExchange.DH,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.ARIA_256,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_384,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DHE_DSS_WITH_ARIA_128_GCM_SHA256 = CipherSuiteParams(
        code=0xc056,
        iana_name='TLS_DHE_DSS_WITH_ARIA_128_GCM_SHA256',
        openssl_name='DHE-DSS-ARIA128-GCM-SHA256',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.DSS,
        bulk_cipher=BlockCipher.ARIA_128,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_256,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DHE_DSS_WITH_ARIA_256_GCM_SHA384 = CipherSuiteParams(
        code=0xc057,
        iana_name='TLS_DHE_DSS_WITH_ARIA_256_GCM_SHA384',
        openssl_name='DHE-DSS-ARIA256-GCM-SHA384',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.DSS,
        bulk_cipher=BlockCipher.ARIA_256,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_384,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DH_DSS_WITH_ARIA_128_GCM_SHA256 = CipherSuiteParams(
        code=0xc058,
        iana_name='TLS_DH_DSS_WITH_ARIA_128_GCM_SHA256',
        openssl_name=None,
        key_exchange=KeyExchange.DH,
        authentication=Authentication.DSS,
        bulk_cipher=BlockCipher.ARIA_128,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_256,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DH_DSS_WITH_ARIA_256_GCM_SHA384 = CipherSuiteParams(
        code=0xc059,
        iana_name='TLS_DH_DSS_WITH_ARIA_256_GCM_SHA384',
        openssl_name=None,
        key_exchange=KeyExchange.DH,
        authentication=Authentication.DSS,
        bulk_cipher=BlockCipher.ARIA_256,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_384,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DH_anon_WITH_ARIA_128_GCM_SHA256 = CipherSuiteParams(  # pylint: disable=invalid-name
        code=0xc05a,
        iana_name='TLS_DH_anon_WITH_ARIA_128_GCM_SHA256',
        openssl_name=None,
        key_exchange=KeyExchange.ADH,
        authentication=Authentication.anon,
        bulk_cipher=BlockCipher.ARIA_128,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_256,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DH_anon_WITH_ARIA_256_GCM_SHA384 = CipherSuiteParams(  # pylint: disable=invalid-name
        code=0xc05b,
        iana_name='TLS_DH_anon_WITH_ARIA_256_GCM_SHA384',
        openssl_name=None,
        key_exchange=KeyExchange.ADH,
        authentication=Authentication.anon,
        bulk_cipher=BlockCipher.ARIA_256,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_384,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_ECDHE_ECDSA_WITH_ARIA_128_GCM_SHA256 = CipherSuiteParams(
        code=0xc05c,
        iana_name='TLS_ECDHE_ECDSA_WITH_ARIA_128_GCM_SHA256',
        openssl_name='ECDHE-ECDSA-ARIA128-GCM-SHA256',
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.ECDSA,
        bulk_cipher=BlockCipher.ARIA_128,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_256,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_ECDHE_ECDSA_WITH_ARIA_256_GCM_SHA384 = CipherSuiteParams(
        code=0xc05d,
        iana_name='TLS_ECDHE_ECDSA_WITH_ARIA_256_GCM_SHA384',
        openssl_name='ECDHE-ECDSA-ARIA256-GCM-SHA384',
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.ECDSA,
        bulk_cipher=BlockCipher.ARIA_256,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_384,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_ECDH_ECDSA_WITH_ARIA_128_GCM_SHA256 = CipherSuiteParams(
        code=0xc05e,
        iana_name='TLS_ECDH_ECDSA_WITH_ARIA_128_GCM_SHA256',
        openssl_name=None,
        key_exchange=KeyExchange.ECDH,
        authentication=Authentication.ECDSA,
        bulk_cipher=BlockCipher.ARIA_128,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_256,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_ECDH_ECDSA_WITH_ARIA_256_GCM_SHA384 = CipherSuiteParams(
        code=0xc05f,
        iana_name='TLS_ECDH_ECDSA_WITH_ARIA_256_GCM_SHA384',
        openssl_name=None,
        key_exchange=KeyExchange.ECDH,
        authentication=Authentication.ECDSA,
        bulk_cipher=BlockCipher.ARIA_256,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_384,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_ECDHE_RSA_WITH_ARIA_128_GCM_SHA256 = CipherSuiteParams(
        code=0xc060,
        iana_name='TLS_ECDHE_RSA_WITH_ARIA_128_GCM_SHA256',
        openssl_name='ECDHE-ARIA128-GCM-SHA256',
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.ARIA_128,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_256,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_ECDHE_RSA_WITH_ARIA_256_GCM_SHA384 = CipherSuiteParams(
        code=0xc061,
        iana_name='TLS_ECDHE_RSA_WITH_ARIA_256_GCM_SHA384',
        openssl_name='ECDHE-ARIA256-GCM-SHA384',
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.ARIA_256,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_384,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_ECDH_RSA_WITH_ARIA_128_GCM_SHA256 = CipherSuiteParams(
        code=0xc062,
        iana_name='TLS_ECDH_RSA_WITH_ARIA_128_GCM_SHA256',
        openssl_name=None,
        key_exchange=KeyExchange.ECDH,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.ARIA_128,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_256,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_ECDH_RSA_WITH_ARIA_256_GCM_SHA384 = CipherSuiteParams(
        code=0xc063,
        iana_name='TLS_ECDH_RSA_WITH_ARIA_256_GCM_SHA384',
        openssl_name=None,
        key_exchange=KeyExchange.ECDH,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.ARIA_256,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_384,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_PSK_WITH_ARIA_128_CBC_SHA256 = CipherSuiteParams(
        code=0xc064,
        iana_name='TLS_PSK_WITH_ARIA_128_CBC_SHA256',
        openssl_name=None,
        key_exchange=KeyExchange.PSK,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.ARIA_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_PSK_WITH_ARIA_256_CBC_SHA384 = CipherSuiteParams(
        code=0xc065,
        iana_name='TLS_PSK_WITH_ARIA_256_CBC_SHA384',
        openssl_name=None,
        key_exchange=KeyExchange.PSK,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.ARIA_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_384,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DHE_PSK_WITH_ARIA_128_CBC_SHA256 = CipherSuiteParams(
        code=0xc066,
        iana_name='TLS_DHE_PSK_WITH_ARIA_128_CBC_SHA256',
        openssl_name=None,
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.ARIA_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DHE_PSK_WITH_ARIA_256_CBC_SHA384 = CipherSuiteParams(
        code=0xc067,
        iana_name='TLS_DHE_PSK_WITH_ARIA_256_CBC_SHA384',
        openssl_name=None,
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.ARIA_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_384,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_RSA_PSK_WITH_ARIA_128_CBC_SHA256 = CipherSuiteParams(
        code=0xc068,
        iana_name='TLS_RSA_PSK_WITH_ARIA_128_CBC_SHA256',
        openssl_name=None,
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.ARIA_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_RSA_PSK_WITH_ARIA_256_CBC_SHA384 = CipherSuiteParams(
        code=0xc069,
        iana_name='TLS_RSA_PSK_WITH_ARIA_256_CBC_SHA384',
        openssl_name=None,
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.ARIA_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_384,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_PSK_WITH_ARIA_128_GCM_SHA256 = CipherSuiteParams(
        code=0xc06a,
        iana_name='TLS_PSK_WITH_ARIA_128_GCM_SHA256',
        openssl_name='PSK-ARIA128-GCM-SHA256',
        key_exchange=KeyExchange.PSK,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.ARIA_128,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_256,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_PSK_WITH_ARIA_256_GCM_SHA384 = CipherSuiteParams(
        code=0xc06b,
        iana_name='TLS_PSK_WITH_ARIA_256_GCM_SHA384',
        openssl_name='PSK-ARIA256-GCM-SHA384',
        key_exchange=KeyExchange.PSK,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.ARIA_256,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_384,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DHE_PSK_WITH_ARIA_128_GCM_SHA256 = CipherSuiteParams(
        code=0xc06c,
        iana_name='TLS_DHE_PSK_WITH_ARIA_128_GCM_SHA256',
        openssl_name='DHE-PSK-ARIA128-GCM-SHA256',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.ARIA_128,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_256,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DHE_PSK_WITH_ARIA_256_GCM_SHA384 = CipherSuiteParams(
        code=0xc06d,
        iana_name='TLS_DHE_PSK_WITH_ARIA_256_GCM_SHA384',
        openssl_name='DHE-PSK-ARIA256-GCM-SHA384',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.ARIA_256,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_384,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_RSA_PSK_WITH_ARIA_128_GCM_SHA256 = CipherSuiteParams(
        code=0xc06e,
        iana_name='TLS_RSA_PSK_WITH_ARIA_128_GCM_SHA256',
        openssl_name='RSA-PSK-ARIA128-GCM-SHA256',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.ARIA_128,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_256,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_RSA_PSK_WITH_ARIA_256_GCM_SHA384 = CipherSuiteParams(
        code=0xc06f,
        iana_name='TLS_RSA_PSK_WITH_ARIA_256_GCM_SHA384',
        openssl_name='RSA-PSK-ARIA256-GCM-SHA384',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.ARIA_256,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_384,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_ECDHE_PSK_WITH_ARIA_128_CBC_SHA256 = CipherSuiteParams(
        code=0xc070,
        iana_name='TLS_ECDHE_PSK_WITH_ARIA_128_CBC_SHA256',
        openssl_name=None,
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.ARIA_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_256,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_ECDHE_PSK_WITH_ARIA_256_CBC_SHA384 = CipherSuiteParams(
        code=0xc071,
        iana_name='TLS_ECDHE_PSK_WITH_ARIA_256_CBC_SHA384',
        openssl_name=None,
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.ARIA_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_384,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_ECDHE_ECDSA_WITH_CAMELLIA_128_CBC_SHA256 = CipherSuiteParams(
        code=0xc072,
        iana_name='TLS_ECDHE_ECDSA_WITH_CAMELLIA_128_CBC_SHA256',
        openssl_name='ECDHE-ECDSA-CAMELLIA128-SHA256',
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.ECDSA,
        bulk_cipher=BlockCipher.CAMELLIA_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_ECDHE_ECDSA_WITH_CAMELLIA_256_CBC_SHA384 = CipherSuiteParams(
        code=0xc073,
        iana_name='TLS_ECDHE_ECDSA_WITH_CAMELLIA_256_CBC_SHA384',
        openssl_name='ECDHE-ECDSA-CAMELLIA256-SHA384',
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.ECDSA,
        bulk_cipher=BlockCipher.CAMELLIA_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_384,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_ECDH_ECDSA_WITH_CAMELLIA_128_CBC_SHA256 = CipherSuiteParams(
        code=0xc074,
        iana_name='TLS_ECDH_ECDSA_WITH_CAMELLIA_128_CBC_SHA256',
        openssl_name='ECDH-ECDSA-CAMELLIA128-SHA256',
        key_exchange=KeyExchange.ECDH,
        authentication=Authentication.ECDSA,
        bulk_cipher=BlockCipher.CAMELLIA_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_ECDH_ECDSA_WITH_CAMELLIA_256_CBC_SHA384 = CipherSuiteParams(
        code=0xc075,
        iana_name='TLS_ECDH_ECDSA_WITH_CAMELLIA_256_CBC_SHA384',
        openssl_name='ECDH-ECDSA-CAMELLIA256-SHA384',
        key_exchange=KeyExchange.ECDH,
        authentication=Authentication.ECDSA,
        bulk_cipher=BlockCipher.CAMELLIA_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_384,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_ECDHE_RSA_WITH_CAMELLIA_128_CBC_SHA256 = CipherSuiteParams(
        code=0xc076,
        iana_name='TLS_ECDHE_RSA_WITH_CAMELLIA_128_CBC_SHA256',
        openssl_name='ECDHE-RSA-CAMELLIA128-SHA256',
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.CAMELLIA_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_ECDHE_RSA_WITH_CAMELLIA_256_CBC_SHA384 = CipherSuiteParams(
        code=0xc077,
        iana_name='TLS_ECDHE_RSA_WITH_CAMELLIA_256_CBC_SHA384',
        openssl_name='ECDHE-RSA-CAMELLIA256-SHA384',
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.CAMELLIA_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_384,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_ECDH_RSA_WITH_CAMELLIA_128_CBC_SHA256 = CipherSuiteParams(
        code=0xc078,
        iana_name='TLS_ECDH_RSA_WITH_CAMELLIA_128_CBC_SHA256',
        openssl_name='ECDH-RSA-CAMELLIA128-SHA256',
        key_exchange=KeyExchange.ECDH,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.CAMELLIA_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_ECDH_RSA_WITH_CAMELLIA_256_CBC_SHA384 = CipherSuiteParams(
        code=0xc079,
        iana_name='TLS_ECDH_RSA_WITH_CAMELLIA_256_CBC_SHA384',
        openssl_name='ECDH-RSA-CAMELLIA256-SHA384',
        key_exchange=KeyExchange.ECDH,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.CAMELLIA_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_384,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_RSA_WITH_CAMELLIA_128_GCM_SHA256 = CipherSuiteParams(
        code=0xc07a,
        iana_name='TLS_RSA_WITH_CAMELLIA_128_GCM_SHA256',
        openssl_name=None,
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.CAMELLIA_128,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_256,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_RSA_WITH_CAMELLIA_256_GCM_SHA384 = CipherSuiteParams(
        code=0xc07b,
        iana_name='TLS_RSA_WITH_CAMELLIA_256_GCM_SHA384',
        openssl_name=None,
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.CAMELLIA_256,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_384,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DHE_RSA_WITH_CAMELLIA_128_GCM_SHA256 = CipherSuiteParams(
        code=0xc07c,
        iana_name='TLS_DHE_RSA_WITH_CAMELLIA_128_GCM_SHA256',
        openssl_name=None,
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.CAMELLIA_128,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_256,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DHE_RSA_WITH_CAMELLIA_256_GCM_SHA384 = CipherSuiteParams(
        code=0xc07d,
        iana_name='TLS_DHE_RSA_WITH_CAMELLIA_256_GCM_SHA384',
        openssl_name=None,
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.CAMELLIA_256,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_384,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DH_RSA_WITH_CAMELLIA_128_GCM_SHA256 = CipherSuiteParams(
        code=0xc07e,
        iana_name='TLS_DH_RSA_WITH_CAMELLIA_128_GCM_SHA256',
        openssl_name=None,
        key_exchange=KeyExchange.DH,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.CAMELLIA_128,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_256,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DH_RSA_WITH_CAMELLIA_256_GCM_SHA384 = CipherSuiteParams(
        code=0xc07f,
        iana_name='TLS_DH_RSA_WITH_CAMELLIA_256_GCM_SHA384',
        openssl_name=None,
        key_exchange=KeyExchange.DH,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.CAMELLIA_256,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_384,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DHE_DSS_WITH_CAMELLIA_128_GCM_SHA256 = CipherSuiteParams(
        code=0xc080,
        iana_name='TLS_DHE_DSS_WITH_CAMELLIA_128_GCM_SHA256',
        openssl_name=None,
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.DSS,
        bulk_cipher=BlockCipher.CAMELLIA_128,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_256,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DHE_DSS_WITH_CAMELLIA_256_GCM_SHA384 = CipherSuiteParams(
        code=0xc081,
        iana_name='TLS_DHE_DSS_WITH_CAMELLIA_256_GCM_SHA384',
        openssl_name=None,
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.DSS,
        bulk_cipher=BlockCipher.CAMELLIA_256,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_384,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DH_DSS_WITH_CAMELLIA_128_GCM_SHA256 = CipherSuiteParams(
        code=0xc082,
        iana_name='TLS_DH_DSS_WITH_CAMELLIA_128_GCM_SHA256',
        openssl_name=None,
        key_exchange=KeyExchange.DH,
        authentication=Authentication.DSS,
        bulk_cipher=BlockCipher.CAMELLIA_128,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_256,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DH_DSS_WITH_CAMELLIA_256_GCM_SHA384 = CipherSuiteParams(
        code=0xc083,
        iana_name='TLS_DH_DSS_WITH_CAMELLIA_256_GCM_SHA384',
        openssl_name=None,
        key_exchange=KeyExchange.DH,
        authentication=Authentication.DSS,
        bulk_cipher=BlockCipher.CAMELLIA_256,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_384,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DH_anon_WITH_CAMELLIA_128_GCM_SHA256 = CipherSuiteParams(  # pylint: disable=invalid-name
        code=0xc084,
        iana_name='TLS_DH_anon_WITH_CAMELLIA_128_GCM_SHA256',
        openssl_name=None,
        key_exchange=KeyExchange.ADH,
        authentication=Authentication.anon,
        bulk_cipher=BlockCipher.CAMELLIA_128,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_256,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DH_anon_WITH_CAMELLIA_256_GCM_SHA384 = CipherSuiteParams(  # pylint: disable=invalid-name
        code=0xc085,
        iana_name='TLS_DH_anon_WITH_CAMELLIA_256_GCM_SHA384',
        openssl_name=None,
        key_exchange=KeyExchange.ADH,
        authentication=Authentication.anon,
        bulk_cipher=BlockCipher.CAMELLIA_256,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_384,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_ECDHE_ECDSA_WITH_CAMELLIA_128_GCM_SHA256 = CipherSuiteParams(
        code=0xc086,
        iana_name='TLS_ECDHE_ECDSA_WITH_CAMELLIA_128_GCM_SHA256',
        openssl_name=None,
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.ECDSA,
        bulk_cipher=BlockCipher.CAMELLIA_128,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_256,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_ECDHE_ECDSA_WITH_CAMELLIA_256_GCM_SHA384 = CipherSuiteParams(
        code=0xc087,
        iana_name='TLS_ECDHE_ECDSA_WITH_CAMELLIA_256_GCM_SHA384',
        openssl_name=None,
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.ECDSA,
        bulk_cipher=BlockCipher.CAMELLIA_256,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_384,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_ECDH_ECDSA_WITH_CAMELLIA_128_GCM_SHA256 = CipherSuiteParams(
        code=0xc088,
        iana_name='TLS_ECDH_ECDSA_WITH_CAMELLIA_128_GCM_SHA256',
        openssl_name=None,
        key_exchange=KeyExchange.ECDH,
        authentication=Authentication.ECDSA,
        bulk_cipher=BlockCipher.CAMELLIA_128,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_256,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_ECDH_ECDSA_WITH_CAMELLIA_256_GCM_SHA384 = CipherSuiteParams(
        code=0xc089,
        iana_name='TLS_ECDH_ECDSA_WITH_CAMELLIA_256_GCM_SHA384',
        openssl_name=None,
        key_exchange=KeyExchange.ECDH,
        authentication=Authentication.ECDSA,
        bulk_cipher=BlockCipher.CAMELLIA_256,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_384,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_ECDHE_RSA_WITH_CAMELLIA_128_GCM_SHA256 = CipherSuiteParams(
        code=0xc08a,
        iana_name='TLS_ECDHE_RSA_WITH_CAMELLIA_128_GCM_SHA256',
        openssl_name=None,
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.CAMELLIA_128,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_256,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_ECDHE_RSA_WITH_CAMELLIA_256_GCM_SHA384 = CipherSuiteParams(
        code=0xc08b,
        iana_name='TLS_ECDHE_RSA_WITH_CAMELLIA_256_GCM_SHA384',
        openssl_name=None,
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.CAMELLIA_256,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_384,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_ECDH_RSA_WITH_CAMELLIA_128_GCM_SHA256 = CipherSuiteParams(
        code=0xc08c,
        iana_name='TLS_ECDH_RSA_WITH_CAMELLIA_128_GCM_SHA256',
        openssl_name=None,
        key_exchange=KeyExchange.ECDH,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.CAMELLIA_128,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_256,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_ECDH_RSA_WITH_CAMELLIA_256_GCM_SHA384 = CipherSuiteParams(
        code=0xc08d,
        iana_name='TLS_ECDH_RSA_WITH_CAMELLIA_256_GCM_SHA384',
        openssl_name=None,
        key_exchange=KeyExchange.ECDH,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.CAMELLIA_256,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_384,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_PSK_WITH_CAMELLIA_128_GCM_SHA256 = CipherSuiteParams(
        code=0xc08e,
        iana_name='TLS_PSK_WITH_CAMELLIA_128_GCM_SHA256',
        openssl_name=None,
        key_exchange=KeyExchange.PSK,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.CAMELLIA_128,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_256,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_PSK_WITH_CAMELLIA_256_GCM_SHA384 = CipherSuiteParams(
        code=0xc08f,
        iana_name='TLS_PSK_WITH_CAMELLIA_256_GCM_SHA384',
        openssl_name=None,
        key_exchange=KeyExchange.PSK,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.CAMELLIA_256,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_384,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DHE_PSK_WITH_CAMELLIA_128_GCM_SHA256 = CipherSuiteParams(
        code=0xc090,
        iana_name='TLS_DHE_PSK_WITH_CAMELLIA_128_GCM_SHA256',
        openssl_name=None,
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.CAMELLIA_128,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_256,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DHE_PSK_WITH_CAMELLIA_256_GCM_SHA384 = CipherSuiteParams(
        code=0xc091,
        iana_name='TLS_DHE_PSK_WITH_CAMELLIA_256_GCM_SHA384',
        openssl_name=None,
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.CAMELLIA_256,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_384,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_RSA_PSK_WITH_CAMELLIA_128_GCM_SHA256 = CipherSuiteParams(
        code=0xc092,
        iana_name='TLS_RSA_PSK_WITH_CAMELLIA_128_GCM_SHA256',
        openssl_name=None,
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.CAMELLIA_128,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_256,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_RSA_PSK_WITH_CAMELLIA_256_GCM_SHA384 = CipherSuiteParams(
        code=0xc093,
        iana_name='TLS_RSA_PSK_WITH_CAMELLIA_256_GCM_SHA384',
        openssl_name=None,
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.CAMELLIA_256,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_384,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_PSK_WITH_CAMELLIA_128_CBC_SHA256 = CipherSuiteParams(
        code=0xc094,
        iana_name='TLS_PSK_WITH_CAMELLIA_128_CBC_SHA256',
        openssl_name='PSK-CAMELLIA128-SHA256',
        key_exchange=KeyExchange.PSK,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.CAMELLIA_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_PSK_WITH_CAMELLIA_256_CBC_SHA384 = CipherSuiteParams(
        code=0xc095,
        iana_name='TLS_PSK_WITH_CAMELLIA_256_CBC_SHA384',
        openssl_name='PSK-CAMELLIA256-SHA384',
        key_exchange=KeyExchange.PSK,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.CAMELLIA_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_384,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DHE_PSK_WITH_CAMELLIA_128_CBC_SHA256 = CipherSuiteParams(
        code=0xc096,
        iana_name='TLS_DHE_PSK_WITH_CAMELLIA_128_CBC_SHA256',
        openssl_name='DHE-PSK-CAMELLIA128-SHA256',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.CAMELLIA_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DHE_PSK_WITH_CAMELLIA_256_CBC_SHA384 = CipherSuiteParams(
        code=0xc097,
        iana_name='TLS_DHE_PSK_WITH_CAMELLIA_256_CBC_SHA384',
        openssl_name='DHE-PSK-CAMELLIA256-SHA384',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.CAMELLIA_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_384,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_RSA_PSK_WITH_CAMELLIA_128_CBC_SHA256 = CipherSuiteParams(
        code=0xc098,
        iana_name='TLS_RSA_PSK_WITH_CAMELLIA_128_CBC_SHA256',
        openssl_name='RSA-PSK-CAMELLIA128-SHA256',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.CAMELLIA_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_RSA_PSK_WITH_CAMELLIA_256_CBC_SHA384 = CipherSuiteParams(
        code=0xc099,
        iana_name='TLS_RSA_PSK_WITH_CAMELLIA_256_CBC_SHA384',
        openssl_name='RSA-PSK-CAMELLIA256-SHA384',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.CAMELLIA_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_384,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_ECDHE_PSK_WITH_CAMELLIA_128_CBC_SHA256 = CipherSuiteParams(
        code=0xc09a,
        iana_name='TLS_ECDHE_PSK_WITH_CAMELLIA_128_CBC_SHA256',
        openssl_name='ECDHE-PSK-CAMELLIA128-SHA256',
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.CAMELLIA_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_ECDHE_PSK_WITH_CAMELLIA_256_CBC_SHA384 = CipherSuiteParams(
        code=0xc09b,
        iana_name='TLS_ECDHE_PSK_WITH_CAMELLIA_256_CBC_SHA384',
        openssl_name='ECDHE-PSK-CAMELLIA256-SHA384',
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.CAMELLIA_256,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA2_384,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_RSA_WITH_AES_128_CCM = CipherSuiteParams(
        code=0xc09c,
        iana_name='TLS_RSA_WITH_AES_128_CCM',
        openssl_name='AES128-CCM',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.CCM,
        mac=None,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_RSA_WITH_AES_256_CCM = CipherSuiteParams(
        code=0xc09d,
        iana_name='TLS_RSA_WITH_AES_256_CCM',
        openssl_name='AES256-CCM',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.CCM,
        mac=None,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DHE_RSA_WITH_AES_128_CCM = CipherSuiteParams(
        code=0xc09e,
        iana_name='TLS_DHE_RSA_WITH_AES_128_CCM',
        openssl_name='DHE-RSA-AES128-CCM',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.CCM,
        mac=None,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DHE_RSA_WITH_AES_256_CCM = CipherSuiteParams(
        code=0xc09f,
        iana_name='TLS_DHE_RSA_WITH_AES_256_CCM',
        openssl_name='DHE-RSA-AES256-CCM',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.CCM,
        mac=None,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_RSA_WITH_AES_128_CCM_8 = CipherSuiteParams(
        code=0xc0a0,
        iana_name='TLS_RSA_WITH_AES_128_CCM_8',
        openssl_name='AES128-CCM8',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.CCM_8,
        mac=None,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_RSA_WITH_AES_256_CCM_8 = CipherSuiteParams(
        code=0xc0a1,
        iana_name='TLS_RSA_WITH_AES_256_CCM_8',
        openssl_name='AES256-CCM8',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.CCM_8,
        mac=None,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DHE_RSA_WITH_AES_128_CCM_8 = CipherSuiteParams(
        code=0xc0a2,
        iana_name='TLS_DHE_RSA_WITH_AES_128_CCM_8',
        openssl_name='DHE-RSA-AES128-CCM8',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.CCM_8,
        mac=None,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DHE_RSA_WITH_AES_256_CCM_8 = CipherSuiteParams(
        code=0xc0a3,
        iana_name='TLS_DHE_RSA_WITH_AES_256_CCM_8',
        openssl_name='DHE-RSA-AES256-CCM8',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.CCM_8,
        mac=None,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_PSK_WITH_AES_128_CCM = CipherSuiteParams(
        code=0xc0a4,
        iana_name='TLS_PSK_WITH_AES_128_CCM',
        openssl_name='PSK-AES128-CCM',
        key_exchange=KeyExchange.PSK,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.CCM,
        mac=None,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_PSK_WITH_AES_256_CCM = CipherSuiteParams(
        code=0xc0a5,
        iana_name='TLS_PSK_WITH_AES_256_CCM',
        openssl_name='PSK-AES256-CCM',
        key_exchange=KeyExchange.PSK,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.CCM,
        mac=None,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DHE_PSK_WITH_AES_128_CCM = CipherSuiteParams(
        code=0xc0a6,
        iana_name='TLS_DHE_PSK_WITH_AES_128_CCM',
        openssl_name='DHE-PSK-AES128-CCM',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.CCM,
        mac=None,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DHE_PSK_WITH_AES_256_CCM = CipherSuiteParams(
        code=0xc0a7,
        iana_name='TLS_DHE_PSK_WITH_AES_256_CCM',
        openssl_name='DHE-PSK-AES256-CCM',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.CCM,
        mac=None,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_PSK_WITH_AES_128_CCM_8 = CipherSuiteParams(
        code=0xc0a8,
        iana_name='TLS_PSK_WITH_AES_128_CCM_8',
        openssl_name='PSK-AES128-CCM8',
        key_exchange=KeyExchange.PSK,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.CCM_8,
        mac=None,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_PSK_WITH_AES_256_CCM_8 = CipherSuiteParams(
        code=0xc0a9,
        iana_name='TLS_PSK_WITH_AES_256_CCM_8',
        openssl_name='PSK-AES256-CCM8',
        key_exchange=KeyExchange.PSK,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.CCM_8,
        mac=None,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DHE_PSK_WITH_AES_128_CCM_8 = CipherSuiteParams(
        code=0xc0aa,
        iana_name='TLS_DHE_PSK_WITH_AES_128_CCM_8',
        openssl_name='DHE-PSK-AES128-CCM8',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.CCM_8,
        mac=None,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DHE_PSK_WITH_AES_256_CCM_8 = CipherSuiteParams(
        code=0xc0ab,
        iana_name='TLS_DHE_PSK_WITH_AES_256_CCM_8',
        openssl_name='DHE-PSK-AES256-CCM8',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.CCM_8,
        mac=None,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_ECDHE_ECDSA_WITH_AES_128_CCM = CipherSuiteParams(
        code=0xc0ac,
        iana_name='TLS_ECDHE_ECDSA_WITH_AES_128_CCM',
        openssl_name='ECDHE-ECDSA-AES128-CCM',
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.ECDSA,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.CCM,
        mac=None,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_ECDHE_ECDSA_WITH_AES_256_CCM = CipherSuiteParams(
        code=0xc0ad,
        iana_name='TLS_ECDHE_ECDSA_WITH_AES_256_CCM',
        openssl_name='ECDHE-ECDSA-AES256-CCM',
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.ECDSA,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.CCM,
        mac=None,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_ECDHE_ECDSA_WITH_AES_128_CCM_8 = CipherSuiteParams(
        code=0xc0ae,
        iana_name='TLS_ECDHE_ECDSA_WITH_AES_128_CCM_8',
        openssl_name='ECDHE-ECDSA-AES128-CCM8',
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.ECDSA,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.CCM_8,
        mac=None,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_ECDHE_ECDSA_WITH_AES_256_CCM_8 = CipherSuiteParams(
        code=0xc0af,
        iana_name='TLS_ECDHE_ECDSA_WITH_AES_256_CCM_8',
        openssl_name='ECDHE-ECDSA-AES256-CCM8',
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.ECDSA,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.CCM_8,
        mac=None,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_GOSTR341112_256_WITH_KUZNYECHIK_CTR_OMAC = CipherSuiteParams(
        code=0xc100,
        iana_name='TLS_GOSTR341112_256_WITH_KUZNYECHIK_CTR_OMAC',
        openssl_name=None,
        key_exchange=KeyExchange.GOST_R3411_12_256,
        authentication=Authentication.GOST_R3410_12_256,
        bulk_cipher=BlockCipher.GOST_R3412_15_128,
        block_cipher_mode=BlockCipherMode.EAX,
        mac=MAC.GOST_R3412_15_KUZNYECHIK_CTR_OMAC,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_GOSTR341112_256_WITH_MAGMA_CTR_OMAC = CipherSuiteParams(
        code=0xc101,
        iana_name='TLS_GOSTR341112_256_WITH_MAGMA_CTR_OMAC',
        openssl_name=None,
        key_exchange=KeyExchange.GOST_R3411_12_256,
        authentication=Authentication.GOST_R3410_12_256,
        bulk_cipher=BlockCipher.GOST_R3412_15_64,
        block_cipher_mode=BlockCipherMode.EAX,
        mac=MAC.GOST_R3412_15_MAGMA_CTR_OMAC,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_GOSTR341112_256_WITH_28147_CNT_IMIT = CipherSuiteParams(
        code=0xc102,
        iana_name='TLS_GOSTR341112_256_WITH_28147_CNT_IMIT',
        openssl_name='GOST2012-GOST8912-GOST8912',
        key_exchange=KeyExchange.GOST_R3411_12_256,
        authentication=Authentication.GOST_R3410_12_256,
        bulk_cipher=BlockCipher.GOST2814789,
        block_cipher_mode=BlockCipherMode.CNT,
        mac=MAC.IMIT_GOST28147,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_GOSTR341112_256_WITH_KUZNYECHIK_MGM_L = CipherSuiteParams(
        code=0xc103,
        iana_name='TLS_GOSTR341112_256_WITH_KUZNYECHIK_MGM_L',
        openssl_name=None,
        key_exchange=KeyExchange.GOST_R3411_12_256,
        authentication=Authentication.GOST_R3410_12_256,
        bulk_cipher=BlockCipher.GOST_R3412_15_128,
        block_cipher_mode=BlockCipherMode.MGM,
        mac=None,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_3),
    )
    TLS_GOSTR341112_256_WITH_MAGMA_MGM_L = CipherSuiteParams(
        code=0xc104,
        iana_name='TLS_GOSTR341112_256_WITH_MAGMA_MGM_L',
        openssl_name=None,
        key_exchange=KeyExchange.GOST_R3411_12_256,
        authentication=Authentication.GOST_R3410_12_256,
        bulk_cipher=BlockCipher.GOST_R3412_15_64,
        block_cipher_mode=BlockCipherMode.MGM,
        mac=None,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_3),
    )
    TLS_GOSTR341112_256_WITH_KUZNYECHIK_MGM_S = CipherSuiteParams(
        code=0xc105,
        iana_name='TLS_GOSTR341112_256_WITH_KUZNYECHIK_MGM_S',
        openssl_name=None,
        key_exchange=KeyExchange.GOST_R3411_12_256,
        authentication=Authentication.GOST_R3410_12_256,
        bulk_cipher=BlockCipher.GOST_R3412_15_128,
        block_cipher_mode=BlockCipherMode.MGM,
        mac=None,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_3),
    )
    TLS_GOSTR341112_256_WITH_MAGMA_MGM_S = CipherSuiteParams(
        code=0xc106,
        iana_name='TLS_GOSTR341112_256_WITH_MAGMA_MGM_S',
        openssl_name=None,
        key_exchange=KeyExchange.GOST_R3411_12_256,
        authentication=Authentication.GOST_R3410_12_256,
        bulk_cipher=BlockCipher.GOST_R3412_15_64,
        block_cipher_mode=BlockCipherMode.MGM,
        mac=None,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_3),
    )
    OLD_TLS_ECDHE_RSA_WITH_CHACHA20_POLY1305_SHA256 = CipherSuiteParams(
        code=0xcc13,
        iana_name=None,
        openssl_name='ECDHE-RSA-CHACHA20-POLY1305',
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.CHACHA20,
        block_cipher_mode=None,
        mac=MAC.POLY1305,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    OLD_TLS_ECDHE_ECDSA_WITH_CHACHA20_POLY1305_SHA256 = CipherSuiteParams(
        code=0xcc14,
        iana_name=None,
        openssl_name=None,
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.ECDSA,
        bulk_cipher=BlockCipher.CHACHA20,
        block_cipher_mode=None,
        mac=MAC.POLY1305,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    OLD_TLS_DHE_RSA_WITH_CHACHA20_POLY1305_SHA256 = CipherSuiteParams(
        code=0xcc15,
        iana_name=None,
        openssl_name=None,
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.CHACHA20,
        block_cipher_mode=None,
        mac=MAC.POLY1305,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_ECDHE_RSA_WITH_CHACHA20_POLY1305_SHA256 = CipherSuiteParams(
        code=0xcca8,
        iana_name='TLS_ECDHE_RSA_WITH_CHACHA20_POLY1305_SHA256',
        openssl_name='ECDHE-RSA-CHACHA20-POLY1305',
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.CHACHA20,
        block_cipher_mode=None,
        mac=MAC.POLY1305,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_ECDHE_ECDSA_WITH_CHACHA20_POLY1305_SHA256 = CipherSuiteParams(
        code=0xcca9,
        iana_name='TLS_ECDHE_ECDSA_WITH_CHACHA20_POLY1305_SHA256',
        openssl_name='ECDHE-ECDSA-CHACHA20-POLY1305',
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.ECDSA,
        bulk_cipher=BlockCipher.CHACHA20,
        block_cipher_mode=None,
        mac=MAC.POLY1305,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DHE_RSA_WITH_CHACHA20_POLY1305_SHA256 = CipherSuiteParams(
        code=0xccaa,
        iana_name='TLS_DHE_RSA_WITH_CHACHA20_POLY1305_SHA256',
        openssl_name='DHE-RSA-CHACHA20-POLY1305',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.CHACHA20,
        block_cipher_mode=None,
        mac=MAC.POLY1305,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_PSK_WITH_CHACHA20_POLY1305_SHA256 = CipherSuiteParams(
        code=0xccab,
        iana_name='TLS_PSK_WITH_CHACHA20_POLY1305_SHA256',
        openssl_name='PSK-CHACHA20-POLY1305',
        key_exchange=KeyExchange.PSK,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.CHACHA20,
        block_cipher_mode=None,
        mac=MAC.POLY1305,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_ECDHE_PSK_WITH_CHACHA20_POLY1305_SHA256 = CipherSuiteParams(
        code=0xccac,
        iana_name='TLS_ECDHE_PSK_WITH_CHACHA20_POLY1305_SHA256',
        openssl_name='ECDHE-PSK-CHACHA20-POLY1305',
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.CHACHA20,
        block_cipher_mode=None,
        mac=MAC.POLY1305,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_DHE_PSK_WITH_CHACHA20_POLY1305_SHA256 = CipherSuiteParams(
        code=0xccad,
        iana_name='TLS_DHE_PSK_WITH_CHACHA20_POLY1305_SHA256',
        openssl_name='DHE-PSK-CHACHA20-POLY1305',
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.CHACHA20,
        block_cipher_mode=None,
        mac=MAC.POLY1305,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_RSA_PSK_WITH_CHACHA20_POLY1305_SHA256 = CipherSuiteParams(
        code=0xccae,
        iana_name='TLS_RSA_PSK_WITH_CHACHA20_POLY1305_SHA256',
        openssl_name='RSA-PSK-CHACHA20-POLY1305',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.CHACHA20,
        block_cipher_mode=None,
        mac=MAC.POLY1305,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_ECDHE_PSK_WITH_AES_128_GCM_SHA256 = CipherSuiteParams(
        code=0xd001,
        iana_name='TLS_ECDHE_PSK_WITH_AES_128_GCM_SHA256',
        openssl_name=None,
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_256,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_ECDHE_PSK_WITH_AES_256_GCM_SHA384 = CipherSuiteParams(
        code=0xd002,
        iana_name='TLS_ECDHE_PSK_WITH_AES_256_GCM_SHA384',
        openssl_name=None,
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_384,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_ECDHE_PSK_WITH_AES_128_CCM_8_SHA256 = CipherSuiteParams(
        code=0xd003,
        iana_name='TLS_ECDHE_PSK_WITH_AES_128_CCM_8_SHA256',
        openssl_name=None,
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.CCM_8,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_ECDHE_PSK_WITH_AES_128_CCM_SHA256 = CipherSuiteParams(
        code=0xd005,
        iana_name='TLS_ECDHE_PSK_WITH_AES_128_CCM_SHA256',
        openssl_name=None,
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.CCM,
        mac=MAC.SHA2_256,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    TLS_AES_128_GCM_SHA256 = CipherSuiteParams(
        code=0x1301,
        iana_name='TLS_AES_128_GCM_SHA256',
        openssl_name='TLS_AES_128_GCM_SHA256',
        key_exchange=None,
        authentication=None,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_256,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionDraft(15),
    )
    TLS_AES_256_GCM_SHA384 = CipherSuiteParams(
        code=0x1302,
        iana_name='TLS_AES_256_GCM_SHA384',
        openssl_name='TLS_AES_256_GCM_SHA384',
        key_exchange=None,
        authentication=None,
        bulk_cipher=BlockCipher.AES_256,
        block_cipher_mode=BlockCipherMode.GCM,
        mac=MAC.SHA2_384,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionDraft(15),
    )
    TLS_CHACHA20_POLY1305_SHA256 = CipherSuiteParams(
        code=0x1303,
        iana_name='TLS_CHACHA20_POLY1305_SHA256',
        openssl_name='TLS_CHACHA20_POLY1305_SHA256',
        key_exchange=None,
        authentication=None,
        bulk_cipher=BlockCipher.CHACHA20,
        block_cipher_mode=None,
        mac=MAC.POLY1305,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionDraft(15),
    )
    TLS_AES_128_CCM_SHA256 = CipherSuiteParams(
        code=0x1304,
        iana_name='TLS_AES_128_CCM_SHA256',
        openssl_name='TLS_AES_128_CCM_SHA256',
        key_exchange=None,
        authentication=None,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.CCM,
        mac=MAC.SHA2_256,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionDraft(15),
    )
    TLS_AES_128_CCM_8_SHA256 = CipherSuiteParams(
        code=0x1305,
        iana_name='TLS_AES_128_CCM_8_SHA256',
        openssl_name='TLS_AES_128_CCM_8_SHA256',
        key_exchange=None,
        authentication=None,
        bulk_cipher=BlockCipher.AES_128,
        block_cipher_mode=BlockCipherMode.CCM_8,
        mac=MAC.SHA2_256,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionDraft(15),
    )
    TLS_RSA_WITH_ESTREAM_SALSA20_SHA1 = CipherSuiteParams(
        code=0xe410,
        iana_name='TLS_RSA_WITH_ESTREAM_SALSA20_SHA1',
        openssl_name=None,
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.ESTREAM_SALSA20,
        block_cipher_mode=None,
        mac=MAC.SHA1,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_RSA_WITH_SALSA20_SHA1 = CipherSuiteParams(
        code=0xe411,
        iana_name='TLS_RSA_WITH_SALSA20_SHA1',
        openssl_name=None,
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.SALSA20,
        block_cipher_mode=None,
        mac=MAC.SHA1,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_ECDHE_RSA_WITH_ESTREAM_SALSA20_SHA1 = CipherSuiteParams(
        code=0xe412,
        iana_name='TLS_ECDHE_RSA_WITH_ESTREAM_SALSA20_SHA1',
        openssl_name=None,
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.ESTREAM_SALSA20,
        block_cipher_mode=None,
        mac=MAC.SHA1,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_ECDHE_RSA_WITH_SALSA20_SHA1 = CipherSuiteParams(
        code=0xe413,
        iana_name='TLS_ECDHE_RSA_WITH_SALSA20_SHA1',
        openssl_name=None,
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.SALSA20,
        block_cipher_mode=None,
        mac=MAC.SHA1,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_ECDHE_ECDSA_WITH_ESTREAM_SALSA20_SHA1 = CipherSuiteParams(
        code=0xe414,
        iana_name='TLS_ECDHE_ECDSA_WITH_ESTREAM_SALSA20_SHA1',
        openssl_name=None,
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.ECDSA,
        bulk_cipher=BlockCipher.ESTREAM_SALSA20,
        block_cipher_mode=None,
        mac=MAC.SHA1,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_ECDHE_ECDSA_WITH_SALSA20_SHA1 = CipherSuiteParams(
        code=0xe415,
        iana_name='TLS_ECDHE_ECDSA_WITH_SALSA20_SHA1',
        openssl_name=None,
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.ECDSA,
        bulk_cipher=BlockCipher.SALSA20,
        block_cipher_mode=None,
        mac=MAC.SHA1,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_PSK_WITH_ESTREAM_SALSA20_SHA1 = CipherSuiteParams(
        code=0xe416,
        iana_name='TLS_PSK_WITH_ESTREAM_SALSA20_SHA1',
        openssl_name=None,
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.ESTREAM_SALSA20,
        block_cipher_mode=None,
        mac=MAC.SHA1,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_PSK_WITH_SALSA20_SHA1 = CipherSuiteParams(
        code=0xe417,
        iana_name='TLS_PSK_WITH_SALSA20_SHA1',
        openssl_name=None,
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.SALSA20,
        block_cipher_mode=None,
        mac=MAC.SHA1,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_ECDHE_PSK_WITH_ESTREAM_SALSA20_SHA1 = CipherSuiteParams(
        code=0xe418,
        iana_name='TLS_ECDHE_PSK_WITH_ESTREAM_SALSA20_SHA1',
        openssl_name=None,
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.ESTREAM_SALSA20,
        block_cipher_mode=None,
        mac=MAC.SHA1,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_ECDHE_PSK_WITH_SALSA20_SHA1 = CipherSuiteParams(
        code=0xe419,
        iana_name='TLS_ECDHE_PSK_WITH_SALSA20_SHA1',
        openssl_name=None,
        key_exchange=KeyExchange.ECDHE,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.SALSA20,
        block_cipher_mode=None,
        mac=MAC.SHA1,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_RSA_PSK_WITH_ESTREAM_SALSA20_SHA1 = CipherSuiteParams(
        code=0xe41a,
        iana_name='TLS_RSA_PSK_WITH_ESTREAM_SALSA20_SHA1',
        openssl_name=None,
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.ESTREAM_SALSA20,
        block_cipher_mode=None,
        mac=MAC.SHA1,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_RSA_PSK_WITH_SALSA20_SHA1 = CipherSuiteParams(
        code=0xe41b,
        iana_name='TLS_RSA_PSK_WITH_SALSA20_SHA1',
        openssl_name=None,
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.SALSA20,
        block_cipher_mode=None,
        mac=MAC.SHA1,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DHE_PSK_WITH_ESTREAM_SALSA20_SHA1 = CipherSuiteParams(
        code=0xe41c,
        iana_name='TLS_DHE_PSK_WITH_ESTREAM_SALSA20_SHA1',
        openssl_name=None,
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.ESTREAM_SALSA20,
        block_cipher_mode=None,
        mac=MAC.SHA1,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DHE_PSK_WITH_SALSA20_SHA1 = CipherSuiteParams(
        code=0xe41d,
        iana_name='TLS_DHE_PSK_WITH_SALSA20_SHA1',
        openssl_name=None,
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.PSK,
        bulk_cipher=BlockCipher.SALSA20,
        block_cipher_mode=None,
        mac=MAC.SHA1,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DHE_RSA_WITH_ESTREAM_SALSA20_SHA1 = CipherSuiteParams(
        code=0xe41e,
        iana_name='TLS_DHE_RSA_WITH_ESTREAM_SALSA20_SHA1',
        openssl_name=None,
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.ESTREAM_SALSA20,
        block_cipher_mode=None,
        mac=MAC.SHA1,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_DHE_RSA_WITH_SALSA20_SHA1 = CipherSuiteParams(
        code=0xe41f,
        iana_name='TLS_DHE_RSA_WITH_SALSA20_SHA1',
        openssl_name=None,
        key_exchange=KeyExchange.DHE,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.SALSA20,
        block_cipher_mode=None,
        mac=MAC.SHA1,
        authenticated_encryption=True,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_RSA_FIPS_WITH_DES_CBC_SHA = CipherSuiteParams(
        code=0xfefe,
        iana_name='TLS_RSA_FIPS_WITH_DES_CBC_SHA',
        openssl_name=None,
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.DES,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_RSA_FIPS_WITH_3DES_EDE_CBC_SHA = CipherSuiteParams(
        code=0xfeff,
        iana_name='TLS_RSA_FIPS_WITH_3DES_EDE_CBC_SHA',
        openssl_name=None,
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.TRIPLE_DES_EDE,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_RSA_WITH_RC2_CBC_MD5 = CipherSuiteParams(
        code=0xff80,
        iana_name='TLS_RSA_WITH_RC2_CBC_MD5',
        openssl_name=None,
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.RC2,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.MD5,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_RSA_WITH_IDEA_CBC_MD5 = CipherSuiteParams(
        code=0xff81,
        iana_name='TLS_RSA_WITH_IDEA_CBC_MD5',
        openssl_name=None,
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.IDEA,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.MD5,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_RSA_WITH_DES_CBC_MD5 = CipherSuiteParams(
        code=0xff82,
        iana_name='TLS_RSA_WITH_DES_CBC_MD5',
        openssl_name=None,
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.DES,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.MD5,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_RSA_WITH_3DES_EDE_CBC_MD5 = CipherSuiteParams(
        code=0xff83,
        iana_name='TLS_RSA_WITH_3DES_EDE_CBC_MD5',
        openssl_name=None,
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.TRIPLE_DES_EDE,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.MD5,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    OLD_TLS_GOSTR341112_256_WITH_28147_CNT_IMIT = CipherSuiteParams(
        code=0xff85,
        iana_name=None,
        openssl_name='LEGACY-GOST2012-GOST8912-GOST8912',
        key_exchange=KeyExchange.GOST_R3411_12_256,
        authentication=Authentication.GOST_R3410_94,
        bulk_cipher=BlockCipher.GOST2814789,
        block_cipher_mode=None,
        mac=MAC.IMIT_GOST28147,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    TLS_GOSTR341112_256_WITH_NULL_GOSTR3411 = CipherSuiteParams(
        code=0xff87,
        iana_name='TLS_GOSTR341112_256_WITH_NULL_GOSTR3411',
        openssl_name='GOST2012-NULL-GOST12',
        key_exchange=KeyExchange.GOST_R3411_12_256,
        authentication=Authentication.GOST_R3410_94,
        bulk_cipher=None,
        block_cipher_mode=None,
        mac=MAC.GOST_R3411_94,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_2),
    )
    OLD_TLS_RSA_FIPS_WITH_DES_CBC_SHA = CipherSuiteParams(
        code=0xffe0,
        iana_name=None,
        openssl_name=None,
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.DES,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )
    OLD_TLS_RSA_FIPS_WITH_3DES_EDE_CBC_SHA = CipherSuiteParams(
        code=0xffe1,
        iana_name=None,
        openssl_name=None,
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.TRIPLE_DES_EDE,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=TlsProtocolVersionFinal(TlsVersion.TLS1_0),
    )


class SslCipherKindFactory(ThreeByteEnumParsable):
    @classmethod
    def get_enum_class(cls):
        return SslCipherKind

    @abc.abstractmethod
    def compose(self):
        raise NotImplementedError()


class SslCipherKind(Serializable, ThreeByteEnumComposer):
    SSL_CK_NULL_WITH_MD5 = CipherSuiteParams(
        code=0x000000,
        iana_name='SSL_CK_NULL_WITH_MD5',
        openssl_name='NULL-MD5',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=None,
        block_cipher_mode=None,
        mac=MAC.MD5,
        authenticated_encryption=False,
        initial_version=SslProtocolVersion(),
    )
    SSL_CK_RC4_128_WITH_MD5 = CipherSuiteParams(
        code=0x010080,
        iana_name='SSL_CK_RC4_128_WITH_MD5',
        openssl_name='RC4-MD5',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.RC4_128,
        block_cipher_mode=None,
        mac=MAC.MD5,
        authenticated_encryption=False,
        initial_version=SslProtocolVersion(),
    )
    SSL_CK_RC4_128_EXPORT40_WITH_MD5 = CipherSuiteParams(
        code=0x020080,
        iana_name='SSL_CK_RC4_128_EXPORT40_WITH_MD5',
        openssl_name='EXP-RC4-MD5',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.RC4_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.MD5,
        authenticated_encryption=False,
        initial_version=SslProtocolVersion(),
    )
    SSL_CK_RC2_128_CBC_WITH_MD5 = CipherSuiteParams(
        code=0x030080,
        iana_name='SSL_CK_RC2_128_CBC_WITH_MD5',
        openssl_name='RC2-CBC-MD5',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.RC2_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.MD5,
        authenticated_encryption=False,
        initial_version=SslProtocolVersion(),
    )
    SSL_CK_RC2_128_CBC_EXPORT40_WITH_MD5 = CipherSuiteParams(
        code=0x040080,
        iana_name='SSL_CK_RC2_128_CBC_EXPORT40_WITH_MD5',
        openssl_name='EXP-RC2-CBC-MD5',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.RC2_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.MD5,
        authenticated_encryption=False,
        initial_version=SslProtocolVersion(),
    )
    SSL_CK_IDEA_128_CBC_WITH_MD5 = CipherSuiteParams(
        code=0x050080,
        iana_name='SSL_CK_IDEA_128_CBC_WITH_MD5',
        openssl_name='IDEA-CBC-MD5',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.IDEA_128,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.MD5,
        authenticated_encryption=False,
        initial_version=SslProtocolVersion(),
    )
    SSL_CK_DES_64_CBC_WITH_MD5 = CipherSuiteParams(
        code=0x060040,
        iana_name='SSL_CK_DES_64_CBC_WITH_MD5',
        openssl_name='DES-CBC-MD5',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.DES,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.MD5,
        authenticated_encryption=False,
        initial_version=SslProtocolVersion(),
    )
    SSL_CK_DES_64_CBC_WITH_SHA = CipherSuiteParams(
        code=0x060140,
        iana_name='SSL_CK_DES_64_CBC_WITH_SHA',
        openssl_name=None,
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.DES,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=SslProtocolVersion(),
    )
    SSL_CK_DES_192_EDE3_CBC_WITH_MD5 = CipherSuiteParams(
        code=0x0700C0,
        iana_name='SSL_CK_DES_192_EDE3_CBC_WITH_MD5',
        openssl_name=None,
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.TRIPLE_DES_EDE,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.MD5,
        authenticated_encryption=False,
        initial_version=SslProtocolVersion(),
    )
    SSL_CK_DES_192_EDE3_CBC_WITH_SHA = CipherSuiteParams(
        code=0x0701C0,
        iana_name='SSL_CK_DES_192_EDE3_CBC_WITH_SHA',
        openssl_name=None,
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.TRIPLE_DES,
        block_cipher_mode=BlockCipherMode.CBC,
        mac=MAC.SHA1,
        authenticated_encryption=False,
        initial_version=SslProtocolVersion(),
    )
    SSL_CK_RC4_64_WITH_MD5 = CipherSuiteParams(
        code=0x080080,
        iana_name='SSL_CK_RC4_64_WITH_MD5',
        openssl_name='RC4-64-MD5',
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.RC4_64,
        block_cipher_mode=None,
        mac=MAC.MD5,
        authenticated_encryption=False,
        initial_version=SslProtocolVersion(),
    )
    SSL_CK_DES_64_CFB64_WITH_MD5_1 = CipherSuiteParams(
        code=0xFF8000,
        iana_name='SSL_CK_DES_64_CFB64_WITH_MD5_1',
        openssl_name=None,
        key_exchange=KeyExchange.RSA,
        authentication=Authentication.RSA,
        bulk_cipher=BlockCipher.DES,
        block_cipher_mode=BlockCipherMode.CFB,
        mac=MAC.MD5,
        authenticated_encryption=False,
        initial_version=SslProtocolVersion(),
    )
    SSL_CK_NULL = CipherSuiteParams(
        code=0xFF8010,
        iana_name='SSL_CK_NULL',
        openssl_name=None,
        key_exchange=None,
        authentication=None,
        bulk_cipher=None,
        block_cipher_mode=None,
        mac=None,
        authenticated_encryption=False,
        initial_version=SslProtocolVersion(),
    )
