"""
Helper functions
================

Util functions that are useful throughout the MVS

Including:
- find_valvue_by_key(): Finds value of a key in a nested dictionary.
"""

import os

from multi_vector_simulator.utils.constants_json_strings import (
    DSO_FEEDIN_CAP,
    AUTO_CREATED_HIGHLIGHT,
    DSO_CONSUMPTION,
    DSO_FEEDIN,
    DSO_PEAK_DEMAND_PERIOD,
    DSO_PEAK_DEMAND_SUFFIX,
    ENERGY_CONSUMPTION,
    ENERGY_CONVERSION,
    ENERGY_STORAGE,
    ENERGY_PRODUCTION,
    ENERGY_PROVIDERS,
    ENERGY_BUSSES,
    OEMOF_ASSET_TYPE,
    TYPE_ASSET,
    INFLOW_DIRECTION,
    OUTFLOW_DIRECTION,
    ENERGY_VECTOR,
)


def find_value_by_key(data, target, result=None):
    """
    Finds value of a key in a nested dictionary.

    Parameters
    ----------
    data: dict
        Dict to be searched for target key

    target: str
        Key for which the value should be found in data

    result: None, value or list
        Only provided if function loops in itself

    Returns
    -------
    value if the key is only once in data
    list of values if it appears multiple times.
    """
    # check each item-value pair in the level
    for k, v in data.items():
        # if target in keys of level
        if k == target:
            if result is None:
                result = v
            elif isinstance(result, list):
                # Expands list of key finds
                result.append(v)
            else:
                # creates list for multiple key finds
                previous_result = result
                result = []
                result.append(previous_result)
                result.append(v)
        # Check next level for target
        if isinstance(v, dict):
            result = find_value_by_key(data=v, target=target, result=result)
    return result


def translates_epa_strings_to_mvs_readable(folder_name, file_name):
    """
    This function translates the json file generated by the EPA to a file readable by the MVS.
    This is necessary as there are some parameter names whose string representative differs in both tools.

    Parameters
    ----------
    folder_name: path
        Path to the folder with the json file
    file_name: json file name with extension
        Json to be converted

    Returns
    -------
    Stores converted json file to current dict

    Usage:
        `import multi_vector_simulator.utils.helpers as helpers`
        `helpers.translates_epa_strings_to_mvs_readable("./epa_benchmark", "epa_benchmark.json-original")`
    """
    import json
    from multi_vector_simulator.utils.data_parser import convert_epa_params_to_mvs

    with open(os.path.join(folder_name, file_name)) as json_file:
        epa_dict = json.load(json_file)

    dict_values = convert_epa_params_to_mvs(epa_dict)

    with open(os.path.join(folder_name, "mvs_config.json"), "w") as json_file:
        json.dump(dict_values, json_file, indent=4)


def get_item_if_list(list_or_float, index):
    if isinstance(list_or_float, list):
        answer = list_or_float[index]
    else:
        answer = list_or_float
    return answer


def get_length_if_list(list_or_float):
    if isinstance(list_or_float, list):
        answer = len(list_or_float)
    else:
        answer = 0
    return answer


def peak_demand_bus_name(dso_name: str, feedin: bool = False):
    """Name for auto created bus related to peak demand pricing period"""

    if feedin is False:
        dso_direction = DSO_CONSUMPTION
    else:
        dso_direction = DSO_FEEDIN

    return (
        f"{dso_name}{dso_direction}_{DSO_PEAK_DEMAND_SUFFIX} {AUTO_CREATED_HIGHLIGHT}"
    )


def peak_demand_transformer_name(
    dso_name: str, peak_number: int = None, feedin: bool = False
):
    """Name for auto created bus related to peak demand pricing period"""
    if feedin is False:
        dso_direction = DSO_CONSUMPTION
    else:
        dso_direction = DSO_FEEDIN
    transformer_name = f"{dso_name}{dso_direction}{DSO_PEAK_DEMAND_PERIOD}"
    if peak_number is not None:
        transformer_name = f"{transformer_name}_{str(peak_number)}"

    return f"{transformer_name} {AUTO_CREATED_HIGHLIGHT}"


def get_asset_types(dict_values):
    """Function which returns records of assets in the energy system"""
    asset_types = []
    for asset_group in (
        ENERGY_CONSUMPTION,
        ENERGY_CONVERSION,
        ENERGY_STORAGE,
        ENERGY_PRODUCTION,
        ENERGY_PROVIDERS,
    ):
        for asset_name, asset_params in dict_values.get(asset_group, {}).items():
            asset_type = {"label": asset_name}
            for param in (OEMOF_ASSET_TYPE, TYPE_ASSET):
                asset_type[param] = asset_params.get(param)
            asset_busses = {}
            input_bus = asset_params.get(INFLOW_DIRECTION)
            if input_bus is not None:
                if not isinstance(input_bus, list):
                    # print("not a list :", input_bus)
                    input_bus = [input_bus]
            else:
                input_bus = []

            output_bus = asset_params.get(OUTFLOW_DIRECTION)
            if output_bus is not None:
                if not isinstance(output_bus, list):
                    # print("not a list: ", output_bus)
                    output_bus = [output_bus]
            else:
                output_bus = []

            for bus in input_bus + output_bus:
                asset_busses[bus] = dict_values[ENERGY_BUSSES][bus].get(ENERGY_VECTOR)
            asset_type["busses"] = asset_busses
            asset_types.append(asset_type)
    return asset_types
