import subprocess
import pkgutil
import os
from pkg_resources import resource_filename
import sys

import NaxToPy.Core.Reference_Finder.__Reference_Finder as RF
from NaxToPy.Core.Errors.N2PLog import N2PLog


# Funcion que busca el path del icono de NaxToPy -----------------------------------------------------------------------
def __ico_path() -> str:
    """ Funcion oculta que busca el icono de NaxToPy

        Returns: 
            icon_path: str
    """
    try:
        ico_path = resource_filename('mypackage', 'icon.ico')
        N2PLog.Info.I200()
        return ico_path

    except:
        search_path = r"C:\GIT_REPOSITORIES\NAXTO\NaxToPy\v.1.0\NaxToPy"
        filename = "NaxToPy.ico"
        for root, directory, files in os.walk(search_path):
            if filename in files:
                N2PLog.Info.I200()
                return str(os.path.join(search_path, filename))

    N2PLog.Error.E304()
    return -1
# ----------------------------------------------------------------------------------------------------------------------


# Funcion que busca si se usa el paquete NaxToPy -----------------------------------------------------------------------
def __busca_n2p(path) -> bool:
    """ Funcion oculta que busca si se usa el modulo NaxToPy en los archivos que se quieren convertir en ejecutables
    """
    with open(path) as f:
        if 'import NaxToPy' in f.read():
            return True
        else:
            return False
# ----------------------------------------------------------------------------------------------------------------------


# Funcion oculta que devuelve la lista de las bibliotecas de las que hace uso NaxToPy ----------------------------------
def __list_ext_libs(libs_path, solver, abq_ver) -> list[str]:
    """ Funcion oculta que devuelve la lista de las bibliotecas de las que hace uso NaxToPy.

        Estan donde NaxToView está instalado. Las devulve con el formato listo para añadirse a la llamada de PyInstaller

        Args:
            libs_path: str

        Returns: 
            dll_list: list
    """
    import NaxToPy.Core.Constants.librerias as lb

    if isinstance(abq_ver, str):
        abq_ver = [abq_ver]

    i = -1
    libs = list()
    directions = list()
    # Bucle que busca todas las versiones de abaqus que hay disponibles para NaxTo y las guarda en directions.
    # También guarda en libs todas las dll de NaxTo para que se guarden en el .exe
    for root, dirnames, filenames in os.walk(libs_path):
        i += 1
        libs += filenames

        if i == 1:
            directions += dirnames
            break

    if abq_ver is None or all(items in directions for items in abq_ver):
        pass
    else:
        N2PLog.Critical.C105(lb.ABAQUS_VERSION)
        # sys.exit(f"C105: THE ARGUMENT ABAQUSVERSION MUST BE ONE OF THE AVAILABLE: {lb.ABAQUS_VERSION}")

    #abq_ver = abq_ver.split(" ")
    abq_libs = list()

    dll_list = list()
    for dll in libs:
        dll_list.append('--add-data')
        dll_list.append(libs_path + "\\" + dll + r";./libraries")

    if abq_ver == "" or abq_ver is None or abq_ver == [""] or abq_ver == []:
        abq_ver = []

    elif abq_ver == ["ALL"] or abq_ver == ["all"]:
        for dirs in directions:
            abq_libs.append("ABQS\\" + dirs)

    else:
        for abq in abq_ver:
            abq_libs.append("ABQS\\" + abq)

    dll_abq_list = list()
    for dll in abq_libs:
        dll_abq_list.append('--add-data')
        dll_abq_list.append(libs_path + "\\" + dll + r";./libraries" + f"\\{dll}")

    dll_list += dll_abq_list

    return dll_list
# ----------------------------------------------------------------------------------------------------------------------


# Funcion principal que usa PyInstaller para generar el .exe -----------------------------------------------------------
def n2ptoexe(path: str, console: bool = True, solver: str = "ALL", abaqusversion: list[str] = None) -> None:
    """Function that creates .exe files of programs that use NaxToPy

    Args:
        path: str -> path of the module that will be used to create the .exe file

        console: bool -> If True (Default), the .exe will opne a console of python.
                         If False, the .exe will not open any python console.

        solver: srt | list[str] -> Default="ALL" Solver or list of solvers that the module will work with.
                                   Posible solvers are:  "NASTRAN", "ABAQUS", "ANSYS", "OPTISTRUCT"

        abaqusversion: str | list[srt] -> Default='2022'. Only when "ABAQUS" is selected a str or a list of
                                          ABAQUS version are aviable. Posible abaqus versions:
                                          ['2016', '2017', '2018', '2019', '2020', '2021', '2022', '6.11', '6.12',
                                          '6.13', '6.14']
    """

    # Paso todoo a mayusculas para que no haya errores
    solver.upper()

    icon = __ico_path()

    try:
        if pkgutil.find_loader("PyInstaller") is None:
            subprocess.call(['pip', 'install', "PyInstaller"])

        import PyInstaller.__main__ as PyIns
    except:
        N2PLog.Critical.C106()
        # sys.exit("C106: THE PACKAGE PyInstaller COULDN'T BE LOADED. PLEASE INSTALL IT MANUALLY")

    # Si queremos que el ejecutable saque una consola de python. Si se piden cosas mediante la funcion input(), o
    # queremos sacar cosas por consola con un print() habrá que marcar console. Si funciona todo por interfaz grafica,
    # usando por ejemplo tkinter o no es necesario interactuar en absoluto es mejor marcar windowed (sin consola).

    if console == True:
        consl = "-c"
    elif console == False:
        consl = "-w"
    else:
        N2PLog.Critical.C104()
        # sys.exit("C104: THE ARGUMENT CONSOLE MUST BE A BOOLEAN: True | False")


    if __busca_n2p(path):

        libs_paths = RF.vizzer_libs()
        libs_path = RF.compatible_vizzer(libs_paths)[:-18]
        dll_list = __list_ext_libs(libs_path, solver, abaqusversion)
        

        # Aqui se cargan las propiedades de la llamada a PyInstaler
        pyinstaller_cmd = [path] + [consl]
        pyinstaller_cmd += dll_list
        if icon == -1:
            pyinstaller_cmd += ["--onefile", "--collect-all", "NaxToPy"]  # "--onefile",
            # pyinstaller_cmd += ["--collect-all", "NaxToPy"]  # "--onefile",
            N2PLog.Warning.W200()
        else:
            pyinstaller_cmd += ["--onefile", "--collect-all", "NaxToPy", '--icon', icon]  # "--onefile",
            # pyinstaller_cmd += ["--collect-all", "NaxToPy", '--icon', icon]  # "--onefile",

        PyIns.run(pyinstaller_cmd)
        N2PLog.Info.I201(os.getcwd())
    else:
        N2PLog.Error.E110(path)

# ----------------------------------------------------------------------------------------------------------------------
