"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const cognito_generated_1 = require("./cognito.generated");
const user_pool_client_1 = require("./user-pool-client");
const user_pool_domain_1 = require("./user-pool-domain");
/**
 * User pool operations to which lambda triggers can be attached.
 */
class UserPoolOperation {
    constructor(operationName) {
        this.operationName = operationName;
    }
    /** A custom user pool operation */
    static of(name) {
        const lowerCamelCase = name.charAt(0).toLowerCase() + name.slice(1);
        return new UserPoolOperation(lowerCamelCase);
    }
}
exports.UserPoolOperation = UserPoolOperation;
/**
 * Creates a challenge in a custom auth flow
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-create-auth-challenge.html
 */
UserPoolOperation.CREATE_AUTH_CHALLENGE = new UserPoolOperation('createAuthChallenge');
/**
 * Advanced customization and localization of messages
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-custom-message.html
 */
UserPoolOperation.CUSTOM_MESSAGE = new UserPoolOperation('customMessage');
/**
 * Determines the next challenge in a custom auth flow
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-define-auth-challenge.html
 */
UserPoolOperation.DEFINE_AUTH_CHALLENGE = new UserPoolOperation('defineAuthChallenge');
/**
 * Event logging for custom analytics
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-post-authentication.html
 */
UserPoolOperation.POST_AUTHENTICATION = new UserPoolOperation('postAuthentication');
/**
 * Custom welcome messages or event logging for custom analytics
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-post-confirmation.html
 */
UserPoolOperation.POST_CONFIRMATION = new UserPoolOperation('postConfirmation');
/**
 * Custom validation to accept or deny the sign-in request
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-authentication.html
 */
UserPoolOperation.PRE_AUTHENTICATION = new UserPoolOperation('preAuthentication');
/**
 * Custom validation to accept or deny the sign-up request
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-sign-up.html
 */
UserPoolOperation.PRE_SIGN_UP = new UserPoolOperation('preSignUp');
/**
 * Add or remove attributes in Id tokens
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-token-generation.html
 */
UserPoolOperation.PRE_TOKEN_GENERATION = new UserPoolOperation('preTokenGeneration');
/**
 * Migrate a user from an existing user directory to user pools
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-migrate-user.html
 */
UserPoolOperation.USER_MIGRATION = new UserPoolOperation('userMigration');
/**
 * Determines if a response is correct in a custom auth flow
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-verify-auth-challenge-response.html
 */
UserPoolOperation.VERIFY_AUTH_CHALLENGE_RESPONSE = new UserPoolOperation('verifyAuthChallengeResponse');
/**
 * The email verification style
 */
var VerificationEmailStyle;
(function (VerificationEmailStyle) {
    /** Verify email via code */
    VerificationEmailStyle["CODE"] = "CONFIRM_WITH_CODE";
    /** Verify email via link */
    VerificationEmailStyle["LINK"] = "CONFIRM_WITH_LINK";
})(VerificationEmailStyle = exports.VerificationEmailStyle || (exports.VerificationEmailStyle = {}));
/**
 * The different ways in which a user pool's MFA enforcement can be configured.
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-mfa.html
 */
var Mfa;
(function (Mfa) {
    /** Users are not required to use MFA for sign in, and cannot configure one. */
    Mfa["OFF"] = "OFF";
    /** Users are not required to use MFA for sign in, but can configure one if they so choose to. */
    Mfa["OPTIONAL"] = "OPTIONAL";
    /** Users are required to configure an MFA, and have to use it to sign in. */
    Mfa["REQUIRED"] = "ON";
})(Mfa = exports.Mfa || (exports.Mfa = {}));
/**
 * Define a Cognito User Pool
 */
class UserPool extends core_1.Resource {
    constructor(scope, id, props = {}) {
        var _a, _b, _c, _d, _e;
        super(scope, id);
        this.triggers = {};
        const signIn = this.signInConfiguration(props);
        if (props.lambdaTriggers) {
            for (const t of Object.keys(props.lambdaTriggers)) {
                const trigger = props.lambdaTriggers[t];
                if (trigger !== undefined) {
                    this.addLambdaPermission(trigger, t);
                    this.triggers[t] = trigger.functionArn;
                }
            }
        }
        const verificationMessageTemplate = this.verificationMessageConfiguration(props);
        let emailVerificationMessage;
        let emailVerificationSubject;
        if (verificationMessageTemplate.defaultEmailOption === VerificationEmailStyle.CODE) {
            emailVerificationMessage = verificationMessageTemplate.emailMessage;
            emailVerificationSubject = verificationMessageTemplate.emailSubject;
        }
        const smsVerificationMessage = verificationMessageTemplate.smsMessage;
        const inviteMessageTemplate = {
            emailMessage: (_a = props.userInvitation) === null || _a === void 0 ? void 0 : _a.emailBody,
            emailSubject: (_b = props.userInvitation) === null || _b === void 0 ? void 0 : _b.emailSubject,
            smsMessage: (_c = props.userInvitation) === null || _c === void 0 ? void 0 : _c.smsMessage,
        };
        const selfSignUpEnabled = props.selfSignUpEnabled !== undefined ? props.selfSignUpEnabled : false;
        const adminCreateUserConfig = {
            allowAdminCreateUserOnly: !selfSignUpEnabled,
            inviteMessageTemplate: props.userInvitation !== undefined ? inviteMessageTemplate : undefined,
        };
        const passwordPolicy = this.configurePasswordPolicy(props);
        const userPool = new cognito_generated_1.CfnUserPool(this, 'Resource', {
            userPoolName: props.userPoolName,
            usernameAttributes: signIn.usernameAttrs,
            aliasAttributes: signIn.aliasAttrs,
            autoVerifiedAttributes: signIn.autoVerifyAttrs,
            lambdaConfig: core_1.Lazy.anyValue({ produce: () => undefinedIfNoKeys(this.triggers) }),
            smsConfiguration: this.smsConfiguration(props),
            adminCreateUserConfig,
            emailVerificationMessage,
            emailVerificationSubject,
            smsVerificationMessage,
            verificationMessageTemplate,
            schema: this.schemaConfiguration(props),
            mfaConfiguration: props.mfa,
            enabledMfas: this.mfaConfiguration(props),
            policies: passwordPolicy !== undefined ? { passwordPolicy } : undefined,
            emailConfiguration: undefinedIfNoKeys({
                from: (_d = props.emailSettings) === null || _d === void 0 ? void 0 : _d.from,
                replyToEmailAddress: (_e = props.emailSettings) === null || _e === void 0 ? void 0 : _e.replyTo,
            }),
        });
        this.userPoolId = userPool.ref;
        this.userPoolArn = userPool.attrArn;
        this.userPoolProviderName = userPool.attrProviderName;
        this.userPoolProviderUrl = userPool.attrProviderUrl;
    }
    /**
     * Import an existing user pool based on its id.
     */
    static fromUserPoolId(scope, id, userPoolId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.userPoolId = userPoolId;
                this.userPoolArn = core_1.Stack.of(this).formatArn({
                    service: 'cognito-idp',
                    resource: 'userpool',
                    resourceName: userPoolId,
                });
            }
            addClient(clientId, options) {
                return new user_pool_client_1.UserPoolClient(this, clientId, {
                    userPool: this,
                    ...options,
                });
            }
        }
        return new Import(scope, id);
    }
    /**
     * Import an existing user pool based on its ARN.
     */
    static fromUserPoolArn(scope, id, userPoolArn) {
        return UserPool.fromUserPoolId(scope, id, core_1.Stack.of(scope).parseArn(userPoolArn).resourceName);
    }
    /**
     * Add a lambda trigger to a user pool operation
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html
     */
    addTrigger(operation, fn) {
        if (operation.operationName in this.triggers) {
            throw new Error(`A trigger for the operation ${operation} already exists.`);
        }
        this.addLambdaPermission(fn, operation.operationName);
        this.triggers[operation.operationName] = fn.functionArn;
    }
    /**
     * Add a new app client to this user pool.
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-client-apps.html
     */
    addClient(id, options) {
        return new user_pool_client_1.UserPoolClient(this, id, {
            userPool: this,
            ...options,
        });
    }
    /**
     * Associate a domain to this user pool.
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-assign-domain.html
     */
    addDomain(id, options) {
        return new user_pool_domain_1.UserPoolDomain(this, id, {
            userPool: this,
            ...options,
        });
    }
    addLambdaPermission(fn, name) {
        const capitalize = name.charAt(0).toUpperCase() + name.slice(1);
        fn.addPermission(`${capitalize}Cognito`, {
            principal: new aws_iam_1.ServicePrincipal('cognito-idp.amazonaws.com'),
            sourceArn: this.userPoolArn,
        });
    }
    verificationMessageConfiguration(props) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k;
        const CODE_TEMPLATE = '{####}';
        const VERIFY_EMAIL_TEMPLATE = '{##Verify Email##}';
        const emailStyle = (_b = (_a = props.userVerification) === null || _a === void 0 ? void 0 : _a.emailStyle) !== null && _b !== void 0 ? _b : VerificationEmailStyle.CODE;
        const emailSubject = (_d = (_c = props.userVerification) === null || _c === void 0 ? void 0 : _c.emailSubject) !== null && _d !== void 0 ? _d : 'Verify your new account';
        const smsMessage = (_f = (_e = props.userVerification) === null || _e === void 0 ? void 0 : _e.smsMessage) !== null && _f !== void 0 ? _f : `The verification code to your new account is ${CODE_TEMPLATE}`;
        if (emailStyle === VerificationEmailStyle.CODE) {
            const emailMessage = (_h = (_g = props.userVerification) === null || _g === void 0 ? void 0 : _g.emailBody) !== null && _h !== void 0 ? _h : `The verification code to your new account is ${CODE_TEMPLATE}`;
            if (emailMessage.indexOf(CODE_TEMPLATE) < 0) {
                throw new Error(`Verification email body must contain the template string '${CODE_TEMPLATE}'`);
            }
            if (smsMessage.indexOf(CODE_TEMPLATE) < 0) {
                throw new Error(`SMS message must contain the template string '${CODE_TEMPLATE}'`);
            }
            return {
                defaultEmailOption: VerificationEmailStyle.CODE,
                emailMessage,
                emailSubject,
                smsMessage,
            };
        }
        else {
            const emailMessage = (_k = (_j = props.userVerification) === null || _j === void 0 ? void 0 : _j.emailBody) !== null && _k !== void 0 ? _k : `Verify your account by clicking on ${VERIFY_EMAIL_TEMPLATE}`;
            if (emailMessage.indexOf(VERIFY_EMAIL_TEMPLATE) < 0) {
                throw new Error(`Verification email body must contain the template string '${VERIFY_EMAIL_TEMPLATE}'`);
            }
            return {
                defaultEmailOption: VerificationEmailStyle.LINK,
                emailMessageByLink: emailMessage,
                emailSubjectByLink: emailSubject,
                smsMessage,
            };
        }
    }
    signInConfiguration(props) {
        var _a;
        let aliasAttrs;
        let usernameAttrs;
        let autoVerifyAttrs;
        const signIn = (_a = props.signInAliases) !== null && _a !== void 0 ? _a : { username: true };
        if (signIn.preferredUsername && !signIn.username) {
            throw new Error('username signIn must be enabled if preferredUsername is enabled');
        }
        if (signIn.username) {
            aliasAttrs = [];
            if (signIn.email) {
                aliasAttrs.push("email" /* EMAIL */);
            }
            if (signIn.phone) {
                aliasAttrs.push("phone_number" /* PHONE_NUMBER */);
            }
            if (signIn.preferredUsername) {
                aliasAttrs.push("preferred_username" /* PREFERRED_USERNAME */);
            }
            if (aliasAttrs.length === 0) {
                aliasAttrs = undefined;
            }
        }
        else {
            usernameAttrs = [];
            if (signIn.email) {
                usernameAttrs.push("email" /* EMAIL */);
            }
            if (signIn.phone) {
                usernameAttrs.push("phone_number" /* PHONE_NUMBER */);
            }
        }
        if (props.autoVerify) {
            autoVerifyAttrs = [];
            if (props.autoVerify.email) {
                autoVerifyAttrs.push("email" /* EMAIL */);
            }
            if (props.autoVerify.phone) {
                autoVerifyAttrs.push("phone_number" /* PHONE_NUMBER */);
            }
        }
        else if (signIn.email || signIn.phone) {
            autoVerifyAttrs = [];
            if (signIn.email) {
                autoVerifyAttrs.push("email" /* EMAIL */);
            }
            if (signIn.phone) {
                autoVerifyAttrs.push("phone_number" /* PHONE_NUMBER */);
            }
        }
        return { usernameAttrs, aliasAttrs, autoVerifyAttrs };
    }
    smsConfiguration(props) {
        var _a;
        if (props.smsRole) {
            return {
                snsCallerArn: props.smsRole.roleArn,
                externalId: props.smsRoleExternalId,
            };
        }
        else {
            const smsRoleExternalId = this.node.uniqueId.substr(0, 1223); // sts:ExternalId max length of 1224
            const smsRole = (_a = props.smsRole) !== null && _a !== void 0 ? _a : new aws_iam_1.Role(this, 'smsRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('cognito-idp.amazonaws.com', {
                    conditions: {
                        StringEquals: { 'sts:ExternalId': smsRoleExternalId },
                    },
                }),
                inlinePolicies: {
                    /*
                     * The UserPool is very particular that it must contain an 'sns:Publish' action as an inline policy.
                     * Ideally, a conditional that restricts this action to 'sms' protocol needs to be attached, but the UserPool deployment fails validation.
                     * Seems like a case of being excessively strict.
                     */
                    'sns-publish': new aws_iam_1.PolicyDocument({
                        statements: [
                            new aws_iam_1.PolicyStatement({
                                actions: ['sns:Publish'],
                                resources: ['*'],
                            }),
                        ],
                    }),
                },
            });
            return {
                externalId: smsRoleExternalId,
                snsCallerArn: smsRole.roleArn,
            };
        }
    }
    mfaConfiguration(props) {
        if (props.mfa === undefined || props.mfa === Mfa.OFF) {
            // since default is OFF, treat undefined and OFF the same way
            return undefined;
        }
        else if (props.mfaSecondFactor === undefined &&
            (props.mfa === Mfa.OPTIONAL || props.mfa === Mfa.REQUIRED)) {
            return ['SMS_MFA'];
        }
        else {
            const enabledMfas = [];
            if (props.mfaSecondFactor.sms) {
                enabledMfas.push('SMS_MFA');
            }
            if (props.mfaSecondFactor.otp) {
                enabledMfas.push('SOFTWARE_TOKEN_MFA');
            }
            return enabledMfas;
        }
    }
    configurePasswordPolicy(props) {
        var _a, _b, _c, _d, _e, _f;
        const tempPasswordValidity = (_a = props.passwordPolicy) === null || _a === void 0 ? void 0 : _a.tempPasswordValidity;
        if (tempPasswordValidity !== undefined && tempPasswordValidity.toDays() > core_1.Duration.days(365).toDays()) {
            throw new Error(`tempPasswordValidity cannot be greater than 365 days (received: ${tempPasswordValidity.toDays()})`);
        }
        const minLength = props.passwordPolicy ? (_b = props.passwordPolicy.minLength) !== null && _b !== void 0 ? _b : 8 : undefined;
        if (minLength !== undefined && (minLength < 6 || minLength > 99)) {
            throw new Error(`minLength for password must be between 6 and 99 (received: ${minLength})`);
        }
        return undefinedIfNoKeys({
            temporaryPasswordValidityDays: tempPasswordValidity === null || tempPasswordValidity === void 0 ? void 0 : tempPasswordValidity.toDays({ integral: true }),
            minimumLength: minLength,
            requireLowercase: (_c = props.passwordPolicy) === null || _c === void 0 ? void 0 : _c.requireLowercase,
            requireUppercase: (_d = props.passwordPolicy) === null || _d === void 0 ? void 0 : _d.requireUppercase,
            requireNumbers: (_e = props.passwordPolicy) === null || _e === void 0 ? void 0 : _e.requireDigits,
            requireSymbols: (_f = props.passwordPolicy) === null || _f === void 0 ? void 0 : _f.requireSymbols,
        });
    }
    schemaConfiguration(props) {
        const schema = [];
        if (props.requiredAttributes) {
            const stdAttributes = [];
            if (props.requiredAttributes.address) {
                stdAttributes.push("address" /* ADDRESS */);
            }
            if (props.requiredAttributes.birthdate) {
                stdAttributes.push("birthdate" /* BIRTHDATE */);
            }
            if (props.requiredAttributes.email) {
                stdAttributes.push("email" /* EMAIL */);
            }
            if (props.requiredAttributes.familyName) {
                stdAttributes.push("family_name" /* FAMILY_NAME */);
            }
            if (props.requiredAttributes.fullname) {
                stdAttributes.push("name" /* NAME */);
            }
            if (props.requiredAttributes.gender) {
                stdAttributes.push("gender" /* GENDER */);
            }
            if (props.requiredAttributes.givenName) {
                stdAttributes.push("given_name" /* GIVEN_NAME */);
            }
            if (props.requiredAttributes.lastUpdateTime) {
                stdAttributes.push("updated_at" /* LAST_UPDATE_TIME */);
            }
            if (props.requiredAttributes.locale) {
                stdAttributes.push("locale" /* LOCALE */);
            }
            if (props.requiredAttributes.middleName) {
                stdAttributes.push("middle_name" /* MIDDLE_NAME */);
            }
            if (props.requiredAttributes.nickname) {
                stdAttributes.push("nickname" /* NICKNAME */);
            }
            if (props.requiredAttributes.phoneNumber) {
                stdAttributes.push("phone_number" /* PHONE_NUMBER */);
            }
            if (props.requiredAttributes.preferredUsername) {
                stdAttributes.push("preferred_username" /* PREFERRED_USERNAME */);
            }
            if (props.requiredAttributes.profilePage) {
                stdAttributes.push("profile" /* PROFILE_URL */);
            }
            if (props.requiredAttributes.profilePicture) {
                stdAttributes.push("picture" /* PICTURE_URL */);
            }
            if (props.requiredAttributes.timezone) {
                stdAttributes.push("zoneinfo" /* TIMEZONE */);
            }
            if (props.requiredAttributes.website) {
                stdAttributes.push("website" /* WEBSITE */);
            }
            schema.push(...stdAttributes.map((attr) => {
                return { name: attr, required: true };
            }));
        }
        if (props.customAttributes) {
            const customAttrs = Object.keys(props.customAttributes).map((attrName) => {
                var _a, _b, _c, _d, _e, _f, _g, _h;
                const attrConfig = props.customAttributes[attrName].bind();
                const numberConstraints = {
                    minValue: (_b = (_a = attrConfig.numberConstraints) === null || _a === void 0 ? void 0 : _a.min) === null || _b === void 0 ? void 0 : _b.toString(),
                    maxValue: (_d = (_c = attrConfig.numberConstraints) === null || _c === void 0 ? void 0 : _c.max) === null || _d === void 0 ? void 0 : _d.toString(),
                };
                const stringConstraints = {
                    minLength: (_f = (_e = attrConfig.stringConstraints) === null || _e === void 0 ? void 0 : _e.minLen) === null || _f === void 0 ? void 0 : _f.toString(),
                    maxLength: (_h = (_g = attrConfig.stringConstraints) === null || _g === void 0 ? void 0 : _g.maxLen) === null || _h === void 0 ? void 0 : _h.toString(),
                };
                return {
                    name: attrName,
                    attributeDataType: attrConfig.dataType,
                    numberAttributeConstraints: (attrConfig.numberConstraints) ? numberConstraints : undefined,
                    stringAttributeConstraints: (attrConfig.stringConstraints) ? stringConstraints : undefined,
                    mutable: attrConfig.mutable,
                };
            });
            schema.push(...customAttrs);
        }
        if (schema.length === 0) {
            return undefined;
        }
        return schema;
    }
}
exports.UserPool = UserPool;
function undefinedIfNoKeys(struct) {
    const allUndefined = Object.values(struct).reduce((acc, v) => acc && (v === undefined), true);
    return allUndefined ? undefined : struct;
}
//# sourceMappingURL=data:application/json;base64,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