"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const cognito_generated_1 = require("./cognito.generated");
/**
 * OAuth scopes that are allowed with this client.
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-app-idp-settings.html
 */
class OAuthScope {
    constructor(scopeName) {
        this.scopeName = scopeName;
    }
    /**
     * Custom scope is one that you define for your own resource server in the Resource Servers.
     * The format is 'resource-server-identifier/scope'.
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-define-resource-servers.html
     */
    static custom(name) {
        return new OAuthScope(name);
    }
}
exports.OAuthScope = OAuthScope;
/**
 * Grants access to the 'phone_number' and 'phone_number_verified' claims.
 * Automatically includes access to `OAuthScope.OPENID`.
 */
OAuthScope.PHONE = new OAuthScope('phone');
/**
 * Grants access to the 'email' and 'email_verified' claims.
 * Automatically includes access to `OAuthScope.OPENID`.
 */
OAuthScope.EMAIL = new OAuthScope('email');
/**
 * Returns all user attributes in the ID token that are readable by the client
 */
OAuthScope.OPENID = new OAuthScope('openid');
/**
 * Grants access to all user attributes that are readable by the client
 * Automatically includes access to `OAuthScope.OPENID`.
 */
OAuthScope.PROFILE = new OAuthScope('profile');
/**
 * Grants access to Amazon Cognito User Pool API operations that require access tokens,
 * such as UpdateUserAttributes and VerifyUserAttribute.
 */
OAuthScope.COGNITO_ADMIN = new OAuthScope('aws.cognito.signin.user.admin');
/**
 * Define a UserPool App Client
 */
class UserPoolClient extends core_1.Resource {
    /*
     * Note to implementers: Two CloudFormation return values Name and ClientSecret are part of the spec.
     * However, they have been explicity not implemented here. They are not documented in CloudFormation, and
     * CloudFormation returns the following the string when these two attributes are 'GetAtt' - "attribute not supported
     * at this time, please use the CLI or Console to retrieve this value".
     * Awaiting updates from CloudFormation.
     */
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id);
        const resource = new cognito_generated_1.CfnUserPoolClient(this, 'Resource', {
            clientName: props.userPoolClientName,
            generateSecret: props.generateSecret,
            userPoolId: props.userPool.userPoolId,
            explicitAuthFlows: this.configureAuthFlows(props),
            allowedOAuthFlows: this.configureOAuthFlows(props.oAuth),
            allowedOAuthScopes: this.configureOAuthScopes(props.oAuth),
            callbackUrLs: (((_a = props.oAuth) === null || _a === void 0 ? void 0 : _a.callbackUrls) && ((_b = props.oAuth) === null || _b === void 0 ? void 0 : _b.callbackUrls.length) > 0) ? (_c = props.oAuth) === null || _c === void 0 ? void 0 : _c.callbackUrls : undefined,
            allowedOAuthFlowsUserPoolClient: props.oAuth ? true : undefined,
            preventUserExistenceErrors: this.configurePreventUserExistenceErrors(props.preventUserExistenceErrors),
        });
        this.userPoolClientId = resource.ref;
        this._userPoolClientName = props.userPoolClientName;
    }
    /**
     * Import a user pool client given its id.
     */
    static fromUserPoolClientId(scope, id, userPoolClientId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.userPoolClientId = userPoolClientId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * The client name that was specified via the `userPoolClientName` property during initialization,
     * throws an error otherwise.
     */
    get userPoolClientName() {
        if (this._userPoolClientName === undefined) {
            throw new Error('userPoolClientName is available only if specified on the UserPoolClient during initialization');
        }
        return this._userPoolClientName;
    }
    configureAuthFlows(props) {
        var _a, _b, _c, _d, _e;
        const authFlows = [];
        if ((_a = props.authFlows) === null || _a === void 0 ? void 0 : _a.userPassword) {
            authFlows.push('ALLOW_USER_PASSWORD_AUTH');
        }
        if ((_b = props.authFlows) === null || _b === void 0 ? void 0 : _b.adminUserPassword) {
            authFlows.push('ALLOW_ADMIN_USER_PASSWORD_AUTH');
        }
        if ((_c = props.authFlows) === null || _c === void 0 ? void 0 : _c.custom) {
            authFlows.push('ALLOW_CUSTOM_AUTH');
        }
        if ((_d = props.authFlows) === null || _d === void 0 ? void 0 : _d.userSrp) {
            authFlows.push('ALLOW_USER_SRP_AUTH');
        }
        if ((_e = props.authFlows) === null || _e === void 0 ? void 0 : _e.refreshToken) {
            authFlows.push('ALLOW_REFRESH_TOKEN_AUTH');
        }
        if (authFlows.length === 0) {
            return undefined;
        }
        return authFlows;
    }
    configureOAuthFlows(oAuth) {
        if ((oAuth === null || oAuth === void 0 ? void 0 : oAuth.flows.authorizationCodeGrant) || (oAuth === null || oAuth === void 0 ? void 0 : oAuth.flows.implicitCodeGrant)) {
            if ((oAuth === null || oAuth === void 0 ? void 0 : oAuth.callbackUrls) === undefined || (oAuth === null || oAuth === void 0 ? void 0 : oAuth.callbackUrls.length) === 0) {
                throw new Error('callbackUrl must be specified when codeGrant or implicitGrant OAuth flows are enabled.');
            }
            if (oAuth === null || oAuth === void 0 ? void 0 : oAuth.flows.clientCredentials) {
                throw new Error('clientCredentials OAuth flow cannot be selected along with codeGrant or implicitGrant.');
            }
        }
        const oAuthFlows = [];
        if (oAuth === null || oAuth === void 0 ? void 0 : oAuth.flows.clientCredentials) {
            oAuthFlows.push('client_credentials');
        }
        if (oAuth === null || oAuth === void 0 ? void 0 : oAuth.flows.implicitCodeGrant) {
            oAuthFlows.push('implicit');
        }
        if (oAuth === null || oAuth === void 0 ? void 0 : oAuth.flows.authorizationCodeGrant) {
            oAuthFlows.push('code');
        }
        if (oAuthFlows.length === 0) {
            return undefined;
        }
        return oAuthFlows;
    }
    configureOAuthScopes(oAuth) {
        const oAuthScopes = new Set(oAuth === null || oAuth === void 0 ? void 0 : oAuth.scopes.map((x) => x.scopeName));
        const autoOpenIdScopes = [OAuthScope.PHONE, OAuthScope.EMAIL, OAuthScope.PROFILE];
        if (autoOpenIdScopes.reduce((agg, s) => agg || oAuthScopes.has(s.scopeName), false)) {
            oAuthScopes.add(OAuthScope.OPENID.scopeName);
        }
        if (oAuthScopes.size > 0) {
            return Array.from(oAuthScopes);
        }
        return undefined;
    }
    configurePreventUserExistenceErrors(prevent) {
        if (prevent === undefined) {
            return undefined;
        }
        return prevent ? 'ENABLED' : 'LEGACY';
    }
}
exports.UserPoolClient = UserPoolClient;
//# sourceMappingURL=data:application/json;base64,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