"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const cdk = require("@aws-cdk/core");
const lib_1 = require("../lib");
const render_util_1 = require("./private/render-util");
describe('Wait State', () => {
    test('wait time from ISO8601 timestamp', () => {
        // GIVEN
        const timestamp = '2025-01-01T00:00:00Z';
        // WHEN
        const waitTime = lib_1.WaitTime.timestamp(timestamp);
        // THEN
        expect(waitTime).toEqual({
            json: {
                Timestamp: '2025-01-01T00:00:00Z',
            },
        });
    });
    test('wait time from seconds path in state object', () => {
        // GIVEN
        const secondsPath = '$.waitSeconds';
        // WHEN
        const waitTime = lib_1.WaitTime.secondsPath(secondsPath);
        // THEN
        expect(waitTime).toEqual({
            json: {
                SecondsPath: '$.waitSeconds',
            },
        });
    });
    test('wait time from timestamp path in state object', () => {
        // GIVEN
        const path = '$.timestampPath';
        // WHEN
        const waitTime = lib_1.WaitTime.timestampPath(path);
        // THEN
        expect(waitTime).toEqual({
            json: {
                TimestampPath: '$.timestampPath',
            },
        });
    });
    test('supports adding a next state', () => {
        // GIVEN
        const stack = new cdk.Stack();
        const chain = new lib_1.Wait(stack, 'myWaitState', {
            time: lib_1.WaitTime.duration(cdk.Duration.seconds(30)),
        });
        // WHEN
        chain.next(new lib_1.Pass(stack, 'final pass', {}));
        // THEN
        expect(render_util_1.render(stack, chain)).toEqual({
            StartAt: 'myWaitState',
            States: {
                'final pass': {
                    End: true,
                    Type: 'Pass',
                },
                'myWaitState': {
                    Next: 'final pass',
                    Seconds: 30,
                    Type: 'Wait',
                },
            },
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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