"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const chain_1 = require("../chain");
const state_graph_1 = require("../state-graph");
const state_type_1 = require("./private/state-type");
const state_1 = require("./state");
/**
 * Returns true if the value passed is a positive integer
 * @param value the value ti validate
 */
exports.isPositiveInteger = (value) => {
    const isFloat = Math.floor(value) !== value;
    const isNotPositiveInteger = value < 0 || value > Number.MAX_SAFE_INTEGER;
    return !isFloat && !isNotPositiveInteger;
};
/**
 * Define a Map state in the state machine
 *
 * A `Map` state can be used to run a set of steps for each element of an input array.
 * A Map state will execute the same steps for multiple entries of an array in the state input.
 *
 * While the Parallel state executes multiple branches of steps using the same input, a Map state
 * will execute the same steps for multiple entries of an array in the state input.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/amazon-states-language-map-state.html
 */
class Map extends state_1.State {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        this.endStates = [this];
        this.maxConcurrency = props.maxConcurrency;
        this.itemsPath = props.itemsPath;
    }
    /**
     * Add retry configuration for this state
     *
     * This controls if and how the execution will be retried if a particular
     * error occurs.
     */
    addRetry(props = {}) {
        super._addRetry(props);
        return this;
    }
    /**
     * Add a recovery handler for this state
     *
     * When a particular error occurs, execution will continue at the error
     * handler instead of failing the state machine execution.
     */
    addCatch(handler, props = {}) {
        super._addCatch(handler.startState, props);
        return this;
    }
    /**
     * Continue normal execution with the given state
     */
    next(next) {
        super.makeNext(next.startState);
        return chain_1.Chain.sequence(this, next);
    }
    /**
     * Define iterator state machine in Map
     */
    iterator(iterator) {
        const name = `Map ${this.stateId} Iterator`;
        super.addIterator(new state_graph_1.StateGraph(iterator.startState, name));
        return this;
    }
    /**
     * Return the Amazon States Language object for this state
     */
    toStateJson() {
        return {
            Type: state_type_1.StateType.MAP,
            Comment: this.comment,
            ResultPath: state_1.renderJsonPath(this.resultPath),
            ...this.renderNextEnd(),
            ...this.renderInputOutput(),
            ...this.renderRetryCatch(),
            ...this.renderIterator(),
            ...this.renderItemsPath(),
            MaxConcurrency: this.maxConcurrency,
        };
    }
    /**
     * Validate this state
     */
    validate() {
        const errors = [];
        if (this.iteration === undefined) {
            errors.push('Map state must have a non-empty iterator');
        }
        if (this.maxConcurrency !== undefined && !exports.isPositiveInteger(this.maxConcurrency)) {
            errors.push('maxConcurrency has to be a positive integer');
        }
        return errors;
    }
    renderItemsPath() {
        return {
            ItemsPath: state_1.renderJsonPath(this.itemsPath),
        };
    }
}
exports.Map = Map;
//# sourceMappingURL=data:application/json;base64,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