from uuid import uuid4
from datetime import datetime as dt

from sqlalchemy import Column, ForeignKey
from sqlalchemy import Integer, String, DateTime
from sqlalchemy.orm import relationship

from metacatalog.db.base import Base


class EntryGroupAssociation(Base):
    __tablename__ = 'nm_entrygroups'

    entry_id = Column(Integer, ForeignKey('entries.id'), primary_key=True)
    group_id = Column(Integer, ForeignKey('entrygroups.id'), primary_key=True)


class EntryGroupType(Base):
    __tablename__ = 'entrygroup_types'

    # columns
    id = Column(Integer, primary_key=True)
    name = Column(String(40), nullable=False)
    description = Column(String, nullable=False)

    # relationships
    entries = relationship("EntryGroup", back_populates='type')

    def to_dict(self, deep=False) -> dict:
        """To dict

        Return the model as a python dictionary.

        Parameters
        ----------
        deep : bool
            If True, all related objects will be included as
            dictionary. Defaults to False

        Returns
        -------
        obj : dict
            The Model as dict

        """
        # base dictionary
        d = dict(
            id=self.id,
            name=self.name,
            description=self.description
        )

        # lazy loading
        if deep:
            d['entries'] = [e.to_dict() for e in self.entries]

        return d

    def __str__(self):
        return "%s <ID=%d>" % (self.name, self.id)


class EntryGroup(Base):
    """
    An EntryGroup is an association object between any number of
    :class:`Entry <metacatalog.models.Entry>` records. The type
    of association is further described by
    :class:`EntryGroupType <EntryGroupType>`.

    Attributes
    ----------
    id : int
        Unique id of the record. If not specified, the database will assign it.
    uuid : str
        .. versionadded:: 0.1.9

        Version 4 UUID string to identify the Entry across installations.
        This field is read-only and will be assigned on creation. It is primarily
        used to export Entry into ISO19115 metadata.
    type : metacatalog.models.EntryGroupType
        The type is most important to give meaning to a group of
        entries. Three types ship with metacatalog ``'Project'``, ``'Composite'`` and ``'Split dataset'``.
    type_id : int
        Foreign key to the EntryGroupType
    title : str
        A descriptive title for the Group. Maximum of 40 letters.
    description : str
        An optional full text description of the EntryGroup. This
        description should contain all information necessary to
        understand the grouping.
    publication : datetime.datetime
        .. versionadded:: 0.1.9

        Autogenerated field giving the UTC timestamp of record creation.
    lastUpdate : datetime.datetime
        .. versionadded: 0.1.9

        Autogenerated field giving the UTC timestamp of last edit.

    """
    __tablename__ = 'entrygroups'

    # columns
    id = Column(Integer, primary_key=True)
    uuid = Column(String(36), nullable=False, default=lambda: str(uuid4()))
    type_id = Column(Integer, ForeignKey('entrygroup_types.id'), nullable=False)
    title = Column(String(250))
    description = Column(String)

    publication = Column(DateTime, default=dt.utcnow)
    lastUpdate = Column(DateTime, default=dt.utcnow, onupdate=dt.utcnow)


    # relationships
    type = relationship("EntryGroupType", back_populates='entries')
    entries = relationship("Entry", secondary="nm_entrygroups", back_populates="associated_groups")

    def to_dict(self, deep=False) -> dict:
        """To dict

        Return the model as a python dictionary.

        Parameters
        ----------
        deep : bool
            If True, all related objects will be included as
            dictionary. Defaults to False

        Returns
        -------
        obj : dict
            The Model as dict

        """
        # base dictionary
        d = dict(
            id=self.id,
            uuid=self.uuid,
            type=self.type.to_dict(deep=False),
            publication=self.publication,
            lastUpdate=self.lastUpdate
        )

        # set optionals
        for attr in ('title', 'description'):
            if hasattr(self, attr) and getattr(self, attr) is not None:
                d[attr] = getattr(self, attr)

        # lazy loading
        if deep:
            d['entries'] = [e.to_dict() for e in self.entries]
        else:
            d['entries'] = [e.uuid for e in self.entries]

        return d

    def __str__(self):
        return "%s%s <ID=%d>" % (
            self.type.name,
            " %s" % self.title[:20] if self.title is not None else '',
            self.id
        )
