<pre class='metadata'>
Title: Bikeshed Documentation
H1: Bikeshed, Your Friendly Spec Generator
Shortname: bikeshed
Level: 1
Status: LS
URL: https://tabatkins.github.io/bikeshed/
Editor: Tab Atkins-Bittner
Repository: tabatkins/bikeshed
Abstract: Bikeshed is a spec-generating tool that takes in lightly-decorated Markdown and spits out a full spec, with cross-spec autolinking, automatic generation of indexes/ToC/etc, and many other features.
Markup Shorthands: css no, markdown yes
Ignored Terms: h1, h2, h3, h4, h5, h6, xmp
</pre>

<pre class=link-defaults>
spec:html; type:element;
	text:a
	text:script
	text:style
	text:title
	text:link
</pre>

<!--
████ ██    ██  ██████  ████████    ███    ██       ██
 ██  ███   ██ ██    ██    ██      ██ ██   ██       ██
 ██  ████  ██ ██          ██     ██   ██  ██       ██
 ██  ██ ██ ██  ██████     ██    ██     ██ ██       ██
 ██  ██  ████       ██    ██    █████████ ██       ██
 ██  ██   ███ ██    ██    ██    ██     ██ ██       ██
████ ██    ██  ██████     ██    ██     ██ ████████ ████████
-->

Installing {#installing}
========================

Using Bikeshed Without A Local Install {#install-no}
----------------------------------------------------

If you use Bikeshed infrequently,
or are okay with requiring a network roundtrip every time you invoke Bikeshed,
you probably want to use [the Bikeshed API instead](https://api.csswg.org/bikeshed/).
In return, the API version is always up-to-date,
so you don't have to remember to update things yourself.
See [[#remote]] for options.

Installing The Right Py3 {#install-py3}
---------------------------------------

Your OS should already be shipping with some version of Python 3.
However, Bikeshed requires Python 3.7,
and that's not always the version installed by default.

You can tell which you're running with:

```bash
python3 --version
```

If it reports 3.7 or later, you're fine.
If it gives 3.5, 3.6, or some lesser version,
you'll need to install 3.7.
This can be a little complicated if you do it directly
(building from source?),
but luckily `pyenv` is now a mature and easy way to do this.

### Installing Pyenv ### {#install-pyenv}

<a href="https://github.com/pyenv/pyenv#installation">https://github.com/pyenv/pyenv#installation</a> gives instructions for installing pyenv on various environments.
It also links to the auto-installer, which can make things even easier.
Windows users have slightly different instructions,
but it links to a windows fork as well.

Note that the final install step currently installs 3.7.8;
that's fine, go ahead and follow it.

If you know your way around pyenv already,
feel free to do what you're used to here.
Otherwise, it's usually easiest to set up 3.7 as your "default" Python:

```bash
pyenv versions
```

will list what versions exist, and then

```bash
pyenv global 3.7.0
```

will set 3.7.0 as your "global" version, used by default.
(Substitute in whatever the name of the version you saw in `versions` was.)

Installing Bikeshed Itself {#install-final}
-------------------------------------------

Once your environment is set up for Python 3.7,
installation is a breeze.
Run the following command:

```bash
pip3 install bikeshed && bikeshed update
```

When this is completed, Bikeshed should be installed,
and the `bikeshed` command should work in your shell.

Note: Depending on your system and python/pip install,
you might instead need to run `pip`, `python -m pip`, or `python3 -m pip`,
instead of the plain `pip`.

Note: If you'd prefer a user-local install rather than a global,
add a `--user` flag to the `pip3 install` invocation.
If you haven't done a user-local install before,
`pip` will warn you about having to add new things to your `$PATH`
before `bikeshed` becomes available to use as a command.

Note: If you're on a Mac,
`bikeshed update` might fail with some errors about certificates.
This is a known issue with the version of Python 3.7 shipped with XCode,
which might be the default on your system.
To fix this, install Python 3.7 with [Brew](https://brew.sh/) instead.

### Installing Bikeshed for Development ### {#install-dev}

If you're installing Bikeshed so you can work on it,
or want to ensure you have the bleeding-edge tip-of-tree version,
the instructions are just a tiny bit more complex.

First, clone the Bikeshed repository:

```bash
git clone https://github.com/tabatkins/bikeshed.git
```

Then navigate to that folder
(by default, it'll be a folder called "bikeshed"
in the current folder you're in)
and run:

```bash
pip3 install -e .
```

This will spam your console with a bunch of install progress.
When it successfully completes,
the `bikeshed` module should be globally available for import in Python,
and a `bikeshed` command should now work in your shell.

Finally,
run the following command to update your data files to the latest version:

```bash
bikeshed update
```

<div class=advisement>
	These install instructions are new,
	and it's possible they may not work on all systems.
	If something doesn't work for you,
	please <a href="https://github.com/tabatkins/bikeshed/issues/new">open an issue</a>
	and let me know what system you're on,
	what you've done so far,
	and what errors you've gotten.
</div>

### Installing With Pipenv ### {#install-pipenv}

If you're a user of `pipenv`,
Bikeshed's folder contains a `Pipfile` and `Pipfile.lock` for you.

Follow the same instructions as above,
but instead of running `pip3 install`,
run:

```bash
pipenv install -e .
pipenv run bikeshed update
```

This will *not* install a `bikeshed` command by default;
instead, you run Bikeshed with `pipenv run bikeshed`.
If you'd like `bikeshed` to work by itself,
either start a pipenv shell with `pipenv shell`,
or add an alias to your machine like:

```bash
bikeshed=pipenv run bikeshed
```


Updating Bikeshed {#updating-bikeshed}
--------------------------------------

To update bikeshed to its latest version at any time,
just run:

```
pip3 install --upgrade bikeshed && bikeshed update
```

This’ll pull the latest version of Bikeshed,
and ensure that you’re looking at the latest version of the data files,
rather than whatever stale version is currently sitting in the repo.

Note: If you did a user-local install up above,
by passing the `--user` flag to `pip3 install`,
pass it here as well to update your local copy;
otherwise this command will install it globally.

Note: If you did an "editable" install up above,
to update it you just run the install command again;
no `--upgrade` flag required.


Travis CI steps {#travis-ci}
----------------------------

To use bikeshed on [Travis CI](https://travis-ci.org/)'s github integration, you'll need the following `.travis.yml` commands:

```
language: python
python:
  - "3.8"
install:
  - pip install bikeshed
  - bikeshed update
script:
  # Invoke bikeshed here, at your own leisure. E.g.:
  - bikeshed spec
```

[Raymond Toy has written a very thoro guide on the full setup you need to auto-build and -publish the generated files to GitHub via Travis, so you only need to check in the source files themselves.](https://github.com/rtoy/auto-deploy-spec)


Building a Bikeshed docker image {#install-docker}
--------------------------------------------------

Note: I don't actually understand Docker or know how to use it,
so this guide is based on one community member's efforts.
If something is wrong or outdated, uh, figure out how to fix it
and tell me so I can fix the docs.

- If needed, download and install [docker community engine](https://hub.docker.com/search/?type=edition&offering=community)

- Build the Bikeshed image from the project's directory:

    ```
    docker build --tag=bikeshed:latest .
    ```

    The above command will cache Bikeshed's datafiles in the Docker layer cache,
    in order to speed up rebuilds.
    If you want to be sure you get the latest datafiles,
    run `bikeshed update` in the container or add the `--no-cache` flag when building the image:

    ```
    docker build --tag=bikeshed:latest --no-cache .
    ```

- Optionally, deploy the Bikeshed image to a docker registry

    For example, on a unix system (for Windows, consider the Windows Subsystem for Linux):

    ```
    TAG=$(docker image inspect --format='{{.Id}}' bikeshed:latest)
    DATE=$(date '+%Y.%m.%d')
    ```

    Login to a docker registry.

    ```
    docker login
    ```

    Tag the image to be pushed as organization/artifact:version

    ```
    docker tag $TAG <organization>/bikeshed:$DATE
    ```

    Push the tagged image to the docker registry

    ```
    docker push <organization>/bikeshed:$DATE
    ```

- Running Bikeshed using a docker image.

    See [[#cli-docker]]


<!--
 ██████  ██     ██ ████████  ██
██    ██ ██     ██ ██     ██ ██
██       ██     ██ ██     ██ ██
██       ██     ██ ████████  ██
██       ██     ██ ██   ██   ██
██    ██ ██     ██ ██    ██  ██
 ██████   ███████  ██     ██ ████████
-->

Invoking Bikeshed Without Installing {#remote}
============================================

While a locally-installed Bikeshed does provide additional functionality,
if all you need to do is process a spec,
there is an API server you can talk to,
maintained by Peter Linss.

Using Curl {#curl}
------------------

These instructions assume use of the `curl` command,
but you can use any equivalent "talk HTTP at a server" command you might have access to.

<div class=example>
	<pre highlight=bash>
	curl https://api.csswg.org/bikeshed/ -F file=@index.bs -F force=1 > index.html
	</pre>

	Simplest possible usage:

	* passing the source as `-F file=@index.bs`
	* forcing output even if there are errors with `-F force=1`
	* piping the output to the desired filename with `> index.html`.

	Additional possible arguments:
	* `-F output=err` to just receive errors, if any
	* `-F md-Foo=bar` to pass a `Foo: bar` metadata to the spec
		(same as passing `--md-Foo=bar` locally)
	* `-F die-on=[nothing, fatal, link-error, warning, everything]`,
		(same as the `--die-on` local flag)
</div>

<div class=example>
	If your source file is online,
	such as in a git repository,
	you can pass the url directly instead:

	<pre highlight=bash>
	curl http://api.csswg.org/bikeshed/ -F url=http://dev.w3.org/csswg/css-syntax/Overview.bs -F force=1 > Overview.html
	</pre>

	If you are using additional files beyond your plain source file--
	local boilerplate files,
	[[#including|sub-document includes]],
	[[#custom-dfns|custom definition files]],
	etc--
	this is the *only* way to make those still work
	via the curl API;
	Bikeshed will look for the additional files
	relative to the source document's url.

	(Any additional files that get included *implicitly*,
	such as local boilerplates,
	must be listed in the [=Local Boilerplate=]
	or [=External Infotrees=] metadatas
	to get picked up.
	Files explicitly listed in the source file,
	such as in a `<pre class=include>`,
	will be picked up automatically.)
</div>

<div class=example>
	The separate `bikeshed issues-list` command
	(for generating issues lists for a W3C Disposition of Comments)
	can also be invoked via curl:

	<pre highlight=bash>
	curl http://api.csswg.org/bikeshed/ -F file=@Issues.txt -F input=issues > Issues.html
	</pre>
</div>


Using the Web Form {#web}
-------------------------

Alternately, you can just visit [https://api.csswg.org/bikeshed/](https://api.csswg.org/bikeshed/) directly,
and upload a file,
point to a URL,
or directly paste in your spec source.

It defaults to outputting both the HTML and the errors,
but you can switch it to one or the other only if you want.


Automatically When You Commit {#commit}
---------------------------------------

See [[#travis-ci]] for a guide on setting up Bikeshed, GitHub, Travis CI
so that you can just commit the source files
and have the processed HTML automatically generated and committed to your GitHub repository.

<!--
 ██████  ██       ████
██    ██ ██        ██
██       ██        ██
██       ██        ██
██       ██        ██
██    ██ ██        ██
 ██████  ████████ ████
-->

Invoking Bikeshed Locally {#cli}
================================

Locally-installed Bikeshed is invoked via the command-line.
There's a bunch of options,
but you'll only use a few in day-to-day use.

If you've installed Bikeshed via one of the standard local installs above,
you can typically just type `bikeshed` in the folder where your `.bs` file lives
and it'll do its thing,
automatically invoking the `spec` subcommand.

However, Bikeshed has a number of different subcommands,
and both global and subcommand-specific flags that can be invoked.

Invoking Bikeshed using a docker image {#cli-docker}
----------------------------------------------------

Note: As stated up in [[#install-docker]],
I don't know how Docker works
and this guide is based on a community member's experience,
so use at your own risk.

Typically, this requires login to the docker registry and pulling a Bikeshed image.

    For some `<organization>` and some `<date>`:

    ```
    docker login
    docker pull <organization>/bikeshed:<date>
    ```

Regardless of host environment's operating system, running Bikeshed from a docker image
requires two things:
- mapping a host directory to a path in the docker image (e.g. `/data`)
- specifying the location of Bikeshed's input and output relative to the path in the docker image (i.e., `/data`)

Example for a Unix host:

    ```
    docker run --rm -v $(pwd):/data <organization>/bikeshed:<date> bikeshed spec /data/<some *.bs file> [/data/<some output file>]
    ```

Example for a Windows host:

    ```
    docker run --rm --volume C:\mystuff\project1\:/data <organization>/bikeshed:<date> spec /data/Index.bs /data/out/Index.html
    ```

Note that the [[#cli-options]] apply to running Bikeshed from a docker image.
Since the Bikeshed docker image is read-only, it does not make sense to execute the Bikeshed `update` command from a docker image.

Global Options {#cli-options}
-----------------------------

There are a few options that apply to many different options,
so they're specified before the individual commands:

: `-h` or `--help`
:: Shows the help message for the current command.
	Can be run without any command,
	like `bikeshed -h`
	to show the list of valid commands and the global flags,
	or after a command,
	like `bikeshed spec -h`
	to show the help text for that command in particular.

: `--die-on = [ nothing | fatal | link-error | warning | everything ]`
:: Bikeshed categorizes errors into several categories,
	depending on their severity:
	clear mistakes are "fatal",
	errors in resolving autolinks are "link-error",
	and things that look a little questionable but are totally okay to generate a document with are "warning".

	The `--die-on` flag controls which sorts of errors
	cause Bikeshed to immediately die
	and exit with a non-0 status
	(indicating the command failed).

	Possible values are "nothing", "fatal", "link-error", "warning", and "everything",
	with each level including all previous levels.
	Defaults to "fatal".

	It's often useful to set this to "nothing" if you just need to push thru and generate a spec,
	or if you're generating a freshly-written or freshly-ported spec that may have several errors,
	where it's useful to see all the errors at once,
	in case some depend on each other.
	The `-f` flag is a shorthand for this.
	(Pairing this with `-q` or `-qq` might be useful,
	to suppress the non-fatal errors temporarily
	and make the output easier to read.)

	Alternately, it can be useful to set it to a level stricter than "fatal"
	if you want to ensure that,
	for example,
	a CI build fails when link errors creep into your document,
	or if you just generally want to ensure that your document builds "cleanly",
	without any warnings at all.
	"everything" is equivalent to "warning",
	but is guaranteed to always die on *all* error messages,
	even if more levels are added in the future.

: `-f` or `--force`
:: A shorthand for `--die-on=nothing`.

: `-q` or `--quiet`
:: The `-q` flag suppresses one level of error messages.
	It can be passed multiple times to suppress additional levels,
	in the order "warnings", then "link errors", then "fatal errors".

: `-s` or `--silent`
:: The `-s` flag suppresses *all* console output from Bikeshed,
	regardless of source.
	(It's more powerful than `-q`,
	as it'll also suppresses things like the success/failure message.)

: `-d` or `--dry-run`
:: The `-d` flag prevents Bikeshed from actually writing anything to disk.
	It does everything else,
	just skips that final "save" operation.

: `--print= [ plain | console | markup ]`
:: Specifies how Bikeshed should output its messages.
	Default is "console",
	which outputs colored text using console color codes.
	"plain" outputs the same text as "console",
	but without any color codes
	(which look like gibberish if you're not outputting to a console).
	"markup" outputs the text in a light markup structure
	(suitable for parsing as HTML),
	for easier parsing of the output by tools.


`bikeshed spec` {#cli-spec}
---------------------------

The `spec` command is the most common one you'll use.
It turns a Bikeshed source file
into an output HTML file.
The rest of this document mostly describes how to format a source file
for use with this command.

Most of the time you can just run `bikeshed spec`
and things will "just work"
as long as your source file has the `.bs` extension.

Relevant flags:

: `--gh-token=TOKEN`
:: If you're using <a>Inline GitHub Issues</a> heavily,
	you might run into the GitHub rate limit.
	You can generate an OAuth token at <a href="https://github.com/settings/tokens">https://github.com/settings/tokens</a>
	and then pass it to Bikeshed via this flag
	to raise your limit considerably.

: `-l` or `--line-numbers`
:: If you're having trouble locating the source of an error in your code,
	run Bikeshed with this flag
	and *almost* all errors will report the source line number they appear at
	(or somewhere very close - it's per-element).

	This code is unfortunately forced to be fairly hacky,
	and has the potential to lightly corrupt your file
	(inserting extra debugging info into plain text
	that just happens to look like markup),
	so it automatically triggers a dry-run,
	as if you'd specified `--dry-run`.
	When you're done debugging,
	just run again without this flag to actually get some output.

After any flags,
you can optionally specify the input file path and output file path.
Both of these can usually be omitted;
if you omit the output file,
it'll just output to a file with the same name as the input file,
but with the extension changed to `.html`;
if you omit the input file,
it'll search thru the current directory
and assume you want the first file with a Bikeshed extension (`.bs`).

If you want to feed Bikeshed from stdin
or have it output to stdout,
just use `-` as the appropriate filename.


`bikeshed watch` {#cli-watch}
-----------------------------

The `watch` command is identical to the `spec` command,
except it sets up a watcher
and auto-rebuilds the spec every time it changes.
With this turned on,
you can edit with a simple save->refresh cycle,
rather than having to do save->build->refresh.

It accepts all the same arguments as `spec`.
(Tho using stdin/stdout might get a little weird.)

Use Ctrl-C to stop the watcher
(or whatever key combo kills the currently running process in your console).

`bikeshed template` {#cli-template}
-----------------------------------

The `template` command outputs a minimal "skeleton" document to stdout.
It's useful to get started with Bikeshed,
without having to remember what all the required metadata is.

On a Linux-like command line,
it should be used like:

```
bikeshed template > index.bs
```

`bikeshed echidna` {#cli-echidna}
---------------------------------

The `echidna` command hooks into the W3C's Echidna auto-publishing system,
letting you publish W3C specs from the command-line,
rather than having to go thru a staff contact.

Note: Currently, only Working Drafts (**not** including FPWD) and Candidate Recommendation Drafts can be published via Echidna.

It's invoked similarly to `bikeshed spec`,
with three required flags:

: `--u USERNAME`
:: Your W3C username

: `--p PASSWORD`
:: Your W3C password

: `--decision DECISION_URL`
:: A link to a publicly-visible message approving the publication of this draft.

After the flags,
you can optionally specify the input file path,
just like for `bikeshed spec`;
if omitted,
it selects a file automatically in the same way.

Bikeshed then builds the spec normally,
performs a little bit of fixup for common issues that PubRules complains about,
and submits the spec to Echidna.
It will print out a URL for you to check on the progress of your publication;
it should complete in a few seconds.

Most likely you'll have some PubRules errors when you check your progress;
fix those
(ask in irc.w3.org#pub for help if needed)
then just run `bikeshed echidna` again.

Note: If auto-publishing via Travis CI,
[this Travis blog post about encrypted environment variables](https://blog.travis-ci.com/2014-08-22-environment-variables)
will be helpful for safely using your W3C password.

### Testing Publication Locally ### {#echidna-testing}

Publishing via Echidna automatically does some "fixup" on your document,
to fix some common errors that would make the doc fail the W3C's PubRules check.

These fixes are controlled by the <a>Prepare For TR</a> metadata,
so if you want to see precisely what your spec will look like when it's published,
add that to your metadata block manually.

If you want to go further,
and see the actual TAR file that will be sent to the Echidna service,
you can pass `--just-tar` to the command
(and omit the authentication flags that are normally required).
This will cause Bikeshed to create a file named `test.tar` in your current directory.

### Echidna Hooks ### {#echidna-hooks}

Bikeshed's default publication behavior is usually sufficient,
but if you need to customize it in some way,
there are a few hooks you can use
(assuming you can write Python).
In an include file named `bs-extensions.include`,
provide an implementation of the following methods
(check the default version of this file for the default implementations,
for the methods you don't want to change):

: `BSPrepTR(doc)`
:: This method is called after processing is done,
	when the document is ready to be packed up and sent to Echidna.
	Here, you can perform whatever final post-processing is required.

	<div class=example>
		For example, the CSSWG EDs link to a stylesheet
		in the parent directory of the server they're stored on.
		In their `BSPrepTR()`,
		they search for the link to that stylesheet,
		and update it to point to the current directory instead.
	</div>

: `BSPublishAdditionalFiles(files)`
:: This method allows you to specify what files should be included in the publishing bundle.
	It must return an array of additional files/folders;
	each entry must either be a string,
	referring to a file/folder in the spec's directory or subdirectories,
	or a tuple of 2 strings,
	the first of which points to a file outside the spec's directory,
	and the second of which provides the path the file should have within the spec's directory.

	The `files` value provides the default values,
	which you probably want to just extend,
	rather than fully replace.
	It defaults to `["images", "diagrams", "examples"]`,
	indicating that those folders,
	if present,
	will be included.

	<div class=example>
		For example, as stated in the example for `BSPrepTR()`,
		the CSSWG needs to include its stylesheet in with its specs.
		To do so,
		it just needs to add the entry `["../default.css", "default.css"]`
		to the `files` array,
		indicating that Bikeshed should grab the `default.css` file from the parent directory,
		and put it in the spec's directory
		(in the bundle)
		with the same name.
	</div>

`bikeshed update` {#cli-update}
-------------------------------

The `update` command updates Bikeshed's datafiles,
which it uses for autolinking and similar things.
By default it'll update all of its datafiles,
but if you want to update only particular ones,
you can pass any or all of the following flags:

* `--anchors` to update Bikeshed's "anchor data" - a list of all the definitions and headers it knows about.
* `--biblio` to update Bikeshed's bibliography database.
* `--caniuse` to update Bikeshed's CanIUse.com information.
* `--link-defaults` to update Bikeshed's manually-maintained list of special linking rules.
* `--test-suites` to update Bikeshed's database of test-suite information from the Shepherd system.
* `--languages` to update Bikeshed's database of language codes/names (used for the [=Translations=] metadata)
* `--wpt` to update Bikeshed's database of WPT tests; see [[#testing]].

By default, Bikeshed's update system relies on the [Bikeshed-Data](http://github.com/tabatkins/bikeshed-data) project,
which preprocesses the various data sources directly into Bikeshed's data formats,
and prepares a manifest of changed files,
letting Bikeshed quickly download only what it needs.
However, this can be up to 10 minutes out of date
(or longer, if the update process has fallen over);
if you need absolutely up-to-date information *right now*,
pass the `--skip-manifest` flag
to force Bikeshed to do its full manual update process.
(You can combine it with any of the flags above to only get the exact files you need.)


`bikeshed refs` {#cli-refs}
---------------------------

The `refs` command lets you search Bikeshed's anchor database,
letting you see what sort of things it's looking at when doing autolinking.
This can be very useful for debugging,
or just for a quick check of where something is defined.

It's flags are similar to the attributes used on autolinks:

* `--text=FOO` to specify the linktext you want to filter for
* `--type=FOO` to specify the link type
* `--for=FOO` to specify the for value
* `--spec=FOO` to specify the spec
* `--status=foo` to specify the status
* `--exact` to turn off the variations that Bikeshed applies to some link texts
	(such as looking for "book" when you search for "books"),
	and just search for the exact text you specified

Any or all of these flags can be specified,
and Bikeshed will display all the refs it can find matching the criteria.


`bikeshed source` {#cli-source}
-------------------------------

The `source` command applies various transformations to the source document itself,
rather than producing a separate output document.
Its options are described in [[#source]].


`bikeshed issues-list` {#cli-issues-list}
-----------------------------------------

The `issues-list` command processes a plain-text document
in an "issues-list" format pioneered by the CSSWG
into an equivalent HTML version.

Relevant flags:

* `-t` outputs a template issues-list file to stdout,
	making it easier to start a new document
	without having to reference an old one to remember the format.
	Use it like `bikeshed issues-list -t > issues-19700101.txt`.

After the command you can pass the input and output filenames.
As usual, one or both can be omitted:
if you omit the output,
it'll write to a file with the same name as the input,
but a `.html` extension;
it you omit the input,
it will look in the current folder
for any files starting with "issues" and ending in ".txt",
and then extract the digits from those filenames
and select the one with the largest number.
If you name your file as suggested above,
with an ISO date,
it'll correctly always choose the latest issues list.

Issue: Define the issues-list format.
The `-t` output is already more than enough to actually work with,
but it would still be good to describe it more fully.


<!--
██     ██ ████████ ████████    ███    ████████     ███    ████████    ███
███   ███ ██          ██      ██ ██   ██     ██   ██ ██      ██      ██ ██
████ ████ ██          ██     ██   ██  ██     ██  ██   ██     ██     ██   ██
██ ███ ██ ██████      ██    ██     ██ ██     ██ ██     ██    ██    ██     ██
██     ██ ██          ██    █████████ ██     ██ █████████    ██    █████████
██     ██ ██          ██    ██     ██ ██     ██ ██     ██    ██    ██     ██
██     ██ ████████    ██    ██     ██ ████████  ██     ██    ██    ██     ██
-->

Metadata {#metadata}
====================

Crucial to the processor's ability to automatically generate most of a spec's boilerplate is a small metadata section,
typically located at the top of a file.

A metadata block is just a <code>&lt;pre class='metadata'></code> element, with contents like:

```html
Status: UD
TR: http://www.w3.org/TR/css-variables/
ED: http://dev.w3.org/csswg/css-variables/
Shortname: css-variables
Level: 1
Editor: Tab Atkins Jr., Google, http://xanthir.com/contact
Editor: Daniel Glazman, Disruptive Innovations, daniel.glazman@disruptive-innovations.com
Abstract: This module introduces cascading variables as a new primitive value type that is accepted by all CSS properties,
	and custom properties for defining them.
```

The syntax of a metadata block is very simple - it's a line-based format, with each line consisting of a key and a value, separated by a colon.
Or if you're adding multiple lines with the same key, you can just start the subsequent lines with whitespace to have it reuse the last-seen key.

Several keys are required information, and will cause the processor to flag an error if you omit them:

<ul dfn-for=metadata export>
	* <dfn>Title</dfn> is the spec's full title, used to generate both the <{title}> and <{h1}> of the document.
		This can alternately be specified by adding an <{h1}> element as the first line of the spec.

		Note: The optional [=metadata/H1=] metadata lets you specify the text you want for the document's <{h1}>,
		if for some reason you want the <{title}> and <{h1}> to have different text.
		For example, some browsers interpret markup as literal text in <{title}>,
		so you may want to provide "plain text" in the [=metadata/Title=] metadata
		and marked-up text in [=metadata/H1=].
	* <dfn>Status</dfn> is the spec's status. There are a few general abbreviations that can be used any group (or specs without a Group at all), and several that are restricted to particular groups.
		<details>
			<summary>Statuses usable by anyone</summary>

			* DREAM: "A Collection of Interesting Ideas"
			* LS: "Living Standard"
			* LS-COMMIT: "Commit Snapshot"
			* LS-BRANCH: "Branch Snapshot"
			* LD: "Living Document"
			* FINDING: "Finding"
		</details>
		<details>
			<summary>Statuses usable by W3C groups</summary>

			* ED: "Editor's Draft"
			* WD: "W3C Working Draft"
			* FPWD: "W3C First Public Working Draft"
			* LCWD: "W3C Last Call Working Draft"
			* CR: "W3C Candidate Recommendation Snapshot"
			* CRD: "W3C Candidate Recommendation Draft"
			* PR: "W3C Proposed Recommendation"
			* REC: "W3C Recommendation"
			* PER: "W3C Proposed Edited Recommendation"
			* WG-NOTE: "W3C Working Group Note"
			* IG-NOTE: "W3C Interest Group Note"
			* NOTE: "W3C Note" ([unclear what this is used for](https://github.com/w3c/tr-design/issues/124))
			* NOTE-ED: "Editor's Draft" of a Group Note
			* NOTE-WD: "W3C Working Draft" of a Group Note
			* NOTE-FPWD: "W3C First Public Working Draft" of a Group Note
			* MO: "W3C Member-only Draft"
			* UD: "Unofficial Proposal Draft"
			* CG-DRAFT: "Draft Community Group Report"
			* CG-FINAL: "Final Community Group Report"

			Bikeshed has a listing of what Groups are associated with the W3C.
			If your Group isn't on that list,
			it'll complain when you try to use a W3C status.
			Please file a bug on Bikeshed to add your Group,
			and/or prefix your status like `Status: w3c/ED`.
		</details>
		<details>
			<summary>Statuses usable by ISO groups</summary>

			* I: "Issue"
			* DR: "Defect Report"
			* D: "Draft Proposal"
			* P: "Published Proposal"
			* MEET: "Meeting Announcements"
			* RESP: "Records of Response"
			* MIN: "Minutes"
			* ER: "Editor's Report"
			* SD: "Standing Document"
			* PWI: "Preliminary Work Item"
			* NP: "New Proposal"
			* NWIP: "New Work Item Proposal"
			* WD: "Working Draft"
			* CD: "Committee Draft"
			* FCD: "Final Committee Draft"
			* DIS: "Draft International Standard"
			* FDIS: "Final Draft International Standard"
			* PRF: "Proof of a new International Standard"
			* IS: "International Standard"
			* TR: "Technical Report"
			* DTR: "Draft Technical Report"
			* TS: "Technical Specification"
			* DTS: "Draft Technical Specification"
			* PAS: "Publicly Available Specification"
			* TTA: "Technology Trends Assessment"
			* IWA: "International Workshop Agreement"
			* COR: "Technical Corrigendum"
			* GUIDE: "Guidance to Technical Committees"
			* NP-AMD: "New Proposal Amendment"
			* AWI-AMD: "Approved new Work Item Amendment"
			* WD-AMD: "Working Draft Amendment"
			* CD-AMD: "Committee Draft Amendment"
			* PD-AMD: "Proposed Draft Amendment"
			* FPD-AMD: "Final Proposed Draft Amendment"
			* D-AMD: "Draft Amendment"
			* FD-AMD: "Final Draft Amendment"
			* PRF-AMD: "Proof Amendment"
			* AMD: "Amendment"

			Bikeshed has a listing of what Groups are associated with ISO.
			If your Group isn't on that list,
			it'll complain when you try to use an ISO status.
			Please file a bug on Bikeshed to add your Group,
			and/or prefix your status like `Status: iso/ER`.
		</details>
		<details>
			<summary>Statuses usable by the TC39 group</summary>

			* STAGE0
			* STAGE1
			* STAGE2
			* STAGE3
			* STAGE4
		</details>

	* <dfn>ED</dfn> (or its synonym <dfn>URL</dfn>) must contain a link that points to the most current draft
		(typically the "editor's draft", or otherwise the version that's most live and updated).
	* <dfn>Shortname</dfn> must contain the spec's shortname, like "css-lists" or "css-backgrounds".
	* <dfn>Level</dfn> (or <dfn>Revision</dfn>, an alias) must contain the spec's level as an integer or `none` if the spec does not use levels.
	* <dfn>Editor</dfn> must contain an editor's information.
		This has a special format of comma-separated clauses
		the first is required and must contain the editor's name;
		all the rest are optional.

		* If editing a W3C document, your W3C ID (the number at the end of the url when you navigate to <a href="https://www.w3.org/users/myprofile">https://www.w3.org/users/myprofile</a>) can be added as a `w3cid ####` clause.
		* Your affiliation (company you work for, etc), optionally followed by a link to their homepage, like `Example Company http://example.com`. If your affiliation contains a comma, HTML-escape it (`&#44;`).
		* Your email address.
		* Your homepage.

		All of the optional clauses can occur in any order, except that affiliation must occur before email or homepage.

		Multiple <a>Editor</a> lines can be used to supply multiple editors.
	* <dfn>Abstract</dfn> must contain an abstract for the spec, a 1-2 sentence description of what the spec is about.
		Multiple Abstract lines can be used, representing multiple lines of content, as if you'd written those multiple lines directly into the document.
</ul>

There are several additional optional keys:

<ul dfn-for=metadata export>
	* <dfn>H1</dfn> contains the text/markup that goes into the document's <{h1}> element.
		If omitted, this defaults to the value of the [=metadata/Title=] metadata,
		which is usually what you want.
	* <dfn>TR</dfn> must contain a link that points to the latest version on /TR.
	* <dfn>Canonical URL</dfn> defines the canonical URL (as used by search engines) for the document; it can be set to "TR" (the default value if TR is defined), "ED", or a specific URL.
	* <dfn>Favicon</dfn> defines the favicon URL for the document; it only supports one single URL.
	* <dfn>Logo</dfn> defines the logo url for the document; it only supports one single url.
		This URL can be used via the `[LOGO]` text macro ([[#text-macros]]),
		and is by default used by the `logo` boilerplate section ([[#bp-sections]]).
	* <dfn>Former Editor</dfn> must contain a former editor's information, in the same format as <a>Editor</a>.
	* <dfn>Warning</dfn> must contain either "Obsolete", "Not Ready", "New Version XXX", "Replaced by XXX", "Commit deadb33f http://example.com/url/to/deadb33f replaced by XXX", "Branch XXX http://example.com/url/to/XXX replaced by YYY", or "Custom", which triggers the appropriate warning message in the boilerplate.

		If set to "Custom", the metadata keys <a>Custom Warning Title</a> and <a>Custom Warning Text</a> control what the warning contains.
	* <dfn>Previous Version</dfn> can contain either a URL to a previously-published version (such as versions published on w3.org/TR),
		or the value `from biblio` or `from biblio my-spec-1`.

		Links get displayed literally in the spec's metadata section. "from biblio" entries instead auto-generate a link to the latest dated version of the given versioned shortname that appears in the biblio database. (If omitted, the spec's own versioned shortname is implicitly used.)
	* <dfn>At Risk</dfn> must contain an at-risk feature.  You can specify this key more than once for multiple entries.
	* <dfn>Group</dfn> must contain the name of the group the spec is being generated for.  This is used by the boilerplate generation to select the correct file.  If omitted, it defaults to a generic set of boilerplate.
	* <dfn>Status Text</dfn> allows adding an additional customized sentence that can be used in the document status section.
	* <dfn>Ignored Terms</dfn> accepts a comma-separated list of terms and makes Bikeshed not emit warnings or errors when attempting to autolink those terms.  Use this to quiet spurious preprocessor warnings caused by you inventing terms (for example, the Variables spec invents custom properties like 'var-foo'), or as a temporary patch when the spec you want to link to doesn't set up its definitions correctly.
	* <dfn>Date</dfn> must contain a date in YYYY-MM-DD format, which is used instead of today's date for all the date-related stuff in the spec. Alternately it can contain the string `now`, which explicitly indicates a date of precisely when the output is generated.
	* <dfn>Deadline</dfn> is also a YYYY-MM-DD date, which is used for things like the LCWD Status section, to indicate deadlines.
	* <dfn>Expires</dfn> is also a YYYY-MM-DD date,
		or can be the string `now` to indicate today's date,
		or can be an "ISO Duration" like `P1Y` to indicate "+1 year"
		from the [=Date=] metadata
		(or the spec's generation date if that's not specified),
		and indicates when the document should be considered "expired"--
		before the expiration date an (initially closed) warning about the upcoming expiration is automatically included in the document;
		afterwards the warning defaults to being open and annoying.

		It can also take a [=boolish=] false value or `never` to indicate it never expires;
		this is just the default value,
		so these values are only useful for turning off a default expiration
		established by your default metadata.
	* <dfn>Test Suite</dfn> must contain a link to the test suite (like <code>https://test.csswg.org/suites/css-flexbox-1_dev/nightly-unstable/</code>).

		Note: This merely displays a "Test Suite" item in the spec's header-list,
		as linked text.
		It has no effect on the test suite pop-over that appears in some specs;
		that is instead based on the spec's versioned shortname
		matching an entry in Shepherd's testsuite database.
		(However, it does *override* the header-list Test Suite item
		that otherwise gets automatically added
		if your spec is in Shepherd's testsuite database.)
	* <dfn>Mailing List</dfn> must contain an email address to be used for mailing lists.
	* <dfn>Mailing List Archives</dfn> must contain a link to the list archives.
	* <dfn>Issue Tracking</dfn> indicates what and where you track issues,
		and will result in an "Issue Tracking" entry in your spec header.
		It must contain a comma-separated list of locations,
		each of which consists of the name of the location followed by the url.

		Bikeshed will automatically add additional entries to this in some situations:
		if you use any inline issues,
		you'll automatically get an "In Spec" annotation;
		if you develop from within a git repo with a GH remote,
		or have the <a>Repository</a> metadata set,
		you'll automatically get a "GitHub" annotation.

		You can turn off the spec-header entry entirely with `Boilerplate: feedback-header off`,
		or just turn off the GitHub entry specifically with `Boilerplate: repository-issue-tracking off`.
	* <dfn>Issue Tracker Template</dfn> specifies a url template used to point [remote issues](#remote-issues) at your issue tracker of choice. It takes a url, with `{0}` in the place where you want the remote identifier to go.
	* <dfn>Translation</dfn> indicates a translation of the document. At minimum it consists of a language code (BCP47 format) and a url. Optionally, it can contain comma-separated `name [name-in-spec-language]` and `native-name [name-in-translation-language]`, but if Bikeshed knows about that language, these will be filled in for you.  (If Bikeshed doesn't know about a given language, let me know what its name is in English and itself!)
	* <dfn>Indent</dfn> tells Bikeshed how many spaces you prefer to use to indent your source code,
		so it can properly parse your Markdown and data-blocks.
		It takes a single integer,
		and defaults to `4` if unspecified.
		(Of course, using tabs avoids this entirely,
		as one tab is always one indent.)
	* <dfn>No Editor</dfn> lets you omit the `Editor` metadata without an error.
		It takes a <a>boolish</a> value.
		This shouldn't generally be used;
		even if your organization doesn't privilege editors in any way,
		putting the organization itself in the `Editor` field
		meets the intent while still producing useful information for readers of the spec.
	* <dfn>No Abstract</dfn> lets you omit the `Abstract` metadata without an error.
		It takes a <a>boolish</a> value.
		This should only be used for very small and trivial documents, where the abstract would just be repeating the title of the more-or-less.
	* <dfn>Editor Term</dfn> is a pair of comma-separated terms giving the singular and plural terms to refer to editors with, if you want something other than the default "Editor(s)".
	* <dfn>Default Ref Status</dfn> takes the values "current" or "snapshot",
		and selects which URL you want to default to
		for bibliography and autolinks entries that have both "current" and "snapshot" URLs.
		(You can also specify this per-biblio entry or per-autolink.)
	* <dfn>Default Biblio Display</dfn> takes the values "index" (default) or "inline",
		and selects whether biblio autolinks default to
		displaying as their shortname and linking to the bibliography index,
		or displaying as their title and linking straight to the referenced document.
		(You can also specify this per-biblio entry.)
	* <dfn>Markup Shorthands</dfn> lets you specify which categories of markup shorthands you want to use; for example, you can turn off CSS shorthands and reclaim use of single quotes in your spec.  You can still link to things with explicit markup even if the shorthand is turned off.  Its value is a comma-separated list of markup categories and <a>boolish</a> values, like `css no, biblio yes`.  The currently-recognized categories are:

		* `algorithm`, covering the special <{var}> shorthand (the `|foo|` shorthand)
		* `biblio`, covering the <a>dfn autolinks</a> for biblio links and section links (the `[[foo]]` shorthands)
		* `css`, covering the <a>dfn autolinks</a> for CSS types
		* `dfn`, covering the <a>dfn autolinks</a> for "dfn" and "abstract-op" type definitions
		* `idl`, covering the <a>dfn autolinks</a> for WebIDL types (the `{{foo}}` shorthand)
		* `markdown`, covering the various inline Markdown syntaxes (block-level Markdown is always supported)
		* `markup`, covering the <a>dfn autolinks</a> for HTML/etc elements and attributes (the `<{foo}>` shorthand)

		Everything but `markdown` defaults to "on".
	* <dfn>Text Macro</dfn> lets you specify custom text macros, like `[TITLE]` or `[DATE]`, letting you fill in different text when building a spec in multiple ways.  (This is mostly useful as a command-line option.)  Each <a>Text Macro</a> line consists of a macro name, which must be uppercase and alphanumeric, followed by the text it will get replaced with.
	* <dfn>Work Status</dfn> indicates the status of the document, in a way unrelated to the publication status of <a for=metadata>Status</a>.  It must be one of (completed, stable, testing, refining, revising, exploring, rewriting, abandoned), with those terms defined in [Fantasai's blog](http://fantasai.inkedblade.net/weblog/2011/inside-csswg/process). This just sets the `[WORKSTATUS]` text macro to the corresponding word, used in some of the boilerplates to pipe the metadata to scraping tools.
	* <dfn>Repository</dfn> indicates the repository the spec is being tracked in.
		You can specify a url followed by a "short name" for it,
		and it populates the `[REPOSITORY]` and `[REPOSITORYURL]` text macros accordingly.
		If you are using GitHub,
		you can just specify `username/repo`,
		and Bikeshed will infer the rest for you.
		If you're developing *from within a GitHub repo*,
		you don't need to specify this at all,
		as Bikeshed will figure it out automatically!
		(That said, if you're working in a *fork*,
		you should still specify the main repo's information here,
		or else the spec will point to your fork instead.)

		If using GitHub,
		this metadata causes Bikeshed to automatically add a "GitHub" annotation
		to the <a>Issue Tracking</a> entry in your spec header.
		You can turn this off with `Boilerplate: repository-issue-tracking off`.
	* <dfn>Inline Github Issues</dfn> tells Bikeshed whether
		to fetch the text of GitHub issues and inline them into the document, or not.
		It takes a <a>boolish</a> value,
		or the values "full" or "title".
		If it's true or "full",
		any [remote issues](#remote-issues) of the form <code>user/repo#issuenumber</code>
		will have their contents replaced by the contents of the corresponding issue from GitHub;
		if it's "title",
		they'll have their contents replaced by just the title of the remote issue
		(so a large, well-written initial issue post doesn't derail a spec).
		If your <a>Repository</a> is set up to a GitHub repo,
		[remote issues](#remote-issues) with just an issue number
		will also be expanded to the corresponding issue from your repository.
	* <dfn>Opaque Elements</dfn> and <dfn>Block Elements</dfn> are a comma-separated list of custom element names, to help control Bikeshed's parsing and serialization. By default, custom elements are treated as inline; marking an element as "opaque" makes it like <{pre}>, so shorthands and Markdown aren't processed in it, and it's serialized precisely as entered; marking an element as "block" makes it interrupt paragraphs when it starts a line, and causes the pretty-printer to serialize it in a block-like way.
	* <dfn>Note Class</dfn>, <dfn>Issue Class</dfn>, <dfn>Assertion Class</dfn>, and <dfn>Advisement Class</dfn> specify the class name given to paragraphs using the note/issue/advisement <a>markup shorthand</a>.  They default to "note", "issue", and "advisement".
	* <dfn>Informative Classes</dfn> is a comma-separated list of classes that should be considered "informative" or "non-normative"--
		elements with these classes must not contain definitions or requirements,
		and Bikeshed will verify this for you
		(either automatically, or via the `Complain About: accidental-2119` metadata).
		"note", "example", "informative", and "non-normative" all trigger this behavior by default.
	* <dfn>Translate IDs</dfn> maps Bikeshed-generated IDs to manually-specified IDs, for cases where you need to manually specify an ID but it's generated by Bikeshed in a hard-to-override way (like IDL blocks).  It takes a comma separated list of Bikeshed IDs and new IDs.
	* <dfn>Default Highlight</dfn> specifies the default highlighting language. Every <{pre}>, <{xmp}>, and <{code}> will automatically be highlighted according to the specified language, unless they or an ancestor specifies a different `highlight=foo` or `nohighlight` attribute overriding it.
	* <dfn>Line Numbers</dfn> is a <a>boolish</a> specifying whether, by default, code blocks (<{pre}> and <{xmp}>) should get line numbers. Defaults to off.
	* <dfn>Toggle Diffs</dfn> is a <a>boolish</a> that specifies whether Bikeshed should include a 'Hide deleted text' button to toggle hiding <{del}> elements in your spec header. Defaults to off.
	* <dfn>Boilerplate</dfn> toggles the generation of individual [boilerplate sections](#bp-sections). Its value is a comma-separated list of section name and <a>boolish</a> value, e.g. `Boilerplate: issues-index no` to disable the list of issues.
	* <dfn>Local Boilerplate</dfn> tells Bikeshed which boilerplate include files to look for relative to the specification (as opposed to inside the Bikeshed source). Its value is a comma-separated list of include file basename and <a>boolish</a> value, e.g. `Local Boilerplate: footer yes` to look for `footer.include` and `footer-[STATUS].include` next to the specification. This also works for default includes that aren't quite boilerplate like [defaults.include](#default-metadata), [computed-metadata.include](#computed-metadata), and [bs-extensions.include](#echidna-hooks).
	* <dfn>External Infotrees</dfn> lists which of [anchors.bsdata](#custom-dfns) and [link-defaults.infotree](#link-defaults) should be found in files adjacent to the specification instead of in `<pre>` blocks inside it.
	* <dfn>Complain About</dfn> indicates what nits you'd like Bikeshed to complain about, that are normally too noisy to be turned on by default. Its value is a <a>boolish</a> list like <a>Markup Shorthands</a>. Accepted values are:

		: `accidental-2119`
		:: Complains about usage of RFC 2119 terms (may, must, should, etc) in non-normative sections.
			If you *need* to use one of those terms
			(such as in an example about using them),
			you can put a `class=allow-2119` on the text's immediate parent element.
		: `broken-links`
		:: Checks all the external links in the document,
			and verifies that they result in a 200 response code.
			Might take a while if your document links out a lot!
		: `missing-example-ids`
		:: Complains about examples without a manually-specified ID.
			(Examples auto-generate an ID based on their contents otherwise,
			which might not be stable over time.)
	* <dfn>Infer CSS Dfns</dfn> is a <a>boolish</a> that specifies whether Bikeshed should try to "infer" what type a <{dfn}> is by looking at the text,
		using CSS-biased heuristics.
		(In other words, this'll only auto-detect certain CSS types.)
		Defaults to off.
	* <dfn>Assume Explicit For</dfn> is a <a>boolish</a> that specifies whether an omitted `for` attribute on an autolink should imply `for="/"`
		(that is, selecting a definition without a `for` value),
		or just not express a `for` requirement at all
		(that is, selecting a definition without caring about what its `for` value is).
		Truthy values set the former,
		falsey set the latter.
		Defaults to off.

		Note: Turning this on is a better match for some people's mental models,
		but it requires you to specify more `for` values on your autolinks
		even when it's not necessary to disambiguate things,
		so it's off by default.
	* <dfn>Max ToC Depth</dfn> specifies the maximum depth you want the ToC to generate to. It can be the value "none", meaning generate the ToC normally, or an integer between 1 and 5. It defaults to "none".
	* <dfn>Include MDN Panels</dfn> is a <a>soft boolish</a> that specifies whether to include \[MDN](https://developer.mozilla.org/en-US/docs/Web) panels or not.
		If it's turned on, MDN panels will be added automatically in output,
		positioned to line up with the elements which define the features the panels annotate.
		Each MDN panel includes a hyperlink to the MDN article for the feature it annotates,
		as well as browser-support details for the feature
		(similar to what's shown in the panels added when [=Include Can I Use Panels=] is enabled).

		Because this examines MDN links for the IDs to use,
		no configuration is necessary beyond turning the feature on;
		the panels will automatically be added to the definitions or headings
		with the associated IDs.

		If your spec is not *yet* in MDN,
		but you'd like the panels to be added when it shows up,
		you can pass one of the "maybe" values of a [=soft boolish=]
		to allow it to silently upgrade when possible;
		otherwise, if directly turned on,
		it will emit a fatal error if it can't find your spec.
	* Several keys relate to the [Can I Use](https://caniuse.com/) project:
		* <dfn>Include Can I Use Panels</dfn> is a <a>boolish</a> that specifies whether to include [Can I Use](https://caniuse.com/) usage-data panels or not. If it's turned on, elements can take a `caniuse="featureID"` attribute, where the `featureID` is the value specified in the url like `https://caniuse.com/#feat=FEATUREIDHERE`, and the panel will be positioned to line up with that element.
			(When possible, put this on the corresponding <{dfn}>,
			as the panel doesn't have any special indication of what feature it's for;
			if you don't, you must ensure the element has an ID of its own.)
		* <dfn>Can I Use URL</dfn> automatically informs you when/if Can I Use starts tracking new features from your spec, so you don't have to check that yourself.
			It takes a single URL prefix, and can be specified multiple times to provide multiple URLs.
			These URLs are checked against the Can I Use data,
			and if there are any Can I Use features whose URL includes one of the specified URLs as a prefix,
			and the feature isn't already specified somewhere in your spec,
			it will log a warning for you.
		* <dfn>Ignore Can I Use URL Failure</dfn> takes one of the URLs specified in [=Can I Use URL=],
			and suppresses the warning that Bikeshed emits if the URL doesn't show up in the Can I Use database.
			(This error exists to help catch typos,
			rather than just letting it silently fail.)
			This allows you to put your URL in pre-emptively,
			and get an error only when Can I Use *does* include your specification
			(and you thus have features that aren't being tracked in your spec).
	* <dfn>Custom Warning Title</dfn> specifies the title of a "custom" warning,
		requested via `Warning: custom` metadata.
	* <dfn>Custom Warning Text</dfn> specifies the body of a "custom" warning.
		Like <a>Abstract</a>,
		it accepts multiple lines,
		and is parsed as Markdown
		(as if you'd written them directly into the document).
	* <dfn>Prepare For TR</dfn> is a <a>boolish</a> that specifies
		whether or not to apply a bunch of automatic "fixups"
		for common errors that would make a spec fail the W3C's PubRules check.
		This is applied automatically when you run the [[#cli-echidna|Echidna]] command to publish your spec,
		but you can use it manually to check precisely what your spec will look like when published.
	* <dfn>Use Dfn Panels</dfn> is a <a>boolish</a> that determines
		whether "definition panels" are generated for every definition;
		it defaults to true.
		These "panels" show up when you click on a <{dfn}>,
		and show every local reference to the definition.

		It automatically adds some JS and CSS to your page for the panels;
		you can suppress this with `Boilerplate: script-dfn-panel no, style-dfn-panel no`
		and supply your own if desired.
	* <dfn>Metadata Include</dfn> and <dfn>Metadata Order</dfn> are defined in [[#rearranging-metadata]].
	* <dfn>Remove Multiple Links</dfn> is a <a>boolish</a> (defaulting to false)
		that controls whether Bikeshed automatically suppresses multiple autolinks to the same term
		in a single paragraph.
		(That is, if multiple autolinks point to the same URL within a single parent element,
		all but the first will be reverted to plain <{span}>s.)
	* <dfn>Slim Build Artifact</dfn> is a <a>boolish</a> (defaulting to false)
		that specifies whether or not Bikeshed should try and "slim" the output,
		to make it more appropriate as an obvious "build artifact",
		rather than an active usable spec.
		This turns off various usability features which otherwise bloat your HTML,
		and which aren't necessary if the output isn't intended to be actively used by humans.
		The exact set of features that are affected by this will evolve over time.
	* <dfn>Required IDs</dfn> is a list of IDs that the document *must* contain.
		This can be used to ensure that certain links into the document don't rot over time.
	* <dfn>Force Crossorigin</dfn> is a [=boolish=] that auto-adds the `crossorigin=anonymous` attribute
		to every relevant element (<{link}>, <{script}>, <{img}>, <{audio}>, <{video}>).
		You can set a different `crossorigin` value manually on the element,
		or skip the element entirely by putting `nocrossorigin` on it or an ancestor.
	* <dfn>WPT Path Prefix</dfn> is a partial URL path
		that specifies a common path prefix for all of the tests in your <{wpt}> elements.
	* <dfn>WPT Display</dfn> takes the values "none", "closed", "open", or "inline",
		and specifies whether and how <{wpt}> elements display anything in the output document.
	* <dfn>Tracking Vector Class</dfn> is the HTML `class` attribute value used on the link inserted in elements annotated with the `tracking-vector` attribute.
	* <dfn>Tracking Vector Image</dfn> is an image URL that can be used to represent tracking vectors with an external image.
		Otherwise an inline SVG is used. The image is a child of the aforementioned link.
	* <dfn>Tracking Vector Image Width</dfn> is the width of the external image, if any.
	* <dfn>Tracking Vector Image Height</dfn> is the height of the external image, if any.
	* <dfn>Tracking Vector Alt Text</dfn> is the replacement text for the inline SVG or external image.
	* <dfn>Tracking Vector Title</dfn> is the title for the inline SVG or external image.
	* <dfn>Image Auto Size</dfn> is a [=boolish=] (defaulting to true)
		that specifies whether Bikeshed should [[#img-size|automatically detect the size]] of images used in the document.
</ul>

<div class=note>
	A <dfn>boolish</dfn> value is a string representing a boolean value (true or false).
	The string "yes", "on", "true", and "y" all represent true values,
	while the strings "no", "off", "false", and "n" all represent false values.

	A <dfn>soft boolish</dfn> value is either a [=boolish=] value,
	or one of the strings "maybe", "if possible", or "if needed",
	indicating an intermediate intention between something being turned on and off.
</div>

You can also provide custom keys with whatever values you want,
by prefixing the key with `!`,
like `!Issue Tracking: in spec`.
Any custom keys are collected together and formatted as entries in the "spec metadata" boilerplate <{dl}>.
Specifying a custom key multiple times will put all the values as <{dd}>s under a single <{dt}> for the key.
A custom key name equal to one of the auto-generated keys
will add your custom value as an additional <{dd}> under that auto-generated key.

Some of the metadata keys are *deprecated*;
you shouldn't use them,
but just in case you run into them in the wild,
they're documented here for understanding.
Each one recommends what you should be using instead.

<ul dfn-for=metadata export>
	* <dfn>Use &lt;i> Autolinks</dfn> turns on legacy support for using <{i}> elements as "dfn" autolinks.
		It takes a <a>boolish</a> value.

		Instead of using this,
		just use the <{a}> element to autolink, as usual.
	* <dfn>Link Defaults</dfn> lets you specify a default spec for particular autolinks to link to.
		The value is a comma-separated list of entries,
		where each entry is a versioned spec shortname,
		followed by a parenthesized link type,
		followed by a "/"-separated list of link phrases.
		For example, `Link Defaults: html (dfn) allowed to show a popup/in parallel`.

		Instead of using this,
		use a `<pre class=link-defaults>` block,
		as the error message suggests when this is necessary.
	* <dfn>Ignored Vars</dfn> accepts a comma-separated list of variable names,
		and makes Bikeshed not emit warnings or errors about them being used only once in the document/algorithm.

		Instead of using this,
		put an `ignore` attribute on the <{var}> in question.
</ul>


Default Metadata {#default-metadata}
------------------------------------

To specify default metadata for all specs generated for a given group and/or spec status,
add an appropriate `defaults.include` file to the `bikeshed/boilerplate/` folder.
This file must be a JSON file,
with the keys and values all strings matching the above descriptions.

Here's an example file:

```js
{
	"Mailing List": "www-style@w3.org",
	"Mailing List Archives": "http://lists.w3.org/Archives/Public/www-style/"
}
```

Computing Metadata From Other Metadata {#computed-metadata}
-----------------------------------------------------------

If your Group produces several specs,
and they all share some formulaic metadata that could be auto-generated with a bit of string-replacement,
you can achieve this automatically with a "computed-metadata.include" file in the `bikeshed/boilerplate/` folder.
It has the exact same syntax as the `defaults.include` file,
except that you can use [[#text-macros|text macros]] in the file,
based on all the previous metadata
(defaults, `<pre class=metadata>`, and command-line overrides).

Note: Don't forget about custom text macros with the [=Text Macro=] metadata,
if none of the pre-defined ones give you the substitution strings you need.

Note: By necessity from the definition,
these metadata override all previous metadata.
Make sure that everything you set in this file
is *really* something that every single spec in your Group should have.

Note: Bikeshed assumes that any text macros you use will be used *inside of* JSON strings,
and escapes them accordingly.
(In other words, you don't have to worry about JSON-escaping when writing your macros,
but you do need to ensure that all macros occur inside of double-quotes in your computed-metadata.include file.)

Overriding Metadata From The Command Line {#overriding-metadata}
----------------------------------------------------------------

If you want to generate multiple versions of a spec from the same source
(such as a primary spec, plus some snapshots),
you can override the metadata from the command line to generate the alternate forms.

For any metadata key defined above,
just pass it as a `--md-foo=bar` command-line argument.
For example, to override the **Status** metadata,
run `bikeshed spec --md-status=ED`.

(If the metadata name has spaces in it, use dashes to separate the words instead.)

### Known Issues ### {#md-issues}

1. You can't override the `Use <i> Autolinks` status, because you can't input the `<>` characters.
	I don't intend to fix this, as you shouldn't be specifying this in the first place.
2. You can't supply custom metadata keys (ones with a `!` prefix).
	If you want to do this, let me know, and I'll work on it.
3. Passing values with spaces in them is tricky.
	This is [an issue with the argparse library](http://bugs.python.org/issue22909).
	The only way around it is to specify both of the positional arguments (the input and output filenames),
	then put the offending argument after them.

Ordering Between Metadata Sources {#metadata-order}
---------------------------------------------------

When you specify a particular metadata key multiple times in your document,
the exact behavior depends on which metadata key it is,
but all of the behaviors depend on the order things appear in.
For example, if you specify [=metadata/Title=] twice,
the second one is used.

This also happens when metadata keys are duplicated between multiple sources,
like from your `defaults.include` file and your `<pre class=metadata>` block.
It's important, then,
to know what order the different sources are resolved in,
so you know which key "wins",
or in what order a list of things is constructed in.

The ordering is, from first to last:

1. "baseline" metadata (the default values that Bikeshed uses)
2. `defaults.include` metadata
3. `computed-metadata.include` metadata
4. `<pre class=metadata>` blocks in your document
5. command-line metadata

In general, this follows the ordering of least-specific to most-specific.


<!--
██     ██    ███    ████████  ██    ██ ██     ██ ████████
███   ███   ██ ██   ██     ██ ██   ██  ██     ██ ██     ██
████ ████  ██   ██  ██     ██ ██  ██   ██     ██ ██     ██
██ ███ ██ ██     ██ ████████  █████    ██     ██ ████████
██     ██ █████████ ██   ██   ██  ██   ██     ██ ██
██     ██ ██     ██ ██    ██  ██   ██  ██     ██ ██
██     ██ ██     ██ ██     ██ ██    ██  ███████  ██
-->

Markup Shortcuts {#markup-shortcuts}
====================================

Bikeshed's source format is *roughly* HTML,
but it allows you to omit or shorten several verbose/annoying parts of the language,
reducing the amount of format noise in the spec source,
making it easier to read and write.

Markdown {#markdown}
--------

Bikeshed uses a Markdown variant called Bikeshed-flavored Markdown (BSMD).
By default,
it recognizes all of the "block-level" Markdown constructs
defined by [CommonMark](https://commonmark.org/),
except for indented code blocks:

* paragraphs
* lists
* blockquotes
* headings (both underlined and #-surrounded)
* horizontal rules
* fenced code blocks (using `~` or `` ` ``)
* HTML blocks

Because it doesn't have to handle indented code blocks,
BSMD's HTML handling is generally superior to other Markdown implementations.
You can freely switch back and forth between Markdown and HTML as needed,
and indent properly.
(In fact, you *should* properly indent everything;
if you don't, you can sometimes get confusing errors!)

If the `Markup Shorthands: markdown yes` metadata is specified,
it also recognizes several of the Markdown "inline elements":

* *emphasis* (`*emphasis*`) and **strong** (`**strong**`) elements
* inline `code` (`` `code` ``) elements, which can include literal backticks by
    delimiting the content with multiple-backtick sequences: ``` `` ` `` ```
    produces `` ` ``. (Note that spaces at the beginning and end of the code
    element are stripped.)
* inline links, with optional title

It does not recognize "autolinks" (surrounded by &lt;&gt; characters),
images,
or "hard line breaks";
use HTML for these.
It also does not recognize "link references"
(defining the link location elsewhere,
and referring to it by reference instead).

In addition to standard (CommonMark) Markdown,
Bikeshed also recognizes **definition lists**, with the following format:

```
Here's the dl syntax:

: key
:: val
: key 1
: key 2
:: more vals
```

For all three list formats,
on the rare occasions you need to add a class or attribute to the list,
you can wrap it in the appropriate list container, like:

```
<ol class=foo>
	1. first
	2. second
</ol>
```

Bikeshed will use the container you provided,
rather than generating a fresh one like it does by default.

Bikeshed also supports adding IDs to headings,
via the Markdown Extra syntax:

```
Header 1 {#header1}
========

### Header 2 ### {#header2}
```

More of Markdown will be supported in the future,
as I get closer to adhering to the CommonMark specification.


Notes, Issues, Examples, Advisements {#notes-etc}
-------------------------------------------------

The Markdown processor specially recognizes paragraphs starting with
"Issue: ",
"Advisement: ",
"Assertion: ",
"Note: ",
or "Note, ",
and will add a matching class to the paragraph automatically.
These classes default to `.issue`, `.advisement`, `.assertion`, and `.note`,
but can be customized with the metadatas
<a>Issue Class</a>, <a>Advisement Class</a>, <a>Assertion Class</a>, and <a>Note Class</a>.
(However, the default classes receive styling from the default stylesheet,
so make sure you provide your own styling if you change them.)

The default styling of these blocks includes a generated-content "header"
containing the word "NOTE:", etc.
Elements with the `.example` class will also be given a generated-content "header"
in the format `EXAMPLE n`
with an auto-incrementing number,
and have self-links created to allow linking directly to the example.

If you'd like to provide your own custom header,
write out the container element yourself
(rather than using the Markdown shorthand,
just create any element with the appropriate class),
and add a `heading="YOUR TEXT HERE"` attribute to the container.
It will automatically have "NOTE: "/etc prepended to it.
(This is used by [[#remote-issues]] to put the GitHub issue title on the issue containers.)

Alternately, if you want *full* control over the contents of the heading,
write out the container element
and create a `<div class=marker>` element containing the text you want
as its first child.
(This is what the `heading=""` attribute does;
it just automatically prepends "Note:"/etc to the text
to keep it visually in line with other notes/etc by default.)




Typography Fixes {#typography}
----------------

Bikeshed will automatically handle a few typographic niceties for you, ones that it can reliably detect:

* Possessive apostrophes, and most contraction apostrophes, are automatically turned into curly right single quotes (`’`).
* Ending a line with `--` will turn it into an em dash (`—`) and pull the following line upwards so there's no space between the surrounding words and the dash.


Autolink Shortcuts {#autolink-shortcuts}
------------------

There are several autolink shortcuts that make autolinks much shorter to write,
such as `'foo'` to link to a CSS property or descriptor,
or `{{foo}}` to link to an IDL construct.
These are documented in [[#autolinking]].

Other types of linking shorthand also exist,
such as `[[#foo]]` to link to sections of a spec
(documented in [[#section-links]]),
or `[[FOO]]` to link to generate and link to bibliography entries
(documented in [[#biblio]]).

<h4 id="l-element" element lt="l">
Autolink Shortcuts That Work Anywhere: the <{l}> element</h4>

The autolink shorthands don't work in elements like <{pre}>,
because it's too easy for code constructs to look like autolinks.
You can still manually write out your autolink as an <{a}> element,
but that can be much more verbose than the terse syntax that the shorthands provide.
(For example, `{{Foo/bar()}}` vs `<a method for=Foo>bar()</a>`.)

You can avoid this by using the <{l}> element,
and putting an autolink shorthand inside of it,
like `<l>{{Foo/bar()}}</l>`.
This will autolink exactly the same as normal,
but works anywhere in your document.
Extra bonus: it will recognize *all* the autolinking syntaxes,
even if you have some turned off with the [=Markup Shorthands=] metadata.

Note: Currently the CSS value autolinking syntax (`''foo''`) does not work.

The <{l}> element itself is removed from the document and replaced by the autolink,
but if you specify any attributes on the <{l}>,
they'll be transferred to the generated <{a}>.
In particular, this is useful for specifying linking attributes that can't be specified by the shorthand,
like the spec to search for,
that would otherwise require converting all the way to a manual <{a}> autolink.


<{var}> and Algorithms {#var-and-algorithms}
----------------------

The <{var}> element (or its shorthand equivalent, `|foo|`) is often used to mark up "arguments" to a prose algorithm.
Bikeshed explicitly recognizes this,
and has several features related to this.

<dfn>Algorithms</dfn> can be explicitly indicated in your markup
by putting the `algorithm="to foo a bar"` attribute on a container element
or a heading.
All vars within an [=algorithm=] are "scoped" to that algorithm.

Generally, vars are used at least twice in an algorithm:
once to define them,
and at least once to actually use them for something.
If you use a var only once,
there's a good chance it's actually a typo.
Bikeshed will emit a warning if it finds any vars used only once in an algorithm.
If this singular usage is correct,
you can instruct Bikeshed to ignore the error by adding an `ignore` attribute to the <{var}> itself.
(There's no way to do this with the `|foo|` syntax;
you have to convert it back to an actual <{var}> element.)

Code Blocks (<{pre}>, etc) {#code-blocks}
-----------------------------------------

Bikeshed has several nice features related to "code blocks"--
<{pre}>, <{xmp}>, and sometimes <{code}>.

### <{pre}> whitespace stripping ### {#pre-whitespace-stripping}

Using a <{pre}> element in HTML is unsatisfying,
because it forces you to break your indentation strategy,
pulling the content back to the margin edge
(or else employing silly hacks with comments and explicit newlines).
The preprocessor fixes this.

Whenever a <{pre}> element is encountered,
the processor records how much whitespace precedes the first line,
and then strips that much whitespace from it and all following lines.

Additionally, if the closing `</pre>` is on its own line,
the processor automatically pulls it up onto the end of the previous line,
so there's no final blank line in the content.

In other words, you can now write:

<xmp highlight=html>
	<div class='example'>
		<p>
			An example:

		<pre>
			&lt;ul>
				&lt;li>one
				&lt;li>two
			&lt;/ul>
		</pre>
	</div>
</xmp>

The preprocessor will automatically convert it into:

<xmp highlight=html>
	<div class='example'>
		<p>
			An example:

		<pre>
	&lt;ul>
		&lt;li>one
		&lt;li>two
	&lt;/ul></pre>
	</div>
</xmp>


### <{xmp}> To Avoid Escaping Markup ### {#xmp}

The <{xmp}> element is an old HTML element that's now deprecated
(but still required to be supported).
It was intended for markup examples
(thus the name),
and has magical parsing properties,
where its contents
(everything until the `</xmp>` closing tag)
are treated as literal text,
and not interpreted as HTML.
In particular, this means that within an <{xmp}>
you don't need to escape your `<` or `&` characters.

Bikeshed supports using <{xmp}> *anywhere* a <{pre}> can be used;
it converts it into a properly-escaped <{pre}> in the output
to avoid validation errors.
For example, the two markup examples in the previous section are using <{xmp}> in this document's source.

Use of <{xmp}> is particularly useful for IDL blocks,
as IDL uses `<foo>` syntax for higher-order types
(like `sequence<DOMString>`).
If you're using a <{pre}> and don't remember to escape this,
you'll end up with confusingly-broken IDL,
as the `<DOMString>` part is interpreted as an opening HTML tag.
If you use <{xmp}> instead,
there's no need to remember to escape anything;
you can write or copy raw WebIDL into the block
and it'll be interpreted correctly.


### Syntax Highlighting ### {#syntax-highlighting}

You can syntax-highlight code blocks.
Just add either a `highlight="foo"` attribute
or a `lang-foo` class to the element,
or set a [=Default Highlight=] metadata,
and the element will automatically be syntax-highlighted according to the "foo" language rules.
(Setting `no-highlight` closer will turn if off for an element;
it looks for the closest ancestor with one of the attributes.)

The syntax highlighter uses Pygments, which supports a large set of languages.
See <a href="http://pygments.org/docs/lexers/">http://pygments.org/docs/lexers/</a> for the full list.
(Use one of the "short names" of the language for the "foo" value.)

Bikeshed comes with a default color scheme based loosely on that of Prism.js.
If you would like to use your own color scheme,
turn off the automatic styles with a `Boilerplate: style-syntax-highlighting off` metadata,
then supply your own.

Note: If you use `highlight=html`,
<{script}> and <{style}> elements are automatically highlighted with JS and CSS rules.
Normative WebIDL blocks (`class=idl`) are automatically highlighted,
but you can use `highlight=idl` to highlight non-normative ones.

Note: If your code block already has markup in it,
this feature will safely "merge" the highlighting into your existing markup.


### Line Numbers ### {#line-numbers}

You can automatically generate line numbers for code blocks
by adding a `line-numbers` attribute to the element or an ancestor,
or setting the [=Line Numbers=] metadata to affect all elements.
(Setting `no-line-numbers` closer will turn it off for an element;
it looks for the closest ancestor with one of the attributes).
The numbers are added to each line via CSS generated content,
so they'll always stay in sync
and won't get copied when you highlight a chunk of text.

<pre line-numbers>
	Haikus are easy:
	five-seven-five syllables,
	spread over three lines.
</pre>

The lines default to starting at 1;
to change that,
set the `line-start` attribute on the element to your desired starting value.

If you would like to produce your own line-numbering CSS,
turn off the automatic styles with a `Boilerplate: style-line-numbers off` metadata,
then supply your own.

Note: Similar to syntax highlighting,
this feature will safely "merge" the line-numbers markup
into your existing markup.
The only complication is if the original markup
has elements that span more than one line;
in this case, the "line" element will expand to cover all the lines spanned by your original markup.
It will still put a line marker on the first and last line of this span,
so the effect isn't too noticeable
if the original markup only spans two lines.

### Highlighting Particular Lines ### {#line-highlighting}

If you want to bring attention to particular lines in your code snippet,
but need to show more surrounding code for context,
you can highlight the important ones
with a `line-highlight` attribute.

<pre line-highlight=2>
Another haiku.
This is the important line.
Ignore what's down here.
</pre>

The syntax of `line-highlight` is a comma-separated sequence
of either integers (like `5`),
or integer ranges (like `2-4`).
For example, `line-highlight="2-4, 6"`
will highlight lines 2, 3, 4, and 6.

Like `line-numbers` (above),
this defaults to considering the first line of the snippet as line 1,
but the numbering can be changed with the `line-start` attribute.

If you would like to produce your own line-highlighting CSS,
turn off the automatic styles with a `Boilerplate: style-line-highlighting off` metadata,
then supply your own.

Note: This feature has the same restriction on line-spanning elements
as the `line-numbers` feature does.
In particular,
if any line being wrapped in a multi-line "line" element
is highlighted,
the entire "line" element gets highlighted.


Automatic ID Generation {#id-gen}
-----------------------

If any heading, issue, or <{dfn}> element doesn't have an `id` attribute,
one will be automatically generated by the processor,
to ensure it's usable as a link target.

Heading IDs are generated directly from the text contents of the element,
cleaning up the characters to be a valid id.
This often isn't the best for complex heading texts,
so it's not recommended to rely on this.
(Bikeshed will warn you that it's generating IDs, and suggest you supply one manually.)

If a heading changed significantly,
so that you want to change the ID,
but you want links to the old heading ID to still work,
put the old ID in an `oldids=""` attribute on the heading element.
If there are multiple, comma-separate them.

Issues (elements with `class="issue"`) will generate IDs of the form "issue-###",
where "###" is substring of a hash of the issue's contents.
This means that an issue's ID will be stable against changes elsewhere in the document,
including adding or removing issues above it in the source,
but will change if you change the contents of the issue.

Definition IDs are also generated directly from the text contents of the element.
Most definitions additionally get a prefix, such as "propdef-",
to avoid clashes with other definitions.

If an automatically-generated ID would collide with any other ID,
it's automatically de-duped by appending a number to the end.
This isn't very pretty,
so if you want to avoid it,
supply an ID yourself.

Bikeshed recognizes a fake element named `<assert>` for marking "assertions" that tests can refer to.
In the generated output, this is converted to a `<span>` with a unique ID generated from its contents,
like issues (described above).
This ensures that you have a unique ID that won't change arbitrarily,
but *will* change **when the contents of the assertion change**,
making it easier to tell when a test might no longer be testing the assertion it points to
(because it's no longer pointing to a valid target at all!).


Automatic Self-Link Generation {#self-link}
------------------------------

Giving IDs to important things in your document,
like headings and definitions,
is great, but of little use if people don't know they can link to them.
Bikeshed will automatically generate a "self-link"
in the margin next to certain linkable elements
which just links to the element,
so people can click on the link and then just copy the URL from their address bar
to get a link straight to what they care about.

Self-links are currently auto-generated for headings, definitions, and issues,
and notes, examples, `<li>`s, and `<dt>`s that have been given IDs.

Remote Issues {#remote-issues}
-------------

As defined earlier, you can start a paragraph with `Issue: ` to cause Bikeshed to automatically format it as an inline issue paragraph.
You can also refer to remote issues, which are tracked in some other issue tracker.
To do so, instead start your paragraph with `Issue(###): `,
where the `###` is some identifying value for the issue.

If the identifying value is of the form `user/repo#number`,
Bikeshed assumes you are referring to GitHub repository,
and points the issue at the corresponding issue.

If you have **Repository** set up to point to a GitHub repository
(or it was auto-detected as such,
because you're working on the spec from within one),
then a numeric identifying value is assumed to be an issue number for your repository.

Otherwise, you need to tell Bikeshed how to convert the identifying value into a remote issue link.
Specify a format string in the **Issue Tracker Template** metadata,
with a `{0}` in the place where the identifying value should go.
Bikeshed will then point the issue at the generated url.


Including Other Files {#including}
---------------------

Sometimes a spec is too large to easily work with in one file.
Sometimes there's lots of repetitive markup that only changes in a few standard ways.
For whatever reason,
Bikeshed has the ability to include additional files directly into your spec
with a &lt;pre class=include> block:

<xmp highlight=html>
	<pre class=include>
	path: relative/to/spec/location
	</pre>
</xmp>

The included document is parsed just like if it were written in locally,
except that metadata blocks aren't processed.
(For various reasons, they have to be parsed before *any* other processing occurs).
This means that the include file can use markdown,
data blocks of various kinds (&lt;pre class=anchors>, &lt;pre class=railroad>, etc),
and both provide definitions for the outer document
and refer to ones defined by the outer document.

Note: All include paths are processed relative to the outermost spec document's location.
If you're including a document in another folder,
and it includes a document itself,
take care that your paths are correctly relative to the outermost spec,
not to the included file.

If you're including a block of repetitive markup multiple times,
and want to vary how it's displayed,
you can pass additional "local" text macros in the block,
which are valid only inside the included file:

<xmp highlight=html>
	<pre class=include>
	path: template.md
	macros:
		foo: bar
		baz: qux qux qux
	</pre>
</xmp>

With the above code, you can use `[FOO]` and `[BAZ]` macros inside the include file,
and they'll be substituted with "bar" and "qux qux qux", respectively.
(Remember that you can mark text macros as optional by appending a `?`, like `[FOO?]`,
in which case they'll be replaced with the empty string if Bikeshed can't find a definition.)

### Including Code Files ### {#including-code}

While it's easy to include short code snippets in your document inline
with an ordinary `<pre>` block
(particularly with a [highlight attribute](#syntax-highlighting)
so it's formatted nicely),
you sometimes want to instead include snippets of a larger external file.
(For example, so you can keep that file up-to-date easily,
without having to go in and fiddle with all the chunks you've inlined into your document.)

Normal includes don't work for this,
as they include the whole file
and assume that it's more Bikeshed code;
instead, use `<pre class=include-code>`
to include part (or all) of a file
and have it rendered as highlightable source code.

<xmp highlight=html>
	<pre class=include-code>
	path: bigFile.cpp
	highlight: c++
	show: 100-120
	</pre>
</xmp>

This snippet will create a nicely highlighted code block
containing lines 100-120 of the file `bigFile.cpp`.

You can additionally include any of the more advanced highlighting options
([[#line-numbers]], [[#line-highlighting]]),
using the same names and value syntax as you would use when specifying them as attributes:

<xmp highlight=html>
	<pre class=include-code>
	path: someFile.cpp
	highlight: c++
	line-numbers:
	line-highlight: 2-5, 10, 12
	</pre>
</xmp>

If any of your options cause line-numbers to show up
(using `line-numbers` or `line-highlight`),
and you're showing a snippet of the larger file with `show`,
the numbering will default to using the numbering of the original file
(so if you said `show: 6-10`, the first visible line will be numbered "6").
You can override this with `line-start`, as usual.

Note that, unlike `<pre class=include>`,
there is no `macros` option.


### Including Non-Bikeshed HTML/SVG/etc ### {#including-raw}

`<pre class=include>` lets you insert chunks of Bikeshed source code
into a larger outer document.
But what if you want to include some markup
that isn't formatted as Bikeshed code?
For example, SVG outputted by a design tool
might not conform to Bikeshed's syntax--
it might not indent properly,
or it might use markup that will get messed up if parsed as Markdown,
etc.

To avoid this,
you can use `<pre class=include-raw>`.
Just like `<pre class=include>`,
it takes a single `path` line:

<xmp highlight=html>
	<pre class=include-raw>
	path: myDiagram.svg
	</pre>
</xmp>

&hellip;and replaces itself with that markup directly,
without doing any additional Bikeshed syntax processing.

Note: Unlike `<pre class=include>`,
there are no local `macros`,
as macro substitution is part of Bikeshed's syntax processing,
which is skipped here.

Note: Most parts of later Bikeshed processing *do* work,
such as linking to definitions,
etc.,
because inclusions are processed *very* early
in Bikeshed's transformation pipeline.
However,
if you're writing something that's intended to use Bikeshed's features,
you should really write Bikeshed source code
and use `<pre class=include>`,
in case some things move to being part of syntax processing.
For example, inline autolinking shorthands currently aren't,
and so will work in include-raw substitutions,
but that will change in the future.


Tracking Vectors {#tracking-vectors}
----------------

A <a>tracking vector</a> can be annotated with the `tracking-vector` attribute on the enclosing element.
This will result in a link containing an image being prepended to the element.

<div class=example id=example-tracking-vector>
  <xmp highlight=html>
    <p tracking-vector>This is an example of an element annotated with that attribute.
  </xmp>

  <p tracking-vector>This is an example of an element annotated with that attribute.
</div>

The various "Tracking Vector *" metadata keys can be used to customize the appearance of this annotation,
though it's encouraged to leave those in their default state for a consistent appearance across standards.


Image Size Detection {#img-size}
----------------

If an <{img}> element has a neither a <{img/width}> nor a <{img/height}> attribute,
Bikeshed will automatically detect these dimensions from the image file
and insert both attributes into the resulting markup.

<div class=example id=img-size-simple>
  Given this markup:
  <xmp highlight=html><img alt="…" src="img.jpg"></xmp>
  if the <code>img.jpg</code> file is 300 pixels wide and 100 pixels tall,
  bikeshed will generate the following output:
  <xmp highlight=html><img alt="…" src="img.jpg" width="300" height="100"></xmp>
</div>

Bikeshed will also detect and insert the dimensions of the image
if the element has no <{img/src}> attribute but does have a <{img/srcset}> attribute.
In that case,
it accepts a narrow subset of the <{img/srcset}> syntax:
a single <a spec=html>image source</a> URL
followed by a single <a spec=html>pixel density descriptor</a>,
limited to integers.
Since the <{img/src}> attribute is mandatory in HTML,
bikeshed will also add it to the output.

<div class=example id=img-size-srcset>
  In order to specify high resolution images,
  use the following syntax:
  <xmp highlight=html><img alt="…" srcset="highDPI.jpg 2x"></xmp>
  If the <code>highDPI.jpg</code> file is 600 pixels wide and 200 pixels tall,
  bikeshed will generate the following output:
  <xmp highlight=html><img alt="…" src="highDPI.jpg" srcset="highDPI.jpg 2x" width="300" height="100"></xmp>
</div>

Note: Bikeshed will only successfully detect the size of local images,
and will not attempt to fetch resources over the network,
as there is no way for Bikeshed to know
whether the size it would get is stable.
Using local images is generally preferred,
but if you need to use a remote one,
consider setting the <{img/width}> and <{img/height}> manually.

If either of the <{img/width}> and <{img/height}> attributes is already set
(or both are),
or if both the <{img/src}> and <{img/srcset}> attributes are set,
Bikeshed will assume you know what you're doing,
skip detection,
and leave the element unchanged.

To opt out of this automated size detection for a single image,
set the boolean <dfn dfn-for=img element-attr>no-autosize</dfn> attribute on it.
To opt out for the entire document,
specify the [=Image Auto Size=] metadata with the value false.

<!--
████████  ████████ ██    ██
██     ██ ██       ███   ██
██     ██ ██       ████  ██
██     ██ ██████   ██ ██ ██
██     ██ ██       ██  ████
██     ██ ██       ██   ███
████████  ██       ██    ██
-->

Definitions {#definitions}
===========

Defining a term is as easy as wrapping a <{dfn}> element around it.
Most of the time, this is all you'll need to do -
the definition automatically gains an id,
and is usually automatically exposed as an autolink target for local and cross-spec autolinks.

**Autolinking** is a special mechanism in the processor to let you link terms to their definitions without having to explicitly provide a url.
Instead, the text of the link is matched against the text of the definitions,
and if a match is found,
the link's `href` is set up to connect the two.

Conjugating/Pluralizing/etc the Linking Text {#conjugating}
--------------------------------------------

Bikeshed can automatically handle a wide range of English conjugations and pluralizations.
For example, if you define the term "snap",
you can link to it with "snapping" or "snapped"
without having to manually add those variations to your <{dfn}> manually.

As such, it's best to define your term in the "base" form,
singular and present tense.
Use `lt='...'` if necessary to set up the correct "base" linking text,
if your visible text needs to be in a conjugated form due to the surrounding text.

These variations only work for the *first* or the *last* word in a phrase,
so it can conjugate "activate the widget" into "activating the widget", "activate the widgets", or "activating the widgets".
If you have a longer phrase where it's a middle word that conjugates differently,
you do still have to manually handle that,
either by defining multiple linking texts on the <{dfn}>,
or by manually specifying the linking text on the <{a}>.

In addition to English variations,
these "conjugations" handle some IDL variations as well.
For example, IDL terms that match an IDL keyword must be defined in IDL with a leading underscore,
to avoid grammatical ambiguities,
but actually define the term without the underscore.
You can link to the term with either syntax.

Similarly, because methods and attributes technically live in the same namespace,
it's safe to link to a method without the parens after its name.
(This is useful when linking to a method in the middle of prose,
when you might be providing sample arguments that would interfere with linking,
or linking to things in the arguments,
which would interact badly with the arguments themselves being part of the method's link text.)

Changing the Linking Text {#changing-lt}
-------------------------

Sometimes, the text of the definition isn't exactly what you want it to be linked by,
or you may want it to be linkable with more than one phrase.
For example, an algorithm named "Check if three characters would start an identifier"
may also want to be linkable from the phrase "starts with an identifier".
To alter the linking text, simply add an `lt` attribute (for "Linking Text") to the definition;
the linking text is used instead of the text content.
You can separate multiple linking phrases by separating them with the pipe "|" character.

Defining Extra-Short "Local" Linking Texts {#local-lt}
------------------------------------------

Sometimes you want to use an extra-short version of a term for within a spec,
but don't want to confuse things by exporting it globally.
To achieve this, add a `local-lt` attribute with the terms you want to be only usable within the spec;
the syntax is identical to that of the `lt` attribute, described above.

Using local linking text does not disturb the normal linking-text process;
that still takes from either the element text or the `lt` attribute,
as normal.

Definition Types {#dfn-types}
----------------

All definitions have a definition type.
This allows for "namespacing" of the linking text,
so you can define, for example, both a property and a term with the same linking text, such as "direction" or "color".

There are several types for CSS values:

* property
* descriptor (the things inside at-rules like @font-face)
* value (any value that goes inside of a property, at-rule, etc.)
* type (an abstract type for CSS grammars, like `<length>` or `<image>`)
* at-rule
* function (like `counter()` or `linear-gradient()`)
* selector

There are additional types for WebIDL definitions:

* interface
* constructor
* method
* argument
* attribute
* callback
* dictionary
* dict-member
* enum
* enum-value
* exception (for new DOMException names)
* const
* typedef
* stringifier
* serializer
* iterator
* maplike
* setlike
* extended-attribute (things like `[EnforceRange]`)

And for HTML/SVG/etc element definitions:

* element
* element-state (a spec concept, like <{input}> being in the "password state")
* element-attr
* attr-value

A special type for URL schemes, like "http" or "blob":

* scheme

A special type for HTTP headers:

* http-header

A special type just for definitions of operators used in grammar definitions,
like `||` and similar:

* grammar

And finally, some categories for "English" terms:

* abstract-op (for "English-language algorithms")
* dfn (for general terms and phrases, and a catch-all for anything else)

The processor will attempt to infer your definition type from the context and text content of the definition:

* Is it inside a propdef, descdef, or elementdef block?  Then it's a **property**, **descriptor**, or **element**.
* Is it inside an idl block (&lt;pre class='idl'>)?  Then it's an **one of the IDL types, inferred by parsing the IDL**.
* Does it start with an `@`?  Then it's an **at-rule**.
* Is it surrounded by `<>`?  Then it's a **type**.
* Does it start with a `:`?  Then it's a **selector**.
* Does it end with `()`?  Then it's a **function**.
* Is it surrounded by double single quotes in the source, like `''foo''`?  Then it's a **value**.
* Otherwise, it's a **dfn**.

(This auto-detection is obviously skewed towards CSS types; Bikeshed started as a CSS spec preprocessor, and the CSS types are easier to auto-detect syntactically than anything else.)

Note that this auto-detection is a **last-resort** operation.
There are methods (defined below) to explicitly indicate what type a definition is,
and those win over the auto-detection.

If your value doesn't fit one of these categories,
you'll have to tag it manually.
Just add the type as a boolean attribute to the definition, like

```html
  attribute DOMString <dfn attribute for=Foo>name</dfn>;
```

Alternately, if you've got several definitions of the same type that share some container element (such as a `<pre>` or `<dl>`),
just add a `dfn-type="type-goes-here"` attribute to the container.
Anything which isn't explicitly tagged otherwise will take that type by default.

(There are more methods to determine definition type, but they're only meant for legacy content, and so are not documented here.)

Namespacing a Definition {#dfn-for}
-----------------------------------

Some types of definitions are defined relative to a higher construct,
such as values for a particularly property,
or attributes of a particular IDL interface.
This is useful, as it means these names don't have to be globally unique,
but that means your autolinks may have a hard time telling which name you intend to link to.

To fix this, the processor enforces that some types of definitions *must* define what they are for.
This is specified with a `for` attribute on the definition.

Specifically:

* "attribute", "constructor", "method", "const", "event", "serializer", "stringifier", and "iterator" definitions must define what interface they're relative to.
* "argument" definitions must define what method or constructor they're relative to.
* "dict-member" definitions must define what dictionary they're relative to.
* "enum-value" definitions must define what enum they're relative to.
* "element-attr" and "element-state" definitions must define what element they're relative to.
* "attr-value" definitions must define what element and attribute they're relative to.
* "descriptor" definitions must define what at-rule they're relative to.
	(This happens automatically if you add a "For" line to the descdef table.)
* "value" definitions must define what property, descriptor, at-rule, type, selector, or function they're relative to.
	If a value definition is relative to a descriptor, the value must be of the form "@foo/bar", where "@foo" is the at-rule the "bar" descriptor is relative to.

Just like with the definition type, you can instead declare what several definitions are for by putting an attribute on a container.
In this case, just add `dfn-for` to the container.
This is especially useful for property/descriptor values, as they're usually defined in a <{dl}>,
or IDL definitions, as you can just put a `dfn-for`	on the &lt;pre class='idl'>.

If a single definition is "for" multiple things,
you can provide a comma-separated list of values in the attribute.

Exporting Definitions {#dfn-export}
---------------------

Most definitions are automatically "exported",
which means they're made available for other specs to autolink to.
The only exception is "dfn" type definitions, which aren't exported by default.

To force a link to be exported, add an `export` boolean attribute to it.
To force a link *not* to be exported, add a `noexport` boolean attribute instead.
Like the other attributes, you can instead add this to a container to have it be a default for the definitions inside.

Link Attributes On Ancestors {#dfn-ancestors}
----------------------------

All of the dfn attributes
can be specified on an ancestor of the dfn instead,
if you have a group of dfns that all share a particular attribute.

They can be spelled out fully
(like `data-dfn-type`),
or without the `data-` prefix,
like `dfn-type` or `dfn-for`.
As there's no overlap with a similar `data-link-*` attribute,
`dfn-force`
can omit `dfn-` as well,
being spelled as merely `force` and `spec`.

A link will take the "closest" version of each attribute it can find,
so specifying an attribute directly on the link
will override anything coming from an ancestor.

Providing Custom Definitions {#custom-dfns}
----------------------------

If you want to link to dfns in specs that aren't yet part of the autolinking database,
you can provide your own definition data that Bikeshed can use.
Within a `<pre class='anchors'>` element,
define the anchors you need in [InfoTree format](#infotree),
with the following keys:

* **text** - the linking text for the definition. (Exactly 1 required.)
* **type** - the definition's type (dfn, interface, etc)  (Exactly 1 required.)
* **urlPrefix** and/or **url** - define the anchor's url, as described below.  (At least one of `urlPrefix` or `url` must be specified. 0+ `urlPrefix` entries allowed, 0 or 1 `url` entries allowed.)
* **for** - what the definition is for.  (Any number allowed, including 0.)
* **spec** - Which spec the definition comes from. (optional)

To generate the url for the anchor,
first all of the `urlPrefix` entries are concatenated.
If a `url` is provided,
it's appended to the prefixes;
otherwise, the `text` is url-ified and appended.
(Lowercased, spaces converted to dashes, non-alphanumeric characters dropped.)
If neither `urlPrefix` nor `url` had a "#" character in them,
one is inserted between them.

The `spec` attribute is used only for index generation, and has no effect on URL generation.

Example:

<xmp highlight=html>
<pre class="anchors">
urlPrefix: https://encoding.spec.whatwg.org/; type: dfn; spec: ENCODING
	text: ascii whitespace
	text: decoder
url: http://www.unicode.org/reports/tr46/#ToASCII; type: dfn; text: toascii
</pre>

<a>ascii whitespace</a> links now!
</xmp>

Alternately, this data can be provided in a file named `anchors.bsdata`,
in the same folder as the spec source, but this prevents you from using
[the web service](https://api.csswg.org/bikeshed/).


Definitions data model {#dfn-contract}
----------------------

Bikeshed's most important feature is its powerful cross-spec autolinking.  This
is possible due to each definition being annotated with a rich set of metadata,
which is then exposed via custom attributes and picked up by specialized
scrapers (such as Shepherd) that then compile a definition database that
Bikeshed relies on.

If you're writing a spec processor or related tool and would like to
interoperate with the Bikeshed ecosystem, here's the full definition data model
and how to properly expose it.

1. The defining element MUST be a <{dfn}> or <{h2}>...<{h6}>.  No other element
	is recognized as defining a term.
2. The element MUST have an `id` attribute.
3. The linking text defaults to the **text content** of the <{dfn}>/heading.
	If the desired text content isn't suitable for linking text, or you wish to
	provide multiple linking texts, a [`lt` attribute](#changing-lt) containing one or more
	pipe-separated linking texts will override the text content. See also [[#local-lt]]
4. `data-dfn-type` MUST be provided, and set [one of the accepted values](#dfn-types).

	(In an actual Bikeshed document, this can be omitted from <{dfn}> elements
	and it will be inferred from context,
	usually defaulting to "dfn" type.
	But non-Bikeshed documents must specify this,
	or else they'll get their dfn type inferred heuristically,
	which may or may not be correct.)
5. Either <a href="#dfn-export">`data-export` or `data-noexport`</a> MAY be
	provided (both boolean attributes).  If neither is provided, "dfn" type
	definitions default to noexport, while all others default to export.
	Unexported definitions aren't linkable by default.
6. [Several types](#dfn-for) of definitions are namespaced to another
	construct; for example, attribute names are namespaced to an interface.
	These definitions MUST contain a `data-dfn-for` attribute, containing a
	comma-separated list of one or more definitions they're namespaced to.

If you have written a web spec and it conforms to this definition syntax,
[contact the project maintainer](https://github.com/tabatkins/bikeshed/issues/)
and ask them to register your spec in Shepherd, so its definitions will be
available to everyone else.


<!--
   ███    ██     ██ ████████  ███████  ██       ████ ██    ██ ██    ██ ████ ██    ██  ██████
  ██ ██   ██     ██    ██    ██     ██ ██        ██  ███   ██ ██   ██   ██  ███   ██ ██    ██
 ██   ██  ██     ██    ██    ██     ██ ██        ██  ████  ██ ██  ██    ██  ████  ██ ██
██     ██ ██     ██    ██    ██     ██ ██        ██  ██ ██ ██ █████     ██  ██ ██ ██ ██   ████
█████████ ██     ██    ██    ██     ██ ██        ██  ██  ████ ██  ██    ██  ██  ████ ██    ██
██     ██ ██     ██    ██    ██     ██ ██        ██  ██   ███ ██   ██   ██  ██   ███ ██    ██
██     ██  ███████     ██     ███████  ████████ ████ ██    ██ ██    ██ ████ ██    ██  ██████
-->

Autolinking {#autolinking}
===========

The processor supports "autolinks" for easy linking to terms without having to fiddle around with urls.
Instead, just match up the text of the link to the text of the definition!

There are three types of autolink syntaxes you might see/use:

1. [=Dfn autolinks=], like `{{foo}}` or `[=foo=]`, which link to a definition.
	See [[#autolink-microsyntax]] for details.
2. [=biblio autolinks|Spec/section autolinks=], like `[[FOO]]` or `[[#foo]]`, which link to a document or heading.
	See [[#autolink-biblio]] for details.
3. [=manual autolinks|Manual=] <{a}> elements without an `href` attribute,
	which can create either of the preceding two types,
	and are occasionally useful when the preceding syntaxes can't express what you need.
	See [[#autolink-manual]] for details.

You may also see an <{l}> element,
which is used to help augment [=dfn autolinks=],
and can avoid the need to use an <{a}> in many cases.
See [[#l-element]] for details.

Dfn Autolinks {#autolink-microsyntax}
-------------

<dfn lt="dfn autolink">Dfn autolinks</dfn> create links to definitions.

There are two parts to the syntax:

1. The <b>wrapper</b>, like `{{...}}` or `[=...=]`,
	which specifies what type of definition you're linking to.
2. The <b>inside syntax</b>,
	which gives details on what you're trying to link to.


### Wrapper Syntax ### {#autolink-wrappers}

The characters used to wrap an autolink determine what type of definition
(see [[#dfn-types]])
you're trying to link to.
They're arranged into categories that can be individually turned on or off,
if you don't think you'll use a particular type of autolink,
and want to avoid possibly triggering an autolink accidentally:

The **Dfn** varieties (controlled by `Markup Shorthands: dfn yes`):

* `[=foo=]` is an autolink to the "dfn"-type definition "foo".
* `[$foo$]` is an autolink to the "abstract-op"-type definition "foo".

The **IDL** variety (controlled by `Markup Shorthands: idl yes`):

* `{{foo}}` or `{{foo()}}` is an autolink to one of the IDL types (interface, method, dictionary, etc) for the term "foo".

The **markup (HTML/SVG/etc)** varieties (controlled by `Markup Shorthands: markup yes`):

* `<{element}>` is an autolink to the element named "element".
* `<{element/attribute}>` is an autolink to the attribute or element-state named "attribute" for the element "element".
* `<{element/attribute/attribute value}>` is an autolink to the value "attribute value" of an attribute or element-state named "attribute" for the element "element".

The **CSS** varieties (controlled by `Markup Shorthands: css yes`):

* `'foo'` (apostrophes/straight quotes) is an autolink to a property or descriptor named "foo". If there is both a property and a descriptor of a given name, this defaults to linking to the property if used in its bare (`'foo'`) form.
* `''foo''` (double apostrophes) is an autolink to any of the CSS definition types except property and descriptor
* `<<foo>>` is an autolink to a type/production named "&lt;foo>"
	* Bikeshed supports [the numeric-range restriction](https://drafts.csswg.org/css-values/#numeric-ranges) on type autolinks, such as `<<length [0,10px]>>`. Infinities in the range can be written as a literal `∞` character, an `&infin;` HTML character reference, or the literal string `Infinity`.
* `<<'foo'>>` is an autolink to the the property or descriptor named "foo" (used in grammars, where you need `<foo>` for non-terminals).
* `<<foo()>>` is an autolink to the function named "foo" (used in grammars)
* `<<@foo>>` is an autolink to the at-rule named "@foo" (used in grammars)

### Inside Syntax ### {#autolink-inside}

* Put the text you're trying to link to inside the autolink.

	<p class=example>`[=foo=]` will link to a `<dfn dfn>foo</dfn>`.

* If the definition is namespaced with a `for` attribute
	(see [[#dfn-for]]),
	put the `for` value before the text,
	separated by a slash.

	<p class=example>`{{Foo/bar}}` will link to a `<dfn attribute for=Foo>bar</dfn>`.

	To link *specifically* to the version of a definition *without* a `for` value,
	use an empty slash,
	like `[=/url=]`,
	which will link to a `<dfn>url</dfn>`,
	but **not** a `<dfn for=page>url</dfn>`.

	If you omit the `/`,
	it means you don't care what the `for` value of the definition is.
	The [=Assume Explicit For=] metadata can change this
	to make it mean the same as an empty slash.

* If your wrapper characters apply to more than one dfn type
	(like `{{...}}`, which can link to *any* of the IDL dfn types),
	you can append a `!!type` after the linking text
	if you need to differentiate them.
	(This is very rarely necessary.)

	<p class=example>`'foo!!descriptor'` will link to a `<dfn descriptor>foo</dfn>`,
		but *not* a `<dfn property>foo</dfn>`,
		while `'foo'` can potentially link to either.

* If you want the visible text to be different from the linking text,
	you can append `|visible text` after the linking text.

	<p class=example>`[=do foo|when foo is done=]` will link to a `<dfn dfn>do foo</dfn>`,
		but will display "when foo is done" as the visible text of the link.

	Note: Many variations on the linking text are automatically handled by Bikeshed
	and don't need manual adjustment like this;
	see [[#conjugating]] for details.

---------

All together, the order of pieces is:

1. `for` value
2. linking text
3. link type
4. visible text

like `{{Document/ready!!attribute|the Document.ready attribute}}`.

Spec/Section Autolinks {#autolink-biblio}
----------------------

<dfn lt="biblio autolink">Biblio/spec/section autolinks</dfn>
create links to other documents as a whole,
or sections of those documents,
rather than to particular definitions.
They come in three closely-related syntaxes:

1. Biblio/document autolinks, like `[[FOO]]` or `[[!FOO]]`,
	which specify the shortname of a document
	(from [SpecRef](https://www.specref.org/)).
	See [[#biblio]] for more details.

2. Local section autolinks, like `[[#foo]]`,
	which link to another section in your current document by its ID.

3. Cross-spec section autolinks, like `[[FOO#bar]]`,
	which link to a section of another document,
	and mixes the previous two syntaxes.

See [[#section-links]] for details on the latter two.

Like [=dfn autolinks=], if you want to display a specific visual text
rather than what the autolink would normally display,
you can append `|visible text` to the internals,
like `[[CSS-FOO|the CSS Foo spec]]`.

These are all controlled by `Markup Shorthands: biblio yes`.

Manual Autolinks {#autolink-manual}
----------------

If one of the shorthand syntaxes don't work for whatever reason,
even if you use the <{l}> element,
you can use an ordinary <{a}> element to specify a <dfn>manual autolink</dfn>.

Any <{a}> without an `href` attribute is an autolink.

* By default, it's a "dfn" autolink, same as the `[=...=]` syntax.
	To override this, add the link type as an attribute,
	similar to <{dfn}>,
	like `<a attribute>...</a>` to autolink to an attribute.

* By default, the visible text and linking text are the text content.
	To override this,
	specify linking text in an `lt` attribute,
	like `<a lt="do foo">when foo is done</a>`.

* If the definition is namespaced by a `for` attribute,
	add a `for` attribute,
	like `<a attribute for=Document>ready</a>`.

	To link *specifically* to the version of a definition *without* a `for` value,
	use an empty `for` attribute,
	like `<a for>a</a>`,
	which will link to a `<dfn>url</dfn>`,
	but **not** a `<dfn for=page>url</dfn>`.

	If you omit the `for`,
	it means you don't care what the `for` value of the definition is.
	The [=Assume Explicit For=] metadata can change this
	to make it mean the same as an empty slash.

* To link to a specific *spec’s* version of a particular definition,
	add a `spec` attribute with the shortname of the spec you're linking to.

-----------

All together, it might look like <code highlight=html>&lt;a attribute for=Document lt=ready>the Document.ready attribute&lt;/a></code>.





Link Types {#link-types}
----------

Links have the same types as definitions, with a few additional "union" types that are used by the shortcut forms.
While you shouldn't specify them explicitly,
they'll show up in error messages sometimes,
so here's a list of them:

* "propdesc" - used by the `'foo'` shorthand.  A union of "property" and "descriptor".
* "functionish" - used by the `''foo()''` shorthand for things that look like functions.  A union of "function", "method", "constructor", and "stringifier".
* "maybe" - used by the rest of the `''foo''` shorthand values.  A union of "dfn" and all the CSS types except "property" and "descriptor".
	For legacy reasons, this link type has the additional magic that it doesn't flag an error if it can't find any matches,
	because it's also used to annotate inline CSS code fragments.
* "idl" - used by the `{{foo}}` shorthand. A union of all the IDL types.
* "idl-name" - used by the IDL auto-parser. A union of all the IDL types that can declare IDL argument types, like "interface", "enum", or "dictionary".
* "element-sub" - used by the `<{foo/bar}>` shorthand. A union of "element-attr" and "element-state".

When you actually run the processor, you may get errors about there being too many possible references to choose from.
The processor will continue to run anyway, but its default choice might be the wrong definition.
There are three things you might have to do to fix these:

1. Specify the type explicitly, if the link isn't being processed as the correct type.
	Like definitions, this can be done by just adding the type as a boolean attribute on the link,
	or by adding a `link-for` attribute to a container.
	If the link is using shorthand syntax, you can use the `!!type` suffix to specify the type.

2. If the link type corresponds to one of the definition types that needs `for` to be specified,
	you may need to specify `for` on the link as well to narrow down which definition you're referring to.
	For example, many CSS properties define an "auto" value;
	to link to the "auto" value of the 'width' property in particular,
	specify `<a value for=width>auto</a>`,
	or the shorthand syntax `''width/auto''`.
	To refer to a value of a descriptor,
	you *can* be completely explicit and specify the at-rule as well,
	like `<a value for='@counter-style/system'>numeric</a>`,
	but you're allowed to omit the at-rule if there are no other properties or descriptors with the same name,
	like `''system/numeric''`.
	This might trigger errors in the future if a conflicting term gets added later,
	but it keeps your links shorter for now.

	Again, you can specify a `link-for` attribute on a container to default it for all the autolinks inside the container.
	Alternately, you can specify `link-for-hint` on a container,
	which'll use the hint as the for value *if possible*
	(if doing so wouldn't eliminate all the possible links).
	This is useful if some container has a bunch of links for a given property, say,
	but *some* of the links are to other things entirely;
	using `link-for` means you have to manually specify the other links aren't for anything,
	but `link-for-hint` is more "do what I mean".

3. If multiple specs define the same term, you may need to declare which spec you're referring to.
	(The processor is smart enough to automatically figure out which one you probably want in many cases.)
	Just add a `spec` attribute with the spec's shortname to either the link or a container.
	This can also be specified document-wide,
	as described in [[#link-defaults]].
	(You can add this to a [=dfn autolink=] by wrapping it in an <{l}> element,
	or by fully converting to an <{a}> element,
	and adding a `spec` attribute.)

As a final note, the autolinking algorithm will link differently based on whether the spec being processed is a "current" (up-to-date) or "snapshot" (generated for a past date) draft.
If "current" (ED, UD, etc.), it'll prefer to link to other current drafts, and will only link to "snapshot" if no "current" version of that spec exists.
(If a definition only exists in the "snapshot" draft but not the "current" draft,
that almost certainly means it's been deleted since the "snapshot" draft was last published,
and thus shouldn't be linked to.)
On the other hand, "official" (WD, CR, etc.) specs will preferentially link to other official specs.
A future version of the processor will likely enforce the W3C's linking policy more strongly:
preventing CRs from linking to EDs at all,
preventing RECs from linking to anything below CR,
etc.

If you need to override the processor's choice for which status to link to for a particular link,
provide a `status` attribute containing either "ED" or "TR" on the link or a container.

Linking to Unexported Definitions {#linking-to-unexported-definitions}
---------------------------------

Most definition types are automatically exported and made available for cross-linking,
but "dfn" type definitions aren't,
because specs often define terms for their own internal use that aren't meant to be used outside the spec
(and in particular, aren't named in a way so as to avoid collisions).

If a spec contains a "dfn" type definition that you want to link to,
but it's not marked for export
(either intentionally, or because it was accidentally missed and fixing the spec would be time-consuming),
using the `spec` attribute (defined above) will override the lack of an export declaration,
and go ahead and link to it anyway.

Link Attributes On Ancestors {#link-ancestors}
----------------------------

All of the autolinking attributes
can be specified on an ancestor of the autolink instead,
if you have a group of links that all share a particular attribute.

They can be spelled out fully
(like `data-link-type`),
or without the `data-` prefix,
like `link-type`.
As there's no overlap with a similar `data-dfn-*` attribute,
`link-spec` and `link-status`
can omit `link-` as well,
being spelled as merely `status` and `spec`.

A link will take the "closest" version of each attribute it can find,
so specifying an attribute directly on the link
will override anything coming from an ancestor.



Configuring Linking Defaults {#link-defaults}
----------------------------

When there are multiple definitions for a given term
and Bikeshed can't automatically tell which one you want,
it'll emit a warning asking you to specify more explicitly.
You can do this per-link,
but you typically want to make the same choice every time the term is autolinked;
this can be done by adding a `<pre class='link-defaults'>` block,
written in the [InfoTree](#infotree) format.
Each piece of info must have a `spec`, `type`, and `text` line,
and optionally a `for` line if necessary to further disambiguate.

Sometimes this is too fine-grained,
and you'd actually like to completely ignore a given spec when autolinking,
always preferring to link to something else.
To do this, add a `<pre class='ignored-specs'>` block,
written in the [InfoTree](#infotree) format.
Each piece of info must have a `spec` line,
and optionally a `replacedBy` line,
both naming specs.
If the info has just a `spec` line, that spec is ignored totally by default;
linking to it requires you to manually specify a `spec=""` attribute on the autolink.
If the info has a `replacedBy` line,
then whenever an autolink has a choice between the two specs,
it'll delete the `spec` value from consideration,
leaving only the `replacedBy` value
(plus any other specs that might be providing a definition).

Potentially-ambiguous `for`-less Links {#ambi-for}
--------------------------------------------------

Advisement: If you were directed here by a Bikeshed error message,
put an explicit `for` value on your autolink.
(Use `for="/"` to link to a definition without a `for` value of its own).

Bikeshed's autolinking functions as a series of filters,
some of which are optional.
For example,
if you don't specify a `for` value,
then Bikeshed simply doesn't care about the `for` values of definitions as it searches.
However, this doesn't always match people's mental models--
in particular, people sometimes implicitly assume that an autolink without a `for`
will specifically match a *definition* without a `for`.

Usually this is fine;
if there are multiple possible definitions,
Bikeshed will just throw up a linking error
informing you of how to be more specific.
But if you have a definition of that term *with* a `for` value in your local spec
(or your anchors block),
Bikeshed can silently select that as the correct definition to link to,
causing accidental spec corruption if you meant to link to a cross-spec definition without a `for`.

In other words,
if some other spec defines <code>&lt;dfn>term&lt;/dfn></code>,
and then your spec both defines <code>&lt;dfn for=foo>term&lt;/dfn></code>
and links to <code>&lt;a>term&lt;/a></code>
(expecting the link to go to the cross-spec definition),
you'll be disappointed,
but won't know it's wrong unless you manually check your links.

Bikeshed looks out for this situation,
and flags it as a potentially-ambiguous link,
requiring you to specify the `for` value explicitly.
If you want to link to the `for`-less cross-spec definition,
simply add `for="/"` to your autolink,
to indicate explicitly that you want the definition without a `for` value.
Otherwise, add the appropriate `for` value as normal.


Section Links {#section-links}
------------------------------

Sometimes you want to link to a section of a document,
rather than a specific definition.
Bikeshed has section links to handle this case more easily:

```html
[[#heading-id]]
```

renders as:

```html
<a href="#heading-id">§6.1 The Example Section</a>
```

Note that this is quite different from normal autolinks;
rather than matching on text and letting Bikeshed fill in the href,
you match on href and let Bikeshed fill in the text.
This is because section titles change much more often than definition texts,
so using text-based matching is fragile;
on the other hand,
their IDs tend to be stable,
as they're often linked to.
Also, the section titles are often long and annoying to type,
and they move around,
so numbering isn't stable.

You can also use **cross-spec** section links,
as long as the spec is either in Bikeshed's linking database,
or the biblio database.
The syntax is a mixture of a biblio reference and a section link:

```html
[[css-flexbox-1#auto-margins]]
[[CSS-access-19990804#Features]]
```

which renders as:

```html
<a href="https://drafts.csswg.org/css-flexbox-1/#auto-margins">CSS Flexbox 1 §8.1 Aligning with auto margins</a>
<a href="http://www.w3.org/1999/08/NOTE-CSS-access-19990804#Features">Accessibility Features of CSS §Features</a>
```

If Bikeshed knows about the spec,
it link-checks you,
and fills in the section number and heading in the generated text.
If the spec is only in the bibliography database,
Bikeshed just assumes that the link target exists
and uses it directly in the text,
because it has no way to tell what the section is named.

If the spec is multipage, like SVG,
and Bikeshed knows about it,
*most* of the time you don't need to do anything different -
Bikeshed will find the correct page for the heading you're linking to.
On the rare occasions that the same heading id exists in multiple pages of the same spec, tho,
specify the page like `[[svg/intro#toc]]`
(which indicates the #toc heading on the intro.html page).
If the desired heading is on the top-level page,
use an empty page name, like `[[html/#living-standard]]`.
In any case, Bikeshed will throw an error,
and tell you what names it knows about so you can easily correct your link.

<!--
████████  ████ ████████  ██       ████  ███████
██     ██  ██  ██     ██ ██        ██  ██     ██
██     ██  ██  ██     ██ ██        ██  ██     ██
████████   ██  ████████  ██        ██  ██     ██
██     ██  ██  ██     ██ ██        ██  ██     ██
██     ██  ██  ██     ██ ██        ██  ██     ██
████████  ████ ████████  ████████ ████  ███████
-->

Bibliography {#biblio}
============

Bibliographical references form a special class of autolinks.
They're typically added *only* via the shorthands
`[[FOO]]` for informative references
and `[[!FOO]]` for normative references.

(A "normative reference" is a reference that your document depends on;
something in the other document is *required*
in order to implement your document correctly.
An "informative" reference is anything else.
Informative references might be to normative documents,
if your feature doesn't actually *depend* on the referenced document.)

Unlike regular autolinks,
which link to <{dfn}> elements,
biblio autolinks cause the spec to generate entries in its "References" section,
and then link to that instead.

The bibliography database is completely separate from the autolinking database,
and comes from multiple sources.
The default data comes from the [SpecRef project](https://github.com/tobie/specref)
and [the CSSWG's own biblio file](http://dev.w3.org/csswg/biblio.ref)
(preferring SpecRef's information when the same name appears in both).

Biblio Link Modifiers {#biblio-mods}
---------------------

Several aspects of how bibliography autolinks work can be controlled by adding keywords to the autolink,
after the biblio name.

* Some biblio entries come with multiple sets of urls;
	at present, Bikeshed tracks a single "current" url and a single "snapshot" url.
	In the W3C, for example, this maps to Editors Drafts and things in /TR space, respectively.

	You can specify which url to use by specifying `current` or `snapshot` within the biblio shorthand,
	like `[[FOO current]]`,
	or specify the default url to choose for all your biblio refs with the <a>Default Ref Status</a>.

* By default, biblio autolinks are formatted like you typed, just with a single set of square brackets:
	`[[FOO]]` will produce a link with the text "\[FOO]",
	which links to the bibliography.
	Alternately, you can force them to display "inline",
	where the link text is the actual title of the referenced document,
	and the link goes straight to that document,
	rather than going to the bibliography first.
	(It still adds an entry to the bibliography index.)

	You can control this by setting `inline` or `index` in the shorthand,
	like `[[FOO inline]]`,
	or change the default for all biblio links in the document
	by setting the [=metadata/Default Biblio Display=] metadata.

* The bibliography database tracks whether certain documents have been obsoleted by newer documents,
	and if you attempt to link to the the now-obsolete document,
	Bikeshed defaults to linking to the newer document instead
	and asking you to update the reference.

	If you do intentionally need to link to a now-obsolete document
	(for example, if you're talking about a feature in the old document
	that's since been replaced),
	you can tell Bikeshed you do intend to link to that by setting `obsolete` in the shorthand,
	like `[[FOO obsolete]]`.

All of these keywords can be combined, in any order,
such as `[[FOO inline obsolete]]` or `[[FOO index current]]`.


Autolinking and Automatic Bibliography Entries {#biblio-auto}
---------------------------------------------------

Whenever you autolink to a term from another spec
using a standard dfn autolink,
Bikeshed automatically adds a bibliography entry for that document
if one doesn't already exist.

By default, it assumes these are normative references.
However, if the link is in a "non-normative" element,
Bikeshed doesn't put it in the bibliography.
This includes any notes (`class=note`), examples (`class=example`),
or sections manually marked with `class=non-normative`.
(You can override this and get a normative reference
by adding a `class=normative` to the link or an ancestor.)


Manual Biblio Links {#biblio-manual}
-------------------

If, for whatever reason, you need to craft a bibliography link manually,
just create an <{a}> element with `data-link-type=biblio`
and the biblio tag either in the text content of the link or in a `data-lt` attribute,
similar to a manually-crafted autolink.

If you need one of the biblio link modifiers,
add `data-biblio-type=[normative | informative]`,
`data-biblio-status=[current | snapshot]`,
`data-biblio-display=[index | inline]`,
and/or `data-biblio-obsolete=""` to the link as well.


Custom Biblio Entries {#biblio-custom}
---------------------

You can also add your own bibliography data,
following the [SpecRef JSON format](https://github.com/tobie/specref):

```json
{
	"foo-bar": {
		"authors": [
			"Tab Atkins",
			"Bat Snikta"
		],
		"href": "http://www.example.com/foo-bar/",
		"title": "Foo Bar Level 1",
		"status": "CR",
		"publisher": "W3C",
		"deliveredBy": [
			"http://www.w3.org/html/wg/"
		]
	}
}
```

This would allow you to write `[[FOO-BAR]]` into your document,
and have it be recognized as a valid autolink,
with a bibliography entry generated in the index
just like any of the predefined biblio references.

Note: Only the "title" field is strictly necessary;
the rest can be omitted if desired.

This JSON should be inline, in a `<pre class=biblio>` block.
It can also be in a `biblio.json` file in the same folder as the Bikeshed file,
but this is incompatible with [[#install-no|the web service]].

If you find yourself adding this to more than one document,
it might be worth adding to SpecRef itself
so others can benefit from it as well;
file an issue on that project if you're interested.


<!--
████████  ████████ ████████ ████████  ██        ███████   ██████  ██    ██  ██████
██     ██ ██       ██       ██     ██ ██       ██     ██ ██    ██ ██   ██  ██    ██
██     ██ ██       ██       ██     ██ ██       ██     ██ ██       ██  ██   ██
██     ██ ██████   ██████   ████████  ██       ██     ██ ██       █████     ██████
██     ██ ██       ██       ██     ██ ██       ██     ██ ██       ██  ██         ██
██     ██ ██       ██       ██     ██ ██       ██     ██ ██    ██ ██   ██  ██    ██
████████  ████████ ██       ████████  ████████  ███████   ██████  ██    ██  ██████
-->

Definition Tables {#def-blocks}
=================

Several types of definitions have well-established presentations,
as a simple "key/value" table providing all of the details
in a standardized format.

Bikeshed handles these specially,
with a dedicated set of `<pre class=xxxdef>` blocks.
The information in the block is presented in a simple line-based `Key: value` format,
similar to `<pre class=metadata>`,
and this is transformed into a standardized <{table}>.

All of the def tables look like this:

<xmp highlight=html>
	<pre class='propdef'>
	Name: flex-basis
	Value: content | <<'width'>>
	Initial: auto
	Inherited: no
	Applies to: <a>flex items</a>
	Computed value: as specified, with lengths made absolute
	Percentages: relative to the <a>flex container's</a> inner <a>main size</a>
	Animation type: length
	Canonical Order: per grammar
	</pre>
</xmp>

CSS Property Tables {#propdef-block}
-------------------

CSS properties are defined with a `<pre class="propdef">` block.

`Name`, `Value`, `Initial`, and `Inherited` are all required.

* `Name` gives the property name.
	It will be automatically wrapped in a <{dfn}> by Bikeshed,
	so there's no need to do anything manually.
	Multiple comma-separated property names can be given,
	if several properties share identical propdef blocks.
* `Value` gives the grammar for the property,
	written in CSS's [[css-values-4#value-defs|Value Definition Syntax]].
	Anything that looks like a grammar non-terminal (like `<foo>`)
	will automatically be escaped and linked appropriately,
	so you don't need to manually autolink it.
* `Initial` gives the initial value of the property.
* `Inherited` must be "yes" or "no".

All the other keys listed in the block above are optional.
If left unspecified,
they'll be given a usually-reasonable default value:

* `Applies to` defaults to "all elements".
	It specifies what subset of elements the property has meaning for.
* `Percentages` defaults to "n/a".
	It specifies what value percentages are resolved against,
	if the property takes percentage values at all.
* `Computed Value` defaults to "as specified".
	It specifies what, if any, canonicalization occurs
	when transforming specified values to computed values.
	(Complex values should use "see prose" and define it in prose instead.)
* `Canonical Order` defaults to "per grammar".
	It specifies what order values are serialized in,
	if values are rearranged when parsing.
	(The default is almost always correct;
	usually only properties with unusual legacy constraints on their serialization
	will need to specify anything for this.)
* `Animation Type` defaults to "discrete".
	It specifies the [=Animation Type=] of the values.

All values are placed directly into the output HTML as written,
so HTML tags, Bikeshed autolinks, etc can all be used as normal.

If defining a shorthand property,
put a `"shorthand"` class on the block.
This will change the block to only require `Name` and `Value`;
`Canonical Order` is still allowed,
but continues to default to "per grammar" as usual.
All other keys will default to "see individual properties",
which is usually all you need.

If adding some values to a property,
rather than defining the property itself,
put a `"partial"` class on the block.
It will now require a `New Values` key,
rather than `Value`,
but is otherwise normal.

Additional keys beyond what is described here can be provided;
they're placed after all the built-in keys.

The end result looks like:

<pre class='propdef'>
Name: flex-basis
Value: content | <<'width'>>
Initial: auto
Inherited: no
Applies to: <a>flex items</a>
Computed value: as specified, with lengths made absolute
Percentages: relative to the <a>flex container's</a> inner <a>main size</a>
Animation type: length
Canonical Order: per grammar
</pre>


CSS Descriptor Tables {#descdef-block}
---------------------

CSS descriptors
(like properties, but for at-rules)
have def blocks similar to,
but simpler than,
propdef blocks.

`Name`, `Value`, and `Initial` are required keys,
just like propdef blocks.
Additionally, `For` is required,
giving the name of the at-rule the descriptor is for,
like `For: @counter-style`.
The rest of the propdef keys are not meaningful for descriptors.

Like propdef blocks,
a `"partial"` class can be put on the block
to indicate that you're adding values to an existing definition;
`Name`, `For`, and `New Values` must be provided.

While not technically descriptors,
media features can be defined with this block as well,
if an `"mq"` class is put on the block.
It then only requires `Name`, `For`, and `Value`.

Additional keys beyond what is described here can be provided;
they're placed after all the built-in keys.


HTML Element Tables {#elementdef-block}
-------------------

HTML/SVG/etc elements can use a `<pre class="elementdef">` block.
(The content and presentation of this block is similar to that in the HTML standard,
but is closer to that in the SVG standard.)

* **Name** - required.
	Gives the element name.
	Can provide multiple element names, comma-separated.
	The names will automatically be wrapped in a <{dfn}> for you.
* **Categories** - required.
	What [[html#kinds-of-content|element categories]] the element is part of.
	Can provide multiple values, comma-separated.
	The names will automatically be wrapped in links for you,
	of `dfn` type.
* **Contexts** - required.
	The [[html#element-definitions|contexts in which the element can be used]].
	This is freeform HTML.
* **Content model** - required.
	The [[html#element-definitions|types of contents the element can contain]].
	This is freeform HTML.
* **Attributes** and/or **Attribute Groups** - at least one is required.
	A comma-separated list of attribute names
	and/or attribute groups
	that this element can have.
	These will be automatically be wrapped in links for you--
	`element-attr` type for attributes,
	and `dfn` type for attribute groups
	(like "core attributes" in SVG).
* **DOM interfaces** - required.
	A comma-separated list of WebIDL interface names
	corresponding to the elements being defined.
	The names will automatically be wrapped in links for you,
	of `interface` type.

Additional keys beyond what is described here can be provided;
they're placed after all the built-in keys.


IDL Method Argument Tables {#argumentdef-block}
--------------------------

IDL methods can have their arguments defined & documented in a `<pre class="argumentdef">` block:

<xmp highlight=html>
	<pre class=idl>
		interface Foo {
			undefined bar(long arg1, DOMString? arg2);
		};
	</pre>

	<pre class=argumentdef for="Foo/bar()">
	arg1: here's the first arg
	arg2: the second arg is different!
	</pre>
</xmp>

The `argumentdef` block must specify what method the arguments are for
in a `for` attribute on the `<pre>`.
The contents of the element must then,
on each line,
list the argument name,
a colon,
then a description of that argument.

It will produce a table listing each argument,
its description,
and automatically contain its type, nullability, and optionality
(determined from the IDL block),
like:

<pre class=idl style="display:none">
	interface Foo {
		undefined bar(long arg1, DOMString? arg2);
	};
</pre>

<pre class=argumentdef for="Foo/bar()">
arg1: here's the first arg
arg2: the second arg is different!
</pre>

Note: This automatically wraps the argument names in a `<dfn>`;
this is meant to be the canonical location of the argument's definition.



<!--
████ ████████  ██
 ██  ██     ██ ██
 ██  ██     ██ ██
 ██  ██     ██ ██
 ██  ██     ██ ██
 ██  ██     ██ ██
████ ████████  ████████
-->

WebIDL Processing {#idl}
========================

Bikeshed can automatically process IDL blocks,
marking up all relevant terms for you without any intervention,
setting up definitions and autolinks as appropriate.

To activate this behavior,
simply place the IDL in the `<pre class='idl'>` element.
Bikeshed will consume the text content of the element
(ignoring any markup you may currently have)
and replace it with marked-up text containing <{dfn}> and <{a}> elements.

<div class=note>
	As mentioned in [[#xmp]],
	you can use `<xmp class=idl>` as well,
	so you don't have to escape your `sequence<Foo>`/etc types.
	This prevents you from using other markup in your IDL,
	but that's relatively rare.

	For IDL specifically, you can also use `<script type=idl>`.
	This has similar effects to `<xmp>`,
	but it interacts better with the syntax-highlighting of some text editors,
	preventing the contents from being highlighted as HTML.
</div>

In the process of doing this, Bikeshed will also syntax-check your IDL,
and report fatal errors for any mistakes.
Bikeshed's IDL parser, courtesy of Peter Linss, is intended to be forward-compatible with IDL changes,
gracefully emitting unknown constructs unchanged and recovering as well as it can.
If anything isn't recognized when it should be,
or the parser fails in a major, non-graceful way,
please report it as an issue.

Putting Definitions Elsewhere {#idl-elsewhere}
-----------------------------

Quite often, you may want to have the actual definition of an IDL term
(the thing that Bikeshed actually links to)
somewhere in your prose near the full definition,
rather than being in the IDL block.

Bikeshed will automatically produce an <{a}> in your IDL,
rather than a <{dfn}>,
if it can find a pre-existing definition of that IDL term,
including local definitions in the current spec.
However, you have to mark up the definition correctly to get this to work,
or else Bikeshed will fail to recognize there's an external definition
and will mark up the IDL with a <{dfn}> as well.

In particular, method and attribute definitions need to have their `for` value set to the interface they're a part of
(and similar with dictionary members).
Methods have some further complexity -
they should have their definition text set to contain the names of all their arguments.

For example, take the following example IDL:

```
interface Foo {
	undefined bar(DOMString baz, optional long qux);
};
```

To have Bikeshed recognize a definition for the `bar()` method placed elsewhere,
it must look something like `<dfn method for=Foo title="bar(baz, qux)">bar(DOMString baz, optional long qux)</dfn>`.

Additionally, it *should* define alternate linking texts for omittable arguments,
like `<dfn method for=Foo title="bar(baz, qux)|bar(baz)">bar(DOMString baz, optional long qux)</dfn>`.
This way any valid call signature can be used to autolink.
Note that arguments are omittable if they're marked with `optional`, or are variadic (like `long... qux`), or have a default value.
Nullable arguments (like `long? qux`) are not omittable.
(If you are fine with the <{dfn}> being in the IDL block,
Bikeshed will do all of this for you.)

Unless all arguments can be omitted, the definition text *should not* have an alternative with empty args.
For convenience, however, Bikeshed will allow *autolinks* with empty argument lists to work,
as long as it can resolve the link unambiguously.
For example, `{{Foo/bar()}}` will autolink to the method defined above,
despite it not being a valid call signature,
as long as there isn't an overload of `bar()` that it might also apply to.

(The above applies to all functionish types: method, constructor, stringifier, etc.)

Marking up argument definitions is similar.
To mark up the `baz` argument of the above method, for example,
do `<dfn argument for="Foo/bar(baz, qux)">baz</dfn>`.
You *should* use the full call signature of the method.


### Free Type/Etc Documentation on Attributes/Dict Members ### {#attribute-info}

Bikeshed automatically gathers information from your IDL block
about attributes and dictionary members,
such as their type,
their default value,
and whether they're readonly or nullable.
You can have Bikeshed automatically output that information in your definition elsewhere,
by using a specific markup structure:

```
<dl dfn-type=attribute dfn-for=FooInterface>
  <dt><dfn>fooAttr</dfn>
  <dd>...

  <dt><dfn>barAttr</dfn>
  <dd>...
</dl>
```

If an attribute or dictionary member is defined in a <{dt}> element,
and is the sole content of that element,
Bikeshed will automatically add useful information about the attribute following it.

You can also invoke this manually,
by adding a `<span data-attribute-info for="FooInterface/fooAttr"></span>` element anywhere in your spec
(or `data-dict-member-info` for dictionary members).

Linking to Stringifiers {#linking-to-stringifiers}
-----------------------

Linking to a stringifier is a little complicated,
because WebIDL allows *four* different syntaxes for it.

The `stringifier` keyword itself is always linkable;
it's a "dfn" type definition with `for=MyInterface`
and linking text "stringification behavior".
Like any other IDL construct,
you can instead define the term yourself in the same way,
and the IDL will link to your definition instead,
like `<dfn dfn for=MyInterface>stringification behavior</dfn>`.
This is generally what you *should* use to link to the stringifier,
as it'll maintain the links even if you change which syntax form you use.

If you use the "stringifier attribute" form,
like `stringifier attribute DOMString href;`,
you can also just link/dfn the attribute as normal.

If you use the "stringifier method" form,
like `stringifier DOMString foo(long bar);`,
you can also just link/dfn the method as normal,
like `<dfn stringifier for=MyInterface>foo(bar)</dfn>`.
(Note that it's a "stringifier" type definition,
not "method".)

If you use the "*anonymous* stringifier method" form,
like `stringifier DOMString(long bar)`,
you can still technically link/dfn it as a stringifier method.
It doesn't have a name, so we invent one -
it's called `__stringifier__()`, a la Python's magic methods.
(Note the *two* underscores on each side.)
You should *almost* never need to do this;
the only reason to need to specify the method name
(rather than just linking to the keyword, as described above)
is if you're linking/dfning an argument to the method,
and need to specify a `for` value for it.


Turning Off Processing {#no-idl}
----------------------

If for whatever reason you don't want your IDL block to be processed by Bikeshed,
simply use another element, or another class.
If you really want to use `<pre class=idl>`,
you can add a `data-no-idl` attribute to the element.
Bikeshed will leave these elements alone.

Alternately, if your block is IDL,
but it's not meant to be taken literally
(for example, if it shows an example attribute,
then explains in prose the set of actual attribute names to be used,
as in the <a href="https://drafts.csswg.org/cssom/#dom-cssstyledeclaration-_camel_cased_attribute">_camel_cased_attribute in CSSOM</a>),
put a `class=extract` on it.


<!--
██      ██ ████████  ████████
██  ██  ██ ██     ██    ██
██  ██  ██ ██     ██    ██
██  ██  ██ ████████     ██
██  ██  ██ ██           ██
██  ██  ██ ██           ██
 ███  ███  ██           ██
-->

Testing Integration With WPT {#testing}
============================

When writing specifications for the open web,
the canonical way to test new features and ensure interoperability
is to contribute the testsuite to the [Web Platform Tests](https://wpt.fyi/about) project.
Bikeshed has some useful tools to interact with WPT information.

<h3 id="wpt-element" lt="wpt" element>
Annotating Specs with Tests: the <{wpt}> element</h3>

When writing tests,
you can sometimes link to the section of the spec that you're testing,
to make it easier to review.
But when you're actually reading (or updating!) the spec,
you can't tell what sections are tested or untested,
or what tests might need to be updated due to a change you're making.
The <{wpt}> element lets you easily link to your WPT testcases inline,
right next to the text they're testing,
and helps make sure that the testsuite and the spec are kept in-sync.

The <{wpt}> element is a block-level element for parsing purposes;
place it after or between paragraphs/etc.
The contents of the element are line-based:
each line contains a single test path
(the path under the [WPT repo](https://github.com/web-platform-tests/wpt),
so not including the domain or the `/web-platforms-tests/wpt/`)
pointing to the test covering some nearby text.


<div class=example>
	For example:

	<xmp highlight=html>
		Implementations must FOO whenever they would also BAR.

		<wpt>
			/foo-spec/foo-whenever-you-bar.html
			/foo-spec/no-foo-when-no-bar.html
		</wpt>
	</xmp>
</div>

If all or most of your tests have a common path prefix,
you can specify it in [=WPT Path Prefix=] metadata,
and then leave it off of all the individual test lines.
An explicit <dfn element-attr for="wpt, wpt-rest">pathprefix</dfn> attribute
can also provide the path prefix,
and will override the metadata if specified.

<div class=example>
	If the preceding example used [=WPT Path Prefix=],
	the <{wpt}> element could be shorter:

	<xmp highlight=html>
		<pre class=metadata>
		WPT Display: open
		WPT Path Prefix: /foo-spec/
		</pre>

		...

		Implementations must FOO whenever they would also BAR.

		<wpt>
			foo-whenever-you-bar.html
			no-foo-when-no-bar.html
		</wpt>
	</xmp>
</div>

For debugging purposes,
a single <dfn element lt=wpt-rest>&lt;wpt-rest></dfn> element can be used in a spec,
which will act like a <{wpt}> element containing all the tests under the current path prefix
that aren't already specified in a <{wpt}> element.
As this defeats Bikeshed's ability to tell you when you're missing any tests,
Bikeshed will emit a warning as long as a <{wpt-rest}> element is in use.

Bikeshed's knowledge of valid tests allows some useful checks:

* If [=WPT Path Prefix=] is specified,
	then any tests under that prefix that aren't specified in a <{wpt}> element
	will trigger a fatal error,
	so you'll be alerted when new tests are added.
* If you specify a path that doesn't exist
	(or *no longer* exists due to tests being moved or deleted),
	it will trigger a fatal error.

By default, <{wpt}> elements don't display in the output document;
they're intended to annotate the source file to help with spec authoring,
and trigger Bikeshed's checks.
If you want to produce a test-annotated version of the output,
specify the [=WPT Display=] metadata with the value "open" or "closed";
all of the <{wpt}> elements will become usefully-formatted lists of their contained tests,
with links to <a href="https://wpt.fyi">wpt.fyi</a>,
the live test on <a href="https://w3c-test.org">w3c-test.org</a>,
and the source code on [GitHub](https://github.com/web-platform-tests/wpt).
The difference between "open" and "closed" is that
with the former, all <{wpt}> elements are initially displayed in a collapsed state
to avoid getting in the way of reading the document,
while "open" causes them to be expanded initially.
In either case, the use can interactively collapse or expand each element.

<figure>
	<img srcset="images/wpt-display-open.png 2x">
	<figcaption>
		When `WPT Display: open` is set,
		the <{wpt}> element turns into a "TESTS" block.
	</figcaption>
</figure>
<figure>
	<img srcset="images/wpt-display-closed.png 2x">
	<figcaption>
		When `WPT Display: closed` is set,
		the <{wpt}> element also turns into a "TESTS" block,
		but it is initially collapsed.
	</figcaption>
</figure>

Note: For compatibility with earlier versions of Bikshed,
"inline" is accepted as a synonym of "open".

When the  <a element-attr>hidden</a> attribute is set on the <{wpt}> element,
Bikeshed will take the tests listed in that element into account,
but exclude them from the output,
even when producing a test-annotated version.

Note: This can be useful
when tests for all versions or levels of a single specification are stored together
under a single directory.
Using [=WPT Path Prefix=] would cause Bikeshed to complain
if the tests are not listed,
but the tests may only be relevant for a later version of the specification.
In such cases,
listing the tests in a <code highlight=html>&lt;wpt hidden></code> element silences the warnings,
without adding anything irrelevant to the output.

Information about a set of tests described in a <{wpt}> element
can be provided to the reader
by writing it in the <a element-attr for=html-global>title</a> attribute of the <{wpt}> element.
This will be inserted in the output as an introductory paragraph
before the test list.
A <a element-attr for=html-global>lang</a> or <a element-attr>dir</a> set on the <{wpt}> element
will apply to the introductory paragraph if there is one.

<div class=example>
	<xmp highlight=html>
		<h1 id=intro>Introduction</h1>

		This specification introduces…

		<wpt title="The following tests are crash tests
		            that relate to general usage
		            of the features described in this specification
		            but are not tied to any particular normative statement.">
		crash-001.html
		crash-002.html
		</wpt>
	</xmp>

	The test block in the example above would be rendered as follows:

	<figure>
		<img alt="A WPT element with a list of tests, preceded by an introductory paragraph"
		     srcset="images/tests-with-intro.png 2x">
	</figure>
</div>

If multiple <{wpt}> follow each other in the source,
with nothing between them other than whitespace,
they will be fused together in the output,
visually taking less space than if they remained as separate elements.
This will work even
if they have different <a element-attr>pathprefix</a>
or <a element-attr for=html-global>title</a> attributes.
When merging elements with <a element-attr for=html-global>title</a> attributes,
the tests listed in each element will be preceded by the relevant introductory paragraph,
and visually separated from tests listed in different <{wpt}> elements.

Note: More tools and options for integrating with WPT will be developed in the future.


Updating Lists of Tests {#updating-tests}
-----------------------

If Bikeshed reports that a test doesn't exist,
but you're sure that it does
(and have checked the spelling!),
make sure your data files are up-to-date with `bikeshed update`
(see [[#cli-update]]) for details).

If you've *just* pushed a new test to WPT
and Bikeshed's update hasn't picked it up yet,
try doing a manual update with `bikeshed update --wpt --skip-manifest`.




<!--
████████   ███████  ████ ██       ████████ ████████  ████████  ██          ███    ████████ ████████
██     ██ ██     ██  ██  ██       ██       ██     ██ ██     ██ ██         ██ ██      ██    ██
██     ██ ██     ██  ██  ██       ██       ██     ██ ██     ██ ██        ██   ██     ██    ██
████████  ██     ██  ██  ██       ██████   ████████  ████████  ██       ██     ██    ██    ██████
██     ██ ██     ██  ██  ██       ██       ██   ██   ██        ██       █████████    ██    ██
██     ██ ██     ██  ██  ██       ██       ██    ██  ██        ██       ██     ██    ██    ██
████████   ███████  ████ ████████ ████████ ██     ██ ██        ████████ ██     ██    ██    ████████
-->

Boilerplate Generation {#boilerplate}
======================

The processor automatically generates nearly all of a spec's boilerplate,
the text that is repeated nearly identically across all specs.

Generally, you won't need to understand what's going on here in order to use the processor - it'll just automatically do the right thing.

For help in creating *new* boilerplate files for your organization, see [[#bp-new]].

Groups {#bp-groups}
------

Much of the boilerplate is determined based on the <a for=metadata>Group</a> metadata.
If unspecified, it defaults to a generic set of boilerplate that is generally appropriate for most things,
without making reference to any particular standards body or the like.
However, to obtain correct boilerplate for a given standards body,
"Group" can be used.

Several groups are already accommodated with appropriate inclusion files:

* "csswg", as mentioned.
* "dap", for the Devices and Sensors Working Group
* "fxtf", for the FX Task Force
* "houdini", for the Houdini Task Force
* "svg", for the SVG Working Group
* "webappsec", for the WebApps Security Working Group
* "whatwg", for the WHATWG
* "wg14", for the C Standards Committee
* "wg21", for the C++ Standards Committee
* "tc39", for ECMAScript TC-39
* "khronos", for the Khronos Group

You can put whatever value you want into the "Group" value, though.
Unrecognized values will just use the default boilerplate files.
If you want to add specialized boilerplate files for your group,
check out the File-Based Includes section, later in this document,
and write your own files.


Rearranging and Excluding "Spec Metadata" {#rearranging-metadata}
-----------------------------------------

An important part of the boilerplate is the "spec-metadata" section.
This will likely be at the top of your header boilerplate,
as it contains a bunch of useful information about your spec.

Bikeshed generates a lot of these automatically for you,
based on the metadata you provide
and other things detected from your document,
and you can supply "custom" items
as specified near the end of [[#metadata]].
There's a predefined ordering of these,
but if you'd like a slightly different order,
or to omit some of the automatically-generated items,
you can use [=Metadata Include=] and [=Metadata Order=] to control this.

[=Metadata Include=] takes a comma-separated list of names and [=boolish=] values,
where the names are the strings that show up in the <{dt}> in the spec metadata.
Everything defaults to "on" currently,
but you can explicitly turn them "off" to omit them from the spec metadata section.

[=Metadata Order=] instead controls the ordering of the spec metadata section.
It's a comma-separated list of names,
where the names are the strings that show up in the <{dt}> in the spec metadata
(same as [=Metadata Include=]).
Two special "names" are recognized as well:
the `*` name stands in for "all the standard keys that aren't otherwise explicitly specified",
while `!*` stands in for "all the custom keys that aren't otherwise explicitly specified".
(The default value is thus `Metadata Order: *, !*`,
listing all the standard keys in the default order,
followed by all the custom keys in the default order.)

If a name is specially pluralized when there are multiple entries
(such as "Editor" vs "Editors"),
use the singular version in either of these metadatas.

<div class=example>
	For example, `Metadata Include: This version off`
	will make Bikeshed omit the "This version" entry from the spec-metadata section,
	which is otherwise auto-generated by the [=metadata/ED=] metadata.

	If you wanted to make sure that editors were listed before anything else,
	you could set `Metadata Order: Editor, *, !*`.
</div>


Text Macros {#text-macros}
-----------

Several text "macros" are defined by the spec's metadata,
and can be used anywhere in the spec to substitute in the spec they stand for by using the syntax `[FOO]`.
Note that this is similar to the syntax for bibliography references, but it has only a single set of `[]` characters, and the text must be uppercase.
The following macros are defined:

<ul dfn-type=dfn dfn-for=macro export>
	* <dfn>`[TITLE]`</dfn> gives the spec's full title, as extracted from either the H1 or the spec metadata.
	* <dfn>`[H1]`</dfn> gives the desired document heading, in case the in-page title is supposed to be different from the `<title>` element value.
	* <dfn>`[SHORTNAME]`</dfn> gives the document's shortname, like "css-cascade".
	* <dfn>`[VSHORTNAME]`</dfn> gives the "versioned" shortname, like "css-cascade-3".
	* <dfn>`[STATUS]`</dfn> gives the spec's status.
	* <dfn>`[LONGSTATUS]`</dfn> gives a long form of the spec's status, so "ED" becomes "Editor's Draft", for example.
	* <dfn>`[STATUSTEXT]`</dfn> gives an additional status text snippet.
	* <dfn>`[LATEST]`</dfn> gives the link to the undated /TR link, if it exists.
	* <dfn>`[VERSION]`</dfn> gives the link to the ED, if the spec is an ED, and otherwise constructs a dated /TR link from today's date.
	* <dfn>`[ABSTRACT]`</dfn> gives the document's abstract.
	* <dfn>`[ABSTRACTATTR]`</dfn> gives the document's abstract, correctly escaped to be an attribute value.
	* <dfn>`[YEAR]`</dfn> gives the current year.
	* <dfn>`[DATE]`</dfn> or <dfn>`[DATE-DMMY]`</dfn> gives a human-readable date like "30 January 2000".
	* <dfn>`[DATE-MY]`</dfn> gives a partial date like "Jan 2000"
	* <dfn>`[DATE-MMY]`</dfn> gives a partial date like "January 2000"
	* <dfn>`[CDATE]`</dfn> gives a compact date in the format "YYYYMMDD", like "20000130"
	* <dfn>`[ISODATE]`</dfn> gives a compact date in iso format "YYYY-MM-DD", like "2000-01-30"
	* <dfn>`[DEADLINE]`</dfn> gives a human-readable version of the deadline date (formatted like [=\[DATE]=]), if one was specified.
	* <dfn>`[LOGO]`</dfn> gives the url of the spec's logo.
	* <dfn>`[REPOSITORY]`</dfn> gives the name of the VCS repository the spec is located in;
		this is currently only filled when the spec source is in a GitHub repository.
		(Patches welcome for more repo-extraction code!)
</ul>

As these are substituted at the text level, not the higher HTML level, you can use them *anywhere*, including in attribute values.

You can mark a macro as "optional" by appending a `?` to its name,
like `[DATE?]`.
This will cause Bikeshed to just remove it
(replace it with the empty string)
if it can't find a definition,
rather than throwing an error.

Like most other markup shorthands,
text macros can be "escaped" by prepending a backslash,
like `\[TITLE]`.
When Bikeshed sees this,
it will remove the slash and leave the text alone.
This is sometimes necessary when code examples in your doc (such as a regex)
accidentally look like text macros.


Conditional Inclusion {#conditional}
---------------------

If you have multiple boilerplate files for publishing with different document statuses
(like WD vs CR),
but the only difference is that some sections are added/omitted between them,
you can greatly simplify things by instead making a generic boilerplate for everything,
and then adding/removing the variant bits with a *conditional inclusion*.

To mark an element as "conditionally included",
put an <dfn element-attr for=global>include-if</dfn> and/or <dfn element-attr for=global>exclude-if</dfn> attribute on it,
each of which takes a comma-separated list of inclusion conditions.

* If an element has <{global/include-if}>, it will included only if it matches at least one of the conditions.
* If an element has <{global/exclude-if}>, it will be included only it matches *none* of the conditions.
* If an element has both, it will be included only if it passes the inclusion criteria for both.

An element that fails will be removed from the output entirely, along with its contents.

Inclusion conditions can be:
* A [=metadata/Status=] value, like `w3c/ED` or `LS`,
	so you can include/exclude text for certain publication statuses only.
* The existence of a custom text macro (defined by the [=metadata/Text Macro=] metadata),
	keyed by the text macro name,
	like `text macro: FOO` to match a `Text Macro: FOO here's a custom text macro` metadata,
	so you can include text that uses a custom text macro only if that macro is defined.

	Note: For technical reasons, text macros are substituted *before* conditionals can be evaluated.
	If you're conditioning an element on the existence of a text macro
	so you can use that macro only when it is defined,
	make sure the text macro is marked as *optional* when you use it
	(written as `[FOO?]` rather than just `[FOO]`),
	so it doesn't trigger an "unknown text macro" error
	before throwing away the element anyway.

* The existence of a [[#overriding-boilerplate|custom boilerplate element]] in the document,
	keyed by the boilerplate name,
	like `boilerplate: FOO` to match a `<div boilerplate=FOO>...</div>` element,
	so you can include text that pulls in a custom boilerplate only if that boilerplate is present.


-----

Normally the conditional attributes go on an existing element in your page,
which'll show up in the output assuming it passes the tests.
If there's not a natural wrapper element,
or adding such an element would mess with the semantics or styling of the page,
you can instead wrap the conditional content in an <dfn element>if-wrapper</dfn> element.

The <{if-wrapper}> element is a standard block-level element that can contain any flow markup.
It **must** have one of the conditional attributes.
If it fails the conditions, it and its contents are removed from the document, as normal;
if it *passes* the conditions, *it* is still removed from the document,
but its children are left behind in its place.

<div class=note>
	<{if-wrapper}> is still parsed as a standard unknown "custom" element by the HTML parser,
	and so can only appear in places where such elements are allowed;
	it cannot, for example, be used in the <{head}> of a document,
	or as a child of <{table}>.

	It's also considered "block level" by the Markdown parser.
	It *can* be used inline in text
	(tho you probably want to instead make the entire block conditional,
	for readability/editability),
	but it won't parse as intended if it's the very first content in a paragraph.
</div>


Boilerplate Sections {#bp-sections}
--------------------

The location of the boilerplate sections are indicated by elements with `data-fill-with=''` attributes.
If the elements contain anything, they're emptied before being filled with the appropriate boilerplate.
The valid `data-fill-with=''` values are:

* "table-of-contents" for the ToC
* "spec-metadata" for the <{dl}> of spec data that's in the header of all of our specs
* "abstract" for the spec's abstract
* "status" for the status section
* "logo" for the W3C logo
* "copyright" for the W3C copyright statement
* "warning" for the relevant spec warning, if one was indicated in the metadata.
* "references" for the bibliography refs
* "index" for the index of terms (all the <{dfn}> elements in the spec)
* "property-index" for the table summarizing all properties defined in the spec
* "idl-index"
* "issues-index"

Additionally, "header" and "footer" boilerplate files are used to put content at the start and end of your document.
Most or all of the above boilerplate sections should actually show up here, in the header and footer,
rather than being manually specified in your source file.

### Default Boilerplate ### {#default-boilerplate}

Some sections listed above are generated *by default*;
if you don't put an explicitly `data-fill-with` container in your document,
they'll generate anyway (if they have anything to fill themselves with),
appending themselves to the end of the <{body}>.
These sections are:

* all of the indexes: "index", "property-index", "idl-index", and "issues-index"
* "references"

Again, these will only auto-generate if there is something for them to do;
if your spec doesn't define any CSS properties, for example,
the "property-index" boilerplate won't generate.
If you want to suppress their generation even when they do have something to do,
use the <a>Boilerplate</a> metadata, like:

<xmp highlight=html>
  <pre class="metadata">
  Boilerplate: idl-index no, property-index no
  </pre>
</xmp>

### Custom/Overriding Boilerplate ### {#overriding-boilerplate}

Sometimes a file-based boilerplate (see below) that is appropriate for most of the specs in your group
isn't quite right for your specific spec.
Any boilerplate, file-based or Bikeshed-generated,
can be overridden by custom text of your choosing.
Just add an element to your source document with the content you'd like to show up in place of the offending boilerplate,
and add a `boilerplate="foo"` attribute to the container,
specifying which boilerplate section is being replaced.

Bikeshed will automatically remove that element from your document,
and instead inject its contents in place of the boilerplate that it would normally provide.

The `boilerplate` attribute isn't limited to just the predefined boilerplate section names;
you can supply whatever section name you want,
and if there's a `data-fill-with` attribute somewhere in the document with the same name,
the element's contents will be moved there.


Table of Contents {#table-of-contents}
-----------------

The headings in the spec are automatically numbered,
and a table of contents automatically generated.

Any heading <{h2}> to <{h6}>
(that is, skipping only the document-titling <{h1}>)
is automatically numbered by having a `<span class='secno'>...</span>`
prepended to its contents.
You can avoid this behavior for a heading and all of its subsequent subheadings
by adding `class="no-num"` to the heading.

Similarly, a ToC is generated to match.
Headings and their subheadings can be omitted from the ToC
by adding `class="no-toc"` to them.

The processor assumes that your headings are numbered correctly.
It does not yet pay attention to the HTML outline algorithm,
so using a bunch of <{h1}>s nested in <{section}>s will have very wrong effects.

Headings also automatically gain a self-link pointing to themselves,
to enable people to easily link to sections without having to return to the ToC.


File-based Includes {#bp-include}
-------------------

Several of the data-fill-with values
(those that are static, rather than generated from in-document data)
actually come from sets of .include files in the include/ directory.

The base files are simply named "foo.include",
where "foo" is the name of the data-fill-with value.
They can be specialized, however,
to particular working groups,
and to particular document statuses.

Putting the boilerplate in a folder named after the group,
like `csswg/header.include`,
specializes it for that group (specified in the spec's metadata).
Adding a "-STATUS" to the filename specializes it for the status (same).
These can be used together, like "csswg/status-CR.include".

The processor will first look for the "group/foo-STATUS.include" file,
failing over to "group/foo.include",
then "foo-STATUS.include",
and finally "foo.include".


Creating New Boilerplate Files For Your Organization {#bp-new}
----------------------------------------------------

Bikeshed's default boilerplate generates a functional and reasonably attractive spec,
but if your group has specific style requirements,
you can produce your own boilerplate files.
This section is a basic guide to developing these files.

The most important part of the boilerplate is the `header.include` and `footer.include` file.
These define the parts of the spec HTML that precede and follow your actual spec content,
so the source file can contain only the actual spec text,
and all specs in the same organization can look similar.

Here is a basic example `header.include` file:

```html
<!DOCTYPE html>
<html lang="en">
<head>
  <meta http-equiv="Content-Type" content="text/html; charset=utf-8">
  <title>\[TITLE]</title>
  <style>
  ...
  </style>
</head>
<body class="h-entry">
<div class="head">
  <p data-fill-with="logo"></p>
  <h1 id="title" class="p-name no-ref">\[TITLE]</h1>
  <h2 id="subtitle" class="no-num no-toc no-ref">\[LONGSTATUS],
	<span class="dt-updated"><span class="value-title" title="\[CDATE]">\[DATE]</span></h2>
  <div data-fill-with="spec-metadata"></div>
  <div data-fill-with="warning"></div>
  <p class='copyright' data-fill-with='copyright'></p>
  <hr title="Separator for header">
</div>

<div class="p-summary" data-fill-with="abstract"></div>
<div data-fill-with="at-risk"></div>

<nav data-fill-with="table-of-contents" id="toc"></nav>
<main>
```

This uses several of Bikeshed's boilerplating features:

* Text replacement, via the `\[FOO]` macros.
	These macros are prepopulated by Bikeshed,
	either from metadata in the spec (like `[TITLE]`) or from environment data (like `[DATE]`).
	The full list of text macros can be found at [[#text-macros]].

* Boilerplate pieces, via empty container elements with `data-fill-with` attributes.
	The list of Bikeshed-provided `data-fill-with` values can be found at [[#bp-sections]].
	At minimum, you want to include the `abstract`, `table-of-contents`, and `spec-metadata` sections here;
	they're all most useful at the top of the document.

<!--
████████     ███    ████ ██       ████████   ███████     ███    ████████
██     ██   ██ ██    ██  ██       ██     ██ ██     ██   ██ ██   ██     ██
██     ██  ██   ██   ██  ██       ██     ██ ██     ██  ██   ██  ██     ██
████████  ██     ██  ██  ██       ████████  ██     ██ ██     ██ ██     ██
██   ██   █████████  ██  ██       ██   ██   ██     ██ █████████ ██     ██
██    ██  ██     ██  ██  ██       ██    ██  ██     ██ ██     ██ ██     ██
██     ██ ██     ██ ████ ████████ ██     ██  ███████  ██     ██ ████████
-->

Railroad Diagrams {#railroad}
=================

A **railroad diagram** is a particular way of visually representing a structure roughly equivalent to regular expressions, or simple grammars.  They tend to be more readable and easier to grok than their equivalents written in terse regexps, and smaller than their equivalents written in explicit parsers.

Here's an example of a railroad diagram, this one describing the syntax of valid IDENT tokens in CSS:

<pre class='railroad'>
Stack:
	Or: 1
		T: --
		Seq:
			Opt: skip
				T: -
			Or:
				N: a-z A-Z _ or non-ASCII
				N: escape
	Star:
		Or:
			N: a-z A-Z 0-9 _ - or non-ASCII
			N: escape
</pre>

Bikeshed supports the automatic generation of railroad diagrams from a simplified DSL.  To use, simply embed a diagram description in a `<pre class='railroad'>` element - it'll get replaced by an appropriate <{svg}> element.

The Diagram Language {#railroad-lang}
--------------------

Diagrams are described by a custom DSL that somewhat resembles Python.

A railroad diagram consists of a number of nested elements, each of which may contain multiple children.  Each element is specified as a command followed by a colon, possibly followed by additional data (the prelude), and the element's children indented on following lines, like:

```
T: /*
ZeroOrMore:
	N: anything but * followed by /
T: */
```

This draws the following diagram:

<pre class=railroad>
T: /*
ZeroOrMore:
	N: anything but * followed by /
T: */
</pre>

The top-level elements are assumed to be a sequence of elements in the diagram.
Inside of a diagram, any of the elements may be used.
Elements are split into two groups: containers and text.

The containers hold other elements, and modify their semantics:

<ul dfn-for=railroad export>
	* <dfn>Sequence</dfn> (<dfn>And</dfn>, <dfn>Seq</dfn>) - used for sequences of elements which must all be selected in order.  Like concatenation in regexes. Takes 1 or more children.
	* <dfn>Stack</dfn> - A sequence that arranges its children vertically. Useful for preventing diagrams from becoming excessively wide. Takes 1 or more children.
	* <dfn>Choice</dfn> (<dfn>Or</dfn>) - used for a choice between elements.  Like the `|` character in regexes.  Takes 1 or more children.  Optionally, the "default" index may be provided in the prelude (defaulting to 0).
	* <dfn>Optional</dfn> (<dfn>Opt</dfn>)- used for an element that's optional.  Like the `?` character in regexes.  Takes 1 child.  Optionally, the word `skip` may be provided in the prelude to indicate that this term is skipped by default.
	* <dfn>OneOrMore</dfn> (<dfn>Plus</dfn>)- used for an element that can be chosen one or more times.  Like the `+` character in regexes.
		Takes 1 or 2 children: the first child is the element being repeated, and the optional second child is an element repeated between repetitions.
	* <dfn>ZeroOrMore</dfn> (<dfn>Star</dfn>) - same as <a>OneOrMore</a>, but allows the element to be chosen zero times as well (skipped entirely).  Like the `*` character in regexes.
		Like <a>Optional</a>, the keyword `skip` may be provided in the prelude to indicate that the "default option" is to skip it (repeat 0 times).

	The text elements only contain text, not other elements.  Their values are given in their preludes.

	* <dfn>Terminal</dfn> (<dfn>T</dfn>) - represents a "terminal" in the grammar, something that can't be expanded any more.  Generally represents literal text.
	* <dfn>NonTerminal</dfn> (<dfn>N</dfn>) - represents a "non-terminal" in the grammar, something that can be expanded further.
	* <dfn>Comment</dfn> (<dfn>C</dfn>) - represents a comment in the railroad diagram, to aid in reading or provide additional information.  This is often used as the repetition value of a <a>OneOrMore</a> or <a>ZeroOrMore</a> to provide information about the repetitions, like how many are allowed.
	* <dfn>Skip</dfn> (<dfn>S</dfn>) - represents nothing, an empty option.  This is rarely necessary to use explicitly, as containers like <a>Optional</a> use it automatically, but it's occasionally useful when writing out a <a>Choice</a> element where one option is to do nothing.
</ul>

<!--
 ██████   ███████  ██     ██ ████████   ██████  ████████
██    ██ ██     ██ ██     ██ ██     ██ ██    ██ ██
██       ██     ██ ██     ██ ██     ██ ██       ██
 ██████  ██     ██ ██     ██ ████████  ██       ██████
	  ██ ██     ██ ██     ██ ██   ██   ██       ██
██    ██ ██     ██ ██     ██ ██    ██  ██    ██ ██
 ██████   ███████   ███████  ██     ██  ██████  ████████
-->

Source-File Processing: `bikeshed source` {#source}
======================

Sometimes it's the *source* file you want to preprocess,
if there is some feature you want literally in your source
that is hard or annoying to type in yourself.
Bikeshed has some options for doing this as well.

All of these commands are accessed from the `source` sub-command,
like `bikeshed source`.
You can run individual commands by specifying their relevant flag
(see `bikeshed source -h` for a list),
or run all of them by not passing any flags.

Big Text {#big-text}
--------

When editing a large spec,
it's easy to get lost in its length,
and have to spend some time scrolling back and forth to find particular sections.

The Sublime Text editor has a special feature,
the minimap,
which shows an extremely-zoomed out version of your document while you scroll,
so you can recognize where you are in the file by the shape of your code.
This can be made even easier by putting extra-large "ASCII art" text in your source
to label major sections,
so they show up visibly in the minimap as section markers.

Bikeshed can auto-generate this "ASCII art" text for you
with its `--big-text` command.
Just add an HTML comment to your document on its own line that looks like:

<pre highlight=html>
&lt;!-- Big Text: Your Text -->
</pre>

If you run `bikeshed source --big-text`,
Bikeshed will replace it with a comment that looks like:

<pre highlight=html>
&lt;!--
██    ██  ███████  ██     ██ ████████        ████████ ████████ ██     ██ ████████
 ██  ██  ██     ██ ██     ██ ██     ██          ██    ██        ██   ██     ██
  ████   ██     ██ ██     ██ ██     ██          ██    ██         ██ ██      ██
   ██    ██     ██ ██     ██ ████████           ██    ██████      ███       ██
   ██    ██     ██ ██     ██ ██   ██            ██    ██         ██ ██      ██
   ██    ██     ██ ██     ██ ██    ██           ██    ██        ██   ██     ██
   ██     ███████   ███████  ██     ██          ██    ████████ ██     ██    ██
-->
</pre>

Which is clearly visible from Sublime's minimap!

<!--
████ ██    ██ ████████  ███████  ████████ ████████  ████████ ████████
 ██  ███   ██ ██       ██     ██    ██    ██     ██ ██       ██
 ██  ████  ██ ██       ██     ██    ██    ██     ██ ██       ██
 ██  ██ ██ ██ ██████   ██     ██    ██    ████████  ██████   ██████
 ██  ██  ████ ██       ██     ██    ██    ██   ██   ██       ██
 ██  ██   ███ ██       ██     ██    ██    ██    ██  ██       ██
████ ██    ██ ██        ███████     ██    ██     ██ ████████ ████████
-->

Appendix A: Bikeshed’s "InfoTree" Format {#infotree}
====================================================

Bikeshed's custom text formats attempt to be fairly regular;
most of them involve specifying key/value pairs,
and are line-based.
For example, Bikeshed's metadata format is one key/value pair per line,
with a colon between the key and the value.

The InfoTree format,
used by several things in Bikeshed,
is similar.
It's used when you need to specify data consisting of multiple key/value pairs,
where it's common that multiple entries share some of that data.
The InfoTree format makes this easy to read, write, and maintain.

Specifying Information on a Single Line {#infotree-one-line}
---------------------------------------

The simplest way to provide a piece of information is by putting all the key/value pairs on a single line.
In the InfoTree format, this is done by putting a colon between the key and value,
and separating the pairs with semicolons.
For example, here is an example of two "anchor" entries:

```
urlPrefix: https://encoding.spec.whatwg.org/; type: dfn; text: ascii whitespace
urlPrefix: https://encoding.spec.whatwg.org/; type: dfn; text: utf-8
```

This specifies two entries, each with three keys: urlPrefix, type, and text.

Nesting Information to Share Pieces {#infotree-nesting}
-----------------------------------

When multiple pieces of information share some key/value pairs,
you can use nesting to indicate this,
so you don't have to repeat yourself.
Here's the same two entries as before,
but using nesting to share their common information:

```
urlPrefix: https://encoding.spec.whatwg.org/; type: dfn
	text: ascii whitespace
	text: utf-8
```

Just like the previous, this defines two entries, each with three key/value pairs.
Now it's clearer, though, that the two entries share their `urlPrefix` and `type` data,
and you only have to maintain the common data in one place.

Comments {#infotree-comments}
-----------------------------

A line that starts with a `#` character
(with any amount of preceding whitespace)
is a comment;
it will be completely ignored for the purpose of parsing.

Additional Details {#infotree-details}
------------------

The order that keys are specified in is irrelevant.
Feel free to rearrange them for readability or more effective nesting.

You can specify the same key multiple times;
the values will be collected into an array for later processing.
(Each user of InfoTree will define whether multiple values for a key is valid or not, and what it means.)
The order that the values appear in *is* preserved,
as it might be important.
(For example, in the anchor format, multiple urlPrefix values are concatenated together, to help specify urls in multipage specs.)

Additional semicolons are silently ignored;
in other words, empty entries get dropped, so you can put a final semicolon at the end of the line or not, as you prefer.
