"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const cdk = require("@aws-cdk/core");
const index_1 = require("./index");
const app = new cdk.App();
const env = {
    region: process.env.CDK_DEFAULT_REGION,
    account: process.env.CDK_DEFAULT_ACCOUNT,
};
const stack = new cdk.Stack(app, 'demo-stack', { env });
const orderTask = new ecs.FargateTaskDefinition(stack, 'orderTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
const zoneName = 'svc.local';
const internalAlbRecordName = 'internal';
const externalAlbRecordName = 'external';
const internalALBEndpoint = `http://${internalAlbRecordName}.${zoneName}`;
orderTask.addContainer('order', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/OrderService')),
    portMappings: [
        { containerPort: 8080 },
    ],
    environment: {
        PRODUCT_SVC_URL: `${internalALBEndpoint}:9090`,
        CUSTOMER_SVC_URL: `${internalALBEndpoint}:8080`,
        serviceName: 'order',
        versionNum: '1.0',
    },
});
const customerTask = new ecs.FargateTaskDefinition(stack, 'customerTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
customerTask.addContainer('customer', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/CommonService')),
    portMappings: [
        { containerPort: 8080 },
    ],
    environment: {
        PRODUCT_SVC_URL: `${internalALBEndpoint}:9090`,
        CUSTOMER_SVC_URL: `${internalALBEndpoint}:8080`,
        serviceName: 'customer',
        versionNum: '1.0',
    },
});
const productTask = new ecs.FargateTaskDefinition(stack, 'productTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
productTask.addContainer('product', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/CommonService')),
    portMappings: [
        { containerPort: 8080 },
    ],
    environment: {
        PRODUCT_SVC_URL: `${internalALBEndpoint}:9090`,
        CUSTOMER_SVC_URL: `${internalALBEndpoint}:8080`,
        serviceName: 'product',
        versionNum: '1.0',
    },
});
const nginxTask = new ecs.FargateTaskDefinition(stack, 'nginxTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
nginxTask.addContainer('nginx', {
    image: ecs.ContainerImage.fromRegistry('nginx:latest'),
    portMappings: [
        { containerPort: 80 },
    ],
});
const svc = new index_1.DualAlbFargateService(stack, 'Service', {
    spot: true,
    enableExecuteCommand: true,
    tasks: [
        {
            listenerPort: 80,
            task: orderTask,
            desiredCount: 2,
            // customize the service autoscaling policy
            scalingPolicy: {
                maxCapacity: 20,
                requestPerTarget: 1000,
                targetCpuUtilization: 50,
            },
        },
        {
            listenerPort: 8080,
            task: customerTask,
            desiredCount: 10,
            capacityProviderStrategy: [
                {
                    capacityProvider: 'FARGATE',
                    base: 1,
                    weight: 1,
                },
                {
                    capacityProvider: 'FARGATE_SPOT',
                    base: 0,
                    weight: 3,
                },
            ],
        },
        { listenerPort: 9090, task: productTask, desiredCount: 2 },
        { listenerPort: 9091, task: nginxTask, desiredCount: 1 },
    ],
    route53Ops: {
        zoneName,
        externalAlbRecordName,
        internalAlbRecordName,
    },
});
// create a dummy sg
const dummySg = new ec2.SecurityGroup(stack, 'DummySG', {
    vpc: svc.vpc,
});
// allow all traffic from dummy sg to all the services
for (let i = 0; i < svc.service.length; i++) {
    svc.service[i].connections.allowFrom(dummySg, ec2.Port.allTraffic());
}
//# sourceMappingURL=data:application/json;base64,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