import os
import sys
import argparse

from waitress import serve
from dotenv import load_dotenv, set_key, find_dotenv, dotenv_values

from .app import app

load_dotenv()

CURDIR = os.path.split(os.path.realpath(__file__))[0]
VERSION = "0.1.9"

def create_folders():
    if not os.path.exists(os.path.join(CURDIR, 'projects')):
        os.mkdir(os.path.join(CURDIR, 'projects'))
    
    if not os.path.exists(os.path.join(CURDIR, 'accounts')):
        os.mkdir(os.path.join(CURDIR, 'accounts'))

def setup_runit(args):
    '''
    Setup Runit server side
    
    @params args
    @return None
    '''
    global parser
    domain = args.domain if hasattr(args, 'domain') else ''
    allowed = ['DBMS', 'DATABASE_HOST', 'DATATABSE_PORT', 
               'DATABASE_USERNAME', 'DATABASE_PASSWORD', 
               'DATABASE_NAME', 'RUNTIME_PYTHON',
               'RUNTIME_PHP', 'RUNTIME_JAVASCRIPT']
    default_settings = {
        'RUNIT_HOMEDIR': CURDIR,
        'RUNIT_SERVERNAME': '',
        'DBMS': 'mongodb',
        'DATABASE_HOST': 'localhost',
        'DATABASE_PORT': 27017,
        'DATABASE_USERNAME': '',
        'DATABASE_PASSWORD': '',
        'DATABASE_NAME': 'runit',
        'RUNTIME_PYTHON': 'python',
        'RUNTIME_PHP': 'php',
        'RUNTIME_JAVASCRIPT': 'node',
        'SETUP': ''
    }
    env_file = find_dotenv()
    if not env_file:
        with open('.env', 'wt'):
            pass
        env_file = find_dotenv()
        for key, value in default_settings.items():
            set_key(env_file, key, value)
        load_dotenv()
    
    settings = dotenv_values(find_dotenv())
    if not domain:
        default = settings['RUNIT_SERVERNAME']
        domain = input(f'RUNIT_SERVERNAME [{default}]: ')
        domain = domain if domain else default
    
    for key, value in settings.items():
        if key in allowed:
            settings[key] = input(f'{key} [{value}]: ')
            if key != 'DATABASE_USERNAME' and key != 'dbpassword':
                settings[key] = settings[key] if settings[key] else value
    
    settings['RUNIT_SERVERNAME'] = domain
    settings['RUNIT_HOMEDIR'] = os.path.join('..', os.path.realpath(os.path.split(__file__)[0]))
    
    if settings['RUNIT_SERVERNAME'] and settings['DATABASE_PASSWORD'] and \
        settings['DATABASE_HOST'] and settings['DATABASE_PORT'] and \
        settings['DATABASE_NAME']:
        settings['SETUP'] = 'completed'
    
    for key, value in settings.items():
        set_key(find_dotenv(), key, value)

def run_server(args = None):
    if not find_dotenv():
        print('[#] Complete Setup configuration first.\n')
        setup_runit(args)
    else:
        if args and args.production:
            serve(app, listen=f"*:{args.port}")
        else:
            app.run(host=args.host, port=args.port, debug=args.debug)

def get_arguments():
    global parser
    global VERSION
    
    subparsers = parser.add_subparsers()
    
    setup_parser = subparsers.add_parser('setup', help='Runit server-side configuration')
    setup_parser.add_argument('-d', '--domain', type=str, help="Runit server-side domain name")
    setup_parser.add_argument('--dbms', type=str, help="Runit Database System [mongodb|mysql]", choices=['mongodb', 'mysql'])
    setup_parser.add_argument('--dbhost', type=str, help="Database host ip address")
    setup_parser.add_argument('--dbport', type=str, help="Database host port")
    setup_parser.add_argument('--dbusername', type=str, help="Database user username")
    setup_parser.add_argument('--dbpassword', type=str, help="Database user password")
    setup_parser.add_argument('--dbname', type=str, help="Database name")
    setup_parser.set_defaults(func=setup_runit)
    
    parser.add_argument('--host', type=str, default='127.0.0.1', help='Host address to run server on')
    parser.add_argument('--port', type=int, default=9000, help='Host port to run server on')
    parser.add_argument('--debug', type=bool, choices=[True, False], default=True, help="Enable debug mode")
    parser.add_argument('--production', action='store_true', help="Run in production mode")
    parser.add_argument('-v','--version', action='version', version=f'%(prog)s {VERSION}')
    parser.set_defaults(func=run_server)
    return parser.parse_args()

def main():
    global parser
    try:
        parser = argparse.ArgumentParser(description="A terminal client for runit-server")
        args = get_arguments()
        args.func(args)

    except Exception as e:
        print(str(e), '\n')
        parser.print_help()
        sys.exit(1)

if __name__ == "__main__":
    main()
