"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApplicationMultipleTargetGroupsFargateService = void 0;
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const application_multiple_target_groups_service_base_1 = require("../base/application-multiple-target-groups-service-base");
/**
 * A Fargate service running on an ECS cluster fronted by an application load balancer.
 */
class ApplicationMultipleTargetGroupsFargateService extends application_multiple_target_groups_service_base_1.ApplicationMultipleTargetGroupsServiceBase {
    /**
     * Constructs a new instance of the ApplicationMultipleTargetGroupsFargateService class.
     */
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        this.assignPublicIp = props.assignPublicIp !== undefined ? props.assignPublicIp : false;
        if (props.taskDefinition && props.taskImageOptions) {
            throw new Error('You must specify only one of TaskDefinition or TaskImageOptions.');
        }
        else if (props.taskDefinition) {
            this.taskDefinition = props.taskDefinition;
        }
        else if (props.taskImageOptions) {
            const taskImageOptions = props.taskImageOptions;
            this.taskDefinition = new aws_ecs_1.FargateTaskDefinition(this, 'TaskDef', {
                memoryLimitMiB: props.memoryLimitMiB,
                cpu: props.cpu,
                executionRole: taskImageOptions.executionRole,
                taskRole: taskImageOptions.taskRole,
                family: taskImageOptions.family,
            });
            const containerName = taskImageOptions.containerName !== undefined ? taskImageOptions.containerName : 'web';
            const container = this.taskDefinition.addContainer(containerName, {
                image: taskImageOptions.image,
                logging: this.logDriver,
                environment: taskImageOptions.environment,
                secrets: taskImageOptions.secrets,
            });
            if (taskImageOptions.containerPorts) {
                for (const containerPort of taskImageOptions.containerPorts) {
                    container.addPortMappings({
                        containerPort,
                    });
                }
            }
        }
        else {
            throw new Error('You must specify one of: taskDefinition or image');
        }
        if (!this.taskDefinition.defaultContainer) {
            throw new Error('At least one essential container must be specified');
        }
        if (this.taskDefinition.defaultContainer.portMappings.length === 0) {
            this.taskDefinition.defaultContainer.addPortMappings({
                containerPort: 80,
            });
        }
        this.service = this.createFargateService(props);
        if (props.targetGroups) {
            this.addPortMappingForTargets(this.taskDefinition.defaultContainer, props.targetGroups);
            this.targetGroup = this.registerECSTargets(this.service, this.taskDefinition.defaultContainer, props.targetGroups);
        }
        else {
            this.targetGroup = this.listener.addTargets('ECS', {
                targets: [this.service],
                port: this.taskDefinition.defaultContainer.portMappings[0].containerPort,
            });
        }
    }
    createFargateService(props) {
        return new aws_ecs_1.FargateService(this, 'Service', {
            cluster: this.cluster,
            desiredCount: this.desiredCount,
            taskDefinition: this.taskDefinition,
            assignPublicIp: this.assignPublicIp,
            serviceName: props.serviceName,
            healthCheckGracePeriod: props.healthCheckGracePeriod,
            propagateTags: props.propagateTags,
            enableECSManagedTags: props.enableECSManagedTags,
            cloudMapOptions: props.cloudMapOptions,
            platformVersion: props.platformVersion,
        });
    }
}
exports.ApplicationMultipleTargetGroupsFargateService = ApplicationMultipleTargetGroupsFargateService;
//# sourceMappingURL=data:application/json;base64,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