# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['autora',
 'autora.skl',
 'autora.theorist',
 'autora.theorist.darts',
 'autora.variable']

package_data = \
{'': ['*']}

install_requires = \
['graphviz>=0.14.1,<0.15.0',
 'imageio>=2.9.0,<3.0.0',
 'matplotlib>=3.2.1,<4.0.0',
 'numpy>=1.18,<2.0',
 'pandas>=1.4.2,<2.0.0',
 'scikit-learn>=1.1.1,<2.0.0',
 'scipy>=1.7.1,<2.0.0',
 'seaborn>=0.11.1,<0.12.0',
 'torch>=1.11.0,<2.0.0',
 'tqdm>=4.64.0,<5.0.0']

extras_require = \
{'tinkerforge': ['tinkerforge>=2.1.25,<3.0.0']}

setup_kwargs = {
    'name': 'autora',
    'version': '1.0.0a5',
    'description': 'Autonomous Research Assistant (AutoRA) is a framework for implementing machine learning tools which autonomously and iteratively generate 1) new theories to describe real-world data, and 2) experiments to invalidate those theories and seed a new cycle of theory-making. The experiments will be run online via crowd-sourcing platforms (MTurk, Prolific).',
    'long_description': '# Autonomous Research Assistant\nAutonomous Research Assistant (AutoRA) is an open source AI-based system for automating each aspect of empirical research in the behavioral sciences, from the construction of a scientific hypothesis to conducting novel experiments.\n\n# Getting started\n\nYou should be familiar with the command line for your operating system. The topics required are covered in:\n- **macOS**: Joe Kissell. [*Take Control of the Mac Command Line with Terminal, 3rd Edition*]((https://bruknow.library.brown.edu/permalink/01BU_INST/528fgv/cdi_safari_books_v2_9781947282513)). Take Control Books, 2022. Chapters *Read Me First* through *Bring the Command Line Into The Real World*.\n- **Linux**: William E. Shotts. [*The Linux Command Line: a Complete Introduction. 2nd edition.*](https://bruknow.library.brown.edu/permalink/01BU_INST/9mvq88/alma991043239704906966). No Starch Press, 2019. Parts *I: Learning the Shell* and *II: Configuration and the Environment*.\n\nTo use the AutoRA package you need:\n- `python` and packages as specified in the `pyproject.toml` file,\n- `graphviz` for some visualizations.\n\nTo develop the AutoRA package, you also need:\n- `git`, the source control tool,\n- `pre-commit` which is used for handling git pre-commit hooks.\n\nWe recommend setting up your development environment using:\n- `pyenv` which is used for installing different versions of `python`,\n- `poetry`, which handles resolving dependencies between `python` modules and ensures that you are using the same package versions as other members of the development team.\n\nYou should also consider using an IDE. We recommend: \n- PyCharm (academic licenses for PyCharm professional edition are available for free). This is a `python`-specific integrated development environment which comes with extremely powerful tools for changing the structure of `python` code, running tests, etc. \n- Visual Studio Code (free). This is a powerful general text editor with plugins to support `python` development. \n\nThe following sections describe how to install and configure the recommended setup for developing AutoRA.\n\n*Note: For end-users, it may be more appropriate to use an environment manager like `Anaconda` or `Miniconda` instead of `poetry`, but this is not currently supported.*\n\n\n## Development Setup on macOS\n\n### Prerequisites\n\nFor macOS, we strongly recommend using `homebrew` to manage packages.\n\nVisit [https://brew.sh](https://brew.sh) and run the installation instructions.\n\n### Clone Repository\n\nWe recommend using the GitHub CLI to clone the repository. Install it: \n\n```shell\nbrew install gh\n```\n\nClone the repository. Run:\n```shell\ngh repo clone AutoResearch/AutoRA\n```\n\n... and following the prompts to authenticate to GitHub. It should clone the repository to a new directory. This is referred to as the `<project directory>` in the rest of this readme.\n\n### Install Dependencies\n\nOpen the repository directory in the terminal.\n\nInstall the dependencies, which are listed in the [`Brewfile`](./Brewfile) by running:\n\n```shell\nbrew bundle\n```\n\n### Install `python`\n\nWe recommend using `pyenv` to manage `python` versions. \n\n#### Initialize pyenv\nRun the initialization script as follows:\n\n```shell\npyenv init\n``` \n... and follow the instructions to add `pyenv` to the `$PATH`.\n\n#### Restart shell session\n\nAfter making these changes, restart your shell session by executing:\n\n```shell\nexec "$SHELL" \n```\n\n#### Install `python`\n\nInstall a `python` version listed in the [`pyproject.toml`](./pyproject.toml) file. The entry looks like:  \n\n```toml\npython = \'>=3.8.13,<3.11\'\n```\n\nIn this case, you could install version 3.8.13 as follows:\n\n```shell\npyenv install 3.8.13\n```\n\n### Install Pre-Commit Hooks\n\nIf you wish to commit to the repository, you should install the pre-commit hooks with the following command: \n```shell\npre-commit install\n```\n\nFor more information on pre-commit hooks, see [Pre-Commit-Hooks](#pre-commit-hooks)\n\n### Configure your development environment\n\nThere are two suggested options for initializing an environment:\n- _(Recommended)_ Using PyCharm,\n- _(Advanced)_ Using `poetry` from the command line.\n\n#### PyCharm configuration\n\nSet up the Virtual environment – an isolated version of `python` and all the packages required to run AutoRA and develop it further – as follows:\n- Open the `<project directory>` in PyCharm.\n- Navigate to PyCharm > Preferences > Project: AutoRA > Python Interpreter\n- Next to the drop-down list of available interpreters, click the "gear" symbol and choose "Add" to initialize a new interpreter. \n- Select "Poetry environment" in the list on the left. Specify the following:  \n  - Python executable: select the path to the installed `python` version you wish to use, e.g. \n    `~/.pyenv/versions/3.8.13/bin/python3`\n  - Select "install packages from pyproject.toml"\n  - Poetry executable: select the path to the poetry installation you have, e.g. \n    `/opt/homebrew/bin/poetry`\n  - Click "OK" and wait while the environment builds.\n  - Run the "Python tests in tests/" Run/Debug configuration in the PyCharm interface, and check that there are no errors.\n\nAdditional setup steps for PyCharm:\n\n- You can (and should) completely hide the IDE-specific directory for Visual Studio Code in PyCharm by adding `.vscode` to the list of ignored folder names in Preferences > Editor > File Types > Ignored Files and Folders. This only needs to be done once.\n    \n#### Command Line `poetry` Setup\n\nIf you need more control over the `poetry` environment, then you can set up a new environment from the command line.\n\n*Note: Setting up a `poetry` environment on the command line is the only option for VSCode users.*\n\nFrom the `<project directory>`, run the following commands.\n\nActivate the target version of `python` using `pyenv`:\n```shell\npyenv shell 3.8.13\n```\n\nSet up a new poetry environment with that `python` version:\n```shell\npoetry env use $(pyenv which python) \n```\n\nUpdate the installation utilities within the new environment:\n```shell\npoetry run python -m pip install --upgrade pip setuptools wheel\n```\n\nUse the `pyproject.toml` file to resolve and then install all the dependencies\n```shell\npoetry install\n```\n\nOnce this step has been completed, skip to the section [Activating and using the environment](#activating-and-using-the-environment) to test it.\n\n#### Visual Studio Code Configuration\n\nAfter installing Visual Studio Code and the other prerequisites, carry out the following steps:\n\n- Open the `<project directory>` in Visual Studio Code\n- Install the Visual Studio Code plugin recommendations suggested with the project. These include:\n  - `python`\n  - `python-environment-manager`\n- Run the [Command Line poetry Setup](#command-line-poetry-setup) specified above. This can be done in the built-in terminal if desired (Menu: Terminal > New Terminal).\n- Select the `python` option in the vertical bar on the far left of the window (which appear after installing the plugins). Under the title "PYTHON: ENVIRONMENTS" should be a list of `python` environments. If these do not appear:\n  - Refresh the window pane\n  - Ensure the python-environment-manager is installed correctly.\n  - Ensure the python-environment-manager is activated.\n\n- Locate the correct `poetry` environment. Click the "thumbs up" symbol next to the poetry environment name to "set as active workspace interpreter".\n\n- Check that the `poetry` environment is correctly set-up. \n  - Open a new terminal within Visual Studio Code (Menu: Terminal > New Terminal). \n  - It should execute something like `source /Users/me/Library/Caches/pypoetry/virtualenvs/autora-2PgcgopX-py3.8/bin/activate` before offering you a prompt.\n  - If you execute `which python` it should return the path to your python executable in the `.../autora-2PgcgopX-py3.8/bin` directory.\n  - Ensure that there are no errors when you run: \n    ```shell\n    python -m unittest\n    ```\n    in the built-in terminal. \n\n### Activating and using the environment\n\n#### Using `poetry` interactively\n\nTo run interactive commands, you can activate the poetry virtual environment. From the `<project directory>` directory, run:\n\n```shell\npoetry shell\n```\n\nThis spawns a new shell where you have access to the poetry `python` and all the packages installed using `poetry install`. You should see the prompt change:\n\n```\n% poetry shell\nSpawning shell within /Users/me/Library/Caches/pypoetry/virtualenvs/autora-2PgcgopX-py3.8\nRestored session: Fri Jun 24 12:34:56 EDT 2022\n(autora-2PgcgopX-py3.8) % \n```\n\nIf you execute `python` and then `import numpy`, you should be able to see that `numpy` has been imported from the `autora-2PgcgopX-py3.8` environment:\n\n```\n(autora-2PgcgopX-py3.8) % python\nPython 3.8.13 (default, Jun 16 2022, 12:34:56) \n[Clang 13.1.6 (clang-1316.0.21.2.5)] on darwin\nType "help", "copyright", "credits" or "license" for more information.\n>>> import numpy\n>>> numpy\n<module \'numpy\' from \'/Users/me/Library/Caches/pypoetry/virtualenvs/autora-2PgcgopX-py3.8/lib/python3.8/site-packages/numpy/__init__.py\'>\n```\n\nTo deactivate the `poetry` environment, `exit` the session. This should return you to your original prompt, as follows:\n```\n(autora-2PgcgopX-py3.8) % exit\n\nSaving session...\n...saving history...truncating history files...\n...completed.\n% \n```\n\nTo run a script, e.g. the `run_weber_study.py` script in the [`example/weber`](./example/weber) directory, execute: \n\n```shell\npoetry run python example/weber/run_weber_study.py\n```\n\n#### Using `poetry` non-interactively\n\nYou can run python programs using poetry without activating the poetry environment, by using `poetry run {command}`. For example, to run the tests, execute:\n\n```shell\npoetry run python -m unittest\n```\n\nIt should return something like:\n\n```\n% poetry run python -m unittest\n.\n--------------------------------\nRan 1 test in 0.000s\n\nOK\n```\n\n## Development Setup on Windows\n\nWindows is not yet officially supported. You may be able to follow the same approach as for macOS to set up your development environment, with some modifications, e.g.:\n- Using `chocolatey` in place of `homebrew`,\n- Using the GitHub Desktop application in place of the GitHub CLI.\n\nIf you successfully set up AutoRA on Windows, please update this readme.\n\n## Development Practices\n\n### Pre-Commit Hooks\n\nWe use [`pre-commit`](https://pre-commit.com) to manage pre-commit hooks. \n\nPre-commit hooks are programs which run before each git commit, and can read and potentially modify the files which are to be committed. \n\nWe use pre-commit hooks to:\n- enforce coding guidelines, including the `python` style-guide [PEP8](https://peps.python.org/pep-0008/) (`black` and `flake8`), \n- to check the order of `import` statements (`isort`),\n- to check the types of `python` objects (`mypy`).\n\nThe hooks and their settings are specified in [`.pre-commit-config.yaml`](./.pre-commit-config.yaml).\n\nSee the section [Install Pre-commit Hooks](#install-pre-commit-hooks) for installation instructions.\n\n#### Handling Pre-Commit Hook Errors\n\nIf your `git commit` fails because of the pre-commit hook, then you should:\n\n1. Run the pre-commit hooks on the files which you have staged, by running the following command in your terminal: \n    ```zsh\n    $ pre-commit run\n    ```\n\n2. Inspect the output. It might look like this:\n   ```\n   $ pre-commit run\n   black....................Passed\n   isort....................Passed\n   flake8...................Passed\n   mypy.....................Failed\n   - hook id: mypy\n   - exit code: 1\n   \n   example.py:33: error: Need type annotation for "data" (hint: "data: Dict[<type>, <type>] = ...")\n   Found 1 errors in 1 files (checked 10 source files)\n   ```\n3. Fix any errors which are reported.\n   **Important: Once you\'ve changed the code, re-stage the files it to Git. \n   This might mean unstaging changes and then adding them again.**\n4. If you have trouble:\n   - Do a web-search to see if someone else had a similar error in the past.\n   - Check that the tests you\'ve written work correctly.\n   - Check that there aren\'t any other obvious errors with the code.\n   - If you\'ve done all of that, and you still can\'t fix the problem, get help from someone else on the team.\n5. Repeat 1-4 until all hooks return "passed", e.g.\n   ```\n   $ pre-commit run\n   black....................Passed\n   isort....................Passed\n   flake8...................Passed\n   mypy.....................Passed\n   ```\n\nIt\'s easiest to solve these kinds of problems if you make small commits, often.  \n\n# Documentation\n\n## Commenting code\n\nTo help users understand code better, and to make the documentation generation automatic, we have some standards for documenting code. The comments, docstrings, and the structure of the code itself are meant to make life easier for the reader. \n- If something important isn\'t _obvious_ from the code, then it should be _made_ obvious with a comment. \n- Conversely, if something _is_ obvious, then it doesn\'t need a comment.\n\nThese standards are inspired by John Ousterhout. *A Philosophy of Software Design.* Yaknyam Press, 2021. Chapter 12 – 14.\n\n### Every public function, class and method has documentation\n\nWe include docstrings for all public functions, classes, and methods. These docstrings are meant to give a concise, high-level overview of **why** the function exists, **what** it is trying to do, and what is **important** about the code. (Details about **how** the code works are often better placed in detailed comments within the code.)\n\nEvery function, class or method has a one-line **high-level description** which clarifies its intent.   \n\nThe **meaning** and **type** of all the input and output parameters should be described.\n\nThere should be **examples** of how to use the function, class or method, with expected outputs, formatted as ["doctests"](https://docs.python.org/3/library/doctest.html). These should include normal cases for the function, but also include cases where it behaves unexpectedly or fails. \n\nWe follow the [Google Style Python Docstrings](https://sphinxcontrib-napoleon.readthedocs.io/en/latest/example_google.html), as these are supported by the online documentation tool we use (see [Online Documentation](#online-documentation)).\n\nA well documented function looks something like this:\n```python\ndef first_order_linear(\n    x: Union[float, np.ndarray], c: float, m: float\n) -> Union[float, np.ndarray]:\n    """\n    Evaluate a first order linear model of the form y = m x + c.\n\n    Arguments:\n        x: input location(s) on the x-axis\n        c: y-intercept of the linear model\n        m: gradient of the linear model\n\n    Returns:\n        y: result y = m x + c, the same shape and type as x\n\n    Examples:\n        >>> first_order_linear(0. , 1. , 0. )\n        1.0\n        >>> first_order_linear(np.array([-1. , 0. , 1. ]), c=1.0, m=2.0)\n        array([-1.,  1.,  3.])\n    """\n    y = m * x + c\n    return y\n```\n\nFor an example of a file with docstrings, see [docs/docstrings.py](./docs/docstrings.py).\n\n*Pro-Tip: Write the docstring for your new high-level object before starting on the code. In particular, writing examples of how you expect it should be used can help clarify the right level of abstraction.*\n\n## Online Documentation\n\nOnline Documentation is automatically generated using [Material for MkDocs](https://squidfunk.github.io/mkdocs-material/) based on docstrings in files in the `autora/` directory. \n\n### Commands\n\nBuild and serve the documentation using the following commands:\n\n* `poetry run mkdocs serve` - Start the live-reloading docs server.\n* `poetry run mkdocs build` - Build the documentation site.\n* `poetry run mkdocs gh-deploy` - Build the documentation and serve at https://AutoResearch.github.io/AutoRA/\n* `poetry run mkdocs -h` - Print help message and exit.\n\n### Documentation layout\n```\nmkdocs.yml    # The configuration file for the documentation.\ndocs/         # Directory for static pages to be included in the documentation.\n    index.md  # The documentation homepage.\n    ...       # Other markdown pages, images and other files.\nautora/          # The directory containing the source code.\n```\n\n\n',
    'author': 'Sebastian Musslick',
    'author_email': 'sebastian_musslick@brown.edu',
    'maintainer': None,
    'maintainer_email': None,
    'url': None,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
