import os
import tomli
import tomli_w
import yaml
import subprocess
from pathlib import Path
from rich.console import Console

from flowui.cli.utils.constants import COLOR_PALETTE, FLOWUI_HELM_PATH
from yaml.resolver import BaseResolver

from cryptography.hazmat.primitives import serialization as crypto_serialization
from cryptography.hazmat.primitives.asymmetric import rsa
from cryptography.hazmat.backends import default_backend as crypto_default_backend
import base64
import flowui.helm as flowui_helm
from tempfile import NamedTemporaryFile


class AsLiteral(str):
  pass

def represent_literal(dumper, data):
  return dumper.represent_scalar(BaseResolver.DEFAULT_SCALAR_TAG, data, style="|")

yaml.add_representer(AsLiteral, represent_literal)


console = Console()

def create_ssh_pair_key() -> None:
    # Create SSH key pair for GitHub Workflows
    console.print("Generating SSH key pair for GitHub Workflows...")
    key = rsa.generate_private_key(
        backend=crypto_default_backend(),
        public_exponent=65537,
        key_size=4096
    )

    private_key = key.private_bytes(
        crypto_serialization.Encoding.PEM,
        crypto_serialization.PrivateFormat.PKCS8,
        crypto_serialization.NoEncryption()
    )
    public_key = key.public_key().public_bytes(
        crypto_serialization.Encoding.OpenSSH,
        crypto_serialization.PublicFormat.OpenSSH
    )
    return private_key, public_key
    

def prepare_platform(
    cluster_name: str, 
    workflows_repository: str, 
    github_workflows_ssh_private_key: str, 
    github_operators_token: str, 
    github_workflows_token: str
) -> None:
    # Create local configuration file updated with user-provided arguments
    config_file_path = Path(__file__).resolve().parent / "config-flowui-local.toml"
    with open(str(config_file_path), "rb") as f:
        config_dict = tomli.load(f)
    
    running_path = str(Path().cwd().resolve())
    config_dict["path"]["FLOWUI_LOCAL_RUNNING_PATH"] = running_path
    config_dict["kind"]["FLOWUI_KIND_CLUSTER_NAME"] = cluster_name

    config_dict['github']['FLOWUI_GITHUB_WORKFLOWS_REPOSITORY'] = workflows_repository.split("github.com/")[-1].strip('/')

    if not github_workflows_ssh_private_key:
        private_key, public_key = create_ssh_pair_key()
        config_dict["github"]["FLOWUI_GITHUB_WORKFLOWS_SSH_PRIVATE_KEY"] = base64.b64encode(private_key).decode('utf-8')
        config_dict["github"]["FLOWUI_GITHUB_WORKFLOWS_SSH_PUBLIC_KEY"] = public_key.decode("utf-8")
    else:
        config_dict["github"]["FLOWUI_GITHUB_WORKFLOWS_SSH_PRIVATE_KEY"] = github_workflows_ssh_private_key

    config_dict["github"]["FLOWUI_GITHUB_ACCESS_TOKEN_OPERATORS"] = github_operators_token
    config_dict['github']['FLOWUI_GITHUB_ACCESS_TOKEN_WORKFLOWS'] = github_workflows_token
    
    with open("config-flowui-local.toml", "wb") as f:
        tomli_w.dump(config_dict, f)

    console.print("")
    console.print(f"FlowUI is prepared to run at: {running_path}")
    console.print(f"You can check and modify the configuration file at: {running_path}/config-flowui-local.toml")
    console.print("Next, run: `flowui platform create`")
    console.print("")


def create_platform(flowui_frontend_image: str = None, flowui_backend_image: str = None) -> None:
    # Load configuration values
    with open("config-flowui-local.toml", "rb") as f:
        flowui_config = tomli.load(f)

    # Create kind config file and run bash script to create Kind cluster
    kubeadm_config_patches = dict(
        kind="InitConfiguration",
        nodeRegistration=dict(
            kubeletExtraArgs={
                "node-labels": "ingress-ready=true"
            }
        )
    )
    kubeadm_parsed = AsLiteral(yaml.dump(kubeadm_config_patches))
    kind_config = dict(
        kind="Cluster",
        apiVersion="kind.x-k8s.io/v1alpha4",
        nodes=[
            dict(
                role="control-plane",
                kubeadmConfigPatches=[kubeadm_parsed],
                extraPortMappings=[
                    dict(
                        containerPort=80,
                        hostPort=80,
                        listenAddress="0.0.0.0",
                        protocol="TCP"
                    ),
                    dict(
                        containerPort=443,
                        hostPort=443,
                        listenAddress="0.0.0.0",
                        protocol="TCP"
                    )
                ]
            ),
            dict(
                role="worker",
                extraMounts=[
                    dict(
                        hostPath=flowui_config["path"]["FLOWUI_LOCAL_RUNNING_PATH"] + "/workflow_shared_storage",
                        containerPath="/cluster_shared_storage",
                        readOnly=False,
                        propagation="Bidirectional"
                    )
                ]
            ),
        ]
    )
    with open("kind-cluster-config.yaml", "w") as f:
        yaml.dump(kind_config, f)

    cluster_name = flowui_config["kind"]["FLOWUI_KIND_CLUSTER_NAME"]
    
    # Open helm values file and update with user-provided arguments
    flowui_helms_path = Path(os.path.dirname(flowui_helm.__file__))
    flowui_base_helm_path = flowui_helms_path / 'flowui-base'
    flowui_airlflow_helm_path = flowui_helms_path / 'flowui-airflow'

    console.print("Removing previous Kind cluster...")
    subprocess.run(["kind", "delete", "cluster", "--name", cluster_name])
    console.print("Creating new Kind cluster...")
    subprocess.run(["kind", "create", "cluster", "--name", cluster_name, "--config", "kind-cluster-config.yaml"])
    console.print("")
    console.print("Kind cluster created successfully!", style=f"bold {COLOR_PALETTE.get('success')}")

    # TODO - Install services with helm upgrade
    console.print("")
    console.print("Installing NGINX controller...")
    subprocess.run(["kubectl", "apply", "-f", "https://raw.githubusercontent.com/kubernetes/ingress-nginx/main/deploy/static/provider/kind/deploy.yaml"], stdout=subprocess.DEVNULL)
    subprocess.run(["kubectl", "wait", "--namespace", "ingress-nginx", "--for", "condition=ready", "pod", "--selector=app.kubernetes.io/component=controller", "--timeout=90s"])

    
    if flowui_frontend_image:
        console.print(f"Loading local frontend image {flowui_frontend_image} to Kind cluster...")
        subprocess.run(["kind", "load", "docker-image", flowui_frontend_image , "--name", cluster_name, "--nodes", f"{cluster_name}-worker"])
    else:
        flowui_frontend_image = "taufferconsulting/flowui-frontend:latest"
    
    if flowui_backend_image:
        console.print(f"Loading local backend image {flowui_backend_image} to Kind cluster...")
        subprocess.run(["kind", "load", "docker-image", flowui_backend_image , "--name", cluster_name, "--nodes", f"{cluster_name}-worker"])
    else:  
        flowui_backend_image = "taufferconsulting/flowui-backend:latest"

    # TODO separate operators and workflow tokens ?
    # TODO pass workflows repository as env variable
    token_operators = flowui_config["github"]["FLOWUI_GITHUB_ACCESS_TOKEN_OPERATORS"]
    token_workflows = flowui_config["github"]["FLOWUI_GITHUB_ACCESS_TOKEN_WORKFLOWS"]
    console.print("\nInstalling FlowUI services...\n")
    # Install FlowUI Services
    # TODO use remote helm chart
    subprocess.run([
        "helm", "install", 
        "--set", f"github_access_token_operators={token_operators}",
        "--set", f"github_access_token_workflows={token_workflows}",
        "--set", f"frontend.image={flowui_frontend_image}",
        "--set", f"backend.image={flowui_backend_image}",
        "--set", f"backend.workflowsRepository={flowui_config['github']['FLOWUI_GITHUB_WORKFLOWS_REPOSITORY']}",
        "flowui",
        flowui_base_helm_path
        #FLOWUI_HELM_PATH
    ])
    console.print("\nInstalling FlowUI Airflow services...\n")

    # Create temporary airflow values with user provided arguments
    airflow_ssh_config = dict(
        gitSshKey=f"{flowui_config['github']['FLOWUI_GITHUB_WORKFLOWS_SSH_PRIVATE_KEY']}",
    )
    airflow_ssh_config_parsed = AsLiteral(yaml.dump(airflow_ssh_config))
    airflow_values_override_config = {
        "airflow": {
            "enabled": True,
            "images": {
                "useDefaultImageForMigration": False,
                "airflow": {
                    "repository": "taufferconsulting/flowui-airflow-base",
                    "tag": "latest",
                    "pullPolicy": "IfNotPresent"
                }
            },
            "extraSecrets": {
                "airflow-ssh-secret": {
                    "data": airflow_ssh_config_parsed
                }
            },
            "dags": {
                "gitSync": {
                    "enabled": True,
                    "wait": 60,
                    "repo": f"ssh://git@github.com/{flowui_config['github']['FLOWUI_GITHUB_WORKFLOWS_REPOSITORY']}.git",
                    "branch": "main",
                    "subPath": "workflows",
                    "sshKeySecret": "airflow-ssh-secret"
                }
            }
        }
    }

    # Add airflow helm repository
    # TODO check why is not working with remote helm chart
    # subprocess.run([
    #     "helm", "repo", "add", "apache-airflow", "https://airflow.apache.org"
    # ])
    # Write yaml to temp file and install flowui airflow
    with NamedTemporaryFile(suffix='.yaml', mode="w") as fp:
        yaml.dump(airflow_values_override_config, fp)
        # Install airflow
        subprocess.run([
            "helm", "upgrade",
            "-i", "-f", str(fp.name),
            "flowui-airflow",
            flowui_airlflow_helm_path,
        ])

    console.print("")
    console.print("K8s resources created successfully!", style=f"bold {COLOR_PALETTE.get('success')}")

    console.print("You can now access the FlowUI frontend at: http://localhost/")
    console.print("and the Backend API at: http://localhost/api/")