"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const client_secrets_manager_1 = require("@aws-sdk/client-secrets-manager");
const core_1 = require("@octokit/core");
const github_helper_1 = require("../github-helper");
const github_secret_encryptor_1 = require("../github-secret-encryptor");
const github_secret_name_validator_1 = require("../github-secret-name-validator");
const onEvent = async (event) => {
    console.log(`Event: ${JSON.stringify(event)}`);
    github_secret_name_validator_1.validateSecretName(event.ResourceProperties.repositorySecretName);
    const smClient = new client_secrets_manager_1.SecretsManagerClient({ region: event.ResourceProperties.awsRegion });
    const githubTokenSecret = await smClient.send(new client_secrets_manager_1.GetSecretValueCommand({ SecretId: event.ResourceProperties.githubTokenSecret }));
    const octokit = new core_1.Octokit({ auth: githubTokenSecret.SecretString });
    const requestType = event.RequestType;
    switch (requestType) {
        case 'Create':
            return onCreate(event, octokit, smClient);
        case 'Update':
            return onUpdate(event, octokit, smClient);
        case 'Delete':
            return onDelete(event, octokit);
        default:
            throw new Error(`Unexpected request type: '${requestType}'`);
    }
};
const onCreate = async (event, octokit, smClient) => {
    const props = event.ResourceProperties;
    console.log('Create new ActionSecret with props ' + JSON.stringify(props));
    await createOrUpdateRepoSecret(event, octokit, smClient);
    const PhysicalResourceId = await buildPhysicalResourceId(event, octokit);
    return { PhysicalResourceId };
};
const onUpdate = async (event, octokit, smClient) => {
    const props = event.ResourceProperties;
    console.log(`Update ActionSecret ${props.repositorySecretName} with props ${JSON.stringify(props)}`);
    await createOrUpdateRepoSecret(event, octokit, smClient);
    const PhysicalResourceId = await buildPhysicalResourceId(event, octokit);
    return { PhysicalResourceId };
};
const onDelete = async (event, octokit) => {
    console.log('Delete ActionSecret ' + event.ResourceProperties.repositorySecretName);
    await deleteRepoSecret(event, octokit);
    const PhysicalResourceId = await buildPhysicalResourceId(event, octokit);
    return { PhysicalResourceId };
};
const createOrUpdateRepoSecret = async (event, octokit, smClient) => {
    const { repositoryOwner, repositoryName: repo, repositorySecretName: secret_name } = event.ResourceProperties;
    const secretId = event.ResourceProperties.sourceSecretArn;
    const secretToEncrypt = await smClient.send(new client_secrets_manager_1.GetSecretValueCommand({ SecretId: secretId }));
    console.log(`Encrypt value of secret with id: ${secretId}`);
    const secretString = secretToEncrypt.SecretString;
    if (!secretString) {
        throw new Error('SecretString is empty from secret with id: ' + secretId);
    }
    const owner = await github_helper_1.getOwner(octokit, repositoryOwner);
    const { data } = await octokit.request('GET /repos/{owner}/{repo}/actions/secrets/public-key', { owner, repo });
    const encryptedSecret = await github_secret_encryptor_1.encryptValue(secretString, data.key);
    console.log('Encrypted secret, attempting to create/update github secret');
    const secretResponse = await octokit.request('PUT /repos/{owner}/{repo}/actions/secrets/{secret_name}', {
        owner,
        repo,
        secret_name,
        encrypted_value: encryptedSecret,
        key_id: data.key_id,
    });
    console.log(JSON.stringify(secretResponse));
    return secretResponse;
};
const deleteRepoSecret = async (event, octokit) => {
    const { repositoryOwner, repositoryName: repo, repositorySecretName: secret_name } = event.ResourceProperties;
    const owner = await github_helper_1.getOwner(octokit, repositoryOwner);
    const deleteSecretResponse = await octokit.request('DELETE /repos/{owner}/{repo}/actions/secrets/{secret_name}', {
        owner,
        repo,
        secret_name,
    });
    console.log(`Delete: ${JSON.stringify(deleteSecretResponse)}`);
    return deleteSecretResponse;
};
const buildPhysicalResourceId = async (event, octokit) => {
    const { repositorySecretName: secret, repositoryOwner, repositoryName: repo } = event.ResourceProperties;
    const owner = await github_helper_1.getOwner(octokit, repositoryOwner);
    return [secret, owner, repo].join('/');
};
exports.handler = onEvent;
//# sourceMappingURL=data:application/json;base64,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