"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const client_secrets_manager_1 = require("@aws-sdk/client-secrets-manager");
const core_1 = require("@octokit/core");
const github_helper_1 = require("../github-helper");
const github_secret_encryptor_1 = require("../github-secret-encryptor");
const github_secret_name_validator_1 = require("../github-secret-name-validator");
const onEvent = async (event) => {
    console.log(`Event: ${JSON.stringify(event)}`);
    github_secret_name_validator_1.validateSecretName(event.ResourceProperties.repositorySecretName);
    const smClient = new client_secrets_manager_1.SecretsManagerClient({ region: event.ResourceProperties.awsRegion });
    const githubTokenSecret = await smClient.send(new client_secrets_manager_1.GetSecretValueCommand({ SecretId: event.ResourceProperties.githubTokenSecret }));
    const octokit = new core_1.Octokit({ auth: githubTokenSecret.SecretString });
    const requestType = event.RequestType;
    switch (requestType) {
        case 'Create':
            return onCreate(event, octokit, smClient);
        case 'Update':
            return onUpdate(event, octokit, smClient);
        case 'Delete':
            return onDelete(event, octokit);
        default:
            throw new Error(`Unexpected request type: '${requestType}'`);
    }
};
const onCreate = async (event, octokit, smClient) => {
    console.log('Create new ActionEnvironmentSecret with props ' + JSON.stringify(event.ResourceProperties));
    await createOrUpdateEnvironmentSecret(event, octokit, smClient);
    const PhysicalResourceId = await buildPhysicalResourceId(event, octokit);
    return { PhysicalResourceId };
};
const onUpdate = async (event, octokit, smClient) => {
    const props = event.ResourceProperties;
    console.log(`Update ActionEnvironmentSecret ${props.repositorySecretName} with props ${JSON.stringify(props)}`);
    await createOrUpdateEnvironmentSecret(event, octokit, smClient);
    const PhysicalResourceId = await buildPhysicalResourceId(event, octokit);
    return { PhysicalResourceId };
};
const onDelete = async (event, octokit) => {
    console.log('Delete ActionEnvironmentSecret ' + event.ResourceProperties.repositorySecretName);
    await deleteEnvironmentSecret(event, octokit);
    const PhysicalResourceId = await buildPhysicalResourceId(event, octokit);
    return { PhysicalResourceId };
};
const createOrUpdateEnvironmentSecret = async (event, octokit, smClient) => {
    const { repositoryOwner, repositoryName: repo, repositorySecretName: secret_name, environment: environment_name, sourceSecretArn: secretId, } = event.ResourceProperties;
    const secretToEncrypt = await smClient.send(new client_secrets_manager_1.GetSecretValueCommand({ SecretId: secretId }));
    console.log(`Encrypt value of secret with id: ${secretId}`);
    const secretString = secretToEncrypt.SecretString;
    if (!secretString) {
        throw new Error('SecretString is empty from secret with id: ' + secretId);
    }
    const owner = await github_helper_1.getOwner(octokit, repositoryOwner);
    const { data } = await octokit.request('GET /repos/{owner}/{repo}/actions/secrets/public-key', { owner, repo });
    const encryptedSecret = await github_secret_encryptor_1.encryptValue(secretString, data.key);
    console.log('Encrypted secret, attempting to create/update github secret');
    const repository_id = await getRepositoryId(event, octokit, owner);
    const secretResponse = await octokit.request('PUT /repositories/{repository_id}/environments/{environment_name}/secrets/{secret_name}', {
        repository_id,
        environment_name,
        secret_name,
        encrypted_value: encryptedSecret,
        key_id: data.key_id,
    });
    console.log(JSON.stringify(secretResponse));
    return secretResponse;
};
const deleteEnvironmentSecret = async (event, octokit) => {
    const { environment: environment_name, repositorySecretName: secret_name, repositoryOwner } = event.ResourceProperties;
    const repository_id = await getRepositoryId(event, octokit, repositoryOwner);
    const deleteSecretResponse = await octokit.request('DELETE /repositories/{repository_id}/environments/{environment_name}/secrets/{secret_name}', {
        repository_id,
        environment_name,
        secret_name,
    });
    console.log(`Delete: ${JSON.stringify(deleteSecretResponse)}`);
    return deleteSecretResponse;
};
const getRepositoryId = async (event, octokit, repositoryOwner) => {
    const { repositoryName: repo } = event.ResourceProperties;
    const owner = await github_helper_1.getOwner(octokit, repositoryOwner);
    const { data } = await octokit.request('GET /repos/{owner}/{repo}', {
        owner,
        repo,
    });
    return data.id;
};
const buildPhysicalResourceId = async (event, octokit) => {
    const { environment, repositorySecretName: secret, repositoryOwner, repositoryName: repo } = event.ResourceProperties;
    const owner = await github_helper_1.getOwner(octokit, repositoryOwner);
    return [environment, secret, owner, repo].join('/');
};
exports.handler = onEvent;
//# sourceMappingURL=data:application/json;base64,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