# -*- coding: utf-8 -*-
# Copyright 2022 Google LLC
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#
from typing import Callable, Dict, Optional, Sequence, Tuple, Union
import warnings

from google.api_core import gapic_v1, grpc_helpers, operations_v1
import google.auth  # type: ignore
from google.auth import credentials as ga_credentials  # type: ignore
from google.auth.transport.grpc import SslCredentials  # type: ignore
from google.longrunning import operations_pb2  # type: ignore
import grpc  # type: ignore

from google.cloud.gke_multicloud_v1.types import attached_resources, attached_service

from .base import DEFAULT_CLIENT_INFO, AttachedClustersTransport


class AttachedClustersGrpcTransport(AttachedClustersTransport):
    """gRPC backend transport for AttachedClusters.

    The AttachedClusters API provides a single centrally managed
    service to register and manage Anthos attached clusters that run
    on customer's owned infrastructure.

    This class defines the same methods as the primary client, so the
    primary client can load the underlying transport implementation
    and call it.

    It sends protocol buffers over the wire using gRPC (which is built on
    top of HTTP/2); the ``grpcio`` package must be installed.
    """

    _stubs: Dict[str, Callable]

    def __init__(
        self,
        *,
        host: str = "gkemulticloud.googleapis.com",
        credentials: Optional[ga_credentials.Credentials] = None,
        credentials_file: Optional[str] = None,
        scopes: Optional[Sequence[str]] = None,
        channel: Optional[grpc.Channel] = None,
        api_mtls_endpoint: Optional[str] = None,
        client_cert_source: Optional[Callable[[], Tuple[bytes, bytes]]] = None,
        ssl_channel_credentials: Optional[grpc.ChannelCredentials] = None,
        client_cert_source_for_mtls: Optional[Callable[[], Tuple[bytes, bytes]]] = None,
        quota_project_id: Optional[str] = None,
        client_info: gapic_v1.client_info.ClientInfo = DEFAULT_CLIENT_INFO,
        always_use_jwt_access: Optional[bool] = False,
        api_audience: Optional[str] = None,
    ) -> None:
        """Instantiate the transport.

        Args:
            host (Optional[str]):
                 The hostname to connect to.
            credentials (Optional[google.auth.credentials.Credentials]): The
                authorization credentials to attach to requests. These
                credentials identify the application to the service; if none
                are specified, the client will attempt to ascertain the
                credentials from the environment.
                This argument is ignored if ``channel`` is provided.
            credentials_file (Optional[str]): A file with credentials that can
                be loaded with :func:`google.auth.load_credentials_from_file`.
                This argument is ignored if ``channel`` is provided.
            scopes (Optional(Sequence[str])): A list of scopes. This argument is
                ignored if ``channel`` is provided.
            channel (Optional[grpc.Channel]): A ``Channel`` instance through
                which to make calls.
            api_mtls_endpoint (Optional[str]): Deprecated. The mutual TLS endpoint.
                If provided, it overrides the ``host`` argument and tries to create
                a mutual TLS channel with client SSL credentials from
                ``client_cert_source`` or application default SSL credentials.
            client_cert_source (Optional[Callable[[], Tuple[bytes, bytes]]]):
                Deprecated. A callback to provide client SSL certificate bytes and
                private key bytes, both in PEM format. It is ignored if
                ``api_mtls_endpoint`` is None.
            ssl_channel_credentials (grpc.ChannelCredentials): SSL credentials
                for the grpc channel. It is ignored if ``channel`` is provided.
            client_cert_source_for_mtls (Optional[Callable[[], Tuple[bytes, bytes]]]):
                A callback to provide client certificate bytes and private key bytes,
                both in PEM format. It is used to configure a mutual TLS channel. It is
                ignored if ``channel`` or ``ssl_channel_credentials`` is provided.
            quota_project_id (Optional[str]): An optional project to use for billing
                and quota.
            client_info (google.api_core.gapic_v1.client_info.ClientInfo):
                The client info used to send a user-agent string along with
                API requests. If ``None``, then default info will be used.
                Generally, you only need to set this if you're developing
                your own client library.
            always_use_jwt_access (Optional[bool]): Whether self signed JWT should
                be used for service account credentials.

        Raises:
          google.auth.exceptions.MutualTLSChannelError: If mutual TLS transport
              creation failed for any reason.
          google.api_core.exceptions.DuplicateCredentialArgs: If both ``credentials``
              and ``credentials_file`` are passed.
        """
        self._grpc_channel = None
        self._ssl_channel_credentials = ssl_channel_credentials
        self._stubs: Dict[str, Callable] = {}
        self._operations_client: Optional[operations_v1.OperationsClient] = None

        if api_mtls_endpoint:
            warnings.warn("api_mtls_endpoint is deprecated", DeprecationWarning)
        if client_cert_source:
            warnings.warn("client_cert_source is deprecated", DeprecationWarning)

        if channel:
            # Ignore credentials if a channel was passed.
            credentials = False
            # If a channel was explicitly provided, set it.
            self._grpc_channel = channel
            self._ssl_channel_credentials = None

        else:
            if api_mtls_endpoint:
                host = api_mtls_endpoint

                # Create SSL credentials with client_cert_source or application
                # default SSL credentials.
                if client_cert_source:
                    cert, key = client_cert_source()
                    self._ssl_channel_credentials = grpc.ssl_channel_credentials(
                        certificate_chain=cert, private_key=key
                    )
                else:
                    self._ssl_channel_credentials = SslCredentials().ssl_credentials

            else:
                if client_cert_source_for_mtls and not ssl_channel_credentials:
                    cert, key = client_cert_source_for_mtls()
                    self._ssl_channel_credentials = grpc.ssl_channel_credentials(
                        certificate_chain=cert, private_key=key
                    )

        # The base transport sets the host, credentials and scopes
        super().__init__(
            host=host,
            credentials=credentials,
            credentials_file=credentials_file,
            scopes=scopes,
            quota_project_id=quota_project_id,
            client_info=client_info,
            always_use_jwt_access=always_use_jwt_access,
            api_audience=api_audience,
        )

        if not self._grpc_channel:
            self._grpc_channel = type(self).create_channel(
                self._host,
                # use the credentials which are saved
                credentials=self._credentials,
                # Set ``credentials_file`` to ``None`` here as
                # the credentials that we saved earlier should be used.
                credentials_file=None,
                scopes=self._scopes,
                ssl_credentials=self._ssl_channel_credentials,
                quota_project_id=quota_project_id,
                options=[
                    ("grpc.max_send_message_length", -1),
                    ("grpc.max_receive_message_length", -1),
                ],
            )

        # Wrap messages. This must be done after self._grpc_channel exists
        self._prep_wrapped_messages(client_info)

    @classmethod
    def create_channel(
        cls,
        host: str = "gkemulticloud.googleapis.com",
        credentials: Optional[ga_credentials.Credentials] = None,
        credentials_file: Optional[str] = None,
        scopes: Optional[Sequence[str]] = None,
        quota_project_id: Optional[str] = None,
        **kwargs,
    ) -> grpc.Channel:
        """Create and return a gRPC channel object.
        Args:
            host (Optional[str]): The host for the channel to use.
            credentials (Optional[~.Credentials]): The
                authorization credentials to attach to requests. These
                credentials identify this application to the service. If
                none are specified, the client will attempt to ascertain
                the credentials from the environment.
            credentials_file (Optional[str]): A file with credentials that can
                be loaded with :func:`google.auth.load_credentials_from_file`.
                This argument is mutually exclusive with credentials.
            scopes (Optional[Sequence[str]]): A optional list of scopes needed for this
                service. These are only used when credentials are not specified and
                are passed to :func:`google.auth.default`.
            quota_project_id (Optional[str]): An optional project to use for billing
                and quota.
            kwargs (Optional[dict]): Keyword arguments, which are passed to the
                channel creation.
        Returns:
            grpc.Channel: A gRPC channel object.

        Raises:
            google.api_core.exceptions.DuplicateCredentialArgs: If both ``credentials``
              and ``credentials_file`` are passed.
        """

        return grpc_helpers.create_channel(
            host,
            credentials=credentials,
            credentials_file=credentials_file,
            quota_project_id=quota_project_id,
            default_scopes=cls.AUTH_SCOPES,
            scopes=scopes,
            default_host=cls.DEFAULT_HOST,
            **kwargs,
        )

    @property
    def grpc_channel(self) -> grpc.Channel:
        """Return the channel designed to connect to this service."""
        return self._grpc_channel

    @property
    def operations_client(self) -> operations_v1.OperationsClient:
        """Create the client designed to process long-running operations.

        This property caches on the instance; repeated calls return the same
        client.
        """
        # Quick check: Only create a new client if we do not already have one.
        if self._operations_client is None:
            self._operations_client = operations_v1.OperationsClient(self.grpc_channel)

        # Return the client from cache.
        return self._operations_client

    @property
    def create_attached_cluster(
        self,
    ) -> Callable[
        [attached_service.CreateAttachedClusterRequest], operations_pb2.Operation
    ]:
        r"""Return a callable for the create attached cluster method over gRPC.

        Creates a new
        [AttachedCluster][google.cloud.gkemulticloud.v1.AttachedCluster]
        resource on a given GCP project and region.

        If successful, the response contains a newly created
        [Operation][google.longrunning.Operation] resource that can be
        described to track the status of the operation.

        Returns:
            Callable[[~.CreateAttachedClusterRequest],
                    ~.Operation]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "create_attached_cluster" not in self._stubs:
            self._stubs["create_attached_cluster"] = self.grpc_channel.unary_unary(
                "/google.cloud.gkemulticloud.v1.AttachedClusters/CreateAttachedCluster",
                request_serializer=attached_service.CreateAttachedClusterRequest.serialize,
                response_deserializer=operations_pb2.Operation.FromString,
            )
        return self._stubs["create_attached_cluster"]

    @property
    def update_attached_cluster(
        self,
    ) -> Callable[
        [attached_service.UpdateAttachedClusterRequest], operations_pb2.Operation
    ]:
        r"""Return a callable for the update attached cluster method over gRPC.

        Updates an
        [AttachedCluster][google.cloud.gkemulticloud.v1.AttachedCluster].

        Returns:
            Callable[[~.UpdateAttachedClusterRequest],
                    ~.Operation]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "update_attached_cluster" not in self._stubs:
            self._stubs["update_attached_cluster"] = self.grpc_channel.unary_unary(
                "/google.cloud.gkemulticloud.v1.AttachedClusters/UpdateAttachedCluster",
                request_serializer=attached_service.UpdateAttachedClusterRequest.serialize,
                response_deserializer=operations_pb2.Operation.FromString,
            )
        return self._stubs["update_attached_cluster"]

    @property
    def import_attached_cluster(
        self,
    ) -> Callable[
        [attached_service.ImportAttachedClusterRequest], operations_pb2.Operation
    ]:
        r"""Return a callable for the import attached cluster method over gRPC.

        Imports creates a new
        [AttachedCluster][google.cloud.gkemulticloud.v1.AttachedCluster]
        resource by importing an existing Fleet Membership resource.

        Attached Clusters created before the introduction of the Anthos
        Multi-Cloud API can be imported through this method.

        If successful, the response contains a newly created
        [Operation][google.longrunning.Operation] resource that can be
        described to track the status of the operation.

        Returns:
            Callable[[~.ImportAttachedClusterRequest],
                    ~.Operation]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "import_attached_cluster" not in self._stubs:
            self._stubs["import_attached_cluster"] = self.grpc_channel.unary_unary(
                "/google.cloud.gkemulticloud.v1.AttachedClusters/ImportAttachedCluster",
                request_serializer=attached_service.ImportAttachedClusterRequest.serialize,
                response_deserializer=operations_pb2.Operation.FromString,
            )
        return self._stubs["import_attached_cluster"]

    @property
    def get_attached_cluster(
        self,
    ) -> Callable[
        [attached_service.GetAttachedClusterRequest], attached_resources.AttachedCluster
    ]:
        r"""Return a callable for the get attached cluster method over gRPC.

        Describes a specific
        [AttachedCluster][google.cloud.gkemulticloud.v1.AttachedCluster]
        resource.

        Returns:
            Callable[[~.GetAttachedClusterRequest],
                    ~.AttachedCluster]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "get_attached_cluster" not in self._stubs:
            self._stubs["get_attached_cluster"] = self.grpc_channel.unary_unary(
                "/google.cloud.gkemulticloud.v1.AttachedClusters/GetAttachedCluster",
                request_serializer=attached_service.GetAttachedClusterRequest.serialize,
                response_deserializer=attached_resources.AttachedCluster.deserialize,
            )
        return self._stubs["get_attached_cluster"]

    @property
    def list_attached_clusters(
        self,
    ) -> Callable[
        [attached_service.ListAttachedClustersRequest],
        attached_service.ListAttachedClustersResponse,
    ]:
        r"""Return a callable for the list attached clusters method over gRPC.

        Lists all
        [AttachedCluster][google.cloud.gkemulticloud.v1.AttachedCluster]
        resources on a given Google Cloud project and region.

        Returns:
            Callable[[~.ListAttachedClustersRequest],
                    ~.ListAttachedClustersResponse]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "list_attached_clusters" not in self._stubs:
            self._stubs["list_attached_clusters"] = self.grpc_channel.unary_unary(
                "/google.cloud.gkemulticloud.v1.AttachedClusters/ListAttachedClusters",
                request_serializer=attached_service.ListAttachedClustersRequest.serialize,
                response_deserializer=attached_service.ListAttachedClustersResponse.deserialize,
            )
        return self._stubs["list_attached_clusters"]

    @property
    def delete_attached_cluster(
        self,
    ) -> Callable[
        [attached_service.DeleteAttachedClusterRequest], operations_pb2.Operation
    ]:
        r"""Return a callable for the delete attached cluster method over gRPC.

        Deletes a specific
        [AttachedCluster][google.cloud.gkemulticloud.v1.AttachedCluster]
        resource.

        If successful, the response contains a newly created
        [Operation][google.longrunning.Operation] resource that can be
        described to track the status of the operation.

        Returns:
            Callable[[~.DeleteAttachedClusterRequest],
                    ~.Operation]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "delete_attached_cluster" not in self._stubs:
            self._stubs["delete_attached_cluster"] = self.grpc_channel.unary_unary(
                "/google.cloud.gkemulticloud.v1.AttachedClusters/DeleteAttachedCluster",
                request_serializer=attached_service.DeleteAttachedClusterRequest.serialize,
                response_deserializer=operations_pb2.Operation.FromString,
            )
        return self._stubs["delete_attached_cluster"]

    @property
    def get_attached_server_config(
        self,
    ) -> Callable[
        [attached_service.GetAttachedServerConfigRequest],
        attached_resources.AttachedServerConfig,
    ]:
        r"""Return a callable for the get attached server config method over gRPC.

        Returns information, such as supported Kubernetes
        versions, on a given Google Cloud location.

        Returns:
            Callable[[~.GetAttachedServerConfigRequest],
                    ~.AttachedServerConfig]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "get_attached_server_config" not in self._stubs:
            self._stubs["get_attached_server_config"] = self.grpc_channel.unary_unary(
                "/google.cloud.gkemulticloud.v1.AttachedClusters/GetAttachedServerConfig",
                request_serializer=attached_service.GetAttachedServerConfigRequest.serialize,
                response_deserializer=attached_resources.AttachedServerConfig.deserialize,
            )
        return self._stubs["get_attached_server_config"]

    @property
    def generate_attached_cluster_install_manifest(
        self,
    ) -> Callable[
        [attached_service.GenerateAttachedClusterInstallManifestRequest],
        attached_service.GenerateAttachedClusterInstallManifestResponse,
    ]:
        r"""Return a callable for the generate attached cluster
        install manifest method over gRPC.

        Generates the install manifest to be installed on the
        target cluster.

        Returns:
            Callable[[~.GenerateAttachedClusterInstallManifestRequest],
                    ~.GenerateAttachedClusterInstallManifestResponse]:
                A function that, when called, will call the underlying RPC
                on the server.
        """
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "generate_attached_cluster_install_manifest" not in self._stubs:
            self._stubs[
                "generate_attached_cluster_install_manifest"
            ] = self.grpc_channel.unary_unary(
                "/google.cloud.gkemulticloud.v1.AttachedClusters/GenerateAttachedClusterInstallManifest",
                request_serializer=attached_service.GenerateAttachedClusterInstallManifestRequest.serialize,
                response_deserializer=attached_service.GenerateAttachedClusterInstallManifestResponse.deserialize,
            )
        return self._stubs["generate_attached_cluster_install_manifest"]

    def close(self):
        self.grpc_channel.close()

    @property
    def delete_operation(
        self,
    ) -> Callable[[operations_pb2.DeleteOperationRequest], None]:
        r"""Return a callable for the delete_operation method over gRPC."""
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "delete_operation" not in self._stubs:
            self._stubs["delete_operation"] = self.grpc_channel.unary_unary(
                "/google.longrunning.Operations/DeleteOperation",
                request_serializer=operations_pb2.DeleteOperationRequest.SerializeToString,
                response_deserializer=None,
            )
        return self._stubs["delete_operation"]

    @property
    def cancel_operation(
        self,
    ) -> Callable[[operations_pb2.CancelOperationRequest], None]:
        r"""Return a callable for the cancel_operation method over gRPC."""
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "cancel_operation" not in self._stubs:
            self._stubs["cancel_operation"] = self.grpc_channel.unary_unary(
                "/google.longrunning.Operations/CancelOperation",
                request_serializer=operations_pb2.CancelOperationRequest.SerializeToString,
                response_deserializer=None,
            )
        return self._stubs["cancel_operation"]

    @property
    def get_operation(
        self,
    ) -> Callable[[operations_pb2.GetOperationRequest], operations_pb2.Operation]:
        r"""Return a callable for the get_operation method over gRPC."""
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "get_operation" not in self._stubs:
            self._stubs["get_operation"] = self.grpc_channel.unary_unary(
                "/google.longrunning.Operations/GetOperation",
                request_serializer=operations_pb2.GetOperationRequest.SerializeToString,
                response_deserializer=operations_pb2.Operation.FromString,
            )
        return self._stubs["get_operation"]

    @property
    def list_operations(
        self,
    ) -> Callable[
        [operations_pb2.ListOperationsRequest], operations_pb2.ListOperationsResponse
    ]:
        r"""Return a callable for the list_operations method over gRPC."""
        # Generate a "stub function" on-the-fly which will actually make
        # the request.
        # gRPC handles serialization and deserialization, so we just need
        # to pass in the functions for each.
        if "list_operations" not in self._stubs:
            self._stubs["list_operations"] = self.grpc_channel.unary_unary(
                "/google.longrunning.Operations/ListOperations",
                request_serializer=operations_pb2.ListOperationsRequest.SerializeToString,
                response_deserializer=operations_pb2.ListOperationsResponse.FromString,
            )
        return self._stubs["list_operations"]

    @property
    def kind(self) -> str:
        return "grpc"


__all__ = ("AttachedClustersGrpcTransport",)
