# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['neattext',
 'neattext.explainer',
 'neattext.functions',
 'neattext.pattern_data']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'neattext',
    'version': '0.0.4',
    'description': 'Neattext - a simple NLP package for cleaning text',
    'long_description': '# neattext\nNeatText:a simple NLP package for cleaning textual data and text preprocessing\n\n[![Build Status](https://travis-ci.org/Jcharis/neattext.svg?branch=master)](https://travis-ci.org/Jcharis/neattext)\n\n[![GitHub license](https://img.shields.io/github/license/Jcharis/neattext)](https://github.com/Jcharis/neattext/blob/master/LICENSE)\n\n#### Problem\n+ Cleaning of unstructured text data\n+ Reduce noise [special characters,stopwords]\n+ Reducing repetition of using the same code for text preprocessing\n\n#### Solution\n+ convert the already known solution for cleaning text into a reuseable package\n\n\n#### Installation\n```bash\npip install neattext\n```\n\n### Usage\n+ The OOP Way(Object Oriented Way)\n+ NeatText offers 4 main classes for working with text data\n\t- TextFrame : a frame-like object for cleaning text\n\t- TextCleaner: remove or replace specifics\n\t- TextExtractor: extract unwanted text data\n\t- TextMetrics: word stats and metrics\n\n\n### Using TextFrame\n+ Keeps the text as `TextFrame` object. This allows us to do more with our text. \n+ It inherits the benefits of the TextCleaner and the TextMetrics out of the box with some additional features for handling text data.\n+ This is the simplest way for text preprocessing with this library alternatively you can utilize the other classes too.\n\n\n```python\n>>> import neattext as nt \n>> mytext = "This is the mail example@gmail.com ,our WEBSITE is https://example.com 😊."\n>>> docx = nt.TextFrame(text=mytext)\n>>> docx.text \n"This is the mail example@gmail.com ,our WEBSITE is https://example.com 😊."\n>>>\n>>> docx.describe()\nKey      Value          \nLength  : 73             \nvowels  : 21             \nconsonants: 34             \nstopwords: 4              \npunctuations: 8              \nspecial_char: 8              \ntokens(whitespace): 10             \ntokens(words): 14             \n>>> \n>>> docx.length\n73\n>>> docs.head(16)\n\'This is the mail\'\n>>> docx.tail()\n>>> docx.count_vowels()\n>>> docx.count_stopwords()\n>>> docx.count_consonants()\n>>> docx.nlongest()\n>>> docx.nshortest()\n```\n#### Basic NLP Task (Tokenization,Ngram,Text Generation)\n```python\n>>> docx.word_tokens()\n>>>\n>>> docx.sent_tokens()\n>>>\n>>> docx.term_freq()\n>>>\n>>> docx.bow()\n```\n#### Basic Text Preprocessing\n```python\n>>> docx.normalize()\n\'this is the mail example@gmail.com ,our website is https://example.com 😊.\'\n>>> docx.normalize(level=\'deep\')\n\'this is the mail examplegmailcom our website is httpsexamplecom \'\n\n>>> docx.remove_puncts()\n>>> docx.remove_special_characters()\n>>> docx.remove_emojis()\n>>> docx.fix_contractions()\n```\n\n##### Handling Files with NeatText\n+ Read txt file directly into TextFrame\n```python\n>>> import neattext as nt \n>>> docx_df = nt.read_txt(\'file.txt\')\n```\n+ Alternatively you can instantiate a TextFrame and read a text file into it\n```python\n>>> import neattext as nt \n>>> docx_df = nt.TextFrame().read_txt(\'file.txt\')\n```\n\n##### Chaining Methods on TextFrame\n```python\n>>> t1 = "This is the mail example@gmail.com ,our WEBSITE is https://example.com 😊 and it will cost $100 to subscribe."\n>>> docx = TextFrame(t1)\n>>> result = docx.remove_emails().remove_urls().remove_emojis()\n>>> print(result)\n\'This is the mail  ,our WEBSITE is   and it will cost $100 to subscribe.\'\n```\n\n\n\n#### Clean Text\n+ Clean text by removing emails,numbers,stopwords,emojis,etc\n```python\n>>> from neattext import TextCleaner\n>>> docx = TextCleaner()\n>>> docx.text = "This is the mail example@gmail.com ,our WEBSITE is https://example.com 😊."\n>>> docx.clean_text()\n```\n\n#### Remove Emails,Numbers,Phone Numbers \n```python\n>>> print(docx.remove_emails())\n>>> \'This is the mail  ,our WEBSITE is https://example.com 😊.\'\n>>>\n>>> print(docx.remove_stopwords())\n>>> \'This mail example@gmail.com ,our WEBSITE https://example.com 😊.\'\n>>>\n>>> print(docx.remove_numbers())\n>>> docx.remove_phone_numbers()\n```\n\n\n#### Remove Special Characters\n```python\n>>> docx.remove_special_characters()\n```\n\n#### Remove Emojis\n```python\n>>> print(docx.remove_emojis())\n>>> \'This is the mail example@gmail.com ,our WEBSITE is https://example.com .\'\n```\n\n#### Replace Emails,Numbers,Phone Numbers\n```python\n>>> docx.replace_emails()\n>>> docx.replace_numbers()\n>>> docx.replace_phone_numbers()\n```\n\n#### Chain Multiple Methods\n```python\n>>> t1 = "This is the mail example@gmail.com ,our WEBSITE is https://example.com 😊 and it will cost $100 to subscribe."\n>>> docx = TextCleaner(t1)\n>>> result = docx.remove_emails().remove_urls().remove_emojis()\n>>> print(result)\n\'This is the mail  ,our WEBSITE is   and it will cost $100 to subscribe.\'\n\n```\n\n### Using TextExtractor\n+ To Extract emails,phone numbers,numbers,urls,emojis from text\n```python\n>>> from neattext import TextExtractor\n>>> docx = TextExtractor()\n>>> docx.text = "This is the mail example@gmail.com ,our WEBSITE is https://example.com 😊."\n>>> docx.extract_emails()\n>>> [\'example@gmail.com\']\n>>>\n>>> docx.extract_emojis()\n>>> [\'😊\']\n```\n\n\n### Using TextMetrics\n+ To Find the Words Stats such as counts of vowels,consonants,stopwords,word-stats\n```python\n>>> from neattext import TextMetrics\n>>> docx = TextMetrics()\n>>> docx.text = "This is the mail example@gmail.com ,our WEBSITE is https://example.com 😊."\n>>> docx.count_vowels()\n>>> docx.count_consonants()\n>>> docx.count_stopwords()\n>>> docx.word_stats()\n```\n\n### Usage \n+ The MOP(method/function oriented way) Way\n\n```python\n>>> from neattext.functions import clean_text,extract_emails\n>>> t1 = "This is the mail example@gmail.com ,our WEBSITE is https://example.com ."\n>>> clean_text(t1,True)\n>>>\'this is the mail <email> ,our website is <url> .\'\n>>> extract_emails(t1)\n>>> [\'example@gmail.com\']\n```\n\n### Explainer\n+ Explain an emoji or unicode for emoji \n\t- emoji_explainer()\n\t- emojify()\n\t- unicode_2_emoji()\n\n\n```python\n>>> from neattext.explainer import emojify\n>>> emojify(\'Smiley\')\n>>> \'😃\'\n```\n\n```python\n>>> from neattext.explainer import emoji_explainer\n>>> emoji_explainer(\'😃\')\n>>> \'SMILING FACE WITH OPEN MOUTH\'\n```\n\n```python\n>>> from neattext.explainer import unicode_2_emoji\n>>> unicode_2_emoji(\'0x1f49b\')\n\t\'FLUSHED FACE\'\n```\n\n\n\n### Documentation\nPlease read the [documentation](https://github.com/Jcharis/neattext/wiki) for more information on what neattext does and how to use is for your needs.\n\n### More Features To Add\n+ basic nlp task\n+ currency normalizer\n\n#### Acknowledgements\n+ Inspired by packages like `clean-text` from Johannes Fillter and `textify` by JCharisTech\n\n\n#### NB\n+ Contributions Are Welcomed\n+ Notice a bug, please let us know.\n+ Thanks A lot\n\n\n#### By\n+ Jesse E.Agbe(JCharis)\n+ Jesus Saves @JCharisTech\n\n\n\n',
    'author': 'Jesse E.Agbe(JCharis)',
    'author_email': 'jcharistech@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/Jcharis/neattext',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.3,<4.0',
}


setup(**setup_kwargs)
