"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Portfolio = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("aws-cdk-lib/aws-iam");
const cdk = require("aws-cdk-lib");
const association_manager_1 = require("./private/association-manager");
const util_1 = require("./private/util");
const validation_1 = require("./private/validation");
const aws_servicecatalog_1 = require("aws-cdk-lib/aws-servicecatalog");
class PortfolioBase extends cdk.Resource {
    constructor() {
        super(...arguments);
        this.associatedPrincipals = new Set();
    }
    /**
     * (experimental) Associate portfolio with an IAM Role.
     *
     * @experimental
     */
    giveAccessToRole(role) {
        this.associatePrincipal(role.roleArn, role.node.addr);
    }
    /**
     * (experimental) Associate portfolio with an IAM User.
     *
     * @experimental
     */
    giveAccessToUser(user) {
        this.associatePrincipal(user.userArn, user.node.addr);
    }
    /**
     * (experimental) Associate portfolio with an IAM Group.
     *
     * @experimental
     */
    giveAccessToGroup(group) {
        this.associatePrincipal(group.groupArn, group.node.addr);
    }
    /**
     * (experimental) Associate portfolio with the given product.
     *
     * @experimental
     */
    addProduct(product) {
        association_manager_1.AssociationManager.associateProductWithPortfolio(this, product, undefined);
    }
    /**
     * (experimental) Initiate a portfolio share with another account.
     *
     * @experimental
     */
    shareWithAccount(accountId, options = {}) {
        const hashId = this.generateUniqueHash(accountId);
        new aws_servicecatalog_1.CfnPortfolioShare(this, `PortfolioShare${hashId}`, {
            portfolioId: this.portfolioId,
            accountId: accountId,
            shareTagOptions: options.shareTagOptions,
            acceptLanguage: options.messageLanguage,
        });
    }
    /**
     * (experimental) Associate Tag Options.
     *
     * A TagOption is a key-value pair managed in AWS Service Catalog.
     * It is not an AWS tag, but serves as a template for creating an AWS tag based on the TagOption.
     *
     * @experimental
     */
    associateTagOptions(tagOptions) {
        association_manager_1.AssociationManager.associateTagOptions(this, tagOptions);
    }
    /**
     * (experimental) Add a Resource Update Constraint.
     *
     * @experimental
     */
    constrainTagUpdates(product, options = {}) {
        association_manager_1.AssociationManager.constrainTagUpdates(this, product, options);
    }
    /**
     * (experimental) Add notifications for supplied topics on the provisioned product.
     *
     * @experimental
     */
    notifyOnStackEvents(product, topic, options = {}) {
        association_manager_1.AssociationManager.notifyOnStackEvents(this, product, topic, options);
    }
    /**
     * (experimental) Set provisioning rules for the product.
     *
     * @experimental
     */
    constrainCloudFormationParameters(product, options) {
        association_manager_1.AssociationManager.constrainCloudFormationParameters(this, product, options);
    }
    /**
     * (experimental) Force users to assume a certain role when launching a product.
     *
     * This sets the launch role using the role arn which is tied to the account this role exists in.
     * This is useful if you will be provisioning products from the account where this role exists.
     * If you intend to share the portfolio across accounts, use a local launch role.
     *
     * @experimental
     */
    setLaunchRole(product, launchRole, options = {}) {
        association_manager_1.AssociationManager.setLaunchRole(this, product, launchRole, options);
    }
    /**
     * (experimental) Force users to assume a certain role when launching a product.
     *
     * The role will be referenced by name in the local account instead of a static role arn.
     * A role with this name will automatically be created and assumable by Service Catalog in this account.
     * This is useful when sharing the portfolio with multiple accounts.
     *
     * @experimental
     */
    setLocalLaunchRoleName(product, launchRoleName, options = {}) {
        const launchRole = new iam.Role(this, `LaunchRole${launchRoleName}`, {
            roleName: launchRoleName,
            assumedBy: new iam.ServicePrincipal('servicecatalog.amazonaws.com'),
        });
        association_manager_1.AssociationManager.setLocalLaunchRoleName(this, product, launchRole.roleName, options);
        return launchRole;
    }
    /**
     * (experimental) Force users to assume a certain role when launching a product.
     *
     * The role name will be referenced by in the local account and must be set explicitly.
     * This is useful when sharing the portfolio with multiple accounts.
     *
     * @experimental
     */
    setLocalLaunchRole(product, launchRole, options = {}) {
        validation_1.InputValidator.validateRoleNameSetForLocalLaunchRole(launchRole);
        association_manager_1.AssociationManager.setLocalLaunchRoleName(this, product, launchRole.roleName, options);
    }
    /**
     * (experimental) Configure deployment options using AWS Cloudformation StackSets.
     *
     * @experimental
     */
    deployWithStackSets(product, options) {
        association_manager_1.AssociationManager.deployWithStackSets(this, product, options);
    }
    /**
     * Associate a principal with the portfolio.
     * If the principal is already associated, it will skip.
     */
    associatePrincipal(principalArn, principalId) {
        if (!this.associatedPrincipals.has(principalArn)) {
            const hashId = this.generateUniqueHash(principalId);
            new aws_servicecatalog_1.CfnPortfolioPrincipalAssociation(this, `PortolioPrincipalAssociation${hashId}`, {
                portfolioId: this.portfolioId,
                principalArn: principalArn,
                principalType: 'IAM',
            });
            this.associatedPrincipals.add(principalArn);
        }
    }
}
/**
 * (experimental) A Service Catalog portfolio.
 *
 * @experimental
 */
class Portfolio extends PortfolioBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        jsiiDeprecationWarnings._aws_cdk_aws_servicecatalog_alpha_PortfolioProps(props);
        this.validatePortfolioProps(props);
        this.portfolio = new aws_servicecatalog_1.CfnPortfolio(this, 'Resource', {
            displayName: props.displayName,
            providerName: props.providerName,
            description: props.description,
            acceptLanguage: props.messageLanguage,
        });
        this.portfolioId = this.portfolio.ref;
        this.portfolioArn = cdk.Stack.of(this).formatArn({
            service: 'servicecatalog',
            resource: 'portfolio',
            resourceName: this.portfolioId,
        });
        if (props.tagOptions !== undefined) {
            this.associateTagOptions(props.tagOptions);
        }
    }
    /**
     * (experimental) Creates a Portfolio construct that represents an external portfolio.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param portfolioArn the Amazon Resource Name of the existing portfolio.
     * @experimental
     */
    static fromPortfolioArn(scope, id, portfolioArn) {
        const arn = cdk.Stack.of(scope).splitArn(portfolioArn, cdk.ArnFormat.SLASH_RESOURCE_NAME);
        const portfolioId = arn.resourceName;
        if (!portfolioId) {
            throw new Error('Missing required Portfolio ID from Portfolio ARN: ' + portfolioArn);
        }
        class Import extends PortfolioBase {
            constructor() {
                super(...arguments);
                this.portfolioArn = portfolioArn;
                this.portfolioId = portfolioId;
            }
            generateUniqueHash(value) {
                return util_1.hashValues(this.portfolioArn, value);
            }
        }
        return new Import(scope, id, {
            environmentFromArn: portfolioArn,
        });
    }
    /**
     * (experimental) Create a unique id based off the L1 CfnPortfolio or the arn of an imported portfolio.
     *
     * @experimental
     */
    generateUniqueHash(value) {
        return util_1.hashValues(cdk.Names.nodeUniqueId(this.portfolio.node), value);
    }
    validatePortfolioProps(props) {
        validation_1.InputValidator.validateLength(this.node.path, 'portfolio display name', 1, 100, props.displayName);
        validation_1.InputValidator.validateLength(this.node.path, 'portfolio provider name', 1, 50, props.providerName);
        validation_1.InputValidator.validateLength(this.node.path, 'portfolio description', 0, 2000, props.description);
    }
}
exports.Portfolio = Portfolio;
_a = JSII_RTTI_SYMBOL_1;
Portfolio[_a] = { fqn: "@aws-cdk/aws-servicecatalog-alpha.Portfolio", version: "2.0.0-alpha.11" };
//# sourceMappingURL=data:application/json;base64,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