# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['compic10', 'compic10.vm']

package_data = \
{'': ['*']}

install_requires = \
['PyYAML>=6.0,<7.0', 'clipboard>=0.0.4,<0.0.5']

setup_kwargs = {
    'name': 'compic10',
    'version': '1.1.1',
    'description': 'A fully fledged compiler and emulator for the IC10-MIPS processor of the game Stationeers.',
    'long_description': '# compIC10\ncompIC10 is a compiler for the "IC10" from the game Stationeers. It supports common programming language features like\nloops, if-statements, functions, procedures (functions without return value), variables and expressions.\n\n## How to install\nJust install Python3 version 3.8.2 or later (earlier versions probably work too) and the compic10 pip package with `python3 -m pip install compic10`.\n\n## Programming Language\n### Basics\nAll statements must be on a single line, blank lines are allowed.\nEvery line can only contain one statement which means that every statement ends with a linebreak.\nWhen referring to names meant are strings starting with a \\\'\\_\\\' or letter followed by zero or more \\\'\\_\\\', letters or numbers.\nNames can not be keywords.\nThere are no types since every operation in stationeers results in a floating point number.\nEverything is case sensitive. Multiple spaces or tabs are beeing ignored by the compiler meaning indentation is possible but not enforced.\n\nAfter successful compilation the compiler conducts post process optimizations to remove redundant and garbage code from crude compiler output.\n\n### Compilation Errors\nIf there is an error in the source code, the compiler will notice you and tell you at which line the error is.\nIf the compiler yields some python-exception than there may be a bug in compic10 or in the source code.\nBugs within compic10 can be submitted at: gitlab.com/smpkdev/compic10/-/issues\n### Post Processor Errors\nIf you get that error, usage of the --no_post_processing_optimization flag is recommended, to disable post process optimizations. Usually this error should not be appearing.\n### Runtime Errors\nIf the program ends up in a unrecoverable state it sets the state of the IC10 housing to an error code and turns the IC10 off.\nCurrently there is only the error-state -1 which means an function ended without an `return` statement.\nUnfortunately AIMEe doesn\'t support a state so he will just turn off.\n## Compiler\n**Basic usage:**\n\n    python3 -m compic10 [source_file]\nCompiles the source code in file source_file. This doesn\'t save the result anywhere. \nUse the following parameters in order to get the result (version without further parameters still useful to check for errors).\n\n**Parameters:**\n\n    -o [file], --out_file [file]\nSafes generated code in [file].\n\n    -p, --print_output\nPrints code to the stdout.\n\n    -c, --copy\nCopys resulting code to clipboard.\n\n    -a, --annotate\nCreates comments in the generated code in order to easily associate source-code with asm-code.\n\n    -s, --offset_varstack [offset]\nSets the location reserved for variables in the stack.\n\n    -l, --length_varstack [length]\nSets the length of the reserved section in stack for variables. \noffset+length must be less then 512.\n\n    --silent\nSuppress any console output except code output from the -p flag.\n\n    --no_post_processing_optimization\nDisable post processor optimizations.\nPPO might be buggy, if the code does not work properly, try to use this flag to completely disable the PPO.\n\n    \n\n## Statements\n### Comments\nComments start with a `#`. Everything after this symbol on the current line will be ignored by the compiler.\nThe following code is a comment with comment message [comment]:\n\n    # [comment]\nComments on a seperate line will be placed in the resulting asm-code if the `--annotate` flag is set.\n### main\nEvery program starts within the main block given by:\n\n    main\n    [main code]\n    end\n`[main code]` is the source code of main.\n### Number literals\nInteger numbers:\n\n    1\n\nFloating point numbers:\n\n    1.1\nShortcuts like .1 or 1. are possible.\nExponential number literals:\n\n    1.1e5\n    1.1E5\n    1.1e-5\n    1.1E-5\n    1.1e+5\n    1.1E+5\n\nSince Stationeers doesn\'t support scientfic notation, floating point numbers are limited to at most 16 digits behind the comma.\nBinary numbers:\n\n    0b0101\nHexadecimal numbers:\n\n    0xFF\nBooleans:\n\n    true\n    false\nTrue is represented as an 1 and false as an 0.\nNote that all number literals are positive a additional minus (-) before that is will result in an expression of unary - and the number literal.\n### Variables\nThe following code generates a variable with name `[name]` and (1) initial value 0 or (2) initial value beeing the \nresult of the expression `[expression]`:\n\n    var [name]  # (1)\n    var [name] = [expression]  # (2)\nVariables can be used in the current block (namespace) and inner blocks.\nGlobal variables are possible too. Those unfortunately require a bigger overhead (those can not be within registers).\n### Constants\nThe following code generates a constant with name `[name]` and value `[value]`.\n\n    const [name] = [value]\nLike variables, constants will be available in the current block and all inner blocks.\nThe value must be known at compile time (no expressions allowed only number literals) an optional +/- before the number is allowed.\n### Devices \nThe following code assignes device number `[dev]` on the IC10 to the name `[name]`:\n\n    dev [name] = [dev]\nThe \'d\' like in `d0` must be omitted here.\nValues of device `[name]` propertys `[property]` can be modified and written to by:\n\n    [name].[property]\nThis statement can be used like a variable.\nIf the device has slots, a certain slot `[slot]` can be accessed by:\n\n    [name][ [slot] ].[property]\n\nIn order to access reagents use:\n\n    [name].[reagentmode].[property]\n`[reagentmode]` has to be one of: `Contents`, `Required`, `Recipe`.\n\nThe special device `db` can always be accessed through the keyword `self`.\n`[name]` can also be a variable or constant for dynamic device access.\n`[name]` in an expression refers to the pin on the housing set by the device declaration.\nThe special device property `DeviceIsSet` can be used to check if the port is configured on socket.\n### Batch Devices \nThe following code associates all devices on the network with itemhash `[hash]` with the name `[name]`:\n\n    bdev [name] = [hash]\nItemhashes can be found here: https://stationeers-wiki.com/ItemHash\nThe value of batch device `[name]` property `[property]` can be read by:\n\n    [name].[property].[batchmode]\n`[batchmode]` is the function used to combine all the values and must be one of: `Average`, `Sum`, `Minimum`, `Maximum`.\nThe property `[property]` of all devices on the network, associated with `[name]`, can be assigned with the result of an expression `[expression]` by:\n\n    [name].[property] = [expression]\n\n`[name]` in an expression refering to the hash set by the batch device declaration.\n### Functions \nA function with name `[name]` is defined by the following code:\n\n    func [name]([parameters...])\n    [function body]\n    end\n`[function body]` is the source code of the function.\nThe functions has parameters `[parameters...]`, which is a comma seperated collection of names. Default values are supported and added with `param=1`, `param=-1` or `param=+1` \nafter an parameter with a default value, no parameters without a default value are allowed. All default values must be known at compile time.\nAll functions must end with a `return` statement that has a return value.\nAt the end of the function body is a `end` statement.\nFunctions have their own namespace.\nIf a function ends without an return statement, the state of the IC10-housing changes to -1 followed by it shuting down.\nRecursion is supported.\n### Procedures\nA procedure with name `[name]` is defined by the following code:\n\n    proc [name]([parameters...])\n    [procedure body]\n    end\n`[procedure body]` is the source code of the procedure.\nThe procedure has parameters `[parameters...]`, which is a comma seperated collection of names. Default values are supported like `param=1`, `param=-1` or `param=+1`\nafter an parameter with default value, no parameters without a default value are allowed. All default values must be known at compile time.\nAt the end of the procedure body is an `end`.\nProcedures don\'t have to end with an `return` statement.\nProcedures have their own namespace.\nRecursion is supported.\n### return\nFunctions can be returned from everywhere with a return statement by:\n\n    return [expression]\nThe return value is the result of expression `[expression]`.\nSame goes for procedures but without a return value:\n\n    return\n`return` is only allowed within functions or procedures.\n### Expressions\nExpression are mathematical statements where operators are connected with variables, number literals, devices, batch devices, \nsubexpressions within braces () or functions.\nExpressions are always evaluated from left to right for operators with equal priorities. \nAn exception to that are assignment operations. Those are evaluated from right to left when grouped like `a=b=c`.\nSupported operators from higher to lower priority (grouped ones have equal priority):\n\n    -- [operand]  # decrement: [operand] = [operand] - 1\n    ++ [operand]  # increment: [operand] = [operand] + 1\n\n    + [operand]  # * 1 (do nothing)\n    - [operand]  # * -1\n\n    # Currently not implemented but recognized:\n    ~ [operand]  # bitwise negate\n    \n    [operand1] * [operand2]  # multiply\n    [operand1] / [operand2]  # divide\n    [operand1] % [operand2]  # modulus\n\n    [operand1] + [operand2]  # add\n    [operand1] - [operand2]  # subtract\n    \n    # Currently not implemented but recognized:\n    [operand1] << [operand2]  # shift left\n    # Currently not implemented but recognized:\n    [operand1] >> [operand2]  # shift right\n    \n    # Currently not implemented but recognized:\n    [operand1] & [operand2]  # bitwise and \n    \n    # Currently not implemented but recognized:\n    [operand1] ^ [operand2]  # bitwise xor\n    \n    # Currently not implemented but recognized:\n    [operand1] | [operand2]  # bitwise or\n\n    [operand1] == [operand2]  # equal\n    [operand1] != [operand2]  # not equal\n    [operand1] < [operand2]  # less\n    [operand1] <= [operand2]  # less equal\n    [operand1] > [operand2]  # greater\n    [operand1] >= [operand2]  # greater equal\n\n    ! [operand]     # logical not\n    \n    [operand1] && [operand2]  # logical and\n    \n    [operand1] || [operand2]  # logical or\n    \n    [operand1] = [operand2]  # assignment\n    [operand1] += [operand2]  # assign and add:  [operand1] = [operand1] + [operand2]\n    [operand1] -= [operand2]  # assign and subtract:  [operand1] = [operand1] - [operand2]\n    [operand1] *= [operand2]  # assign and multiply:  [operand1] = [operand1] * [operand2]\n    [operand1] /= [operand2]  # assign and divide:  [operand1] = [operand1] / [operand2]\n    [operand1] %= [operand2]  # assign and calculate modulus:  [operand1] = [operand1] % [operand2]\n    [operand1] &&= [operand2]  # assign and and:  [operand1] = [operand1] && [operand2]\n    [operand1] ||= [operand2]  # assign and or:  [operand1] = [operand1] || [operand2]\n### if, elif, else\nIf statements can be implemented by:\n\n    if([expression])\n    [if body]\n    elif([expression])\n    [elif body]\n    else\n    [else body]\n    end\nThe first body with an true (nonzero) expression will be executed or, if none is true, the `else` body will be executed.\nEvery body has its own namespace. Everything except the `if([expression])` and `end` is optional. There can also be more than one elif.\n### while\nWhile loops can be implemented by:\n\n    while([expression])\n    [while body]\n    end\nThe body will be executed until it is leaved by `break` or the expression `[expression]` is not true anymore.\nEvery iteration can be cut short with `continue`. This immediately triggers a new loop iteration.\n### for\nFor loops are implemented by:\n\n    for([expression or variable definition], [cond_expression], [iter_expression])\n    [for_body]\n    end\n\nFor loops lopp until expression cond_expression is false or it is stopped by `break`.\nBefore the loop, an expression is executed or a variable in the embedding scope `[for_body]` is definded by [expression or variable definition].\nEvery iteration can be cut short with `continue`. This immediately triggers a new loop iteration.\n### continue, break\n\n    continue\n    break\n`break` exits a loop immeditly and `continue` skips the remainder of the current iteration.\nBoth are only allowed within loops.\n### Inline assembler\nIt is possible to execute IC10-MIPS assembler code [asm code] within the source code through:\n\n    asm [register list and associated variables]\n    [asm code]\n    end\n\n`[register list and associated variables]` refers to a comma separated list, consisting out of elements representing registers like `$r0`-`$r13` \nand optional associations with variables by `$r0-13=[variable]` where variable refers to the identifier of the variable. All modifications to that register will be saved in the variable.\nasm code must start with a `@` like `@move $r0 $r1`.\nIn asm code registers can be used like usual. However they shouldn\'t be since that may break the program they should be addressed with `$r0` - `$r13` \nand added to the register list by which only the compiler assigned registers will be used that are also safe to use (references to them in the comments within an inline asm will be replaced with the corresponding compiler-assigned registers as well).\nA maximum of 14 registers can be used like that since `sp`, `ra`, `r14` and `r15` are used by the compiler. However, `ra`, `r14` and `r15` are generally save to use within an inline assembler block. Only `sp` must refer to the same value at the end.\nThe registers will in general not be associated with the corresponding registers in IC10 assembler source code, \nsince they could be reserved for variables.\nIt should not be possible to break a program with inline assembler since all used registers get backed up on the stack before the execution of the inline asm block.\nThis is not the case for jumps and branching instructions. Use of those should be limited to labels within the inline asm block. The latter will not be enforced.\n\nIt is possible but not recommended to use asm code outside of a asm block. Everything after a `@` will be placed in the source code without modification (including comments).\n## Examples\n### IC10 giving sphere coordinates for a vector\n\n    dev posx=0\n    dev posy=1\n    dev posz=2\n    dev output_dist=3\n    dev output_phi=4\n    dev output_theta=5\n\n    func atan2(x,y)\n        const pi = 3.14159\n        if(x>0)\n            return atan(y/x)\n        elif(y>0)\n            return pi/2 - atan(x/y)\n        elif(y<0)\n            return -pi/2 - atan(x/y)\n        elif(x<0)\n            return atan(y/x) + pi\n        else\n            return 0\n        end\n        # return is needed here unfortunatly\n        return 0\n    end\n\n    main\n        while(true)\n            # length of vector\n            output_dist.Value = sqrt(posx.Value*posx.Value + posy.Value*posy.Value + posz.Value*posz.Value)\n            # Sphere coordinates\n            output_phi.Value = atan2(posx.Value, posy.Value)\n            output_theta.Value = asin(posz.Value / output_dist.Value)\n            yield()\n        end\n    end\n\n\n### Fibonacci number generator\n    main\n        var l=1\n        var ll=1\n        for(var i=0, i<50, ++i)\n            self.Setting = l+ll\n            ll=l\n            l=self.Setting\n            yield()\n        end\n    end\n\n### Recursive Fibonacci number generator\n\n    func fibonacci(start)\n        if(start <= 2)\n            return 1\n        else\n            return fibonacci(start-2)+fibonacci(start-1)\n        end\n    end\n\n    main\n        self.Setting = fibonacci(9)\n    end\n',
    'author': 'Florian Ullmann',
    'author_email': 'florian.ullmann@protonmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://gitlab.com/smpkdev/compic10',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
