# Copyright (c) 2021-present, Ethan Henderson
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# 1. Redistributions of source code must retain the above copyright notice, this
#    list of conditions and the following disclaimer.
#
# 2. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
#
# 3. Neither the name of the copyright holder nor the names of its
#    contributors may be used to endorse or promote products derived from
#    this software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

from __future__ import annotations

import json
import logging
import pathlib
import typing as t
from dataclasses import dataclass

import aiofiles

from analytix.types import TokenT

log = logging.getLogger(__name__)


@dataclass()
class Tokens:
    """A dataclass representing a set of tokens for a Google Developers
    project. This should generally be created using one of the available
    classmethods.

    Args:
        access_token:
            The token used to access services.
        expires_in:
            The number of seconds until the token expires. This is not
            live updated by analytix, and so should not be used to track
            a token's expiration.
        refresh_token:
            The token used to refresh the access token.
        scope:
            The scopes the token is authorised for. This is provided as
            a single string, where separate scopes are separated by a
            space.
        token_type:
            The type of token. Will probably be "Bearer".
    """

    access_token: str
    expires_in: int
    refresh_token: str
    scope: str
    token_type: str

    def __getitem__(self, key: str) -> TokenT:
        return t.cast(TokenT, getattr(self, key))

    @classmethod
    def from_data(cls, data: dict[str, TokenT]) -> Tokens:
        """Create an instance of this class from a dictionary.

        Args:
            data:
                The tokens as a dictionary.

        Returns:
            The created instance.
        """

        return cls(**data)  # type: ignore

    @classmethod
    def from_file(cls, path: pathlib.Path | str) -> Tokens:
        """Load a set of tokens from a file generated by analytix.

        Args:
            path:
                The path to the tokens file.

        Returns:
            The loaded tokens.
        """

        if not isinstance(path, pathlib.Path):
            path = pathlib.Path(path)

        log.debug(f"Loading tokens from {path.resolve()}...")

        with open(path) as f:
            data = json.load(f)

        log.info("Tokens loaded!")
        return cls(**data)

    @classmethod
    async def afrom_file(cls, path: pathlib.Path | str) -> Tokens:
        """Asynchronously load a set of tokens from a file generated by
        analytix.

        Args:
            path:
                The path to the tokens file.

        Returns:
            The loaded tokens.
        """

        if not isinstance(path, pathlib.Path):
            path = pathlib.Path(path)

        log.debug(f"Loading tokens from {path.resolve()}...")

        async with aiofiles.open(path) as f:
            data = json.loads(await f.read())

        log.info("Tokens loaded!")
        return cls(**data)

    def update(self, data: dict[str, TokenT]) -> None:
        """Update token attributes.

        .. warning::
            This does not update the tokens on disk.

        Args:
            data:
                A dictionary of attributes to be updated.
        """

        for k, v in data.items():
            setattr(self, k, v)

        log.info("Tokens updated!")

    def to_dict(self) -> dict[str, TokenT]:
        """Convert tokens to a dictionary.

        Returns:
            A dictionary of tokens, where the keys are strings, and the
            values are either strings or integers.
        """

        return {
            "access_token": self.access_token,
            "expires_in": self.expires_in,
            "refresh_token": self.refresh_token,
            "scope": self.scope,
            "token_type": self.token_type,
        }

    def write(self, path: pathlib.Path | str) -> None:
        """Write tokens to a file.

        Args:
            path:
                The path to the tokens file.
        """

        if not isinstance(path, pathlib.Path):
            path = pathlib.Path(path)

        with open(path, "w") as f:
            json.dump(self.to_dict(), f)

        log.info(f"Tokens saved to {path.resolve()}")

    async def awrite(self, path: pathlib.Path | str) -> None:
        """Asynchronously write tokens to a file.

        Args:
            path:
                The path to the tokens file.
        """

        if not isinstance(path, pathlib.Path):
            path = pathlib.Path(path)

        async with aiofiles.open(path, "w") as f:
            await f.write(json.dumps(self.to_dict()))

        log.info(f"Tokens saved to {path.resolve()}")
