"""
scaffoldgraph.prioritization.prioritization_rules

Implements abstract rules for scaffold prioritization when constructing scaffold trees.
"""

from abc import ABCMeta, abstractmethod
from itertools import compress

__all__ = [
    'BaseScaffoldFilterRule',
    'ScaffoldFilterRule',
    'ScaffoldMaxFilterRule',
    'ScaffoldMinFilterRule',
]


class BaseScaffoldFilterRule(metaclass=ABCMeta):
    """Abstract base class for defining rules for scaffold prioritization.

    Scaffold filter rules should subclass this base class.
    All base rules should implement the ``name`` property and the
    ``filter`` function.

    """
    @abstractmethod
    def filter(self, child, parents):
        """Filter a set of input scaffolds (parents).

        The filter method should filter a set of input parent scaffolds using
        a defined rule.

        Defined rules may use properties relating to the scaffolds themselves
        or from the child scaffold from which they were obtained. i.e. properties
        of the ring that was removed:

        # Get index of removed ring from a parent scaffold
        >>> removed_ring = parents[0].removed_ring_index

        # Get atoms/bonds in this ring
        >>> ring_atoms = child.rings[removed_ring]

        # calculate property (i.e. number of carbon atoms in ring)
        >>> prop = [a.GetAtomicNum() for a in ring_atoms].count(6)

        Parameters
        ----------
        child : scaffoldgraph.core.Scaffold
            The child scaffold from which the parent scaffolds were obtained.
        parents : iterable
            An iterable of all parent scaffolds generated by a fragmenter.

        """
        raise NotImplementedError()

    @property
    @abstractmethod
    def name(self):
        """Return the name of the filter rule.

        Subclasses should define this, returning a name for the rule.

        """
        raise NotImplementedError()

    def __call__(self, child, parents):
        return self.filter(child, parents)

    def __str__(self):
        return str(self.name)

    def __repr__(self):
        return '<{_cls} at {address}>'.format(
            _cls=self.__class__.__name__,
            address=hex(id(self))
        )


class ScaffoldFilterRule(BaseScaffoldFilterRule):
    """
    Abstract base class for defining rules for scaffold prioritization
    based on a defined True/False condition.

    Subclasses should implement the condition method, where a boolean value
    is returned for a particular input scaffold. Scaffolds with a 'True'
    property will be retained when using the filter method. Subclasses
    should also implement the ``name`` property.

    Example
    -------
    >>> class MyRule(ScaffoldFilterRule):
    ...
    ...    def condition(self, child, parent):
    ...        if parent ... :
    ...            return True
    ...        return False
    ...
    ...    @property
    ...    def name(self):
    ...       return 'my conditional rule'

    """
    def filter(self, child, parents):
        """Filter a set of parent scaffolds using a defined condition.

        Parameters
        ----------
        child : scaffoldgraph.core.Scaffold
            The child scaffold from which the parent scaffolds were obtained.
        parents : iterable
            An iterable of all parent scaffolds generated by a fragmenter.

        """
        return [s for s in parents if self.condition(child, s)]

    @abstractmethod
    def condition(self, child, parent):
        """A Boolean condition for scaffold filtering.

        Subclasses should implement this method.

        Parameters
        ----------
        child : scaffoldgraph.core.Scaffold
            The child scaffold from which the parent scaffolds were obtained.
        parent : scaffoldgraph.core.Scaffold
            A parent scaffold.

        """
        raise NotImplementedError()


class ScaffoldMinFilterRule(BaseScaffoldFilterRule):
    """
    Abstract base class for defining rules for scaffold prioritization
    based on a minimum property value.

    Subclasses should implement the ``get_property method``, where a property value
    is returned for a particular input scaffold. Scaffolds with a property value
    equal to the minimum property value will be retained. Subclasses should also
    implement the ``name`` property.

    Example
    -------
    >>> class MyRule(ScaffoldMinFilterRule):
    ...
    ...    def get_property(self, child, parent):
    ...        prop = get_some_property(parent)
    ...        return prop
    ...
    ...    @property
    ...    def name(self):
    ...        return 'my min conditional rule'

    """
    def filter(self, child, parents):
        """Filter a set of parent scaffolds using a minimum property value.

        Parameters
        ----------
        child : scaffoldgraph.core.Scaffold
            The child scaffold from which the parent scaffolds were obtained.
        parents : iterable
            An iterable of all parent scaffolds generated by a fragmenter.

        """
        props = [self.get_property(child, s) for s in parents]
        min_val = min(props)
        return list(compress(parents, [True if p == min_val else False for p in props]))

    @abstractmethod
    def get_property(self, child, parent):
        """Return a property value for a child/parent scaffold.

        Subclasses should implement this method.

        Parameters
        ----------
        child : scaffoldgraph.core.Scaffold
            The child scaffold from which the parent scaffolds were obtained.
        parent : scaffoldgraph.core.Scaffold
            A parent scaffold.

        """
        raise NotImplementedError()


class ScaffoldMaxFilterRule(BaseScaffoldFilterRule):
    """Abstract base class for defining rules for scaffold prioritization
    based on a maximum property value.

    Subclasses should implement the ``get_property`` method, where a property value
    is returned for a particular input scaffold. Scaffolds with a property value
    equal to the maximum property value will be retained. Subclasses should also
    implement the ``name`` property.

    Example
    -------
    >>> class MyRule(ScaffoldMaxFilterRule):
    ...
    ...    def get_property(self, child, parent):
    ...        prop = get_some_property(parent)
    ...        return prop
    ...
    ...    @property
    ...    def name(self):
    ...        return 'my min conditional rule'

    """
    def filter(self, child, parents):
        """Filter a set of parent scaffolds using a maximum property value.

        Parameters
        ----------
        child : scaffoldgraph.core.Scaffold
            The child scaffold from which the parent scaffolds were obtained.
        parents : iterable
            An iterable of all parent scaffolds generated by a fragmenter.

        """
        props = [self.get_property(child, s) for s in parents]
        max_val = max(props)
        return list(compress(parents, [True if p == max_val else False for p in props]))

    @abstractmethod
    def get_property(self, child, parent):
        """Return a property value for a child/parent scaffold.

        Subclasses should implement this method.

        Parameters
        ----------
        child : scaffoldgraph.core.Scaffold
            The child scaffold from which the parent scaffolds were obtained.
        parent : scaffoldgraph.core.Scaffold
            A parent scaffold.

        """
        raise NotImplementedError()
