"""
scaffoldgraph.core.fragment
"""
import re
from abc import ABC, abstractmethod

from loguru import logger

from rdkit.Chem import (
    Mol,
    Atom,
    RWMol,
    MolToSmiles,
    rdmolops,
    SanitizeMol,
    GetMolFrags,
    BondType,
    GetSymmSSSR,
    RemoveHs,MolFromSmarts,
    Descriptors,
    MurckoDecompose,
    MolStandardize,
    SetAromaticity,
    AROMATICITY_SIMPLE,AROMATICITY_RDKIT,AROMATICITY_MDL,
    CHI_UNSPECIFIED,
    SANITIZE_ALL,
    SANITIZE_CLEANUP,
    SANITIZE_CLEANUPCHIRALITY,
    SANITIZE_FINDRADICALS,MolToMolBlock,
    MolToMolBlock, MolFromMolBlock

)

from scaffoldgraph.core.scaffold import Scaffold
from scaffoldgraph.utils import suppress_rdlogger


class Fragmenter(ABC):
    """Abstract base class for scaffold fragmentation methods.

    Fragmenters should be designed to be used for generating
    scaffold graphs. subclasses may use attributes to store an
    internal state or property used during fragmentation.

    Subclasses should define the fragment method which takes a
    scaffold ``scaffoldgraph.core.Scaffold`` as an argument
    and returns the next set of parent scaffolds after
    fragmentation of the child scaffold. Subclasses may also
    define an __init__ method to supply any arguments which may
    be used during fragmentation. Note: it is expected that
    the fragment method return scaffolds as
    scaffoldgraph.core.Scaffold objects.

    The __call__ method can be used as an alias to the fragment
    function.

    """
    def __call__(self, scaffold):
        return self.fragment(scaffold)

    @abstractmethod
    def fragment(self, scaffold):
        """Subclasses should implement this method.

        Parameters
        ----------
        scaffold : scaffoldgraph.core.Scaffold
            Child scaffold to be fragmented.

        Returns
        -------
        list
            This method should return the next set of parent scaffolds
            after fragmentation of the child scaffold.

        """
        raise NotImplementedError()

    def __repr__(self):
        return '<{_cls} at {address}>'.format(
            _cls=self.__class__.__name__,
            address=hex(id(self))
        )


class MurckoRingFragmenter(Fragmenter):
    """
    A Fragmenter class for the removal of peripheral rings from a
    Murcko scaffold.

    This fragmenter is hierarchical, returning only parent scaffolds
    that are one hierarchy below the supplied child scaffold (i.e.
    one ring less).

    Notes
    -----
    This paticular implementation will also fragment fused ring
    systems. If this behaviour is not desired the
    ``MurckoRingSystemFragmenter`` may be used instead.

    See Also
    --------
    scaffoldgraph.core.fragment.MurckoRingSystemFragmenter

    """
    def __init__(self, use_scheme_4=False):
        """
        Parameters
        ----------
        use_scheme_4 : bool
            if True use scheme 4 from the paper:
            `The Scaffold Tree − Visualization of the Scaffold Universe
            by Hierarchical Scaffold Classification.` This scheme should
            be used when generating scaffold trees with the original
            prioritization rules.

        Notes
        -----
        Scheme 4 (`description taken from paper`):
            The fusion bond connecting a three-membered ring with other
            rings is converted into a double bond. This rule is intended
            to deal with epoxides and aziridines. This rule treats such
            systems as functional groups which are removed beforehand,
            rather than as rings. This reflects the situation that epoxides
            are usually generated by the oxidation of a double bond, and
            also many natural products exist often in forms with and
            without epoxidized double bonds.

        """
        super(MurckoRingFragmenter, self).__init__()
        self.use_scheme_4 = use_scheme_4

    def fragment(self, scaffold):
        """Fragment a scaffold into its next set of Murcko fragments.

        Parameters
        ----------
        scaffold : scaffoldgraph.core.Scaffold
            Child scaffold to be fragmented.

        Returns
        -------
        list
            A list of parent scaffolds representing the next hierarchy.

        """
        #print('MurckoRingFragmenter');
        parents = []  # container for parent scaffolds
        rings = scaffold.rings  # ring information

        for rix, ring in enumerate(rings):  # Loop through all rings and remove
            edit = RWMol(scaffold.mol)  # Editable molecule
            editMolBlk = MolToMolBlock(edit, kekulize=False);
            molA = MolFromMolBlock(editMolBlk);
            SetAromaticity(molA, AROMATICITY_SIMPLE);
            editArmMolBlk = MolToMolBlock(molA, kekulize=False);
            # find all bonds with 4 / trouver toutes les liasons avec 4
            #matches = re.findall(".{6}(?=  4  \d)", editArmMolBlk)
            molQTest = MolFromSmarts("[#6,#7,#8;a]")
            arom_atom_indices_ = molA.GetSubstructMatches(molQTest)#.map((s) => s[0]);
            arom_atom_indices = sorted(list(map(lambda x: x[0], arom_atom_indices_)))
            
            #print('Before setting 11111')
            #print(editMolBlk)
            idxV2000 = editArmMolBlk.index("V2000\n")
            idxV2000ToEnd = editArmMolBlk.index("\n", idxV2000) +1
            count = 0;
            for idx_sorted in arom_atom_indices: 
                while count < idx_sorted: 
                    count = count + 1
                    idxV2000ToEnd = editArmMolBlk.index("\n", idxV2000ToEnd) +1
                                
                #editArmMolBlk = replace_str_index(editArmMolBlk, idxV2000ToEnd + 56, "1")
                editArmMolBlk = editArmMolBlk[:idxV2000ToEnd + 56] + "1" + editArmMolBlk[idxV2000ToEnd + 57:]
                
            #print(editArmMolBlk)
            #print(idxV2000ToEnd)
            #print(editArmMolBlk[idxV2000ToEnd : idxV2000ToEnd + 30])

            editArmMolBlkMarked = editArmMolBlk;#re.sub("(?<=(\d  ){7})0(?=(  \d){4})", "1", editArmMolBlk)
            
            #print(editArmMolBlkMarked)
            molARW = RWMol(MolFromMolBlock(editArmMolBlkMarked))
            editAromMolBlkMrk = MolToMolBlock(molARW, kekulize=False)
            #print(editAromMolBlkMrk);
            # End edit - molecule / La FIN
            
            # Collect all removable atoms in the molecule
            remove_atoms = set()
            for index, atom in zip(ring.aix, ring.atoms):
                if rings.info.NumAtomRings(index) == 1:
                    if atom.GetDegree() > 2:  # Evoke linker collection
                        collect_linker_atoms(edit.GetAtomWithIdx(index), remove_atoms)
                    else:  # Add ring atom to removable set
                        remove_atoms.add(index)
                else:  # Atom is shared between multiple rings
                    correct_atom_props(edit.GetAtomWithIdx(index))

            # Collect removable bonds (this needs to be done to prevent the case where when deleting
            # a ring two atoms belonging to the same bond are also part of separate other rings.
            # This bond must be broken to prevent an incorrect output)
            remove_bonds = set()
            for bix in {x for x in ring.bix if rings.info.NumBondRings(x) == 1}:
                bond = edit.GetBondWithIdx(bix)
                b_x, b_y = bond.GetBeginAtomIdx(), bond.GetEndAtomIdx()
                if b_x not in remove_atoms and b_y not in remove_atoms:
                    remove_bonds.add((b_x, b_y))
                    correct_atom_props(edit.GetAtomWithIdx(b_x))
                    correct_atom_props(edit.GetAtomWithIdx(b_y))

            # Scheme 4 (scaffold tree rule)
            if self.use_scheme_4 is not False and len(ring) == 3:
                atomic_nums = [a.GetAtomicNum() for a in ring.atoms]
                if len([a for a in atomic_nums if a != 1 and a != 6]) == 1:
                    shared = {x for x in ring.bix if rings.info.NumBondRings(x) > 1}
                    if len(shared) == 1:
                        bond = edit.GetBondWithIdx(shared.pop())
                        bond.SetBondType(BondType.DOUBLE)
                        print('DOUBLE')

            # Remove collected atoms and bonds
            for bix in remove_bonds:
                molARW.RemoveBond(*bix)  #edit.RemoveBond(*bix)
            for aix in sorted(remove_atoms, reverse=True):
                molARW.RemoveAtom(aix)    #edit.RemoveAtom(aix)

            # Add new parent scaffolds to parent list
            #print('molEdit')
            #molEditArom = SetAromaticity(edit, AROMATICITY_SIMPLE);
            #molEditBlock = MolToMolBlock(molEditArom, , kekulize=False);
           
            #print('molEditArom')
            for parent in get_scaffold_frags(molARW): #(edit)
                #print('parent')
                molPBlok = MolToMolBlock(parent.mol, kekulize=False);
                #print(molPBlok)

                arom_atoms = []
                atomCount = len(parent.mol.GetAtoms())
                bondCount = len(parent.mol.GetBonds())
                #print(atomCount)
                idxV2000 = molPBlok.index("V2000\n")
                idxV2000ToEnd = molPBlok.index("\n", idxV2000) +1
                for idx_atom in range(atomCount):
                    idxV2000ToEnd = idxV2000ToEnd + 56
                    if molPBlok[idxV2000ToEnd] == "1":
                        arom_atoms.append(idx_atom +1)
                    idxV2000ToEnd = molPBlok.index("\n", idxV2000ToEnd) +1
                #print(arom_atoms)
               
                for idx_bond in range(bondCount):
                    atom_1 = int(molPBlok[idxV2000ToEnd: idxV2000ToEnd+3])
                    atom_2 = int(molPBlok[idxV2000ToEnd+4: idxV2000ToEnd+6])
                    #print(atom_1)
                    #print(atom_2)
                    if atom_1 in arom_atoms and atom_2 in arom_atoms:
                        molPBlok = molPBlok[:idxV2000ToEnd + 8] + "4" + molPBlok[idxV2000ToEnd + 9:]
                    idxV2000ToEnd = molPBlok.index("\n", idxV2000ToEnd) +1

                #print(molPBlok)
                #test = MolFromMolBlock(molPBlok, False, True, False)
                #print('Test');
                #print(MolToMolBlock(test, kekulize = False))
                parent.mol = MolFromMolBlock(molPBlok, False, True, False)
                if parent.rings.count == len(rings) - 1:
                    parent.removed_ring_idx = rix
                    parents.append(parent)
                
        return parents


class MurckoRingSystemFragmenter(Fragmenter):
    """A Fragmenter class for the removal of peripheral ring systems from a
    Murcko scaffold.

    This fragmenter is hierarchical, returning only parent scaffolds
    that are one hierarchy below the supplied child scaffold. In this
    case the hierarchy below is one ring system less rather than one
    ring less.

    Notes
    -----
    This paticular implementation will not fragment fused ring
    systems. If this behaviour is not desired the
    ``MurckoRingFragmenter`` may be used instead.

    See Also
    --------
    scaffoldgraph.core.fragment.MurckoRingFragmenter

    """
    def __init__(self):
        super(MurckoRingSystemFragmenter, self).__init__()

    def fragment(self, scaffold):
        """Fragment a scaffold into its next set of Murcko fragments.

        Parameters
        ----------
        scaffold : scaffoldgraph.core.Scaffold
            Child scaffold to be fragmented.

        Returns
        -------
        list
            A list of parent scaffolds representing the next hierarchy.

        """

        print('MurckoRingSystemFragmenter');
        parents = []
        rings = scaffold.ring_systems  # ring system information
        info = scaffold.rings.info

        if rings.count == 1:
            return []
        for rix, ring in enumerate(rings):
            edit = RWMol(scaffold.mol)
            remove_atoms = set()
            for index, atom in zip(ring.aix, ring.atoms):
                if info.NumAtomRings(index) == 1 or any([not b.IsInRing() for b in atom.GetBonds()]):
                    if atom.GetDegree() > 2:  # Evoke linker collection
                        collect_linker_atoms(edit.GetAtomWithIdx(index), remove_atoms)
                    else:
                        remove_atoms.add(index)
                else:
                    remove_atoms.add(index)

            for aix in sorted(remove_atoms, reverse=True):
                edit.RemoveAtom(aix)

            for parent in get_scaffold_frags(edit):
                if parent.ring_systems.count == len(rings) - 1:
                    parent.removed_ring_idx = rix
                    parents.append(parent)

        return parents


def collect_linker_atoms(origin, remove_atoms, include_origin=True):
    """Collect atoms that are part of a linker.

    This function is primarily used during molecular fragmentation.
    Linker atoms are added to the `remove_atoms` set supplied. The
    funtion also returns any ring attachment points encountered.

    Parameters
    ----------
    origin : rdkit.Chem.rdchem.Atom
        Origin atom from where the linker will be collected.
    remove_atoms : set
        A set to which linker atoms indexes will be added may
        or may not contain other atom indexes.
    include_origin : bool
        If True add the origin atoms index to the `remove_atoms`
        set supplied.

    Returns
    -------
    set
        The returned set contains any ring attachment points
        encountered. The linker atoms are added to the
        `remove_atoms` set that is supplied.

    """
    visited = set()  # Visited bond indexes
    ring_attachments = set()  # Linker ring attachments

    def collect(origin_atom):

        for bond in origin_atom.GetBonds():
            bond_id = bond.GetIdx()
            if bond_id in visited or bond.IsInRing():
                continue

            other_atom = bond.GetOtherAtom(origin_atom)
            other_degree = other_atom.GetDegree()

            if other_degree == 1:  # Terminal side-chain
                remove_atoms.add(origin_atom.GetIdx())
                remove_atoms.add(other_atom.GetIdx())
                correct_atom_props(origin_atom)
                visited.add(bond_id)

            elif other_degree == 2:  # Two neighboring atoms (remove)
                remove_atoms.add(origin_atom.GetIdx())
                visited.add(bond_id)
                collect(other_atom)

            elif other_degree > 2:  # Branching point

                # Determine number of non-terminal branches
                non_terminal_branches = 0
                for neighbor in other_atom.GetNeighbors():
                    if neighbor.GetDegree() != 1:
                        non_terminal_branches += 1

                if non_terminal_branches < 3:  # Continue with deletion
                    remove_atoms.add(origin_atom.GetIdx())
                    visited.add(bond_id)
                    collect(other_atom)

                else:  # Branching point links two rings
                    # Test for exolinker double bond
                    if not bond.GetBondType() == BondType.DOUBLE:
                        remove_atoms.add(origin_atom.GetIdx())
                        correct_atom_props(other_atom)
                        visited.add(bond_id)
                    if other_atom.IsInRing():
                        ring_attachments.add(other_atom.GetIdx())

    # Linker is recursively collected
    # Linker atoms are added to the existing set 'remove_atoms'
    collect(origin)

    if include_origin is False:
        remove_atoms.discard(origin.GetIdx())
    if origin.IsInRing():
        ring_attachments.add(origin.GetIdx())

    return ring_attachments


def get_scaffold_frags(frag, hash_func=None):
    """Get fragments from a disconnected structure.

    This function is used primarily used during molecular fragmentation.

    Parameters
    ----------
    frag : rdkit.Chem.rdchem.Mol
        An rdkit Mol containing disconeccted structures.
    hash_func : callable, optional
        Scaffold hash function, the default is the canonical smiles.

    Returns
    -------
    set
        A set of scaffoldgraph.core.Scaffold objects which are unique
        fragments obtained from the disconnected structures in the
        input molecule.

    Notes
    -----
    The function also performs a `partial sanitization` of the input.
    If the molecule fails to sanitize the function will return an
    empty set. This case can occur when an aromatic ring system is
    dissected in a way in which the resultant system is no longer
    aromatic.

    """
    try:
        # frag.ClearComputedProps()
        # frag.UpdatePropertyCache()
        # FastFindRings(frag)
        partial_sanitization(frag)
    except ValueError as e:
        # This error is caught as dissecting an aromatic ring system,
        # may lead to an undefined state where the resultant system
        # is no longer aromatic. We make no attempt to prevent this
        # but log it for reference.
        # This behaviour may be desirable for a scaffold tree and is
        # equivalent to the behavior of SNG (I believe...)
        logger.debug(e)
        return set()
    frags = {Scaffold(f, hash_func) for f in GetMolFrags(frag, True, False)}
    return frags


def correct_atom_props(atom):
    """
    Used during fragmentation to correct atom properties where an
    adjacent atom is removed.

    Parameters
    ----------
    atom : rdkit.Chem.rdchem.Atom
        Atom to correct.

    """
    if atom.GetIsAromatic() and atom.GetAtomicNum() != 6:
        atom.SetNumExplicitHs(1)
    elif atom.GetNoImplicit() or atom.GetChiralTag() != CHI_UNSPECIFIED:
        atom.SetNoImplicit(False)
        atom.SetNumExplicitHs(0)
        atom.SetChiralTag(CHI_UNSPECIFIED)


def partial_sanitization(mol):
    """Partially sanitize a molecule.

    Parameters
    ----------
    mol : rdkit.Chem.rdchem.Mol
        Molecule to sanitize.

    """
    SanitizeMol(
        mol, sanitizeOps=SANITIZE_ALL ^
        SANITIZE_CLEANUP ^
        SANITIZE_CLEANUPCHIRALITY ^
        SANITIZE_FINDRADICALS
    )


def flatten_isotopes(mol):
    """Remove specific isotopes from a molecule.

    This operation is performed in-place.

    Parameters
    ----------
    mol : rdkit.Chem.rdchem.Mol

    """
    for atom in mol.GetAtoms():
        atom.SetIsotope(0)


def keep_largest_fragment(mol):
    """Return the largest fragment in a disconnected molecule.

    The largest fragment is simply considered to be the
    fragment with the largest number of atoms.

    Parameters
    ----------
    mol : rdkit.Chem.rdchem.Mol
        rdkit molecule containg disconnected fragments.

    Returns
    -------
    mol : rdkit.Chem.rdchem.Mol
        Molecule containing the largest disconnected
        fragment.

    """
    frags = GetMolFrags(mol, asMols=True)
    if len(frags) <= 1:
        return mol
    return max(frags, key=lambda x: x.GetNumAtoms())


def convert_radicals_to_hydrogen(mol):
    """Converts radical electrons in a molecule into bonds to hydrogens.

    Parameters
    ----------
    mol : rdkit.Chem.rdchem.Mol

    Returns
    -------
    mol : rdkit.Chem.rdchem.Mol
        New molecule with radical electrons converted into
        bonds to hydrogens.

    """
    m = Mol(mol)
    if Descriptors.NumRadicalElectrons(m) == 0:
        return m
    else:
        for atom in m.GetAtoms():
            nradicals = atom.GetNumRadicalElectrons()
            if nr > 0:
                atom.SetNumRadicalElectrons(0)
                atom.SetNumExplicitHs(nradicals)
    return m


@suppress_rdlogger()
def discharge_molecule(mol):
    """Discharge molecules using the RDKit Uncharger class.

    Uncharges molecules by adding and/or removing hydrogens.
    For zwitterions, hydrogens are moved to eliminate charges where possible.
    In cases where there is a positive charge that is not neutralizable,
    an attempt is made to also preserve the corresponding negative charge.

    Parameters
    ----------
    mol : rdkit.Chem.rdchem.Mol

    Returns
    -------
    mol : rdkit.Chem.rdchem.Mol
        New molecule with charges eliminated where possible.

    """
    uncharger = MolStandardize.charge.Uncharger()
    out = uncharger.uncharge(mol)
    return out


def discharge_and_deradicalize(mol):
    """Discharge and remove radicals.

    Parameters
    ----------
    mol : rdkit.Chem.rdchem.Mol

    Returns
    -------
    mol : rdkit.Chem.rdchem.Mol
        New molecule with charges eliminated where possible
        and radicals removed.

    """
    out = convert_radicals_to_hydrogen(mol)
    out = discharge_molecule(out)
    return out
        

def remove_exocyclic_attachments(mol):
    """
    Remove exocyclic and exolinker attachments from
    a molecule.

    Parameters
    ----------
    mol : rdkit.Chem.rdchem.Mol

    Returns
    -------
    rdkit.Chem.rdchem.Mol
        Molecule with exocyclic/exolinker attachments
        removed.

    """
    edit = RWMol(mol)
    remove_atoms = set()
    for atom in edit.GetAtoms():
        degree = atom.GetDegree()
        if degree == 1:
            bond = atom.GetBonds()[0]
            if bond.GetBondTypeAsDouble() == 2.0:
                nbr = bond.GetOtherAtom(atom)
                hcount = nbr.GetTotalNumHs()
                nbr.SetNumExplicitHs(hcount + 2)
                nbr.SetNoImplicit(True)
                remove_atoms.add(atom.GetIdx())
    for aix in sorted(remove_atoms, reverse=True):
        edit.RemoveAtom(aix)
    rdmolops.AssignRadicals(edit)
    GetSymmSSSR(edit)
    return edit.GetMol()


def genericise_scaffold(mol):
    """Make a scaffold generic.

    Parameters
    ----------
    mol : rdkit.Chem.rdchem.Mol
        Molecule to make generic.

    Returns
    -------
    rdkit.Chem.rdchem.Mol
        Genericised scaffold.

    Notes
    -----
    Copy pasta'd from rdkit Murcko Scaffold module.
    Adds a degree check to make sure output will
    not fail sanitization when an atom has a degree
    > 4. Achieved by using a dummy atom to replace
    such atoms.

    """
    print('genericise_scaffold');
    out = Mol(mol)
    for atom in out.GetAtoms():
        if atom.GetAtomicNum() != 1:
            if atom.GetDegree() <= 4:
                atom.SetAtomicNum(6)
            else:
                atom.SetAtomicNum(0)
        atom.SetIsAromatic(False)
        atom.SetFormalCharge(0)
        atom.SetChiralTag(CHI_UNSPECIFIED)
        atom.SetNoImplicit(0)
        atom.SetNumExplicitHs(0)
    for bond in out.GetBonds():
        bond.SetBondType(BondType.SINGLE)
        bond.SetIsAromatic(False)
    return RemoveHs(out)


def _collapse_linker_bonds(mol, retain_het=False):
    """Private. condense linkers into a single chain.

    Used when constructing collapsed linker Murcko
    scaffolds and ring topology scaffolds.

    Parameters
    ----------
    mol : rdkit.Chem.rdchem.Mol
    retain_het : bool, optional
        If True retain heteroatoms in the linker.
        The default is False.

    Returns
    -------
    rdkit.Chem.rdchem.RWMol
        Mol object with collapsed linkers.

    """

    def collapse(edit):
        for atom in edit.GetAtoms():
            if atom.IsInRing():
                continue
            nbrs = atom.GetNeighbors()
            if len(nbrs) == 2 and (
                retain_het is False or
                nbrs[0].GetAtomicNum() == atom.GetAtomicNum() or
                nbrs[1].GetAtomicNum() == atom.GetAtomicNum()
            ):
                nix = map(lambda x: x.GetIdx(), nbrs)
                edit.AddBond(*nix, BondType.SINGLE)
                edit.RemoveAtom(atom.GetIdx())
                return collapse(edit)
        return edit

    edit = RWMol(mol)
    edit = collapse(edit)
    return edit


def get_murcko_scaffold(mol, generic=False, remove_exocyclic=False, collapse_linkers=False):
    """Get the murcko scaffold for an input molecule.

    Parameters
    ----------
    mol : rdkit.rdchem.Chem.Mol
    generic : bool, optional
        If True return a generic scaffold (CSK).
        The default is False.
    remove_exocyclic : bool, optional
        If True remove all exocyclic/exolinker
        attachments. The default is False.
    collapse_linkers : bool, optional
        If True collapse linkers into a single
        chain. The default is False.

    Returns
    -------
    murcko : rdkit.Chem.rdchem.Mol
        Murcko scaffold.

    Notes
    -----
    If aromaticity is due to exocyclic attachments
    and these are removed, the molecule will fail
    any attmept to kekulize. Removing these however
    may be desirable when generating generic scaffolds.

    """
    #print('collapse_linkers')
    #print(collapse_linkers)
    murcko = MurckoDecompose(mol)
    if remove_exocyclic:
        murcko = remove_exocyclic_attachments(murcko)
    if generic:
        murcko = genericise_scaffold(murcko)
    if collapse_linkers:
        murcko = _collapse_linker_bonds(murcko).GetMol()
    murcko.ClearComputedProps()
    murcko.UpdatePropertyCache()
    GetSymmSSSR(murcko)
    return murcko


def get_annotated_murcko_scaffold(mol, scaffold=None, as_mol=True):
    """
    Return an annotated murcko scaffold where side chains are replaced
    with a dummy atom ('*').

    Parameters
    ----------
    mol : rdkit.Chem.rdchem.Mol
    scaffold : rdkit.Chem.rdchem.Mol, optional
        If a murcko scaffold is already calculated for the `mol`,
        this can be supplied as a template. The default is None.

    as_mol : bool, optional
        If True return rdkit.Chem.rdchem.Mol object else return
        a SMILES string representation. The default is True.

    Returns
    -------
    {str, rdkit.Chem.rdchem.Mol}
        Annotated Murcko scaffold.
MolToSmiles
    """
    if not scaffold:
        scaffold = get_murcko_scaffold(mol)
    annotated = rdmolops.ReplaceSidechains(mol, scaffold)
    if as_mol:
        return annotated
    if annotated is None:
        return ''
    return MolToMolBlock(annotated)


@suppress_rdlogger()
def get_next_murcko_fragments(murcko_scaffold, break_fused_rings=True):
    """
    Fragment a scaffold into its next set of Murcko fragments.

    The fragmenter assumes that a Murcko scaffold is supplied. If not
    the behaviour may be undesirable.

    Parameters
    ----------
    murcko_scaffold : rdkit.Chem.rdchem.Mol
        A Murcko scaffold to fragment
    break_fused_rings : bool, optional
        If True dissect fused rings. The default is True.

    Returns
    -------
    parents : list
        A list of parent scaffolds (next hierarchy [num_rings - 1])

    """
    if break_fused_rings:
        fragmenter = MurckoRingFragmenter()
    else:
        fragmenter = MurckoRingSystemFragmenter()
    parents = [f.mol for f in set(fragmenter.fragment(Scaffold(murcko_scaffold)))]
    return parents


# fragmenter is hierarchical so all fragments may not be returned (fix?).
@suppress_rdlogger()
def get_all_murcko_fragments(mol, break_fused_rings=True):
    """
    Get all possible murcko fragments from a molecule through
    recursive removal of peripheral rings.

    Parameters
    ----------
    mol : rdkit.Chem.rdchem.Mol
    break_fused_rings : bool, optional
        If True dissect fused rings. The default is True.

    Returns
    -------
    list
        A list of Murcko fragments for the input molecule.

    Examples
    --------
    Generating Murcko fragments:

    >>> from rdkit import Chem
    >>> smiles = 'Cc1[nH]cnc1Cn1cccc(-c2ccccc2O)c1=O'
    >>> molecule = Chem.MolFromSmiles(smiles)
    >>> frags = get_all_murcko_fragments(molecule)

    """
    if break_fused_rings:
        fragmenter = MurckoRingFragmenter()
    else:
        fragmenter = MurckoRingSystemFragmenter()
    mol = get_murcko_scaffold(mol)
    rdmolops.RemoveStereochemistry(mol)
    scaffold = Scaffold(mol)
    parents = {scaffold}

    def recursive_generation(child):
        for parent in fragmenter.fragment(child):
            if parent in parents:
                continue
            parents.add(parent)
            recursive_generation(parent)

    recursive_generation(scaffold)
    return [f.mol for f in parents]


def _minimize_rings(mol):
    """Private: Minimize rings in a scaffold.

    In this process, all remaining vertices/atoms of degree two are
    removed by performing an edge merging operation. The only
    exception being when both vertices neighbours are connected
    (i.e. we have a triangle), when edge merging would lead to the
    loss of a cycle. The result is a minimum cycle topological
    representation of the original molecule. This function is used
    in the computation of ring topology scaffolds (Oprea).

    If a ring contains a non-carbon atom, this atom is maintained.
    Neighbouring ring atoms which are of the same type are merged
    together into a single atom of the corresponding type.

    Parameters
    ----------
    mol : rdkit.Chem.rdchem.Mol

    Returns
    -------
    rdkit.Chem.rdchem.RWMol
        Minimum cycle topological graph.

    """
    edit = RWMol(mol)
    remove_atoms = set()
    for atom in edit.GetAtoms():
        if atom.GetDegree() == 2:
            n1, n2 = atom.GetNeighbors()
            n1_idx, n2_idx = n1.GetIdx(), n2.GetIdx()
            connected = edit.GetBondBetweenAtoms(n1_idx, n2_idx)
            if not connected and (
                n1.GetAtomicNum() == atom.GetAtomicNum() or
                n2.GetAtomicNum() == atom.GetAtomicNum()
            ):
                a_idx = atom.GetIdx()
                edit.RemoveBond(n1_idx, a_idx)
                edit.RemoveBond(n2_idx, a_idx)
                edit.AddBond(n1_idx, n2_idx, BondType.SINGLE)
                remove_atoms.add(a_idx)
    for a_idx in sorted(remove_atoms, reverse=True):
        edit.RemoveAtom(a_idx)
    return edit


def get_ring_toplogy_scaffold(mol):
    """Generate a ring topology scaffold (Oprea scaffold).

    This type of scaffold was originally described by Pollock
    and co-workers. The result is a minimum cycle topological
    representation of the original molecule.

    Parameters
    ----------
    mol : rdkit.Chem.rdchem.Mol
        Input molecule for computing a ring topology scaffold.

    Returns
    -------
    rdkit.Chem.rdchem.Mol
        An rdkit molecule object containing a ring topology
        scaffold (Minimum cycle topological graph).

    References
    ----------
    .. [1] Pollock, S.N., Coutsias, E.A., Wester, M.J. & Oprea, T.I.
           (2008) Scaffold topologies. 1. Exhaustive enumeration up
           to eight rings. J. Chem. Inf. Model. 48, 1304-1310.
           PMID: 18605680.

    """
    precursor = get_murcko_scaffold(mol, True, True, True)
    rts = _minimize_rings(precursor)
    GetSymmSSSR(rts)
    return rts.GetMol()


def _get_rings_with_atom(scaffold, aix):
    """list: Get indexes of rings containing an atom."""
    return [rix for rix, r in enumerate(scaffold.rings) if aix in r.aix]


def _contract_rings(mol):
    """Private: Return a molecule with rings contracted.

    Create a new molecule, replacing each ring with one
    atom. Atoms are connected if the rings they represent
    are connected by a bond not in any ring. If rings
    share a common bond, the bond added is double. If rings
    share an common atom (i.e. spiro rings) the rings are
    connected with a single bond.

    Used for generating ring connectivity scaffolds.

    Parameters
    ----------
    mol : rdkit.Chem.rdchem.Mol
        Scaffold template for ring contraction.

    Returns
    -------
    rcs : rdkit.Chem.rdchem.Mol
        New molecule with contracted rings.

    Notes
    -----
    Dummy atoms are used instead of carbon atoms, so that
    valence constraints are not violated. In ring connectivity
    scaffolds the valence of a vertex is occaisionaly > 4.

    See Also
    --------
    get_ring_connectivity_scaffold

    """
    # Use a Scaffold object for ring + ring system information.
    scf, rcs = Scaffold(mol), RWMol()

    # Add a dummy atom for each ring to the empty RWMol.
    for _ in range(scf.rings.count):
        rcs.AddAtom(Atom(0))

    # Create bonds between atoms (ring vertices).
    weak_connections, visited = set(), set()
    for system in scf.ring_systems:
        bc = set(system.get_bond_connectivity())
        ac = set(system.get_atom_connectivity())
        visited.update(system.aix)

        # link strongly connected rings (DOUBLE)
        for connection in ac.intersection(bc):
            rcs.AddBond(*connection, BondType.DOUBLE)

        # link weakly connected rings (SINGLE)
        for connection in ac.difference(bc):
            rcs.AddBond(*connection, BondType.SINGLE)

        # link rings connected by a linker (SINGLE)
        for rix, ring in zip(system.rix, system):
            to_visit = ring.get_attachment_points()
            while to_visit:
                aix = to_visit.pop()
                for nbr in scf.atoms[aix].GetNeighbors():
                    idx = nbr.GetIdx()
                    if idx in visited:
                        continue
                    elif nbr.IsInRing():
                        visited.add(idx)
                        for nix in _get_rings_with_atom(scf, idx):
                            c = tuple(sorted((nix, rix)))
                            weak_connections.add(c)
                        continue
                    else:
                        to_visit.add(idx)
                        visited.add(idx)

    # Add remaining weak ring connections (SINGLE)
    for connection in weak_connections:
        rcs.AddBond(*connection, BondType.SINGLE)

    return rcs.GetMol()


def get_ring_connectivity_scaffold(mol, all_single_bonds=False):
    """Generate a ring connectivity scaffold for a molecule.

    In a ring connectivity scaffold vertices correspond to rings
    rather than atoms. The bonds between these vertices represent
    connections between rings. There are two types of connectivity:

    - Strong connectivity: When the connected rings share a common
    bond in the original molecule.

    - Weak connectivity: When the two rings share an atom (i.e. spiro)
    or when they are connected by a bond or set of bonds which are not
    part of any ring (i.e. a linker).

    Parameters
    ----------
    mol : rdkit.Chem.rdchem.Mol
        Input for generating a ring connectivity scaffold.
    all_single_bonds : bool, optional
        If True make all bonds within the ring connectivity
        scaffold single.

    Returns
    -------
    rdkit.Chem.rdchem.Mol
        An rdkit molecule containing a ring connectivity scaffold.

    Notes
    -----
    Dummy atoms are used instead of carbon atoms, so that
    valence constraints are not violated. In ring connectivity
    scaffolds the valence of a vertex is occaisionaly > 4.

    """
    murcko = get_murcko_scaffold(mol, generic=True, remove_exocyclic=True)
    rcs = _contract_rings(murcko)
    print('all single bonds');
    if all_single_bonds:
        for bond in rcs.GetBonds():
            bond.SetBondType(BondType.SINGLE)
    return rcs
