""" The DAO for the flags file. """

import json
import pathlib
from typing import List

from work_components.consts import FLAG_FILE


class Flags:
    """Read and modify flags."""

    def __init__(self) -> None:
        self._file: pathlib.Path = FLAG_FILE
        self._flags: List[str] = []

        if self._file.exists():
            self._load()

    def _load(self) -> None:
        """Load flags into memory."""

        if not self._file.exists():
            return

        with self._file.open(mode="r", encoding="utf-8") as flag_file:
            try:
                self._flags = json.load(flag_file)
            except json.JSONDecodeError:
                self._file.unlink()
                return

    def is_set(self, key: str) -> bool:
        """Return if given flag is set."""
        return key in self._flags

    def set(self, *flags: str) -> None:
        """Add given flag(s) to flag store."""
        self._flags.extend(flags)
        self._write()

    def remove(self, *flags: str) -> None:
        """Remove given flag(s) from flag store. Accepts nonexistent flags."""
        for flag in filter(lambda f: f in self._flags, flags):
            self._flags.remove(flag)
        self._write()

    def _write(self) -> None:
        """Write flags to disk."""
        # Ensure that the state directory exists before writing
        self._file.parent.mkdir(parents=True, exist_ok=True)

        with self._file.open(mode="w", encoding="utf-8") as flag_file:
            json.dump(self._flags, flag_file, indent="\t")
