import argparse
from renogymodbus import RenogyChargeController, RenogySmartBattery
from renogymodbus.find_slaveaddress import find_slaveaddress

def main():
    parser = argparse.ArgumentParser()
    parser.add_argument(
        "--portname", help="Port name for example /dev/ttyUSB0", default="/dev/ttyUSB0"
    )
    parser.add_argument(
        "--slaveaddress", help="Slave address 1-247", default=1, type=int
    )
    parser.add_argument(
        "charge-controller", help="Show charge controller real time data", action="store_true", default=False
    )
    parser.add_argument(
        "smart-battery", help="Show smart battery real time data", action="store_true", default=False
    )
    parser.add_argument(
        "find-slave-address", help="Find slave address of modbus device", action="store_true", default=False
    )
    args = parser.parse_args()

    if args.charge_controller:
        print_charge_controller_output(args)
    elif args.smart_battery:
        print_smart_battery_output(args)
    elif args.find_slave_address:
        print("Finding slave addresses...")
        addresses = find_slaveaddress(args.portname)

        if len(addresses) == 0:
            print("No modbus devices found")
        else:
            print("Found modbus devices at addresses:")
            for address in addresses:
                print(f"{address}")
    else:
        parser.print_help()


def print_charge_controller_output(args):
    controller = RenogyChargeController(args.portname, args.slaveaddress)

    print("Real Time Charge Controller Data")
    print(f"Solar voltage: {controller.get_solar_voltage()}V")
    print(f"Solar current: {controller.get_solar_current()}A")
    print(f"Solar power: {controller.get_solar_power()}W")
    print(f"Load voltage: {controller.get_load_voltage()}V")
    print(f"Load current: {controller.get_load_current()}A")
    print(f"Load power: {controller.get_load_power()}W")
    print(f"Battery voltage: {controller.get_battery_voltage()}V")
    print(f"Battery state of charge: {controller.get_battery_state_of_charge()}%")
    print(f"Battery temperature: {controller.get_battery_temperature()}°C")
    print(f"Controller temperature: {controller.get_controller_temperature()}°C")
    print(f"Maximum solar power today: { controller.get_maximum_solar_power_today()}W")
    print(f"Minimum solar power today: { controller.get_minimum_solar_power_today()}W")
    print(f"Maximum battery voltage today: { controller.get_maximum_battery_voltage_today()}V")
    print(f"Minimum battery voltage today: { controller.get_minimum_battery_voltage_today()}V")

def print_smart_battery_output(args):
    battery = RenogySmartBattery(args.portname, args.slaveaddress)

    print("Real Time Smart Battery Data")
    print(f"Voltage: {battery.get_voltage()}V")
    print(f"Current: {battery.get_current()}A")
    print(f"Capacity: {battery.get_capacity()}mAh")
    print(f"Remaining charge: {battery.get_remaining_charge()}mAh")
    print(f"State of charge: {battery.get_state_of_charge()}%")

if __name__ == "__main__":
    main()
