# SPDX-FileCopyrightText: 2018 Scott Shawcroft for Adafruit Industries
# SPDX-FileCopyrightText: 2022 Matt Land
# SPDX-FileCopyrightText: Brooke Storm
# SPDX-FileCopyrightText: Sam McGahan
#
# SPDX-License-Identifier: MIT

"""
`adafruit_imageload.pnm.ppm_ascii`
====================================================

Load pixel values (indices or colors) into a bitmap and for an ascii ppm,
return None for pallet.

* Author(s):  Matt Land, Brooke Storm, Sam McGahan

"""

__version__ = "1.16.4"
__repo__ = "https://github.com/adafruit/Adafruit_CircuitPython_ImageLoad.git"

try:
    from typing import (
        Tuple,
        Iterator,
        Optional,
        List,
        Set,
    )
    from io import BufferedReader
    from displayio import Palette, Bitmap
    from ..displayio_types import PaletteConstructor, BitmapConstructor
except ImportError:
    pass


def load(
    file: BufferedReader,
    width: int,
    height: int,
    bitmap: BitmapConstructor = None,
    palette: PaletteConstructor = None,
) -> Tuple[Optional[Bitmap], Optional[Palette]]:
    """
    :param stream file: infile with the position set at start of data
    :param int width:
    :param int height:
    :param int max_colors: color space of file
    :param bitmap: displayio.Bitmap class
    :param palette: displayio.Palette class
    :return tuple:
    """
    palette_colors = set()  # type: Set[bytes]
    data_start = file.tell()
    for triplet in read_three_colors(file):
        palette_colors.add(triplet)

    if palette:
        palette = palette(len(palette_colors))  # type: Palette
        for counter, color in enumerate(palette_colors):
            palette[counter] = color
    if bitmap:
        file.seek(data_start)
        bitmap = bitmap(width, height, len(palette_colors))  # type: Bitmap
        palette_colors = list(palette_colors)  # type: List[bytes]
        for y in range(height):
            for x in range(width):
                for color in read_three_colors(file):
                    bitmap[x, y] = palette_colors.index(color)
                    break  # exit the inner generator
    return bitmap, palette


def read_three_colors(file: BufferedReader) -> Iterator[bytes]:
    """
    Generator to read integer values from file, in groups of three.
    Each value can be len 1-3, for values 0 - 255, space padded.
    :return tuple[int]:
    """
    triplet = []  # type: List[int]
    color = bytearray()
    while True:
        this_byte = file.read(1)
        if this_byte.isdigit():
            color += this_byte
        # not a digit means we completed one number (found a space separator or EOF)
        elif color or (triplet and this_byte == b""):
            triplet.append(int("".join(["%c" % char for char in color])))
            color = bytearray()
        if len(triplet) == 3:  # completed one pixel
            yield bytes(tuple(triplet))
            triplet = []
        # short circuit must be after all other cases, so we yield the last pixel before returning
        if this_byte == b"":
            return
