import terrascript.core as core


@core.resource(type="aws_iam_service_specific_credential", namespace="iam")
class ServiceSpecificCredential(core.Resource):
    """
    The combination of `service_name` and `user_name` as such: `service_name:user_name:service_specific_
    credential_id`.
    """

    id: str | core.StringOut = core.attr(str, computed=True)

    """
    (Required) The name of the AWS service that is to be associated with the credentials. The service yo
    u specify here is the only service that can be accessed using these credentials.
    """
    service_name: str | core.StringOut = core.attr(str)

    """
    The generated password for the service-specific credential.
    """
    service_password: str | core.StringOut = core.attr(str, computed=True)

    """
    The unique identifier for the service-specific credential.
    """
    service_specific_credential_id: str | core.StringOut = core.attr(str, computed=True)

    """
    The generated user name for the service-specific credential. This value is generated by combining th
    e IAM user's name combined with the ID number of the AWS account, as in `jane-at-123456789012`, for
    example.
    """
    service_user_name: str | core.StringOut = core.attr(str, computed=True)

    """
    (Optional) The status to be assigned to the service-specific credential. Valid values are `Active` a
    nd `Inactive`. Default value is `Active`.
    """
    status: str | core.StringOut | None = core.attr(str, default=None)

    """
    (Required) The name of the IAM user that is to be associated with the credentials. The new service-s
    pecific credentials have the same permissions as the associated user except that they can be used on
    ly to access the specified service.
    """
    user_name: str | core.StringOut = core.attr(str)

    def __init__(
        self,
        resource_name: str,
        *,
        service_name: str | core.StringOut,
        user_name: str | core.StringOut,
        status: str | core.StringOut | None = None,
        depends_on: list[str] | core.ArrayOut[core.StringOut] | None = None,
        provider: str | core.StringOut | None = None,
        lifecycle: core.Lifecycle | None = None,
    ):
        super().__init__(
            name=resource_name,
            args=ServiceSpecificCredential.Args(
                service_name=service_name,
                user_name=user_name,
                status=status,
                depends_on=depends_on,
                provider=provider,
                lifecycle=lifecycle,
            ),
        )

    @core.schema_args
    class Args(core.Resource.Args):
        service_name: str | core.StringOut = core.arg()

        status: str | core.StringOut | None = core.arg(default=None)

        user_name: str | core.StringOut = core.arg()
