import terrascript.core as core


@core.resource(type="aws_cloudwatch_event_api_destination", namespace="eventbridge")
class CloudwatchEventApiDestination(core.Resource):
    """
    The Amazon Resource Name (ARN) of the event API Destination.
    """

    arn: str | core.StringOut = core.attr(str, computed=True)

    """
    (Required) ARN of the EventBridge Connection to use for the API Destination.
    """
    connection_arn: str | core.StringOut = core.attr(str)

    """
    (Optional) The description of the new API Destination. Maximum of 512 characters.
    """
    description: str | core.StringOut | None = core.attr(str, default=None)

    """
    (Required) Select the HTTP method used for the invocation endpoint, such as GET, POST, PUT, etc.
    """
    http_method: str | core.StringOut = core.attr(str)

    id: str | core.StringOut = core.attr(str, computed=True)

    """
    (Required) URL endpoint to invoke as a target. This could be a valid endpoint generated by a partner
    service. You can include "*" as path parameters wildcards to be set from the Target HttpParameters.
    """
    invocation_endpoint: str | core.StringOut = core.attr(str)

    """
    (Optional) Enter the maximum number of invocations per second to allow for this destination. Enter a
    value greater than 0 (default 300).
    """
    invocation_rate_limit_per_second: int | core.IntOut | None = core.attr(int, default=None)

    """
    (Required) The name of the new API Destination. The name must be unique for your account. Maximum of
    64 characters consisting of numbers, lower/upper case letters, .,-,_.
    """
    name: str | core.StringOut = core.attr(str)

    def __init__(
        self,
        resource_name: str,
        *,
        connection_arn: str | core.StringOut,
        http_method: str | core.StringOut,
        invocation_endpoint: str | core.StringOut,
        name: str | core.StringOut,
        description: str | core.StringOut | None = None,
        invocation_rate_limit_per_second: int | core.IntOut | None = None,
        depends_on: list[str] | core.ArrayOut[core.StringOut] | None = None,
        provider: str | core.StringOut | None = None,
        lifecycle: core.Lifecycle | None = None,
    ):
        super().__init__(
            name=resource_name,
            args=CloudwatchEventApiDestination.Args(
                connection_arn=connection_arn,
                http_method=http_method,
                invocation_endpoint=invocation_endpoint,
                name=name,
                description=description,
                invocation_rate_limit_per_second=invocation_rate_limit_per_second,
                depends_on=depends_on,
                provider=provider,
                lifecycle=lifecycle,
            ),
        )

    @core.schema_args
    class Args(core.Resource.Args):
        connection_arn: str | core.StringOut = core.arg()

        description: str | core.StringOut | None = core.arg(default=None)

        http_method: str | core.StringOut = core.arg()

        invocation_endpoint: str | core.StringOut = core.arg()

        invocation_rate_limit_per_second: int | core.IntOut | None = core.arg(default=None)

        name: str | core.StringOut = core.arg()
