# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['wdata']

package_data = \
{'': ['*']}

install_requires = \
['numpy>=1.19.4,<2.0.0',
 'pytz>=2020.4,<2021.0',
 'setuptools>=54.2.0,<55.0.0',
 'tzlocal>=2.1,<3.0',
 'zope.interface>=5.2.0,<6.0.0']

setup_kwargs = {
    'name': 'wdata',
    'version': '0.1.5',
    'description': 'W-Data format for superfluid dynamics and the W-SLDA Toolkit.',
    'long_description': 'W-data Format\n=============\n<!-- [Documentation Status](https://readthedocs.org/projects/wdata/badge/?version=latest)](https://wdata.readthedocs.io/en/latest/?badge=latest) -->\n[![Language grade: Python](https://img.shields.io/lgtm/grade/python/g/forbes-group/wdata.svg)](https://lgtm.com/projects/g/forbes-group/wdata/context:python)\n[![Tests](https://github.com/forbes-group/wdata/actions/workflows/tests.yml/badge.svg)](https://github.com/forbes-group/wdata/actions/workflows/tests.yml)\n[![codecov](https://codecov.io/gh/forbes-group/wdata/branch/branch/default/graph/badge.svg?token=U1GRP9B06D)](https://codecov.io/gh/forbes-group/wdata)\n[![Pypi](https://img.shields.io/pypi/v/wdata.svg)](https://pypi.python.org/pypi/wdata)\n[![pyversions](https://img.shields.io/pypi/pyversions/wdata.svg)](https://pypi.python.org/pypi/wdata)[![Code style: black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black)\n\nThis project contains tools for working with and manipulating the\nW-data format used for analyzing superfluid data generated by the [W-SLDA\nToolkit](https://wslda.fizyka.pw.edu.pl/).\n\nThis format was originally derived from the W-SLDA project led by Gabriel Wlazlowski as\ndocumented here:\n\n* [W-SLDA Toolkit](https://wslda.fizyka.pw.edu.pl)\n* [Original W-data format](https://gitlab.fizyka.pw.edu.pl/gabrielw/wslda/-/wikis/W-data%20format)\n\nHere we augment this format slightly to facilitate working with Python.\n\nGeneralizations\n---------------\n\nThe original format required a `.wtxt` file with lots of relevant\ninformation.  Here we generalize the format to allow this information\nto be specified in the data files, which we allow to be in the NPY\nformat.\n\nInstallation\n------------\n\n```bash\npython3 -m pip install wdata\n```\n\nBasic Usage\n-----------\n\nThe W-data format stores various arrays representing physical\nquantities such as the density (real), pairing field (complex),\ncurrents (3-component real vectors) etc. on a regular lattice of shape\n`Nxyz = (Nx, Ny, Nz)` at a bunch of `Nt` times.\n\nThe data is represented by two classes: \n\n* `Var`: These are the data variables such as density, currents,\n    etc. with additional metadata (ee the `wdata.io.IVar` interface for\n    details):\n    * `Var.name`: Name of variable as it will appear in VisIt for example.\n    * `Var.data`: The actual data as a NumPy array.\n    * `Var.description`: Description.\n    * `Var.filename`: The file where the data is stored on disk.\n    * `Var.unit`: Unit (mainly for use in VisIt... does not affect the data.)\n* `WData`: This represents a complete dataset.  Some relevant attributes are (see\n    `wdata.io.IWData` for details): \n    * `WData.infofile`: Location of the infofile (see below).  This is where the\n        metadata will be stored or loaded from.\n    * `WData.variables`: List of `Var` variables.\n    * `WData.xyz`: Abscissa `(x, y, z)` shaped so that they can be used with\n        broadcasting.  I.e. `r = np.sqrt(x**2+y**2+z**2)`. \n    * `WData.t`: Array of times.\n    * `WData.dim`: Dimension of dataset.  I.e. `dim==1` for 1D simulations, `dim==3` for\n        3D simulations. \n    * `WData.aliases`: Dictionary of aliases.  Convenience for providing alternative\n        data access in VisIt.\n    * `WData.constants`: Dictionary of constants such as `kF`, `eF`.\n\n**Minimal Example**:\n\nHere is a minimal set of data:\n\n```python\nimport numpy as np\nnp.random.seed(3)\nfrom wdata.io import WData, Var\n\nNt = 10 \nNxyz = (4, 8, 16)\ndxyz = (0.3, 0.2, 0.1)\ndt = 0.1\nNtxyz = (Nt,) + Nxyz\n\ndensity = np.random.random(Ntxyz)\n\ndata = WData(prefix=\'dataset\', data_dir=\'_example_wdata\',\n             Nxyz=Nxyz, dxyz=dxyz,\n             variables=[Var(density=density)],\n             Nt=Nt)\ndata.save(force=True)\n```\n\nThis will make a directory `_example_wdata` with infofile\n`_example_wdata/dataset.wtxt`:\n\n```bash\n$ tree _example_wdata\n_example_wdata\n|-- dataset.wtxt\n`-- dataset_density.wdat\n\n0 directories, 2 files\n$ cat _example_wdata/dataset.wtxt\n# Generated by wdata.io: [2020-12-18 06:41:29 UTC+0000 = 2020-12-17 22:41:29 PST-0800]\n\nNX               4    # Lattice size in X direction\nNY               8    #             ... Y ...\nNZ              16    #             ... Z ...\nDX             0.3    # Spacing in X direction\nDY             0.2    #        ... Y ...\nDZ             0.1    #        ... Z ...\nprefix     dataset    # datafile prefix: <prefix>_<var>.<format>\ndatadim          3    # Block size: 1:NX, 2:NX*NY, 3:NX*NY*NZ\ncycles          10    # Number Nt of frames/cycles per dataset\nt0               0    # Time value of first frame\ndt               1    # Time interval between frames\n\n# variables\n# tag       name    type    unit    format    # description\nvar      density    real    none      wdat    # density\n```\n\nThe data can be loaded by specifying the infofile:\n\n```python\nfrom wdata.io import WData\ndata = WData.load(\'_example_wdata/dataset.wtxt\')\n```\n\nThe data could be plotted using [PyVista](https://docs.pyvista.org)\nfor example (the random data will not look so good...):\n\n```python\nimport numpy as np\nimport pyvista as pv\nfrom wdata.io import WData\n\ndata = WData.load(\'_example_wdata/dataset.wtxt\')\nn = data.density[0]\n\ngrid = pv.StructuredGrid(*np.meshgrid(*data.xyz))\ngrid["vol"] = n.flatten(order="F")\ncontours = grid.contour(np.linspace(n.min(), n.max(), 5))\n\np = pv.Plotter()\np.add_mesh(contours, scalars=contours.points[:, 2])\np.show()\n```\n\nThe recommended way to save data is to create variables for the data,\ntimes, and abscissa, then store this:\n\n```bash\nimport numpy as np\nfrom wdata.io import WData, Var\n\nnp.random.seed(3)\n\nNt = 10\nNxyz = (32, 32, 32)\ndxyz = (10.0/32, 10.0/32, 10.0/32)\ndt = 0.1\n\n# Abscissa.  Not strictly needed, but if you have them, then use them\n# instead.\nt = np.arange(Nt)*dt\nxyz = np.meshgrid(*[(np.arange(_N)-_N/2)*_dx\n                    for _N, _dx in zip(Nxyz, dxyz)],\n                  sparse=True, indexing=\'ij\')\n\n# Now make the WData object and save the data.\nNtxyz = (Nt,) + Nxyz\nw = np.pi/t.max()\nws = [1.0 + 0.5*np.cos(w*t), \n      1.0 + 0.5*np.sin(w*t),\n      1.0 + 0*t]\ndensity = np.exp(-sum((_x[None,...].T*_w).T**2/2 for _x, _w in zip(xyz, ws)))\ndelta = np.random.random(Ntxyz) + np.random.random(Ntxyz)*1j - 0.5 - 0.5j\ncurrent = np.random.random((Nt, 3,) + Nxyz) - 0.5\n\nvariables = [\n    Var(density=density),\n    Var(delta=delta),\n    Var(current=current)\n]\n    \ndata = WData(prefix=\'dataset2\', \n             data_dir=\'_example_wdata/\',\n             xyz=xyz, t=t,\n             variables=variables)\ndata.save()\n```\n\nNow load and plot the data:\n\n```bash\nimport numpy as np\nimport pyvista as pv\n\nfrom wdata.io import WData\ndata = WData.load(infofile=\'_example_wdata/dataset2.wtxt\')\n\nn = data.density[0]\n\ngrid = pv.StructuredGrid(*np.meshgrid(*data.xyz))\ngrid["vol"] = n.flatten(order="F")\ncontours = grid.contour(np.linspace(n.min(), n.max(), 5))\n\np = pv.Plotter()\np.add_mesh(contours, scalars=contours.points[:, 2])\np.show()\n```\n\nNote: the actual data is loaded into python using [memory-mapped\narrays](https://numpy.org/doc/stable/reference/generated/numpy.memmap.html).  This\nallows you to refer to very large data-sets without loading the entire data into\nmemory.  This will delay loading until a copy of the array is made.  For example:\n\n```bash\nimport numpy as np\nfrom wdata.io import WData\ndata = WData.load(infofile=\'_example_wdata/dataset2.wtxt\')\n\n# At this point, the data has not been fully loaded.  You can\n# work with subsets efficiently.  For example, the following will\n# only load the first frame of data:\n\nn = data.density[0]\n\n# Beware: if you make a copy of the data, explicitly *or implicitly* then it will get\n# loaded.  The following will load the full array into memory so that np.cos can do its\n# computations.\n\nsum_cos_n = np.sum(np.cos(data.density))\n\n# If this is too big, you may want to process each slice independently.  The previous\n# example could be more efficiently computed using the following loop:\n\nsum_cos_n = sum(np.cos(_n).sum() for _n in data.density)\n\n# The Dask package may be useful for such processing in more complicated settings.\n```\n\nSee Also\n--------\n* [NumPy memory-mapped files](https://numpy.org/doc/stable/reference/generated/numpy.memmap.html)\n* [Dask](https://dask.org)\n\nDeveloper Notes\n===============\n\nTesting\n-------\n\nFor distribution we use [poetry](https://python-poetry.org) and for testing we use\n[nox](https://nox.thea.codes).  To test the code:\n\n```bash\nnox\n```\n\nDocumentation\n-------------\nFor documentation, we use [Sphinx](https://www.sphinx-doc.org).  To build this run:\n\n```bash\npoetry install  # Install all of the developer dependencies\npoetry run make -C docs html\n```\n\n* `__init__()`: The default behavior of [autodoc](https://www.sphinx-doc.org/en/master/usage/extensions/autodoc.html#confval-autoclass_content)\nis to merge the documentation of `__init__` methods with the class since the user never\ndirectly calls `__init__()`.  Keep this in mind when writing the docstrings.\n\nChanges\n=======\n## 0.1.5\n* Resolne issue #13: `WData` can now load read-only files.\n\n## 0.1.4\n* Resolve issue #8.  Vectors can have `Nv <= dim`.  Also, keep `Nxyz` info even if\n  `dim<3`: this is how plane-wave approximations are used sometimes.\n* Fixed many small bugs discovered by 100% coverage testing.\n* Pass-through `kwargs` from `io.WData.load()` etc. to constructor.\n* Added `check_data` flag to optionally disable testing of data.\n* Remove item-access.  Use attribute access instead: `data.x` or `getattr(data, \'x\')`.\n\n## 0.1.3\n* Address issue #4 for loading large datasets.  We now use memory mapped files.\n* Started adding Sphinx documentation.  Not complete (`sphinxcontrib.zopeext` needs\n  updating... something is wrong.)\n\n## 0.1.2\n* Fixed issue #2.  `datadim < 3` now works properly.\n* Started working on documentation (incomplete).\n',
    'author': 'Michael McNeil Forbes',
    'author_email': 'mforbes@alum.mit.edu',
    'maintainer': None,
    'maintainer_email': None,
    'url': None,
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6.1,<4.0.0',
}


setup(**setup_kwargs)
