# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pyroots']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'pyroots',
    'version': '0.5.0',
    'description': 'Pure python single variable function solvers',
    'long_description': '![image](https://travis-ci.org/pmav99/pyroots.svg?branch=master%0A%20:target:%20https://travis-ci.org/pmav99/pyroots)\n\npyroots\n=======\n\nAbstract\n--------\n\nA Python library implementing various root finding methods for\nsingle-variable functions.\n\nCurrently the following methods have been implemented:\n\n-   The [bisect](http://en.wikipedia.org/wiki/Bisection_method) method.\n-   The [ridder](http://en.wikipedia.org/wiki/Ridders%27_method) method.\n-   The [brent](http://en.wikipedia.org/wiki/Brent%27s_method) method.\n\nWith regard to `Brent`\'s method, there are two implementations, the\nfirst one uses inverse quadratic extrapolation (`Brentq`) while the\nother ones uses hyperbolic extrapolation (`Brenth`).\n\nIf you don\'t know which method to use, you should probably use `Brentq`.\nThat being said, `Bisect` method is safe and slow (i.e. lots of iterations).\n\nExample\n-------\n\n```python\n# define the function whose root you are searching\ndef f(x, a):\n    return x ** 2 - a + 1\n\n# Create the Solver object (instead of Brentq you could also import Brenth/Ridder/Bisect)\nfrom pyroots import Brentq\nbrent = Brentq(epsilon=1e-5)\n\n# solve the function in `[-3, 0]` while `a` is equal to 2\nresult = brent(f, -3, 0, a=2)\nprint(result)\n```\n\nwill output:\n\n```\n converged : True\n   message : Solution converged.\niterations :   6\nfunc calls :   9\n        x0 :    -1.0000000748530762\n      xtol :     0.0000000000000002\n     f(x0) :     0.0000001497061579\n   epsilon :     0.0000100000000000\n   x_steps : [-3, 0, -0.3333333333333333, -1.6666666666666665, -0.7777777777777779, -1.0686868686868687, -0.9917335278385606, -0.9997244260982788, -1.0000000748530762]\n  fx_steps : [8, -1, -0.8888888888888888, 1.7777777777777772, -0.3950617283950615, 0.14209162330374459, -0.01646460976088293, -0.0005510718624670563, 1.4970615791476405e-07]\n```\n\nRationale\n---------\n\nThe functionality of `pyroots` is already implemented in `scipy`, so the\nnatural question is why rediscover the wheel?\n\nWell, the main reason is that `scipy` is a huge dependency. `Pyroots` on\nthe other hand is just a single package that is easily installed and\nthat you can easily bundle with `py2exe` or similar projects. It doesn\'t\neven need to get installed, just throw the `pyroots` folder in your\nproject and you are ready to go.\n\nApart from that, the API used by `scipy`\'s functions is not very\nuser-friendly. For example you can\'t use keyword arguments for your\nfunctions. Moreover, in `scipy` there is no reliable way to define how\nmany digits of accuracy you want in the obtained root. For example, you\nmay ask for 6 digits, but scipy may calculate up to 14 (or 12 or\nwhatever) digits. The main implication of this "glitch" is that scipy\'s\nmethod may evaluate the function more times than those really needed. If\nthe function calculates something trivial like the functions in the\nfollowing examples, then these extra function calls are no big deal, but\nif your functions take significant time to evaluate ,e.g. more than\nseconds, then this can quickly become annoying, or even, simply\nunacceptable, e.g. the function takes some minutes to return a value.\n\nInstallation\n------------\n\nwith pip:\n\n    pip install pyroots\n\nUsage\n-----\n\nAll the solvers share the same API, so you can easily switch between the\nvarious methods.\n\n### Function\n\nThe function whose root you are searching must take at least a single\nargument and return a single number. This first argument is also the\ndependent variable and, apart from that, the function can also take any\nnumber of positional/keyword arguments. For example the following\nfunctions are totally valid ones:\n\n```python\ndef f(x, a):\n    return x ** 2 - a + 1\n\ndef g(x, a, b, c=3):\n    return x ** 2 + a ** b - c\n```\n\nSolver Objects\n--------------\n\nThe first thing you have to do is to create a `Solver` object for the\nmethod you want to use:\n\n```python\nfrom pyroots import Brentq\n\nbrent = Brentq()\n```\n\nWhen you create the `Solver` object, you can specify several parameters\nthat will affect the convergence. The most important are:\n\n-   epsilon which specifies the number of digits that will be taken\n    under consideration when checking for convergence. It defaults to\n    `1e-6`.\n-   `raise_on_fail` which will raise an exception if convergence failed.\n    It defaults to True.\n\nUsing the above function definitions, in order to find the root of `f`\nyou must first define an interval that contains the root. Let\'s say that\nthis interval is defined as `[xa, xb]`. In this case you will call the\nsolver like this:\n\n``` python\ndef f(x, a):\n    return x ** 2 - a + 1\n\nsolver = Brentq()\nresult = solver(f, xa, xb, a=3)\n```\n\nResult Objects\n--------------\n\nAll the methods return a `Result` object that has the following\nattributes:\n\n```\nresult.x0               # the root\nresult.fx0              # the value of ``f(x0)`\nresult.convergence      # True/False\nresult.iterations       # the number of iterations\nresult.func_calls       # the number of function evaluations.\nresult.msg              # a descriptive message regarding the convergence (or the failure of convergence)\nresult.x_steps          # a list containing the x values  that have been tried while the solver run\nresult.fx_steps         # a list containing the f(x) values that have been calculated while the solver run\n```\n\nIf, for some reason, convergence cannot be achieved, then a\n`ConvergenceError` is raised. If you don\'t want that to happen, then you\nhave to pass `False` as the value of `raise_on_fail` argument:\n\n```python\ndef f(x):\n    return x ** 2 - 1\n\nresult = brent(f, xa=-10, xb=-5, raise_on_fail=False)\nprint(result)\n```\n\nAPI\n---\n\nEach solver factory has the following signature:\n\n```python\nSolverFactory(epsilon=1e-6, xtol=EPS, max_iter=500, raise_on_fail=True, debug_precision=10)\n```\n\nwhere:\n\n-   `epsilon` is the required precision of the solution, i.e. a solution\n    is achieved when `|f(x0)|` is smaller than `epsilon`.\n-   `max_iter` is the maximum allowed number of iterations.\n-   `raise_on_fail` is a boolean flag indicating whether or not an\n    exception should be raised if convergence fails. It defaults to True\n\nEach solver object has the following signature:\n\n```python\nsolver_object(f, xa, xb, *args, **kwargs)\n```\n\nwhere:\n\n-   `f` is the function whose root we are searching.\n-   `xa` is the lower bracket of the interval of the solution we search.\n-   `xb` is the upper bracket of the interval of the solution we search.\n-   `*args` are passed as positional arguments when `f` is evaluated.\n-   `**kwargs` are passed as keyword arguments when `f` is evaluated.\n\nDocumentation\n-------------\n\nFor the time being documentation is not yet ready, but the examples in\nthe README should be enough to get your feet wet.\n\nThe source code repository of pyroots can be found at:\n<https://github.com/pmav99/pyroots>\n\nFeedback and contributions are greatly appreciated.\n\npmav99 \\<gmail\\>\n\n<https://www.embeddedrelated.com/showarticle/855.php>\n',
    'author': 'Panos Mavrogiorgos',
    'author_email': 'pmav99@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/pmav99/pyroots',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=2.7, !=3.0.*, !=3.1.*, !=3.2.*, !=3.3.*',
}


setup(**setup_kwargs)
