"use strict";
const assert_1 = require("@aws-cdk/assert");
const logs = require("@aws-cdk/aws-logs");
const cdk = require("@aws-cdk/core");
const stepfunctions = require("../lib");
module.exports = {
    'Instantiate Default State Machine'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new stepfunctions.StateMachine(stack, 'MyStateMachine', {
            stateMachineName: 'MyStateMachine',
            definition: stepfunctions.Chain.start(new stepfunctions.Pass(stack, 'Pass'))
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::StepFunctions::StateMachine', {
            StateMachineName: 'MyStateMachine',
            DefinitionString: '{"StartAt":"Pass","States":{"Pass":{"Type":"Pass","End":true}}}'
        }));
        test.done();
    },
    'Instantiate Standard State Machine'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new stepfunctions.StateMachine(stack, 'MyStateMachine', {
            stateMachineName: 'MyStateMachine',
            definition: stepfunctions.Chain.start(new stepfunctions.Pass(stack, 'Pass')),
            stateMachineType: stepfunctions.StateMachineType.STANDARD
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::StepFunctions::StateMachine', {
            StateMachineName: 'MyStateMachine',
            StateMachineType: 'STANDARD',
            DefinitionString: '{"StartAt":"Pass","States":{"Pass":{"Type":"Pass","End":true}}}'
        }));
        test.done();
    },
    'Instantiate Express State Machine'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new stepfunctions.StateMachine(stack, 'MyStateMachine', {
            stateMachineName: 'MyStateMachine',
            definition: stepfunctions.Chain.start(new stepfunctions.Pass(stack, 'Pass')),
            stateMachineType: stepfunctions.StateMachineType.EXPRESS
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::StepFunctions::StateMachine', {
            StateMachineName: 'MyStateMachine',
            StateMachineType: 'EXPRESS',
            DefinitionString: '{"StartAt":"Pass","States":{"Pass":{"Type":"Pass","End":true}}}'
        }));
        test.done();
    },
    'log configuration'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const logGroup = new logs.LogGroup(stack, 'MyLogGroup');
        new stepfunctions.StateMachine(stack, 'MyStateMachine', {
            definition: stepfunctions.Chain.start(new stepfunctions.Pass(stack, 'Pass')),
            logs: {
                destination: logGroup,
                level: stepfunctions.LogLevel.FATAL,
                includeExecutionData: false
            }
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::StepFunctions::StateMachine', {
            DefinitionString: '{"StartAt":"Pass","States":{"Pass":{"Type":"Pass","End":true}}}',
            LoggingConfiguration: {
                Destinations: [{
                        CloudWatchLogsLogGroup: {
                            LogGroupArn: {
                                'Fn::GetAtt': ['MyLogGroup5C0DAD85', 'Arn']
                            }
                        }
                    }],
                IncludeExecutionData: false,
                Level: 'FATAL'
            }
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [{
                        Action: [
                            'logs:CreateLogDelivery',
                            'logs:GetLogDelivery',
                            'logs:UpdateLogDelivery',
                            'logs:DeleteLogDelivery',
                            'logs:ListLogDeliveries',
                            'logs:PutResourcePolicy',
                            'logs:DescribeResourcePolicies',
                            'logs:DescribeLogGroups'
                        ],
                        Effect: 'Allow',
                        Resource: '*'
                    }],
                Version: '2012-10-17'
            },
            PolicyName: 'MyStateMachineRoleDefaultPolicyE468EB18',
            Roles: [
                {
                    Ref: 'MyStateMachineRoleD59FFEBC'
                }
            ]
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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