"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const stepfunctions = require("../lib");
module.exports = {
    'Tasks can add permissions to the execution role'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const task = new stepfunctions.Task(stack, 'Task', {
            task: {
                bind: () => ({
                    resourceArn: 'resource',
                    policyStatements: [new iam.PolicyStatement({
                            actions: ['resource:Everything'],
                            resources: ['resource']
                        })],
                })
            }
        });
        // WHEN
        new stepfunctions.StateMachine(stack, 'SM', {
            definition: task
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [
                    {
                        Action: 'resource:Everything',
                        Effect: 'Allow',
                        Resource: 'resource'
                    }
                ],
            }
        }));
        test.done();
    },
    'Tasks hidden inside a Parallel state are also included'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const task = new stepfunctions.Task(stack, 'Task', {
            task: {
                bind: () => ({
                    resourceArn: 'resource',
                    policyStatements: [
                        new iam.PolicyStatement({
                            actions: ['resource:Everything'],
                            resources: ['resource']
                        })
                    ]
                })
            }
        });
        const para = new stepfunctions.Parallel(stack, 'Para');
        para.branch(task);
        // WHEN
        new stepfunctions.StateMachine(stack, 'SM', {
            definition: para
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [
                    {
                        Action: 'resource:Everything',
                        Effect: 'Allow',
                        Resource: 'resource'
                    }
                ],
            }
        }));
        test.done();
    },
    'Task should render InputPath / Parameters / OutputPath correctly'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const task = new stepfunctions.Task(stack, 'Task', {
            inputPath: '$',
            outputPath: '$.state',
            task: {
                bind: () => ({
                    resourceArn: 'resource',
                    parameters: {
                        'input.$': '$',
                        'stringArgument': 'inital-task',
                        'numberArgument': 123,
                        'booleanArgument': true,
                        'arrayArgument': ['a', 'b', 'c']
                    }
                })
            }
        });
        // WHEN
        const taskState = task.toStateJson();
        // THEN
        test.deepEqual(taskState, { End: true,
            Retry: undefined,
            Catch: undefined,
            InputPath: '$',
            Parameters: { 'input.$': '$',
                'stringArgument': 'inital-task',
                'numberArgument': 123,
                'booleanArgument': true,
                'arrayArgument': ['a', 'b', 'c'] },
            OutputPath: '$.state',
            Type: 'Task',
            Comment: undefined,
            Resource: 'resource',
            ResultPath: undefined,
            TimeoutSeconds: undefined,
            HeartbeatSeconds: undefined
        });
        test.done();
    },
    'Task combines taskobject parameters with direct parameters'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const task = new stepfunctions.Task(stack, 'Task', {
            inputPath: '$',
            outputPath: '$.state',
            task: {
                bind: () => ({
                    resourceArn: 'resource',
                    parameters: {
                        a: 'aa',
                    }
                })
            },
            parameters: {
                b: 'bb'
            }
        });
        // WHEN
        const taskState = task.toStateJson();
        // THEN
        test.deepEqual(taskState, { End: true,
            Retry: undefined,
            Catch: undefined,
            InputPath: '$',
            Parameters: { a: 'aa',
                b: 'bb', },
            OutputPath: '$.state',
            Type: 'Task',
            Comment: undefined,
            Resource: 'resource',
            ResultPath: undefined,
            TimeoutSeconds: undefined,
            HeartbeatSeconds: undefined
        });
        test.done();
    },
    'Created state machine can grant start execution to a role'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const task = new stepfunctions.Task(stack, 'Task', {
            task: {
                bind: () => ({ resourceArn: 'resource' })
            }
        });
        const stateMachine = new stepfunctions.StateMachine(stack, 'StateMachine', {
            definition: task
        });
        const role = new iam.Role(stack, 'Role', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com')
        });
        // WHEN
        stateMachine.grantStartExecution(role);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'states:StartExecution',
                        Effect: 'Allow',
                        Resource: {
                            Ref: 'StateMachine2E01A3A5'
                        }
                    }
                ],
                Version: '2012-10-17',
            },
            PolicyName: 'RoleDefaultPolicy5FFB7DAB',
            Roles: [
                {
                    Ref: 'Role1ABCC5F0'
                }
            ]
        }));
        test.done();
    },
    'Imported state machine can grant start execution to a role'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const stateMachineArn = 'arn:aws:states:::my-state-machine';
        const stateMachine = stepfunctions.StateMachine.fromStateMachineArn(stack, 'StateMachine', stateMachineArn);
        const role = new iam.Role(stack, 'Role', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com')
        });
        // WHEN
        stateMachine.grantStartExecution(role);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'states:StartExecution',
                        Effect: 'Allow',
                        Resource: stateMachineArn
                    }
                ],
                Version: '2012-10-17',
            },
            PolicyName: 'RoleDefaultPolicy5FFB7DAB',
            Roles: [
                {
                    Ref: 'Role1ABCC5F0'
                }
            ]
        }));
        test.done();
    },
    'Pass should render InputPath / Parameters / OutputPath correctly'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const task = new stepfunctions.Pass(stack, 'Pass', {
            inputPath: '$',
            outputPath: '$.state',
            parameters: {
                'input.$': '$',
                'stringArgument': 'inital-task',
                'numberArgument': 123,
                'booleanArgument': true,
                'arrayArgument': ['a', 'b', 'c']
            }
        });
        // WHEN
        const taskState = task.toStateJson();
        // THEN
        test.deepEqual(taskState, { End: true,
            InputPath: '$',
            OutputPath: '$.state',
            Parameters: { 'input.$': '$',
                'stringArgument': 'inital-task',
                'numberArgument': 123,
                'booleanArgument': true,
                'arrayArgument': ['a', 'b', 'c'] },
            Type: 'Pass',
            Comment: undefined,
            Result: undefined,
            ResultPath: undefined,
        });
        test.done();
    },
    'State machines must depend on their roles'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const task = new stepfunctions.Task(stack, 'Task', {
            task: {
                bind: () => ({
                    resourceArn: 'resource',
                    policyStatements: [
                        new iam.PolicyStatement({
                            resources: ['resource'],
                            actions: ['lambda:InvokeFunction'],
                        })
                    ],
                })
            }
        });
        new stepfunctions.StateMachine(stack, 'StateMachine', {
            definition: task
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::StepFunctions::StateMachine', {
            DependsOn: [
                'StateMachineRoleDefaultPolicyDF1E6607',
                'StateMachineRoleB840431D'
            ]
        }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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