"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const chain_1 = require("../chain");
const state_graph_1 = require("../state-graph");
const state_type_1 = require("./private/state-type");
const state_1 = require("./state");
/**
 * Define a Parallel state in the state machine
 *
 * A Parallel state can be used to run one or more state machines at the same
 * time.
 *
 * The Result of a Parallel state is an array of the results of its substatemachines.
 */
class Parallel extends state_1.State {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        this.endStates = [this];
    }
    /**
     * Add retry configuration for this state
     *
     * This controls if and how the execution will be retried if a particular
     * error occurs.
     */
    addRetry(props = {}) {
        super._addRetry(props);
        return this;
    }
    /**
     * Add a recovery handler for this state
     *
     * When a particular error occurs, execution will continue at the error
     * handler instead of failing the state machine execution.
     */
    addCatch(handler, props = {}) {
        super._addCatch(handler.startState, props);
        return this;
    }
    /**
     * Continue normal execution with the given state
     */
    next(next) {
        super.makeNext(next.startState);
        return chain_1.Chain.sequence(this, next);
    }
    /**
     * Define one or more branches to run in parallel
     */
    branch(...branches) {
        for (const branch of branches) {
            const name = `Parallel '${this.stateId}' branch ${this.branches.length + 1}`;
            super.addBranch(new state_graph_1.StateGraph(branch.startState, name));
        }
        return this;
    }
    /**
     * Return the Amazon States Language object for this state
     */
    toStateJson() {
        return {
            Type: state_type_1.StateType.PARALLEL,
            Comment: this.comment,
            ResultPath: state_1.renderJsonPath(this.resultPath),
            ...this.renderNextEnd(),
            ...this.renderInputOutput(),
            ...this.renderRetryCatch(),
            ...this.renderBranches(),
        };
    }
    /**
     * Validate this state
     */
    validate() {
        if (this.branches.length === 0) {
            return ['Parallel must have at least one branch'];
        }
        return [];
    }
}
exports.Parallel = Parallel;
//# sourceMappingURL=data:application/json;base64,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