import logging

from .error import ThreeCommasError
from .model import Bot
from typing import List
from .sys_utils import get_parent_function_name

logger = logging.getLogger(__name__)


class BotMath:
    # Legend: mabu is max amount bot usage
    @staticmethod
    def get_bot_math_arguments(bot: Bot) -> dict:
        return {
            'bo': bot.get_base_order_volume(),
            'so':  bot.get_safety_order_volume(),
            'max_so':  bot.get_max_safety_orders(),
            'martingale':  bot.get_martingale_volume_coefficient(),
            'mad':  bot.get_max_active_deals(),
        }

    @staticmethod
    def get_max_bot_usage(bot: Bot) -> float:
        return BotMath.calculate_max_bot_usage(**BotMath.get_bot_math_arguments(bot))

    @staticmethod
    def calculate_max_bot_usage(bo: float, so: float, max_so: int, martingale: float, mad: int) -> float:
        return (bo + so * BotMath.calculate_so_multiplicator(max_so, martingale)) * mad

    @staticmethod
    def calculate_so_multiplicator(max_so: float, martingale: float) -> float:
        return max_so if martingale == 1 else (martingale ** max_so - 1) / (martingale-1)

    @staticmethod
    def calculate_bo(mabu: float, mad: int, max_so: int, martingale: float, so: float) -> float:
        return mabu / mad - so * BotMath.calculate_so_multiplicator(max_so, martingale)

    @staticmethod
    def calculate_bo_with_so_bo_ratio(mabu: float, mad: int, max_so: int, martingale: float, so_bo_ratio: float) -> float:
        return mabu / mad / (1 + so_bo_ratio * BotMath.calculate_so_multiplicator(max_so, martingale))

    @staticmethod
    def calculate_mad(mabu: float, max_so: int, martingale: float, bo: float, so: float):
        return mabu / (bo + so * BotMath.calculate_so_multiplicator(max_so, martingale))


def get_base_from_3c_pair(tc_pair: str, account_market_code: str) -> str:
    if account_market_code in {'ftx', 'binance', 'paper_trading'}:
        return tc_pair.split('_')[1].upper()
    elif account_market_code in {'ftx_futures'}:
        return tc_pair.split('_')[1].split('-')[0]
    else:
        raise RuntimeError(f'Not known market code {account_market_code} in get_base_from_3c_pair')


def filter_market_pairs_with_quote(market_pairs: List[str], quote: str):
    return [pair for pair in market_pairs if pair.upper().startswith(quote.upper())]


def get_bot_quote(bot: Bot):
    pairs = bot.get_pairs()
    if not pairs:
        return None
    return pairs[0].split('_')[0].upper()


def get_quote_from_3c_pair(tc_pair: str) -> str:
    return tc_pair.split('_')[0].upper()


def pair_is_quote(tc_pair: str, quote: str) -> bool:
    return get_quote_from_3c_pair(tc_pair).upper() == quote.upper()


def map_spot_tc_pairs_to_bases(tc_pairs: list, account_market_code: str) -> list:
    return list(map(lambda pair: get_base_from_3c_pair(tc_pair=pair, account_market_code=account_market_code), tc_pairs))


def filter_tc_pairs_by_quote(pairs: list, quote: str) -> list:
    return list(filter(lambda pair: pair_is_quote(tc_pair=pair, quote=quote), pairs))


def construct_pair_from_quote_and_base(quote: str, base: str) -> str:
    return f"{quote.upper()}_{base.upper()}"


def construct_futures_pair_from_base(base: str, account_market_code: str) -> str:
    if account_market_code in {'ftx_futures'}:
        return f'USD_{base.upper()}-PERP'
    else:
        raise RuntimeError(f'Not known market code {account_market_code} in construct_futures_pair_from_quote_and_base')


def filter_list_bot_having_pair(list_bot_show: List[Bot], pair: str) -> List[Bot]:
    return [bot_model for bot_model in list_bot_show if bot_model.has_pair(pair)]




