"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3CodeCommitBackup = void 0;
const core_1 = require("@aws-cdk/core");
const aws_codebuild_1 = require("@aws-cdk/aws-codebuild");
const aws_events_1 = require("@aws-cdk/aws-events");
const aws_events_targets_1 = require("@aws-cdk/aws-events-targets");
const aws_iam_1 = require("@aws-cdk/aws-iam");
class S3CodeCommitBackup extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const { backupBucket, schedule, computeType, repository } = props;
        const { repositoryName, repositoryCloneUrlHttp, repositoryArn, } = repository;
        const buildImage = aws_codebuild_1.LinuxBuildImage.STANDARD_2_0;
        this.backupProject = new aws_codebuild_1.Project(this, 'BackupProject', {
            environment: {
                buildImage,
                computeType: computeType || buildImage.defaultComputeType,
            },
            buildSpec: aws_codebuild_1.BuildSpec.fromObject({
                version: '0.2',
                env: {
                    'git-credential-helper': 'yes',
                },
                phases: {
                    pre_build: {
                        commands: [
                            `echo "[===== Clone repository: ${repositoryName} =====]"`,
                            `git clone "${repositoryCloneUrlHttp}"`,
                        ],
                    },
                    build: {
                        commands: [
                            "dt=$(date -u '+%Y_%m_%d_%H_%M')",
                            `zipfile="${repositoryName}_backup_\${dt}_UTC.tar.gz"`,
                            `echo "Compressing repository: ${repositoryName} into file: \${zipfile} and uploading to S3 bucket: ${backupBucket.bucketName}/${repositoryName}"`,
                            `tar -zcvf "\${zipfile}" "${repositoryName}/"`,
                            `aws s3 cp "\${zipfile}" "s3://${backupBucket.bucketName}/${repositoryName}/\${zipfile}"`,
                        ],
                    },
                },
            }),
        });
        backupBucket.grantPut(this.backupProject);
        this.backupProject.addToRolePolicy(new aws_iam_1.PolicyStatement({
            resources: [repositoryArn],
            actions: [
                'codecommit:BatchGet*',
                'codecommit:Get*',
                'codecommit:Describe*',
                'codecommit:List*',
                'codecommit:GitPull',
            ],
        }));
        new aws_events_1.Rule(this, 'ScheduleRule', {
            schedule,
            targets: [new aws_events_targets_1.CodeBuildProject(this.backupProject)],
        });
    }
    /**
     * Defines an event rule which triggers when a backup fails.
     */
    onBackupFailed(id, options) {
        return this.backupProject.onBuildFailed(id, options);
    }
    /**
     * Defines an event rule which triggers when a backup starts.
     */
    onBackupStarted(id, options) {
        return this.backupProject.onBuildStarted(id, options);
    }
    /**
     * Defines an event rule which triggers when a backup complets successfully.
     */
    onBackupSucceeded(id, options) {
        return this.backupProject.onBuildSucceeded(id, options);
    }
}
exports.S3CodeCommitBackup = S3CodeCommitBackup;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiczMtY29kZWNvbW1pdC1iYWNrdXAuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi9zcmMvczMtY29kZWNvbW1pdC1iYWNrdXAudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsd0NBQTBDO0FBQzFDLDBEQUtnQztBQUVoQyxvREFBcUU7QUFDckUsb0VBQStEO0FBRS9ELDhDQUFtRDtBQTJCbkQsTUFBYSxrQkFBbUIsU0FBUSxnQkFBUztJQUcvQyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQThCO1FBQ3RFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFakIsTUFBTSxFQUFFLFlBQVksRUFBRSxRQUFRLEVBQUUsV0FBVyxFQUFFLFVBQVUsRUFBRSxHQUFHLEtBQUssQ0FBQztRQUVsRSxNQUFNLEVBQ0osY0FBYyxFQUNkLHNCQUFzQixFQUN0QixhQUFhLEdBQ2QsR0FBRyxVQUFVLENBQUM7UUFFZixNQUFNLFVBQVUsR0FBRywrQkFBZSxDQUFDLFlBQVksQ0FBQztRQUVoRCxJQUFJLENBQUMsYUFBYSxHQUFHLElBQUksdUJBQU8sQ0FBQyxJQUFJLEVBQUUsZUFBZSxFQUFFO1lBQ3RELFdBQVcsRUFBRTtnQkFDWCxVQUFVO2dCQUNWLFdBQVcsRUFBRSxXQUFXLElBQUksVUFBVSxDQUFDLGtCQUFrQjthQUMxRDtZQUNELFNBQVMsRUFBRSx5QkFBUyxDQUFDLFVBQVUsQ0FBQztnQkFDOUIsT0FBTyxFQUFFLEtBQUs7Z0JBQ2QsR0FBRyxFQUFFO29CQUNILHVCQUF1QixFQUFFLEtBQUs7aUJBQy9CO2dCQUNELE1BQU0sRUFBRTtvQkFDTixTQUFTLEVBQUU7d0JBQ1QsUUFBUSxFQUFFOzRCQUNSLGtDQUFrQyxjQUFjLFVBQVU7NEJBQzFELGNBQWMsc0JBQXNCLEdBQUc7eUJBQ3hDO3FCQUNGO29CQUNELEtBQUssRUFBRTt3QkFDTCxRQUFRLEVBQUU7NEJBQ1IsaUNBQWlDOzRCQUNqQyxZQUFZLGNBQWMsNEJBQTRCOzRCQUN0RCxpQ0FBaUMsY0FBYyx1REFBdUQsWUFBWSxDQUFDLFVBQVUsSUFBSSxjQUFjLEdBQUc7NEJBQ2xKLDRCQUE0QixjQUFjLElBQUk7NEJBQzlDLGlDQUFpQyxZQUFZLENBQUMsVUFBVSxJQUFJLGNBQWMsZUFBZTt5QkFDMUY7cUJBQ0Y7aUJBQ0Y7YUFDRixDQUFDO1NBQ0gsQ0FBQyxDQUFDO1FBRUgsWUFBWSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLENBQUM7UUFFMUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxlQUFlLENBQ2hDLElBQUkseUJBQWUsQ0FBQztZQUNsQixTQUFTLEVBQUUsQ0FBQyxhQUFhLENBQUM7WUFDMUIsT0FBTyxFQUFFO2dCQUNQLHNCQUFzQjtnQkFDdEIsaUJBQWlCO2dCQUNqQixzQkFBc0I7Z0JBQ3RCLGtCQUFrQjtnQkFDbEIsb0JBQW9CO2FBQ3JCO1NBQ0YsQ0FBQyxDQUNILENBQUM7UUFFRixJQUFJLGlCQUFJLENBQUMsSUFBSSxFQUFFLGNBQWMsRUFBRTtZQUM3QixRQUFRO1lBQ1IsT0FBTyxFQUFFLENBQUMsSUFBSSxxQ0FBZ0IsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLENBQUM7U0FDcEQsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOztPQUVHO0lBQ0ksY0FBYyxDQUFDLEVBQVUsRUFBRSxPQUF3QjtRQUN4RCxPQUFPLElBQUksQ0FBQyxhQUFhLENBQUMsYUFBYSxDQUFDLEVBQUUsRUFBRSxPQUFPLENBQUMsQ0FBQztJQUN2RCxDQUFDO0lBRUQ7O09BRUc7SUFDSSxlQUFlLENBQUMsRUFBVSxFQUFFLE9BQXdCO1FBQ3pELE9BQU8sSUFBSSxDQUFDLGFBQWEsQ0FBQyxjQUFjLENBQUMsRUFBRSxFQUFFLE9BQU8sQ0FBQyxDQUFDO0lBQ3hELENBQUM7SUFFRDs7T0FFRztJQUNJLGlCQUFpQixDQUFDLEVBQVUsRUFBRSxPQUF3QjtRQUMzRCxPQUFPLElBQUksQ0FBQyxhQUFhLENBQUMsZ0JBQWdCLENBQUMsRUFBRSxFQUFFLE9BQU8sQ0FBQyxDQUFDO0lBQzFELENBQUM7Q0FDRjtBQXZGRCxnREF1RkMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7XG4gIEJ1aWxkU3BlYyxcbiAgTGludXhCdWlsZEltYWdlLFxuICBQcm9qZWN0LFxuICBDb21wdXRlVHlwZSxcbn0gZnJvbSAnQGF3cy1jZGsvYXdzLWNvZGVidWlsZCc7XG5pbXBvcnQgeyBJUmVwb3NpdG9yeSB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1jb2RlY29tbWl0JztcbmltcG9ydCB7IFJ1bGUsIFNjaGVkdWxlLCBPbkV2ZW50T3B0aW9ucyB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1ldmVudHMnO1xuaW1wb3J0IHsgQ29kZUJ1aWxkUHJvamVjdCB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1ldmVudHMtdGFyZ2V0cyc7XG5pbXBvcnQgeyBCdWNrZXQgfSBmcm9tICdAYXdzLWNkay9hd3MtczMnO1xuaW1wb3J0IHsgUG9saWN5U3RhdGVtZW50IH0gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSc7XG5cbmV4cG9ydCBpbnRlcmZhY2UgUzNDb2RlQ29tbWl0QmFja3VwUHJvcHMge1xuICAvKipcbiAgICogQnVja2V0IGZvciBzdG9yaW5nIHRoZSBiYWNrdXBzLlxuICAgKi9cbiAgcmVhZG9ubHkgYmFja3VwQnVja2V0OiBCdWNrZXQ7XG5cbiAgLyoqXG4gICAqIFNjaGVkdWxlIGZvciBiYWNrdXBzLlxuICAgKi9cbiAgcmVhZG9ubHkgc2NoZWR1bGU6IFNjaGVkdWxlO1xuXG4gIC8qKlxuICAgKiBSZXBvc2l0b3J5IHRvIGJlIGJhY2tlZCB1cFxuICAgKi9cbiAgcmVhZG9ubHkgcmVwb3NpdG9yeTogSVJlcG9zaXRvcnk7XG5cbiAgLyoqXG4gICAqIFRoZSB0eXBlIG9mIGNvbXB1dGUgdG8gdXNlIGZvciBiYWNrdXAgdGhlIHJlcG9zaXRvcmllcy5cbiAgICogU2VlIHRoZSB7QGxpbmsgQ29tcHV0ZVR5cGV9IGVudW0gZm9yIHRoZSBwb3NzaWJsZSB2YWx1ZXMuXG4gICAqXG4gICAqIEBkZWZhdWx0IHRha2VuIGZyb20ge0BsaW5rICNidWlsZEltYWdlI2RlZmF1bHRDb21wdXRlVHlwZX1cbiAgICovXG4gIHJlYWRvbmx5IGNvbXB1dGVUeXBlPzogQ29tcHV0ZVR5cGU7XG59XG5cbmV4cG9ydCBjbGFzcyBTM0NvZGVDb21taXRCYWNrdXAgZXh0ZW5kcyBDb25zdHJ1Y3Qge1xuICBwcml2YXRlIHJlYWRvbmx5IGJhY2t1cFByb2plY3Q6IFByb2plY3Q7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IFMzQ29kZUNvbW1pdEJhY2t1cFByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIGNvbnN0IHsgYmFja3VwQnVja2V0LCBzY2hlZHVsZSwgY29tcHV0ZVR5cGUsIHJlcG9zaXRvcnkgfSA9IHByb3BzO1xuXG4gICAgY29uc3Qge1xuICAgICAgcmVwb3NpdG9yeU5hbWUsXG4gICAgICByZXBvc2l0b3J5Q2xvbmVVcmxIdHRwLFxuICAgICAgcmVwb3NpdG9yeUFybixcbiAgICB9ID0gcmVwb3NpdG9yeTtcblxuICAgIGNvbnN0IGJ1aWxkSW1hZ2UgPSBMaW51eEJ1aWxkSW1hZ2UuU1RBTkRBUkRfMl8wO1xuXG4gICAgdGhpcy5iYWNrdXBQcm9qZWN0ID0gbmV3IFByb2plY3QodGhpcywgJ0JhY2t1cFByb2plY3QnLCB7XG4gICAgICBlbnZpcm9ubWVudDoge1xuICAgICAgICBidWlsZEltYWdlLFxuICAgICAgICBjb21wdXRlVHlwZTogY29tcHV0ZVR5cGUgfHwgYnVpbGRJbWFnZS5kZWZhdWx0Q29tcHV0ZVR5cGUsXG4gICAgICB9LFxuICAgICAgYnVpbGRTcGVjOiBCdWlsZFNwZWMuZnJvbU9iamVjdCh7XG4gICAgICAgIHZlcnNpb246ICcwLjInLFxuICAgICAgICBlbnY6IHtcbiAgICAgICAgICAnZ2l0LWNyZWRlbnRpYWwtaGVscGVyJzogJ3llcycsXG4gICAgICAgIH0sXG4gICAgICAgIHBoYXNlczoge1xuICAgICAgICAgIHByZV9idWlsZDoge1xuICAgICAgICAgICAgY29tbWFuZHM6IFtcbiAgICAgICAgICAgICAgYGVjaG8gXCJbPT09PT0gQ2xvbmUgcmVwb3NpdG9yeTogJHtyZXBvc2l0b3J5TmFtZX0gPT09PT1dXCJgLFxuICAgICAgICAgICAgICBgZ2l0IGNsb25lIFwiJHtyZXBvc2l0b3J5Q2xvbmVVcmxIdHRwfVwiYCxcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgfSxcbiAgICAgICAgICBidWlsZDoge1xuICAgICAgICAgICAgY29tbWFuZHM6IFtcbiAgICAgICAgICAgICAgXCJkdD0kKGRhdGUgLXUgJyslWV8lbV8lZF8lSF8lTScpXCIsXG4gICAgICAgICAgICAgIGB6aXBmaWxlPVwiJHtyZXBvc2l0b3J5TmFtZX1fYmFja3VwX1xcJHtkdH1fVVRDLnRhci5nelwiYCxcbiAgICAgICAgICAgICAgYGVjaG8gXCJDb21wcmVzc2luZyByZXBvc2l0b3J5OiAke3JlcG9zaXRvcnlOYW1lfSBpbnRvIGZpbGU6IFxcJHt6aXBmaWxlfSBhbmQgdXBsb2FkaW5nIHRvIFMzIGJ1Y2tldDogJHtiYWNrdXBCdWNrZXQuYnVja2V0TmFtZX0vJHtyZXBvc2l0b3J5TmFtZX1cImAsXG4gICAgICAgICAgICAgIGB0YXIgLXpjdmYgXCJcXCR7emlwZmlsZX1cIiBcIiR7cmVwb3NpdG9yeU5hbWV9L1wiYCxcbiAgICAgICAgICAgICAgYGF3cyBzMyBjcCBcIlxcJHt6aXBmaWxlfVwiIFwiczM6Ly8ke2JhY2t1cEJ1Y2tldC5idWNrZXROYW1lfS8ke3JlcG9zaXRvcnlOYW1lfS9cXCR7emlwZmlsZX1cImAsXG4gICAgICAgICAgICBdLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICB9KSxcbiAgICB9KTtcblxuICAgIGJhY2t1cEJ1Y2tldC5ncmFudFB1dCh0aGlzLmJhY2t1cFByb2plY3QpO1xuXG4gICAgdGhpcy5iYWNrdXBQcm9qZWN0LmFkZFRvUm9sZVBvbGljeShcbiAgICAgIG5ldyBQb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICByZXNvdXJjZXM6IFtyZXBvc2l0b3J5QXJuXSxcbiAgICAgICAgYWN0aW9uczogW1xuICAgICAgICAgICdjb2RlY29tbWl0OkJhdGNoR2V0KicsXG4gICAgICAgICAgJ2NvZGVjb21taXQ6R2V0KicsXG4gICAgICAgICAgJ2NvZGVjb21taXQ6RGVzY3JpYmUqJyxcbiAgICAgICAgICAnY29kZWNvbW1pdDpMaXN0KicsXG4gICAgICAgICAgJ2NvZGVjb21taXQ6R2l0UHVsbCcsXG4gICAgICAgIF0sXG4gICAgICB9KSxcbiAgICApO1xuXG4gICAgbmV3IFJ1bGUodGhpcywgJ1NjaGVkdWxlUnVsZScsIHtcbiAgICAgIHNjaGVkdWxlLFxuICAgICAgdGFyZ2V0czogW25ldyBDb2RlQnVpbGRQcm9qZWN0KHRoaXMuYmFja3VwUHJvamVjdCldLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIERlZmluZXMgYW4gZXZlbnQgcnVsZSB3aGljaCB0cmlnZ2VycyB3aGVuIGEgYmFja3VwIGZhaWxzLlxuICAgKi9cbiAgcHVibGljIG9uQmFja3VwRmFpbGVkKGlkOiBzdHJpbmcsIG9wdGlvbnM/OiBPbkV2ZW50T3B0aW9ucyk6IFJ1bGUge1xuICAgIHJldHVybiB0aGlzLmJhY2t1cFByb2plY3Qub25CdWlsZEZhaWxlZChpZCwgb3B0aW9ucyk7XG4gIH1cblxuICAvKipcbiAgICogRGVmaW5lcyBhbiBldmVudCBydWxlIHdoaWNoIHRyaWdnZXJzIHdoZW4gYSBiYWNrdXAgc3RhcnRzLlxuICAgKi9cbiAgcHVibGljIG9uQmFja3VwU3RhcnRlZChpZDogc3RyaW5nLCBvcHRpb25zPzogT25FdmVudE9wdGlvbnMpOiBSdWxlIHtcbiAgICByZXR1cm4gdGhpcy5iYWNrdXBQcm9qZWN0Lm9uQnVpbGRTdGFydGVkKGlkLCBvcHRpb25zKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBEZWZpbmVzIGFuIGV2ZW50IHJ1bGUgd2hpY2ggdHJpZ2dlcnMgd2hlbiBhIGJhY2t1cCBjb21wbGV0cyBzdWNjZXNzZnVsbHkuXG4gICAqL1xuICBwdWJsaWMgb25CYWNrdXBTdWNjZWVkZWQoaWQ6IHN0cmluZywgb3B0aW9ucz86IE9uRXZlbnRPcHRpb25zKTogUnVsZSB7XG4gICAgcmV0dXJuIHRoaXMuYmFja3VwUHJvamVjdC5vbkJ1aWxkU3VjY2VlZGVkKGlkLCBvcHRpb25zKTtcbiAgfVxufVxuIl19