"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MachineImageType = exports.DefaultCapacityType = exports.CoreDnsComputeType = exports.CpuArch = exports.NodeType = exports.EksOptimizedImage = exports.Cluster = exports.KubernetesVersion = exports.EndpointAccess = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const autoscaling = require("@aws-cdk/aws-autoscaling");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const ssm = require("@aws-cdk/aws-ssm");
const core_1 = require("@aws-cdk/core");
const constructs_1 = require("constructs");
const YAML = require("yaml");
const aws_auth_1 = require("./aws-auth");
const cluster_resource_1 = require("./cluster-resource");
const fargate_profile_1 = require("./fargate-profile");
const helm_chart_1 = require("./helm-chart");
const instance_types_1 = require("./instance-types");
const k8s_manifest_1 = require("./k8s-manifest");
const k8s_object_value_1 = require("./k8s-object-value");
const k8s_patch_1 = require("./k8s-patch");
const kubectl_provider_1 = require("./kubectl-provider");
const managed_nodegroup_1 = require("./managed-nodegroup");
const oidc_provider_1 = require("./oidc-provider");
const bottlerocket_1 = require("./private/bottlerocket");
const service_account_1 = require("./service-account");
const user_data_1 = require("./user-data");
// defaults are based on https://eksctl.io
const DEFAULT_CAPACITY_COUNT = 2;
const DEFAULT_CAPACITY_TYPE = ec2.InstanceType.of(ec2.InstanceClass.M5, ec2.InstanceSize.LARGE);
/**
 * Endpoint access characteristics.
 *
 * @stability stable
 */
class EndpointAccess {
    constructor(
    /**
     * Configuration properties.
     *
     * @internal
     */
    _config) {
        this._config = _config;
        if (!_config.publicAccess && _config.publicCidrs && _config.publicCidrs.length > 0) {
            throw new Error('CIDR blocks can only be configured when public access is enabled');
        }
    }
    /**
     * Restrict public access to specific CIDR blocks.
     *
     * If public access is disabled, this method will result in an error.
     *
     * @param cidr CIDR blocks.
     * @stability stable
     */
    onlyFrom(...cidr) {
        if (!this._config.privateAccess) {
            // when private access is disabled, we can't restric public
            // access since it will render the kubectl provider unusable.
            throw new Error('Cannot restric public access to endpoint when private access is disabled. Use PUBLIC_AND_PRIVATE.onlyFrom() instead.');
        }
        return new EndpointAccess({
            ...this._config,
            // override CIDR
            publicCidrs: cidr,
        });
    }
}
exports.EndpointAccess = EndpointAccess;
_a = JSII_RTTI_SYMBOL_1;
EndpointAccess[_a] = { fqn: "@aws-cdk/aws-eks.EndpointAccess", version: "1.128.0" };
/**
 * The cluster endpoint is accessible from outside of your VPC.
 *
 * Worker node traffic will leave your VPC to connect to the endpoint.
 *
 * By default, the endpoint is exposed to all adresses. You can optionally limit the CIDR blocks that can access the public endpoint using the `PUBLIC.onlyFrom` method.
 * If you limit access to specific CIDR blocks, you must ensure that the CIDR blocks that you
 * specify include the addresses that worker nodes and Fargate pods (if you use them)
 * access the public endpoint from.
 *
 * @stability stable
 */
EndpointAccess.PUBLIC = new EndpointAccess({ privateAccess: false, publicAccess: true });
/**
 * The cluster endpoint is only accessible through your VPC.
 *
 * Worker node traffic to the endpoint will stay within your VPC.
 *
 * @stability stable
 */
EndpointAccess.PRIVATE = new EndpointAccess({ privateAccess: true, publicAccess: false });
/**
 * The cluster endpoint is accessible from outside of your VPC.
 *
 * Worker node traffic to the endpoint will stay within your VPC.
 *
 * By default, the endpoint is exposed to all adresses. You can optionally limit the CIDR blocks that can access the public endpoint using the `PUBLIC_AND_PRIVATE.onlyFrom` method.
 * If you limit access to specific CIDR blocks, you must ensure that the CIDR blocks that you
 * specify include the addresses that worker nodes and Fargate pods (if you use them)
 * access the public endpoint from.
 *
 * @stability stable
 */
EndpointAccess.PUBLIC_AND_PRIVATE = new EndpointAccess({ privateAccess: true, publicAccess: true });
/**
 * Kubernetes cluster version.
 *
 * @stability stable
 */
class KubernetesVersion {
    /**
     *
     * @param version cluster version number
     */
    constructor(version) {
        this.version = version;
    }
    /**
     * Custom cluster version.
     *
     * @param version custom version number.
     * @stability stable
     */
    static of(version) { return new KubernetesVersion(version); }
}
exports.KubernetesVersion = KubernetesVersion;
_b = JSII_RTTI_SYMBOL_1;
KubernetesVersion[_b] = { fqn: "@aws-cdk/aws-eks.KubernetesVersion", version: "1.128.0" };
/**
 * Kubernetes version 1.14.
 *
 * @stability stable
 */
KubernetesVersion.V1_14 = KubernetesVersion.of('1.14');
/**
 * Kubernetes version 1.15.
 *
 * @stability stable
 */
KubernetesVersion.V1_15 = KubernetesVersion.of('1.15');
/**
 * Kubernetes version 1.16.
 *
 * @stability stable
 */
KubernetesVersion.V1_16 = KubernetesVersion.of('1.16');
/**
 * Kubernetes version 1.17.
 *
 * @stability stable
 */
KubernetesVersion.V1_17 = KubernetesVersion.of('1.17');
/**
 * Kubernetes version 1.18.
 *
 * @stability stable
 */
KubernetesVersion.V1_18 = KubernetesVersion.of('1.18');
/**
 * Kubernetes version 1.19.
 *
 * @stability stable
 */
KubernetesVersion.V1_19 = KubernetesVersion.of('1.19');
/**
 * Kubernetes version 1.20.
 *
 * @stability stable
 */
KubernetesVersion.V1_20 = KubernetesVersion.of('1.20');
/**
 * Kubernetes version 1.21.
 *
 * @stability stable
 */
KubernetesVersion.V1_21 = KubernetesVersion.of('1.21');
class ClusterBase extends core_1.Resource {
    /**
     * Defines a Kubernetes resource in this cluster.
     *
     * The manifest will be applied/deleted using kubectl as needed.
     *
     * @param id logical id of this manifest.
     * @param manifest a list of Kubernetes resource specifications.
     * @returns a `KubernetesResource` object.
     * @stability stable
     */
    addManifest(id, ...manifest) {
        return new k8s_manifest_1.KubernetesManifest(this, `manifest-${id}`, { cluster: this, manifest });
    }
    /**
     * Defines a Helm chart in this cluster.
     *
     * @param id logical id of this chart.
     * @param options options of this chart.
     * @returns a `HelmChart` construct
     * @stability stable
     */
    addHelmChart(id, options) {
        return new helm_chart_1.HelmChart(this, `chart-${id}`, { cluster: this, ...options });
    }
    /**
     * Defines a CDK8s chart in this cluster.
     *
     * @param id logical id of this chart.
     * @param chart the cdk8s chart.
     * @returns a `KubernetesManifest` construct representing the chart.
     * @stability stable
     */
    addCdk8sChart(id, chart) {
        const cdk8sChart = chart;
        // see https://github.com/awslabs/cdk8s/blob/master/packages/cdk8s/src/chart.ts#L84
        if (typeof cdk8sChart.toJson !== 'function') {
            throw new Error(`Invalid cdk8s chart. Must contain a 'toJson' method, but found ${typeof cdk8sChart.toJson}`);
        }
        return this.addManifest(id, ...cdk8sChart.toJson());
    }
    /**
     * Creates a new service account with corresponding IAM Role (IRSA).
     *
     * @stability stable
     */
    addServiceAccount(id, options = {}) {
        return new service_account_1.ServiceAccount(this, id, {
            ...options,
            cluster: this,
        });
    }
    /**
     * Installs the AWS spot instance interrupt handler on the cluster if it's not
     * already added.
     */
    addSpotInterruptHandler() {
        if (!this._spotInterruptHandler) {
            this._spotInterruptHandler = this.addHelmChart('spot-interrupt-handler', {
                chart: 'aws-node-termination-handler',
                version: '0.13.2',
                repository: 'https://aws.github.io/eks-charts',
                namespace: 'kube-system',
                values: {
                    nodeSelector: {
                        lifecycle: user_data_1.LifecycleLabel.SPOT,
                    },
                },
            });
        }
        return this._spotInterruptHandler;
    }
    /**
     * Connect capacity in the form of an existing AutoScalingGroup to the EKS cluster.
     *
     * The AutoScalingGroup must be running an EKS-optimized AMI containing the
     * /etc/eks/bootstrap.sh script. This method will configure Security Groups,
     * add the right policies to the instance role, apply the right tags, and add
     * the required user data to the instance's launch configuration.
     *
     * Spot instances will be labeled `lifecycle=Ec2Spot` and tainted with `PreferNoSchedule`.
     * If kubectl is enabled, the
     * [spot interrupt handler](https://github.com/awslabs/ec2-spot-labs/tree/master/ec2-spot-eks-solution/spot-termination-handler)
     * daemon will be installed on all spot instances to handle
     * [EC2 Spot Instance Termination Notices](https://aws.amazon.com/blogs/aws/new-ec2-spot-instance-termination-notices/).
     *
     * Prefer to use `addAutoScalingGroupCapacity` if possible.
     *
     * @param autoScalingGroup [disable-awslint:ref-via-interface].
     * @param options options for adding auto scaling groups, like customizing the bootstrap script.
     * @see https://docs.aws.amazon.com/eks/latest/userguide/launch-workers.html
     * @stability stable
     */
    connectAutoScalingGroupCapacity(autoScalingGroup, options) {
        var _e, _f, _g;
        // self rules
        autoScalingGroup.connections.allowInternally(ec2.Port.allTraffic());
        // Cluster to:nodes rules
        autoScalingGroup.connections.allowFrom(this, ec2.Port.tcp(443));
        autoScalingGroup.connections.allowFrom(this, ec2.Port.tcpRange(1025, 65535));
        // Allow HTTPS from Nodes to Cluster
        autoScalingGroup.connections.allowTo(this, ec2.Port.tcp(443));
        // Allow all node outbound traffic
        autoScalingGroup.connections.allowToAnyIpv4(ec2.Port.allTcp());
        autoScalingGroup.connections.allowToAnyIpv4(ec2.Port.allUdp());
        autoScalingGroup.connections.allowToAnyIpv4(ec2.Port.allIcmp());
        // allow traffic to/from managed node groups (eks attaches this security group to the managed nodes)
        autoScalingGroup.addSecurityGroup(this.clusterSecurityGroup);
        const bootstrapEnabled = (_e = options.bootstrapEnabled) !== null && _e !== void 0 ? _e : true;
        if (options.bootstrapOptions && !bootstrapEnabled) {
            throw new Error('Cannot specify "bootstrapOptions" if "bootstrapEnabled" is false');
        }
        if (bootstrapEnabled) {
            const userData = options.machineImageType === MachineImageType.BOTTLEROCKET ?
                user_data_1.renderBottlerocketUserData(this) :
                user_data_1.renderAmazonLinuxUserData(this, autoScalingGroup, options.bootstrapOptions);
            autoScalingGroup.addUserData(...userData);
        }
        autoScalingGroup.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSWorkerNodePolicy'));
        autoScalingGroup.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKS_CNI_Policy'));
        autoScalingGroup.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEC2ContainerRegistryReadOnly'));
        // EKS Required Tags
        // https://docs.aws.amazon.com/eks/latest/userguide/worker.html
        core_1.Tags.of(autoScalingGroup).add(`kubernetes.io/cluster/${this.clusterName}`, 'owned', {
            applyToLaunchedInstances: true,
            // exclude security groups to avoid multiple "owned" security groups.
            // (the cluster security group already has this tag)
            excludeResourceTypes: ['AWS::EC2::SecurityGroup'],
        });
        // do not attempt to map the role if `kubectl` is not enabled for this
        // cluster or if `mapRole` is set to false. By default this should happen.
        let mapRole = (_f = options.mapRole) !== null && _f !== void 0 ? _f : true;
        if (mapRole && !(this instanceof Cluster)) {
            // do the mapping...
            core_1.Annotations.of(autoScalingGroup).addWarning('Auto-mapping aws-auth role for imported cluster is not supported, please map role manually');
            mapRole = false;
        }
        if (mapRole) {
            // see https://docs.aws.amazon.com/en_us/eks/latest/userguide/add-user-role.html
            this.awsAuth.addRoleMapping(autoScalingGroup.role, {
                username: 'system:node:{{EC2PrivateDNSName}}',
                groups: [
                    'system:bootstrappers',
                    'system:nodes',
                ],
            });
        }
        else {
            // since we are not mapping the instance role to RBAC, synthesize an
            // output so it can be pasted into `aws-auth-cm.yaml`
            new core_1.CfnOutput(autoScalingGroup, 'InstanceRoleARN', {
                value: autoScalingGroup.role.roleArn,
            });
        }
        const addSpotInterruptHandler = (_g = options.spotInterruptHandler) !== null && _g !== void 0 ? _g : true;
        // if this is an ASG with spot instances, install the spot interrupt handler (only if kubectl is enabled).
        if (autoScalingGroup.spotPrice && addSpotInterruptHandler) {
            this.addSpotInterruptHandler();
        }
    }
}
/**
 * A Cluster represents a managed Kubernetes Service (EKS).
 *
 * This is a fully managed cluster of API Servers (control-plane)
 * The user is still required to create the worker nodes.
 *
 * @stability stable
 */
class Cluster extends ClusterBase {
    /**
     * Initiates an EKS Cluster with the supplied arguments.
     *
     * @param scope a Construct, most likely a cdk.Stack created.
     * @param id the id of the Construct to create.
     * @param props properties in the IClusterProps interface.
     * @stability stable
     */
    constructor(scope, id, props) {
        var _e, _f, _g, _h, _j, _k, _l, _m;
        super(scope, id, {
            physicalName: props.clusterName,
        });
        /**
         * If the cluster has one (or more) FargateProfiles associated, this array
         * will hold a reference to each.
         */
        this._fargateProfiles = [];
        const stack = core_1.Stack.of(this);
        this.prune = (_e = props.prune) !== null && _e !== void 0 ? _e : true;
        this.vpc = props.vpc || new ec2.Vpc(this, 'DefaultVpc');
        this.version = props.version;
        this.tagSubnets();
        // this is the role used by EKS when interacting with AWS resources
        this.role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('eks.amazonaws.com'),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSClusterPolicy'),
            ],
        });
        const securityGroup = props.securityGroup || new ec2.SecurityGroup(this, 'ControlPlaneSecurityGroup', {
            vpc: this.vpc,
            description: 'EKS Control Plane Security Group',
        });
        this.vpcSubnets = (_f = props.vpcSubnets) !== null && _f !== void 0 ? _f : [{ subnetType: ec2.SubnetType.PUBLIC }, { subnetType: ec2.SubnetType.PRIVATE }];
        const selectedSubnetIdsPerGroup = this.vpcSubnets.map(s => this.vpc.selectSubnets(s).subnetIds);
        if (selectedSubnetIdsPerGroup.some(core_1.Token.isUnresolved) && selectedSubnetIdsPerGroup.length > 1) {
            throw new Error('eks.Cluster: cannot select multiple subnet groups from a VPC imported from list tokens with unknown length. Select only one subnet group, pass a length to Fn.split, or switch to Vpc.fromLookup.');
        }
        // Get subnetIds for all selected subnets
        const subnetIds = Array.from(new Set(flatten(selectedSubnetIdsPerGroup)));
        this.endpointAccess = (_g = props.endpointAccess) !== null && _g !== void 0 ? _g : EndpointAccess.PUBLIC_AND_PRIVATE;
        this.kubectlEnvironment = props.kubectlEnvironment;
        this.kubectlLayer = props.kubectlLayer;
        this.onEventLayer = props.onEventLayer;
        this.kubectlMemory = props.kubectlMemory;
        const privateSubents = this.selectPrivateSubnets().slice(0, 16);
        const publicAccessDisabled = !this.endpointAccess._config.publicAccess;
        const publicAccessRestricted = !publicAccessDisabled
            && this.endpointAccess._config.publicCidrs
            && this.endpointAccess._config.publicCidrs.length !== 0;
        // validate endpoint access configuration
        if (privateSubents.length === 0 && publicAccessDisabled) {
            // no private subnets and no public access at all, no good.
            throw new Error('Vpc must contain private subnets when public endpoint access is disabled');
        }
        if (privateSubents.length === 0 && publicAccessRestricted) {
            // no private subents and public access is restricted, no good.
            throw new Error('Vpc must contain private subnets when public endpoint access is restricted');
        }
        const placeClusterHandlerInVpc = (_h = props.placeClusterHandlerInVpc) !== null && _h !== void 0 ? _h : false;
        if (placeClusterHandlerInVpc && privateSubents.length === 0) {
            throw new Error('Cannot place cluster handler in the VPC since no private subnets could be selected');
        }
        const resource = this._clusterResource = new cluster_resource_1.ClusterResource(this, 'Resource', {
            name: this.physicalName,
            environment: props.clusterHandlerEnvironment,
            roleArn: this.role.roleArn,
            version: props.version.version,
            resourcesVpcConfig: {
                securityGroupIds: [securityGroup.securityGroupId],
                subnetIds,
            },
            ...(props.secretsEncryptionKey ? {
                encryptionConfig: [{
                        provider: {
                            keyArn: props.secretsEncryptionKey.keyArn,
                        },
                        resources: ['secrets'],
                    }],
            } : {}),
            endpointPrivateAccess: this.endpointAccess._config.privateAccess,
            endpointPublicAccess: this.endpointAccess._config.publicAccess,
            publicAccessCidrs: this.endpointAccess._config.publicCidrs,
            secretsEncryptionKey: props.secretsEncryptionKey,
            vpc: this.vpc,
            subnets: placeClusterHandlerInVpc ? privateSubents : undefined,
            onEventLayer: this.onEventLayer,
        });
        if (this.endpointAccess._config.privateAccess && privateSubents.length !== 0) {
            // when private access is enabled and the vpc has private subnets, lets connect
            // the provider to the vpc so that it will work even when restricting public access.
            // validate VPC properties according to: https://docs.aws.amazon.com/eks/latest/userguide/cluster-endpoint.html
            if (this.vpc instanceof ec2.Vpc && !(this.vpc.dnsHostnamesEnabled && this.vpc.dnsSupportEnabled)) {
                throw new Error('Private endpoint access requires the VPC to have DNS support and DNS hostnames enabled. Use `enableDnsHostnames: true` and `enableDnsSupport: true` when creating the VPC.');
            }
            this.kubectlPrivateSubnets = privateSubents;
            // the vpc must exist in order to properly delete the cluster (since we run `kubectl delete`).
            // this ensures that.
            this._clusterResource.node.addDependency(this.vpc);
        }
        this.adminRole = resource.adminRole;
        // we use an SSM parameter as a barrier because it's free and fast.
        this._kubectlReadyBarrier = new core_1.CfnResource(this, 'KubectlReadyBarrier', {
            type: 'AWS::SSM::Parameter',
            properties: {
                Type: 'String',
                Value: 'aws:cdk:eks:kubectl-ready',
            },
        });
        // add the cluster resource itself as a dependency of the barrier
        this._kubectlReadyBarrier.node.addDependency(this._clusterResource);
        this.clusterName = this.getResourceNameAttribute(resource.ref);
        this.clusterArn = this.getResourceArnAttribute(resource.attrArn, cluster_resource_1.clusterArnComponents(this.physicalName));
        this.clusterEndpoint = resource.attrEndpoint;
        this.clusterCertificateAuthorityData = resource.attrCertificateAuthorityData;
        this.clusterSecurityGroupId = resource.attrClusterSecurityGroupId;
        this.clusterEncryptionConfigKeyArn = resource.attrEncryptionConfigKeyArn;
        this.clusterSecurityGroup = ec2.SecurityGroup.fromSecurityGroupId(this, 'ClusterSecurityGroup', this.clusterSecurityGroupId);
        this.connections = new ec2.Connections({
            securityGroups: [this.clusterSecurityGroup, securityGroup],
            defaultPort: ec2.Port.tcp(443),
        });
        // we can use the cluster security group since its already attached to the cluster
        // and configured to allow connections from itself.
        this.kubectlSecurityGroup = this.clusterSecurityGroup;
        // use the cluster creation role to issue kubectl commands against the cluster because when the
        // cluster is first created, that's the only role that has "system:masters" permissions
        this.kubectlRole = this.adminRole;
        this._kubectlResourceProvider = this.defineKubectlProvider();
        const updateConfigCommandPrefix = `aws eks update-kubeconfig --name ${this.clusterName}`;
        const getTokenCommandPrefix = `aws eks get-token --cluster-name ${this.clusterName}`;
        const commonCommandOptions = [`--region ${stack.region}`];
        if (props.outputClusterName) {
            new core_1.CfnOutput(this, 'ClusterName', { value: this.clusterName });
        }
        // if an explicit role is not configured, define a masters role that can
        // be assumed by anyone in the account (with sts:AssumeRole permissions of
        // course)
        const mastersRole = (_j = props.mastersRole) !== null && _j !== void 0 ? _j : new iam.Role(this, 'MastersRole', {
            assumedBy: new iam.AccountRootPrincipal(),
        });
        // map the IAM role to the `system:masters` group.
        this.awsAuth.addMastersRole(mastersRole);
        if (props.outputMastersRoleArn) {
            new core_1.CfnOutput(this, 'MastersRoleArn', { value: mastersRole.roleArn });
        }
        commonCommandOptions.push(`--role-arn ${mastersRole.roleArn}`);
        // allocate default capacity if non-zero (or default).
        const minCapacity = (_k = props.defaultCapacity) !== null && _k !== void 0 ? _k : DEFAULT_CAPACITY_COUNT;
        if (minCapacity > 0) {
            const instanceType = props.defaultCapacityInstance || DEFAULT_CAPACITY_TYPE;
            this.defaultCapacity = props.defaultCapacityType === DefaultCapacityType.EC2 ?
                this.addAutoScalingGroupCapacity('DefaultCapacity', { instanceType, minCapacity }) : undefined;
            this.defaultNodegroup = props.defaultCapacityType !== DefaultCapacityType.EC2 ?
                this.addNodegroupCapacity('DefaultCapacity', { instanceTypes: [instanceType], minSize: minCapacity }) : undefined;
        }
        const outputConfigCommand = (_l = props.outputConfigCommand) !== null && _l !== void 0 ? _l : true;
        if (outputConfigCommand) {
            const postfix = commonCommandOptions.join(' ');
            new core_1.CfnOutput(this, 'ConfigCommand', { value: `${updateConfigCommandPrefix} ${postfix}` });
            new core_1.CfnOutput(this, 'GetTokenCommand', { value: `${getTokenCommandPrefix} ${postfix}` });
        }
        this.defineCoreDnsComputeType((_m = props.coreDnsComputeType) !== null && _m !== void 0 ? _m : CoreDnsComputeType.EC2);
    }
    /**
     * Import an existing cluster.
     *
     * @param scope the construct scope, in most cases 'this'.
     * @param id the id or name to import as.
     * @param attrs the cluster properties to use for importing information.
     * @stability stable
     */
    static fromClusterAttributes(scope, id, attrs) {
        return new ImportedCluster(scope, id, attrs);
    }
    /**
     * Fetch the load balancer address of a service of type 'LoadBalancer'.
     *
     * @param serviceName The name of the service.
     * @param options Additional operation options.
     * @stability stable
     */
    getServiceLoadBalancerAddress(serviceName, options = {}) {
        const loadBalancerAddress = new k8s_object_value_1.KubernetesObjectValue(this, `${serviceName}LoadBalancerAddress`, {
            cluster: this,
            objectType: 'service',
            objectName: serviceName,
            objectNamespace: options.namespace,
            jsonPath: '.status.loadBalancer.ingress[0].hostname',
            timeout: options.timeout,
        });
        return loadBalancerAddress.value;
    }
    /**
     * Add nodes to this EKS cluster.
     *
     * The nodes will automatically be configured with the right VPC and AMI
     * for the instance type and Kubernetes version.
     *
     * Note that if you specify `updateType: RollingUpdate` or `updateType: ReplacingUpdate`, your nodes might be replaced at deploy
     * time without notice in case the recommended AMI for your machine image type has been updated by AWS.
     * The default behavior for `updateType` is `None`, which means only new instances will be launched using the new AMI.
     *
     * Spot instances will be labeled `lifecycle=Ec2Spot` and tainted with `PreferNoSchedule`.
     * In addition, the [spot interrupt handler](https://github.com/awslabs/ec2-spot-labs/tree/master/ec2-spot-eks-solution/spot-termination-handler)
     * daemon will be installed on all spot instances to handle
     * [EC2 Spot Instance Termination Notices](https://aws.amazon.com/blogs/aws/new-ec2-spot-instance-termination-notices/).
     *
     * @stability stable
     */
    addAutoScalingGroupCapacity(id, options) {
        if (options.machineImageType === MachineImageType.BOTTLEROCKET && options.bootstrapOptions !== undefined) {
            throw new Error('bootstrapOptions is not supported for Bottlerocket');
        }
        const asg = new autoscaling.AutoScalingGroup(this, id, {
            ...options,
            vpc: this.vpc,
            machineImage: options.machineImageType === MachineImageType.BOTTLEROCKET ?
                new bottlerocket_1.BottleRocketImage({
                    kubernetesVersion: this.version.version,
                }) :
                new EksOptimizedImage({
                    nodeType: nodeTypeForInstanceType(options.instanceType),
                    cpuArch: cpuArchForInstanceType(options.instanceType),
                    kubernetesVersion: this.version.version,
                }),
            updateType: options.updateType,
            instanceType: options.instanceType,
        });
        this.connectAutoScalingGroupCapacity(asg, {
            mapRole: options.mapRole,
            bootstrapOptions: options.bootstrapOptions,
            bootstrapEnabled: options.bootstrapEnabled,
            machineImageType: options.machineImageType,
            spotInterruptHandler: options.spotInterruptHandler,
        });
        if (nodeTypeForInstanceType(options.instanceType) === NodeType.INFERENTIA) {
            this.addNeuronDevicePlugin();
        }
        return asg;
    }
    /**
     * Add managed nodegroup to this Amazon EKS cluster.
     *
     * This method will create a new managed nodegroup and add into the capacity.
     *
     * @param id The ID of the nodegroup.
     * @param options options for creating a new nodegroup.
     * @see https://docs.aws.amazon.com/eks/latest/userguide/managed-node-groups.html
     * @stability stable
     */
    addNodegroupCapacity(id, options) {
        return new managed_nodegroup_1.Nodegroup(this, `Nodegroup${id}`, {
            cluster: this,
            ...options,
        });
    }
    /**
     * Lazily creates the AwsAuth resource, which manages AWS authentication mapping.
     *
     * @stability stable
     */
    get awsAuth() {
        if (!this._awsAuth) {
            this._awsAuth = new aws_auth_1.AwsAuth(this, 'AwsAuth', { cluster: this });
        }
        return this._awsAuth;
    }
    /**
     * If this cluster is kubectl-enabled, returns the OpenID Connect issuer url.
     *
     * This is because the values is only be retrieved by the API and not exposed
     * by CloudFormation. If this cluster is not kubectl-enabled (i.e. uses the
     * stock `CfnCluster`), this is `undefined`.
     *
     * @stability stable
     * @attribute true
     */
    get clusterOpenIdConnectIssuerUrl() {
        return this._clusterResource.attrOpenIdConnectIssuerUrl;
    }
    /**
     * If this cluster is kubectl-enabled, returns the OpenID Connect issuer.
     *
     * This is because the values is only be retrieved by the API and not exposed
     * by CloudFormation. If this cluster is not kubectl-enabled (i.e. uses the
     * stock `CfnCluster`), this is `undefined`.
     *
     * @stability stable
     * @attribute true
     */
    get clusterOpenIdConnectIssuer() {
        return this._clusterResource.attrOpenIdConnectIssuer;
    }
    /**
     * An `OpenIdConnectProvider` resource associated with this cluster, and which can be used to link this cluster to AWS IAM.
     *
     * A provider will only be defined if this property is accessed (lazy initialization).
     *
     * @stability stable
     */
    get openIdConnectProvider() {
        if (!this._openIdConnectProvider) {
            this._openIdConnectProvider = new oidc_provider_1.OpenIdConnectProvider(this, 'OpenIdConnectProvider', {
                url: this.clusterOpenIdConnectIssuerUrl,
            });
        }
        return this._openIdConnectProvider;
    }
    /**
     * Adds a Fargate profile to this cluster.
     *
     * @param id the id of this profile.
     * @param options profile options.
     * @see https://docs.aws.amazon.com/eks/latest/userguide/fargate-profile.html
     * @stability stable
     */
    addFargateProfile(id, options) {
        return new fargate_profile_1.FargateProfile(this, `fargate-profile-${id}`, {
            ...options,
            cluster: this,
        });
    }
    /**
     * Internal API used by `FargateProfile` to keep inventory of Fargate profiles associated with
     * this cluster, for the sake of ensuring the profiles are created sequentially.
     *
     * @returns the list of FargateProfiles attached to this cluster, including the one just attached.
     * @internal
     */
    _attachFargateProfile(fargateProfile) {
        this._fargateProfiles.push(fargateProfile);
        // add all profiles as a dependency of the "kubectl-ready" barrier because all kubectl-
        // resources can only be deployed after all fargate profiles are created.
        this._kubectlReadyBarrier.node.addDependency(fargateProfile);
        return this._fargateProfiles;
    }
    /**
     * Adds a resource scope that requires `kubectl` to this cluster and returns
     * the `KubectlProvider` which is the custom resource provider that should be
     * used as the resource provider.
     *
     * Called from `HelmResource` and `KubernetesResource`
     *
     * @param resourceScope the construct scope in which kubectl resources are defined.
     *
     * @internal
     */
    _attachKubectlResourceScope(resourceScope) {
        constructs_1.Node.of(resourceScope).addDependency(this._kubectlReadyBarrier);
        return this._kubectlResourceProvider;
    }
    defineKubectlProvider() {
        const uid = '@aws-cdk/aws-eks.KubectlProvider';
        // since we can't have the provider connect to multiple networks, and we
        // wanted to avoid resource tear down, we decided for now that we will only
        // support a single EKS cluster per CFN stack.
        if (this.stack.node.tryFindChild(uid)) {
            throw new Error('Only a single EKS cluster can be defined within a CloudFormation stack');
        }
        return new kubectl_provider_1.KubectlProvider(this.stack, uid, { cluster: this });
    }
    selectPrivateSubnets() {
        const privateSubnets = [];
        const vpcPrivateSubnetIds = this.vpc.privateSubnets.map(s => s.subnetId);
        const vpcPublicSubnetIds = this.vpc.publicSubnets.map(s => s.subnetId);
        for (const placement of this.vpcSubnets) {
            for (const subnet of this.vpc.selectSubnets(placement).subnets) {
                if (vpcPrivateSubnetIds.includes(subnet.subnetId)) {
                    // definitely private, take it.
                    privateSubnets.push(subnet);
                    continue;
                }
                if (vpcPublicSubnetIds.includes(subnet.subnetId)) {
                    // definitely public, skip it.
                    continue;
                }
                // neither public and nor private - what is it then? this means its a subnet instance that was explicitly passed
                // in the subnet selection. since ISubnet doesn't contain information on type, we have to assume its private and let it
                // fail at deploy time :\ (its better than filtering it out and preventing a possibly successful deployment)
                privateSubnets.push(subnet);
            }
        }
        return privateSubnets;
    }
    /**
     * Installs the Neuron device plugin on the cluster if it's not
     * already added.
     */
    addNeuronDevicePlugin() {
        if (!this._neuronDevicePlugin) {
            const fileContents = fs.readFileSync(path.join(__dirname, 'addons/neuron-device-plugin.yaml'), 'utf8');
            const sanitized = YAML.parse(fileContents);
            this._neuronDevicePlugin = this.addManifest('NeuronDevicePlugin', sanitized);
        }
        return this._neuronDevicePlugin;
    }
    /**
     * Opportunistically tag subnets with the required tags.
     *
     * If no subnets could be found (because this is an imported VPC), add a warning.
     *
     * @see https://docs.aws.amazon.com/eks/latest/userguide/network_reqs.html
     */
    tagSubnets() {
        const tagAllSubnets = (type, subnets, tag) => {
            for (const subnet of subnets) {
                // if this is not a concrete subnet, attach a construct warning
                if (!ec2.Subnet.isVpcSubnet(subnet)) {
                    // message (if token): "could not auto-tag public/private subnet with tag..."
                    // message (if not token): "count not auto-tag public/private subnet xxxxx with tag..."
                    const subnetID = core_1.Token.isUnresolved(subnet.subnetId) || core_1.Token.isUnresolved([subnet.subnetId]) ? '' : ` ${subnet.subnetId}`;
                    core_1.Annotations.of(this).addWarning(`Could not auto-tag ${type} subnet${subnetID} with "${tag}=1", please remember to do this manually`);
                    continue;
                }
                core_1.Tags.of(subnet).add(tag, '1');
            }
        };
        // https://docs.aws.amazon.com/eks/latest/userguide/network_reqs.html
        tagAllSubnets('private', this.vpc.privateSubnets, 'kubernetes.io/role/internal-elb');
        tagAllSubnets('public', this.vpc.publicSubnets, 'kubernetes.io/role/elb');
    }
    /**
     * Patches the CoreDNS deployment configuration and sets the "eks.amazonaws.com/compute-type"
     * annotation to either "ec2" or "fargate". Note that if "ec2" is selected, the resource is
     * omitted/removed, since the cluster is created with the "ec2" compute type by default.
     */
    defineCoreDnsComputeType(type) {
        // ec2 is the "built in" compute type of the cluster so if this is the
        // requested type we can simply omit the resource. since the resource's
        // `restorePatch` is configured to restore the value to "ec2" this means
        // that deletion of the resource will change to "ec2" as well.
        if (type === CoreDnsComputeType.EC2) {
            return;
        }
        // this is the json patch we merge into the resource based off of:
        // https://docs.aws.amazon.com/eks/latest/userguide/fargate-getting-started.html#fargate-gs-coredns
        const renderPatch = (computeType) => ({
            spec: {
                template: {
                    metadata: {
                        annotations: {
                            'eks.amazonaws.com/compute-type': computeType,
                        },
                    },
                },
            },
        });
        new k8s_patch_1.KubernetesPatch(this, 'CoreDnsComputeTypePatch', {
            cluster: this,
            resourceName: 'deployment/coredns',
            resourceNamespace: 'kube-system',
            applyPatch: renderPatch(CoreDnsComputeType.FARGATE),
            restorePatch: renderPatch(CoreDnsComputeType.EC2),
        });
    }
}
exports.Cluster = Cluster;
_c = JSII_RTTI_SYMBOL_1;
Cluster[_c] = { fqn: "@aws-cdk/aws-eks.Cluster", version: "1.128.0" };
/**
 * Import a cluster to use in another stack
 */
class ImportedCluster extends ClusterBase {
    constructor(scope, id, props) {
        var _e, _f;
        super(scope, id);
        this.props = props;
        this.connections = new ec2.Connections();
        this.clusterName = props.clusterName;
        this.clusterArn = this.stack.formatArn(cluster_resource_1.clusterArnComponents(props.clusterName));
        this.kubectlRole = props.kubectlRoleArn ? iam.Role.fromRoleArn(this, 'KubectlRole', props.kubectlRoleArn) : undefined;
        this.kubectlSecurityGroup = props.kubectlSecurityGroupId ? ec2.SecurityGroup.fromSecurityGroupId(this, 'KubectlSecurityGroup', props.kubectlSecurityGroupId) : undefined;
        this.kubectlEnvironment = props.kubectlEnvironment;
        this.kubectlPrivateSubnets = props.kubectlPrivateSubnetIds ? props.kubectlPrivateSubnetIds.map((subnetid, index) => ec2.Subnet.fromSubnetId(this, `KubectlSubnet${index}`, subnetid)) : undefined;
        this.kubectlLayer = props.kubectlLayer;
        this.onEventLayer = props.onEventLayer;
        this.kubectlMemory = props.kubectlMemory;
        this.prune = (_e = props.prune) !== null && _e !== void 0 ? _e : true;
        let i = 1;
        for (const sgid of (_f = props.securityGroupIds) !== null && _f !== void 0 ? _f : []) {
            this.connections.addSecurityGroup(ec2.SecurityGroup.fromSecurityGroupId(this, `SecurityGroup${i}`, sgid));
            i++;
        }
        if (props.clusterSecurityGroupId) {
            this._clusterSecurityGroup = ec2.SecurityGroup.fromSecurityGroupId(this, 'ClusterSecurityGroup', this.clusterSecurityGroupId);
            this.connections.addSecurityGroup(this._clusterSecurityGroup);
        }
    }
    get vpc() {
        if (!this.props.vpc) {
            throw new Error('"vpc" is not defined for this imported cluster');
        }
        return this.props.vpc;
    }
    get clusterSecurityGroup() {
        if (!this._clusterSecurityGroup) {
            throw new Error('"clusterSecurityGroup" is not defined for this imported cluster');
        }
        return this._clusterSecurityGroup;
    }
    get clusterSecurityGroupId() {
        if (!this.props.clusterSecurityGroupId) {
            throw new Error('"clusterSecurityGroupId" is not defined for this imported cluster');
        }
        return this.props.clusterSecurityGroupId;
    }
    get clusterEndpoint() {
        if (!this.props.clusterEndpoint) {
            throw new Error('"clusterEndpoint" is not defined for this imported cluster');
        }
        return this.props.clusterEndpoint;
    }
    get clusterCertificateAuthorityData() {
        if (!this.props.clusterCertificateAuthorityData) {
            throw new Error('"clusterCertificateAuthorityData" is not defined for this imported cluster');
        }
        return this.props.clusterCertificateAuthorityData;
    }
    get clusterEncryptionConfigKeyArn() {
        if (!this.props.clusterEncryptionConfigKeyArn) {
            throw new Error('"clusterEncryptionConfigKeyArn" is not defined for this imported cluster');
        }
        return this.props.clusterEncryptionConfigKeyArn;
    }
    get openIdConnectProvider() {
        if (!this.props.openIdConnectProvider) {
            throw new Error('"openIdConnectProvider" is not defined for this imported cluster');
        }
        return this.props.openIdConnectProvider;
    }
    get awsAuth() {
        throw new Error('"awsAuth" is not supported on imported clusters');
    }
}
/**
 * Construct an Amazon Linux 2 image from the latest EKS Optimized AMI published in SSM.
 *
 * @stability stable
 */
class EksOptimizedImage {
    /**
     * Constructs a new instance of the EcsOptimizedAmi class.
     *
     * @stability stable
     */
    constructor(props = {}) {
        var _e, _f, _g;
        this.nodeType = (_e = props.nodeType) !== null && _e !== void 0 ? _e : NodeType.STANDARD;
        this.cpuArch = (_f = props.cpuArch) !== null && _f !== void 0 ? _f : CpuArch.X86_64;
        this.kubernetesVersion = (_g = props.kubernetesVersion) !== null && _g !== void 0 ? _g : LATEST_KUBERNETES_VERSION;
        // set the SSM parameter name
        this.amiParameterName = `/aws/service/eks/optimized-ami/${this.kubernetesVersion}/`
            + (this.nodeType === NodeType.STANDARD ? this.cpuArch === CpuArch.X86_64 ?
                'amazon-linux-2/' : 'amazon-linux-2-arm64/' : '')
            + (this.nodeType === NodeType.GPU ? 'amazon-linux-2-gpu/' : '')
            + (this.nodeType === NodeType.INFERENTIA ? 'amazon-linux-2-gpu/' : '')
            + 'recommended/image_id';
    }
    /**
     * Return the correct image.
     *
     * @stability stable
     */
    getImage(scope) {
        const ami = ssm.StringParameter.valueForStringParameter(scope, this.amiParameterName);
        return {
            imageId: ami,
            osType: ec2.OperatingSystemType.LINUX,
            userData: ec2.UserData.forLinux(),
        };
    }
}
exports.EksOptimizedImage = EksOptimizedImage;
_d = JSII_RTTI_SYMBOL_1;
EksOptimizedImage[_d] = { fqn: "@aws-cdk/aws-eks.EksOptimizedImage", version: "1.128.0" };
// MAINTAINERS: use ./scripts/kube_bump.sh to update LATEST_KUBERNETES_VERSION
const LATEST_KUBERNETES_VERSION = '1.14';
/**
 * Whether the worker nodes should support GPU or just standard instances.
 *
 * @stability stable
 */
var NodeType;
(function (NodeType) {
    NodeType["STANDARD"] = "Standard";
    NodeType["GPU"] = "GPU";
    NodeType["INFERENTIA"] = "INFERENTIA";
})(NodeType = exports.NodeType || (exports.NodeType = {}));
/**
 * CPU architecture.
 *
 * @stability stable
 */
var CpuArch;
(function (CpuArch) {
    CpuArch["ARM_64"] = "arm64";
    CpuArch["X86_64"] = "x86_64";
})(CpuArch = exports.CpuArch || (exports.CpuArch = {}));
/**
 * The type of compute resources to use for CoreDNS.
 *
 * @stability stable
 */
var CoreDnsComputeType;
(function (CoreDnsComputeType) {
    CoreDnsComputeType["EC2"] = "ec2";
    CoreDnsComputeType["FARGATE"] = "fargate";
})(CoreDnsComputeType = exports.CoreDnsComputeType || (exports.CoreDnsComputeType = {}));
/**
 * The default capacity type for the cluster.
 *
 * @stability stable
 */
var DefaultCapacityType;
(function (DefaultCapacityType) {
    DefaultCapacityType[DefaultCapacityType["NODEGROUP"] = 0] = "NODEGROUP";
    DefaultCapacityType[DefaultCapacityType["EC2"] = 1] = "EC2";
})(DefaultCapacityType = exports.DefaultCapacityType || (exports.DefaultCapacityType = {}));
/**
 * The machine image type.
 *
 * @stability stable
 */
var MachineImageType;
(function (MachineImageType) {
    MachineImageType[MachineImageType["AMAZON_LINUX_2"] = 0] = "AMAZON_LINUX_2";
    MachineImageType[MachineImageType["BOTTLEROCKET"] = 1] = "BOTTLEROCKET";
})(MachineImageType = exports.MachineImageType || (exports.MachineImageType = {}));
function nodeTypeForInstanceType(instanceType) {
    return instance_types_1.INSTANCE_TYPES.gpu.includes(instanceType.toString().substring(0, 2)) ? NodeType.GPU :
        instance_types_1.INSTANCE_TYPES.inferentia.includes(instanceType.toString().substring(0, 4)) ? NodeType.INFERENTIA :
            NodeType.STANDARD;
}
function cpuArchForInstanceType(instanceType) {
    return instance_types_1.INSTANCE_TYPES.graviton2.includes(instanceType.toString().substring(0, 3)) ? CpuArch.ARM_64 :
        instance_types_1.INSTANCE_TYPES.graviton.includes(instanceType.toString().substring(0, 2)) ? CpuArch.ARM_64 :
            CpuArch.X86_64;
}
function flatten(xss) {
    return Array.prototype.concat.call([], ...xss);
}
//# sourceMappingURL=data:application/json;base64,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