"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Nodegroup = exports.CapacityType = exports.NodegroupAmiType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const cluster_1 = require("./cluster");
const eks_generated_1 = require("./eks.generated");
const instance_types_1 = require("./instance-types");
/**
 * The AMI type for your node group.
 *
 * GPU instance types should use the `AL2_x86_64_GPU` AMI type, which uses the
 * Amazon EKS-optimized Linux AMI with GPU support. Non-GPU instances should use the `AL2_x86_64` AMI type, which
 * uses the Amazon EKS-optimized Linux AMI.
 *
 * @stability stable
 */
var NodegroupAmiType;
(function (NodegroupAmiType) {
    NodegroupAmiType["AL2_X86_64"] = "AL2_x86_64";
    NodegroupAmiType["AL2_X86_64_GPU"] = "AL2_x86_64_GPU";
    NodegroupAmiType["AL2_ARM_64"] = "AL2_ARM_64";
})(NodegroupAmiType = exports.NodegroupAmiType || (exports.NodegroupAmiType = {}));
/**
 * Capacity type of the managed node group.
 *
 * @stability stable
 */
var CapacityType;
(function (CapacityType) {
    CapacityType["SPOT"] = "SPOT";
    CapacityType["ON_DEMAND"] = "ON_DEMAND";
})(CapacityType = exports.CapacityType || (exports.CapacityType = {}));
/**
 * The Nodegroup resource class.
 *
 * @stability stable
 */
class Nodegroup extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g, _h;
        super(scope, id, {
            physicalName: props.nodegroupName,
        });
        this.cluster = props.cluster;
        this.desiredSize = (_c = (_b = props.desiredSize) !== null && _b !== void 0 ? _b : props.minSize) !== null && _c !== void 0 ? _c : 2;
        this.maxSize = (_d = props.maxSize) !== null && _d !== void 0 ? _d : this.desiredSize;
        this.minSize = (_e = props.minSize) !== null && _e !== void 0 ? _e : 1;
        if (this.desiredSize > this.maxSize) {
            throw new Error(`Desired capacity ${this.desiredSize} can't be greater than max size ${this.maxSize}`);
        }
        if (this.desiredSize < this.minSize) {
            throw new Error(`Minimum capacity ${this.minSize} can't be greater than desired size ${this.desiredSize}`);
        }
        if (props.launchTemplateSpec && props.diskSize) {
            // see - https://docs.aws.amazon.com/eks/latest/userguide/launch-templates.html
            // and https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-nodegroup.html#cfn-eks-nodegroup-disksize
            throw new Error('diskSize must be specified within the launch template');
        }
        if (props.instanceType && props.instanceTypes) {
            throw new Error('"instanceType is deprecated, please use "instanceTypes" only.');
        }
        if (props.instanceType) {
            core_1.Annotations.of(this).addWarning('"instanceType" is deprecated and will be removed in the next major version. please use "instanceTypes" instead');
        }
        const instanceTypes = (_f = props.instanceTypes) !== null && _f !== void 0 ? _f : (props.instanceType ? [props.instanceType] : undefined);
        let expectedAmiType = undefined;
        if (instanceTypes && instanceTypes.length > 0) {
            // if the user explicitly configured instance types, we can calculate the expected ami type.
            expectedAmiType = getAmiType(instanceTypes);
            // if the user explicitly configured an ami type, make sure its the same as the expected one.
            if (props.amiType && props.amiType !== expectedAmiType) {
                throw new Error(`The specified AMI does not match the instance types architecture, either specify ${expectedAmiType} or dont specify any`);
            }
        }
        if (!props.nodeRole) {
            const ngRole = new aws_iam_1.Role(this, 'NodeGroupRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
            });
            ngRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSWorkerNodePolicy'));
            ngRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKS_CNI_Policy'));
            ngRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEC2ContainerRegistryReadOnly'));
            this.role = ngRole;
        }
        else {
            this.role = props.nodeRole;
        }
        const resource = new eks_generated_1.CfnNodegroup(this, 'Resource', {
            clusterName: this.cluster.clusterName,
            nodegroupName: props.nodegroupName,
            nodeRole: this.role.roleArn,
            subnets: this.cluster.vpc.selectSubnets(props.subnets).subnetIds,
            // if a launch template is configured, we cannot apply a default since it
            // might exist in the launch template as well, causing a deployment failure.
            amiType: props.launchTemplateSpec !== undefined ? props.amiType : ((_g = props.amiType) !== null && _g !== void 0 ? _g : expectedAmiType),
            capacityType: props.capacityType ? props.capacityType.valueOf() : undefined,
            diskSize: props.diskSize,
            forceUpdateEnabled: (_h = props.forceUpdate) !== null && _h !== void 0 ? _h : true,
            // note that we don't check if a launch template is configured here (even though it might configure instance types as well)
            // because this doesn't have a default value, meaning the user had to explicitly configure this.
            instanceTypes: instanceTypes === null || instanceTypes === void 0 ? void 0 : instanceTypes.map(t => t.toString()),
            labels: props.labels,
            launchTemplate: props.launchTemplateSpec,
            releaseVersion: props.releaseVersion,
            remoteAccess: props.remoteAccess ? {
                ec2SshKey: props.remoteAccess.sshKeyName,
                sourceSecurityGroups: props.remoteAccess.sourceSecurityGroups ?
                    props.remoteAccess.sourceSecurityGroups.map(m => m.securityGroupId) : undefined,
            } : undefined,
            scalingConfig: {
                desiredSize: this.desiredSize,
                maxSize: this.maxSize,
                minSize: this.minSize,
            },
            tags: props.tags,
        });
        // managed nodegroups update the `aws-auth` on creation, but we still need to track
        // its state for consistency.
        if (this.cluster instanceof cluster_1.Cluster) {
            // see https://docs.aws.amazon.com/en_us/eks/latest/userguide/add-user-role.html
            this.cluster.awsAuth.addRoleMapping(this.role, {
                username: 'system:node:{{EC2PrivateDNSName}}',
                groups: [
                    'system:bootstrappers',
                    'system:nodes',
                ],
            });
        }
        this.nodegroupArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'eks',
            resource: 'nodegroup',
            resourceName: this.physicalName,
        });
        this.nodegroupName = this.getResourceNameAttribute(resource.ref);
    }
    /**
     * Import the Nodegroup from attributes.
     *
     * @stability stable
     */
    static fromNodegroupName(scope, id, nodegroupName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.nodegroupName = nodegroupName;
            }
        }
        return new Import(scope, id);
    }
}
exports.Nodegroup = Nodegroup;
_a = JSII_RTTI_SYMBOL_1;
Nodegroup[_a] = { fqn: "@aws-cdk/aws-eks.Nodegroup", version: "1.109.0" };
function getAmiTypeForInstanceType(instanceType) {
    return instance_types_1.INSTANCE_TYPES.graviton2.includes(instanceType.toString().substring(0, 3)) ? NodegroupAmiType.AL2_ARM_64 :
        instance_types_1.INSTANCE_TYPES.graviton.includes(instanceType.toString().substring(0, 2)) ? NodegroupAmiType.AL2_ARM_64 :
            instance_types_1.INSTANCE_TYPES.gpu.includes(instanceType.toString().substring(0, 2)) ? NodegroupAmiType.AL2_X86_64_GPU :
                instance_types_1.INSTANCE_TYPES.inferentia.includes(instanceType.toString().substring(0, 4)) ? NodegroupAmiType.AL2_X86_64_GPU :
                    NodegroupAmiType.AL2_X86_64;
}
// this function examines the CPU architecture of every instance type and determines
// what ami type is compatible for all of them. it either throws or produces a single value because
// instance types of different CPU architectures are not supported.
function getAmiType(instanceTypes) {
    const amiTypes = new Set(instanceTypes.map(i => getAmiTypeForInstanceType(i)));
    if (amiTypes.size == 0) { // protective code, the current implementation will never result in this.
        throw new Error(`Cannot determine any ami type comptaible with instance types: ${instanceTypes.map(i => i.toString).join(',')}`);
    }
    if (amiTypes.size > 1) {
        throw new Error('instanceTypes of different CPU architectures is not allowed');
    }
    return amiTypes.values().next().value;
}
//# sourceMappingURL=data:application/json;base64,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