"""Rebase script generation and execution."""

import os
from datetime import datetime
from typing import List

from ..models import CommitGroup


class RebaseScriptGenerator:
    """Generates rebase scripts for dry-run mode."""
    
    def __init__(self, repo_path: str = "."):
        self.repo_path = repo_path
    
    def generate_rebase_script(self, groups: List[CommitGroup], output_path: str) -> str:
        """Generate a shell script that performs the rebase operations."""
        script_lines = [
            "#!/bin/bash",
            "# Generated by git-smart-squash",
            f"# Generated on: {datetime.now().isoformat()}",
            "",
            "set -e  # Exit on any error",
            "",
            "echo 'Starting git smart squash operations...'",
            ""
        ]
        
        # Add backup creation
        script_lines.extend([
            "# Create backup branch",
            'BACKUP_BRANCH="backup_$(git branch --show-current)_$(date +%Y%m%d_%H%M%S)"',
            'git branch "$BACKUP_BRANCH"',
            'echo "Created backup branch: $BACKUP_BRANCH"',
            ""
        ])
        
        # Generate rebase instructions for each group
        for i, group in enumerate(groups):
            if len(group.commits) > 1:
                script_lines.extend([
                    f"# Group {i+1}: {group.rationale}",
                    f"# Files: {', '.join(list(group.files_touched)[:3])}",
                    f"# New message: {group.suggested_message}",
                ])
                
                # Create the git rebase interactive sequence
                oldest_commit = group.commits[0]
                script_lines.extend([
                    f'git rebase -i {oldest_commit.parent_hash} --exec \'echo "Processing group {i+1}"\'',
                    ""
                ])
        
        script_lines.extend([
            "echo 'Git smart squash completed successfully!'",
            'echo "Backup branch created: $BACKUP_BRANCH"',
            'echo "To restore backup: git reset --hard $BACKUP_BRANCH"'
        ])
        
        # Write script to file
        with open(output_path, 'w') as f:
            f.write('\n'.join(script_lines))
        
        # Make script executable
        os.chmod(output_path, 0o755)
        
        return output_path