"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const cdk = require("@aws-cdk/core");
const stepfunctions = require("../lib");
describe('States Language', () => {
    test('A single task is a State Machine', () => {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const chain = new stepfunctions.Pass(stack, 'Some State');
        // THEN
        expect(render(chain)).toStrictEqual({
            StartAt: 'Some State',
            States: {
                'Some State': { Type: 'Pass', End: true },
            },
        });
    }),
        test('A sequence of two tasks is a State Machine', () => {
            // GIVEN
            const stack = new cdk.Stack();
            // WHEN
            const task1 = new stepfunctions.Pass(stack, 'State One');
            const task2 = new stepfunctions.Pass(stack, 'State Two');
            const chain = stepfunctions.Chain
                .start(task1)
                .next(task2);
            // THEN
            expect(render(chain)).toStrictEqual({
                StartAt: 'State One',
                States: {
                    'State One': { Type: 'Pass', Next: 'State Two' },
                    'State Two': { Type: 'Pass', End: true },
                },
            });
        }),
        test('You dont need to hold on to the state to render the entire state machine correctly', () => {
            const stack = new cdk.Stack();
            // WHEN
            const task1 = new stepfunctions.Pass(stack, 'State One');
            const task2 = new stepfunctions.Pass(stack, 'State Two');
            task1.next(task2);
            // THEN
            expect(render(task1)).toStrictEqual({
                StartAt: 'State One',
                States: {
                    'State One': { Type: 'Pass', Next: 'State Two' },
                    'State Two': { Type: 'Pass', End: true },
                },
            });
        }),
        test('A chain can be appended to', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const task1 = new stepfunctions.Pass(stack, 'State One');
            const task2 = new stepfunctions.Pass(stack, 'State Two');
            const task3 = new stepfunctions.Pass(stack, 'State Three');
            // WHEN
            const chain = stepfunctions.Chain
                .start(task1)
                .next(task2)
                .next(task3);
            // THEN
            expect(render(chain)).toStrictEqual({
                StartAt: 'State One',
                States: {
                    'State One': { Type: 'Pass', Next: 'State Two' },
                    'State Two': { Type: 'Pass', Next: 'State Three' },
                    'State Three': { Type: 'Pass', End: true },
                },
            });
        }),
        test('A state machine can be appended to another state machine', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const task1 = new stepfunctions.Pass(stack, 'State One');
            const task2 = new stepfunctions.Pass(stack, 'State Two');
            const task3 = new stepfunctions.Wait(stack, 'State Three', {
                time: stepfunctions.WaitTime.duration(cdk.Duration.seconds(10)),
            });
            // WHEN
            const chain = stepfunctions.Chain
                .start(task1)
                .next(stepfunctions.Chain.start(task2).next(task3));
            // THEN
            expect(render(chain)).toStrictEqual({
                StartAt: 'State One',
                States: {
                    'State One': { Type: 'Pass', Next: 'State Two' },
                    'State Two': { Type: 'Pass', Next: 'State Three' },
                    'State Three': { Type: 'Wait', End: true, Seconds: 10 },
                },
            });
        }),
        test('A state machine definition can be instantiated and chained', () => {
            const stack = new cdk.Stack();
            const before = new stepfunctions.Pass(stack, 'Before');
            const after = new stepfunctions.Pass(stack, 'After');
            // WHEN
            const chain = before.next(new ReusableStateMachine(stack, 'Reusable')).next(after);
            // THEN
            expect(render(chain)).toStrictEqual({
                StartAt: 'Before',
                States: {
                    'Before': { Type: 'Pass', Next: 'Choice' },
                    'Choice': {
                        Type: 'Choice',
                        Choices: [
                            { Variable: '$.branch', StringEquals: 'left', Next: 'Left Branch' },
                            { Variable: '$.branch', StringEquals: 'right', Next: 'Right Branch' },
                        ],
                    },
                    'Left Branch': { Type: 'Pass', Next: 'After' },
                    'Right Branch': { Type: 'Pass', Next: 'After' },
                    'After': { Type: 'Pass', End: true },
                },
            });
        }),
        test('A success state cannot be chained onto', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const succeed = new stepfunctions.Succeed(stack, 'Succeed');
            const pass = new stepfunctions.Pass(stack, 'Pass');
            // WHEN
            expect(() => pass.next(succeed).next(pass)).toThrow();
        }),
        test('A failure state cannot be chained onto', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const fail = new stepfunctions.Fail(stack, 'Fail', { error: 'X', cause: 'Y' });
            const pass = new stepfunctions.Pass(stack, 'Pass');
            // WHEN
            expect(() => pass.next(fail).next(pass)).toThrow();
        }),
        test('Parallels can contain direct states', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const one = new stepfunctions.Pass(stack, 'One');
            const two = new stepfunctions.Pass(stack, 'Two');
            const three = new stepfunctions.Pass(stack, 'Three');
            // WHEN
            const para = new stepfunctions.Parallel(stack, 'Parallel');
            para.branch(one.next(two));
            para.branch(three);
            // THEN
            expect(render(para)).toStrictEqual({
                StartAt: 'Parallel',
                States: {
                    Parallel: {
                        Type: 'Parallel',
                        End: true,
                        Branches: [
                            {
                                StartAt: 'One',
                                States: {
                                    One: { Type: 'Pass', Next: 'Two' },
                                    Two: { Type: 'Pass', End: true },
                                },
                            },
                            {
                                StartAt: 'Three',
                                States: {
                                    Three: { Type: 'Pass', End: true },
                                },
                            },
                        ],
                    },
                },
            });
        }),
        test('Parallels can contain instantiated reusable definitions', () => {
            // GIVEN
            const stack = new cdk.Stack();
            // WHEN
            const para = new stepfunctions.Parallel(stack, 'Parallel');
            para.branch(new ReusableStateMachine(stack, 'Reusable1').prefixStates('Reusable1/'));
            para.branch(new ReusableStateMachine(stack, 'Reusable2').prefixStates('Reusable2/'));
            // THEN
            expect(render(para)).toStrictEqual({
                StartAt: 'Parallel',
                States: {
                    Parallel: {
                        Type: 'Parallel',
                        End: true,
                        Branches: [
                            {
                                StartAt: 'Reusable1/Choice',
                                States: {
                                    'Reusable1/Choice': {
                                        Type: 'Choice',
                                        Choices: [
                                            { Variable: '$.branch', StringEquals: 'left', Next: 'Reusable1/Left Branch' },
                                            { Variable: '$.branch', StringEquals: 'right', Next: 'Reusable1/Right Branch' },
                                        ],
                                    },
                                    'Reusable1/Left Branch': { Type: 'Pass', End: true },
                                    'Reusable1/Right Branch': { Type: 'Pass', End: true },
                                },
                            },
                            {
                                StartAt: 'Reusable2/Choice',
                                States: {
                                    'Reusable2/Choice': {
                                        Type: 'Choice',
                                        Choices: [
                                            { Variable: '$.branch', StringEquals: 'left', Next: 'Reusable2/Left Branch' },
                                            { Variable: '$.branch', StringEquals: 'right', Next: 'Reusable2/Right Branch' },
                                        ],
                                    },
                                    'Reusable2/Left Branch': { Type: 'Pass', End: true },
                                    'Reusable2/Right Branch': { Type: 'Pass', End: true },
                                },
                            },
                        ],
                    },
                },
            });
        }),
        test('State Machine Fragments can be wrapped in a single state', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const reusable = new SimpleChain(stack, 'Hello');
            const state = reusable.toSingleState();
            expect(render(state)).toStrictEqual({
                StartAt: 'Hello',
                States: {
                    Hello: {
                        Type: 'Parallel',
                        End: true,
                        Branches: [
                            {
                                StartAt: 'Hello: Task1',
                                States: {
                                    'Hello: Task1': { Type: 'Task', Next: 'Hello: Task2', Resource: 'resource' },
                                    'Hello: Task2': { Type: 'Task', End: true, Resource: 'resource' },
                                },
                            },
                        ],
                    },
                },
            });
        }),
        test('Chaining onto branched failure state ignores failure state', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const yes = new stepfunctions.Pass(stack, 'Yes');
            const no = new stepfunctions.Fail(stack, 'No', { error: 'Failure', cause: 'Wrong branch' });
            const enfin = new stepfunctions.Pass(stack, 'Finally');
            const choice = new stepfunctions.Choice(stack, 'Choice')
                .when(stepfunctions.Condition.stringEquals('$.foo', 'bar'), yes)
                .otherwise(no);
            // WHEN
            choice.afterwards().next(enfin);
            // THEN
            expect(render(choice)).toStrictEqual({
                StartAt: 'Choice',
                States: {
                    Choice: {
                        Type: 'Choice',
                        Choices: [
                            { Variable: '$.foo', StringEquals: 'bar', Next: 'Yes' },
                        ],
                        Default: 'No',
                    },
                    Yes: { Type: 'Pass', Next: 'Finally' },
                    No: { Type: 'Fail', Error: 'Failure', Cause: 'Wrong branch' },
                    Finally: { Type: 'Pass', End: true },
                },
            });
        }),
        test('Can include OTHERWISE transition for Choice in afterwards()', () => {
            // GIVEN
            const stack = new cdk.Stack();
            // WHEN
            const chain = new stepfunctions.Choice(stack, 'Choice')
                .when(stepfunctions.Condition.stringEquals('$.foo', 'bar'), new stepfunctions.Pass(stack, 'Yes'))
                .afterwards({ includeOtherwise: true })
                .next(new stepfunctions.Pass(stack, 'Finally'));
            // THEN
            expect(render(chain)).toStrictEqual({
                StartAt: 'Choice',
                States: {
                    Choice: {
                        Type: 'Choice',
                        Choices: [
                            { Variable: '$.foo', StringEquals: 'bar', Next: 'Yes' },
                        ],
                        Default: 'Finally',
                    },
                    Yes: { Type: 'Pass', Next: 'Finally' },
                    Finally: { Type: 'Pass', End: true },
                },
            });
        }),
        test('State machines can have unconstrainted gotos', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const one = new stepfunctions.Pass(stack, 'One');
            const two = new stepfunctions.Pass(stack, 'Two');
            // WHEN
            const chain = one.next(two).next(one);
            // THEN
            expect(render(chain)).toStrictEqual({
                StartAt: 'One',
                States: {
                    One: { Type: 'Pass', Next: 'Two' },
                    Two: { Type: 'Pass', Next: 'One' },
                },
            });
        }),
        test('States can have error branches', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const task1 = new stepfunctions.Task(stack, 'Task1', { task: new FakeTask() });
            const failure = new stepfunctions.Fail(stack, 'Failed', { error: 'DidNotWork', cause: 'We got stuck' });
            // WHEN
            const chain = task1.addCatch(failure);
            // THEN
            expect(render(chain)).toStrictEqual({
                StartAt: 'Task1',
                States: {
                    Task1: {
                        Type: 'Task',
                        Resource: 'resource',
                        End: true,
                        Catch: [
                            { ErrorEquals: ['States.ALL'], Next: 'Failed' },
                        ],
                    },
                    Failed: {
                        Type: 'Fail',
                        Error: 'DidNotWork',
                        Cause: 'We got stuck',
                    },
                },
            });
        }),
        test('Retries and errors with a result path', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const task1 = new stepfunctions.Task(stack, 'Task1', { task: new FakeTask() });
            const failure = new stepfunctions.Fail(stack, 'Failed', { error: 'DidNotWork', cause: 'We got stuck' });
            // WHEN
            const chain = task1.addRetry({ errors: ['HTTPError'], maxAttempts: 2 }).addCatch(failure, { resultPath: '$.some_error' }).next(failure);
            // THEN
            expect(render(chain)).toStrictEqual({
                StartAt: 'Task1',
                States: {
                    Task1: {
                        Type: 'Task',
                        Resource: 'resource',
                        Catch: [{ ErrorEquals: ['States.ALL'], Next: 'Failed', ResultPath: '$.some_error' }],
                        Retry: [{ ErrorEquals: ['HTTPError'], MaxAttempts: 2 }],
                        Next: 'Failed',
                    },
                    Failed: {
                        Type: 'Fail',
                        Error: 'DidNotWork',
                        Cause: 'We got stuck',
                    },
                },
            });
        }),
        test('Can wrap chain and attach error handler', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const task1 = new stepfunctions.Task(stack, 'Task1', { task: new FakeTask() });
            const task2 = new stepfunctions.Task(stack, 'Task2', { task: new FakeTask() });
            const errorHandler = new stepfunctions.Pass(stack, 'ErrorHandler');
            // WHEN
            const chain = task1.next(task2).toSingleState('Wrapped').addCatch(errorHandler);
            // THEN
            expect(render(chain)).toStrictEqual({
                StartAt: 'Wrapped',
                States: {
                    Wrapped: {
                        Type: 'Parallel',
                        Branches: [
                            {
                                StartAt: 'Task1',
                                States: {
                                    Task1: {
                                        Type: 'Task',
                                        Resource: 'resource',
                                        Next: 'Task2',
                                    },
                                    Task2: {
                                        Type: 'Task',
                                        Resource: 'resource',
                                        End: true,
                                    },
                                },
                            },
                        ],
                        Catch: [
                            { ErrorEquals: ['States.ALL'], Next: 'ErrorHandler' },
                        ],
                        End: true,
                    },
                    ErrorHandler: { Type: 'Pass', End: true },
                },
            });
        }),
        test('Chaining does not chain onto error handler state', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const task1 = new stepfunctions.Task(stack, 'Task1', { task: new FakeTask() });
            const task2 = new stepfunctions.Task(stack, 'Task2', { task: new FakeTask() });
            const errorHandler = new stepfunctions.Pass(stack, 'ErrorHandler');
            // WHEN
            const chain = task1.addCatch(errorHandler).next(task2);
            // THEN
            expect(render(chain)).toStrictEqual({
                StartAt: 'Task1',
                States: {
                    Task1: {
                        Type: 'Task',
                        Resource: 'resource',
                        Next: 'Task2',
                        Catch: [
                            { ErrorEquals: ['States.ALL'], Next: 'ErrorHandler' },
                        ],
                    },
                    Task2: { Type: 'Task', Resource: 'resource', End: true },
                    ErrorHandler: { Type: 'Pass', End: true },
                },
            });
        }),
        test('Chaining does not chain onto error handler, extended', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const task1 = new stepfunctions.Task(stack, 'Task1', { task: new FakeTask() });
            const task2 = new stepfunctions.Task(stack, 'Task2', { task: new FakeTask() });
            const task3 = new stepfunctions.Task(stack, 'Task3', { task: new FakeTask() });
            const errorHandler = new stepfunctions.Pass(stack, 'ErrorHandler');
            // WHEN
            const chain = task1.addCatch(errorHandler)
                .next(task2.addCatch(errorHandler))
                .next(task3.addCatch(errorHandler));
            // THEN
            const sharedTaskProps = { Type: 'Task', Resource: 'resource', Catch: [{ ErrorEquals: ['States.ALL'], Next: 'ErrorHandler' }] };
            expect(render(chain)).toStrictEqual({
                StartAt: 'Task1',
                States: {
                    Task1: { Next: 'Task2', ...sharedTaskProps },
                    Task2: { Next: 'Task3', ...sharedTaskProps },
                    Task3: { End: true, ...sharedTaskProps },
                    ErrorHandler: { Type: 'Pass', End: true },
                },
            });
        }),
        test('Error handler with a fragment', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const task1 = new stepfunctions.Task(stack, 'Task1', { task: new FakeTask() });
            const task2 = new stepfunctions.Task(stack, 'Task2', { task: new FakeTask() });
            const errorHandler = new stepfunctions.Pass(stack, 'ErrorHandler');
            // WHEN
            task1.addCatch(errorHandler)
                .next(new SimpleChain(stack, 'Chain').catch(errorHandler))
                .next(task2.addCatch(errorHandler));
        }),
        test('Can merge state machines with shared states', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const task1 = new stepfunctions.Task(stack, 'Task1', { task: new FakeTask() });
            const task2 = new stepfunctions.Task(stack, 'Task2', { task: new FakeTask() });
            const failure = new stepfunctions.Fail(stack, 'Failed', { error: 'DidNotWork', cause: 'We got stuck' });
            // WHEN
            task1.addCatch(failure);
            task2.addCatch(failure);
            task1.next(task2);
            // THEN
            expect(render(task1)).toStrictEqual({
                StartAt: 'Task1',
                States: {
                    Task1: {
                        Type: 'Task',
                        Resource: 'resource',
                        Next: 'Task2',
                        Catch: [
                            { ErrorEquals: ['States.ALL'], Next: 'Failed' },
                        ],
                    },
                    Task2: {
                        Type: 'Task',
                        Resource: 'resource',
                        End: true,
                        Catch: [
                            { ErrorEquals: ['States.ALL'], Next: 'Failed' },
                        ],
                    },
                    Failed: {
                        Type: 'Fail',
                        Error: 'DidNotWork',
                        Cause: 'We got stuck',
                    },
                },
            });
        }),
        test('No duplicate state IDs', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const intermediateParent = new cdk.Construct(stack, 'Parent');
            const state1 = new stepfunctions.Pass(stack, 'State');
            const state2 = new stepfunctions.Pass(intermediateParent, 'State');
            state1.next(state2);
            // WHEN
            expect(() => render(state1)).toThrow();
        }),
        test('No duplicate state IDs even across Parallel branches', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const intermediateParent = new cdk.Construct(stack, 'Parent');
            const state1 = new stepfunctions.Pass(stack, 'State');
            const state2 = new stepfunctions.Pass(intermediateParent, 'State');
            const parallel = new stepfunctions.Parallel(stack, 'Parallel')
                .branch(state1)
                .branch(state2);
            // WHEN
            expect(() => render(parallel)).toThrow();
        }),
        test('No cross-parallel jumps', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const state1 = new stepfunctions.Pass(stack, 'State1');
            const state2 = new stepfunctions.Pass(stack, 'State2');
            expect(() => new stepfunctions.Parallel(stack, 'Parallel')
                .branch(state1.next(state2))
                .branch(state2)).toThrow();
        });
});
class ReusableStateMachine extends stepfunctions.StateMachineFragment {
    constructor(scope, id) {
        super(scope, id);
        const choice = new stepfunctions.Choice(this, 'Choice')
            .when(stepfunctions.Condition.stringEquals('$.branch', 'left'), new stepfunctions.Pass(this, 'Left Branch'))
            .when(stepfunctions.Condition.stringEquals('$.branch', 'right'), new stepfunctions.Pass(this, 'Right Branch'));
        this.startState = choice;
        this.endStates = choice.afterwards().endStates;
    }
}
class SimpleChain extends stepfunctions.StateMachineFragment {
    constructor(scope, id) {
        super(scope, id);
        const task1 = new stepfunctions.Task(this, 'Task1', { task: new FakeTask() });
        this.task2 = new stepfunctions.Task(this, 'Task2', { task: new FakeTask() });
        task1.next(this.task2);
        this.startState = task1;
        this.endStates = [this.task2];
    }
    catch(state, props) {
        this.task2.addCatch(state, props);
        return this;
    }
}
function render(sm) {
    return new cdk.Stack().resolve(new stepfunctions.StateGraph(sm.startState, 'Test Graph').toGraphJson());
}
class FakeTask {
    bind(_task) {
        return {
            resourceArn: 'resource',
        };
    }
}
//# sourceMappingURL=data:application/json;base64,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