"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
/**
 * Metrics on the rate limiting performed on state machine execution.
 *
 * These rate limits are shared across all state machines.
 */
class StateTransitionMetric {
    /**
     * Return the given named metric for the service's state transition metrics
     *
     * @default average over 5 minutes
     */
    static metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/States',
            metricName,
            dimensions: { ServiceMetric: 'StateTransition' },
            ...props,
        });
    }
    /**
     * Metric for the number of available state transitions.
     *
     * @default average over 5 minutes
     */
    static metricProvisionedBucketSize(props) {
        return StateTransitionMetric.metric('ProvisionedBucketSize', props);
    }
    /**
     * Metric for the provisioned steady-state execution rate
     *
     * @default average over 5 minutes
     */
    static metricProvisionedRefillRate(props) {
        return StateTransitionMetric.metric('ProvisionedRefillRate', props);
    }
    /**
     * Metric for the number of available state transitions per second
     *
     * @default average over 5 minutes
     */
    static metricConsumedCapacity(props) {
        return StateTransitionMetric.metric('ConsumedCapacity', props);
    }
    /**
     * Metric for the number of throttled state transitions
     *
     * @default sum over 5 minutes
     */
    static metricThrottledEvents(props) {
        return StateTransitionMetric.metric('ThrottledEvents', { statistic: 'sum', ...props });
    }
}
exports.StateTransitionMetric = StateTransitionMetric;
//# sourceMappingURL=data:application/json;base64,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