"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const fields_1 = require("./fields");
/**
 * Type union for task classes that accept multiple types of payload
 */
class TaskInput {
    /**
     *
     * @param type type of task input
     * @param value payload for the corresponding input type.
     * It can be a JSON-encoded object, context, data, etc.
     */
    constructor(type, value) {
        this.type = type;
        this.value = value;
    }
    /**
     * Use a literal string as task input
     *
     * This might be a JSON-encoded object, or just a text.
     */
    static fromText(text) {
        return new TaskInput(InputType.TEXT, text);
    }
    /**
     * Use an object as task input
     *
     * This object may contain Data and Context fields
     * as object values, if desired.
     */
    static fromObject(obj) {
        return new TaskInput(InputType.OBJECT, obj);
    }
    /**
     * Use a part of the execution data as task input
     *
     * Use this when you want to use a subobject or string from
     * the current state machine execution as complete payload
     * to a task.
     */
    static fromDataAt(path) {
        return new TaskInput(InputType.TEXT, fields_1.Data.stringAt(path));
    }
    /**
     * Use a part of the task context as task input
     *
     * Use this when you want to use a subobject or string from
     * the current task context as complete payload
     * to a task.
     */
    static fromContextAt(path) {
        return new TaskInput(InputType.TEXT, fields_1.Context.stringAt(path));
    }
}
exports.TaskInput = TaskInput;
/**
 * The type of task input
 */
var InputType;
(function (InputType) {
    /**
     * Use a literal string
     * This might be a JSON-encoded object, or just text.
     * valid JSON text: standalone, quote-delimited strings; objects; arrays; numbers; Boolean values; and null.
     *
     * example: `literal string`
     * example: {"json": "encoded"}
     */
    InputType[InputType["TEXT"] = 0] = "TEXT";
    /**
     * Use an object which may contain Data and Context fields
     * as object values, if desired.
     *
     * example:
     * {
     *  literal: 'literal',
     *  SomeInput: sfn.Data.stringAt('$.someField')
     * }
     *
     * @see https://docs.aws.amazon.com/step-functions/latest/dg/concepts-state-machine-data.html
     * @see https://docs.aws.amazon.com/step-functions/latest/dg/input-output-contextobject.html
     */
    InputType[InputType["OBJECT"] = 1] = "OBJECT";
})(InputType = exports.InputType || (exports.InputType = {}));
//# sourceMappingURL=data:application/json;base64,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