import numpy as np
import tqdm as tq

from .sol_ode import sol_ode
from .delay import delay

def PID_controller(model, var0, t, param, ylimit, w, y0 = 0, PID = [0,0,0], Tt = 0, x_index = -1, y_index = -1, n = 1, n_meas = 0):

    """
    Create a PID controller to adjust the parameter of input ode function,
    calculate the dataframe using the adjusted parameter.

    Args:
    
        model (callable(y,t,...)): The function computes the derivative of y at t.
        var0 (array): Initial condition on var.
        t (array): A sequence of time points for which to solve for y. The initial
                    value point should be the first element of this sequence.
        param (array, optional): Input parameters to model function           
        ylimit (array, optional): Upper and lower setpoint for control element.
                    
        w (float or int, optional): The w value of the measured element function.      
        y0 (int, optional): First value of the controlled parameter. Defaults to 0.
        PID (array, optional): Parameter of the PID controller. Defaults to 0.
        Tt (float, optional): Specified dead time point.
        
        x_index (int, optional): Index of the measured element in var0.
        y_index (int, optional): Index of control element in param.
        
        n (int, optional): How many reactors are there in total.
        n_meas (int,optional): In which reactor does the measurment take place

    Returns:
        results: The solved variables using adjusted control element.
        y_final: The log of the adjusted control element.
        e_final: The log of error signal
    """

#    warnings.filterwarnings("ignore")


    # create the y value list with initial value.
    y = y0
    y_final = np.zeros(len(t))
    x = np.zeros(len(t)-1)
    e_final = np.zeros(len(t))
    e = w
    KP, KI, KD = PID
    Int_e = 0
    
    # empty dataframe for saving results
    results = list(range(len(t)))

    #  store the value calculated from last time.
    for i in tq.tqdm(range(len(t)-1),position=0,leave=True):
        
        y_final[i] = y
        param[y_index] = y

        #   get the solved results and get the new kla
        df = sol_ode(model, var0, t[i:i+2], param)

        # save the solved results, drop the duplicated value with the same time index,
        # only keeping the last one.
        
        
        if n == 1:
            results[i] = np.transpose(df)[-1]
            var0 = np.transpose(df)[-1]
            x[i] = df[x_index][-1]
        else:
            results[i] =np.array(df)[:,-1]
            var0 = np.array(df)[:,-1]
            x[i] = df[x_index][-1,n_meas-1]
            
        
        if Tt != 0:
            x_delay = delay(t, x, Tt, value = 0)

        else:
            x_delay = x

        e_temp = w - x_delay[i]
        # D component
        D = KD * (e - e_temp) / (t[i+1]-t[i])
        # P component
        e = e_temp
        e_final[i] = e_temp
        P = KP * e
        if y<ylimit[1] and y>ylimit[0]:
            Int_e = Int_e + e *(t[i+1]-t[i])
        else: Int_e = Int_e + 0
        # I component
        I = KI*Int_e
        
        
        
        y = y0 + P + I - D
        y = max(min(y, ylimit[1]), ylimit[0])
        
    results[len(t)-1] = results[len(t)-2]
    e_final[len(t)-1] = e_final[len(t)-2]
    y_final[len(t)-1] = y_final[len(t)-2]
    
    if n != 1: 
        r = np.array(results)
        results = r.transpose(1,0,2)
    else: results = np.transpose(results)
    return results, y_final, e_final