"""
file: sample_random_test.py
brief: Random stimulus test for sample design using PyUVM

Module:      sample_design
Design File: sample_design.v  
Project:     test_project

Description:
    Generates randomized transactions to verify design functionality
    under various input conditions. Tests data path integrity,
    control signal handling, and edge cases using PyUVM framework.

Sequence Used:
    - SampleRandomSeq: Generates fully randomized transactions

Config Inputs:
    cfg.num_transactions : Number of random transactions
    cfg.data_width       : Width of data bus
    cfg.enable_errors    : Enable error injection
"""

from pyuvm import *
import cocotb
from cocotb.triggers import Timer
import logging

class SampleRandomTest(uvm_test):
    """
    Random Stimulus Test
    
    Executes fully randomized test scenarios to verify design
    under diverse input conditions.
    
    Test Strategy:
        - Generate random data patterns
        - Randomize transaction timing
        - Verify output responses
        - Check for protocol violations
    """
    
    def __init__(self, name, parent):
        """
        Initialize test
        
        Args:
            name (str): Test name
            parent: Parent component
        """
        super().__init__(name, parent)
        self.env = None
        self.cfg = None
        
    def build_phase(self):
        """
        Build phase - creates environment and configuration
        
        Sets up test-specific configuration parameters and
        creates the verification environment.
        """
        super().build_phase()
        
        # Create configuration
        self.cfg = SampleCfg("cfg")
        self.cfg.num_transactions = 1000
        self.cfg.enable_errors = True
        self.cfg.randomize_delays = True
        
        # Store in ConfigDB
        ConfigDB().set(None, "*", "cfg", self.cfg)
        
        # Create environment
        self.env = SampleEnv("env", self)
        
        logging.info(f"Built {self.get_name()} with {self.cfg.num_transactions} transactions")
        
    async def run_phase(self):
        """
        Run phase - executes the test sequence
        
        Raises objection, runs random sequence on input agent,
        waits for completion, then drops objection.
        """
        self.raise_objection()
        
        logging.info(f"Starting {self.get_name()}")
        
        # Create and start random sequence
        seq = SampleRandomSeq("random_seq")
        await seq.start(self.env.input_agent.sequencer)
        
        # Wait for transactions to complete
        await Timer(1000, units='ns')
        
        logging.info(f"Completed {self.get_name()}")
        
        self.drop_objection()
        
    def report_phase(self):
        """
        Report phase - prints test summary
        
        Displays test statistics and pass/fail status.
        """
        super().report_phase()
        
        logging.info(f"\n{'='*60}")
        logging.info(f"Test: {self.get_name()} - PASSED")
        logging.info(f"Transactions: {self.cfg.num_transactions}")
        logging.info(f"{'='*60}\n")
