"""
file: sample_env.py
brief: Sample PyUVM environment for documentation testing

class SampleEnv

brief: Top-level PyUVM verification environment

::

    Module       : sample_design
    Design File  : sample_design.v
    Project      : test_project

This environment integrates agents, scoreboard, and coverage collectors
to create a complete verification environment using PyUVM.
"""

from pyuvm import *
import logging

class SampleEnv(uvm_env):
    """
    Sample Verification Environment
    
    Top-level environment that instantiates and connects all verification
    components including agents, scoreboards, and coverage collectors.
    
    Attributes:
        input_agent: Agent for input interface
        output_agent: Agent for output interface
        scoreboard: Reference model and checker
        coverage: Functional coverage collector
        cfg: Environment configuration object
    """
    
    def __init__(self, name, parent):
        """
        Initialize the environment
        
        Args:
            name (str): Environment instance name
            parent: Parent component in UVM hierarchy
        """
        super().__init__(name, parent)
        self.input_agent = None
        self.output_agent = None
        self.scoreboard = None
        self.coverage = None
        self.cfg = None
        
    def build_phase(self):
        """
        Build phase - creates all environment components
        
        Instantiates agents, scoreboard, and coverage components.
        Retrieves configuration from ConfigDB.
        """
        super().build_phase()
        
        # Get configuration
        self.cfg = ConfigDB().get(self, "", "cfg")
        
        # Create agents
        self.input_agent = SampleAgent("input_agent", self)
        self.output_agent = SampleAgent("output_agent", self)
        
        # Create scoreboard
        self.scoreboard = SampleScoreboard("scoreboard", self)
        
        # Create coverage
        if self.cfg.enable_coverage:
            self.coverage = SampleCoverage("coverage", self)
            
        logging.info(f"{self.get_name()}: Built environment components")
        
    def connect_phase(self):
        """
        Connect phase - connects monitors to scoreboard and coverage
        
        Establishes analysis port connections from monitors to
        scoreboard FIFOs and coverage collectors.
        """
        super().connect_phase()
        
        # Connect monitors to scoreboard
        self.input_agent.monitor.ap.connect(self.scoreboard.input_fifo.analysis_export)
        self.output_agent.monitor.ap.connect(self.scoreboard.output_fifo.analysis_export)
        
        # Connect coverage if enabled
        if self.cfg.enable_coverage:
            self.input_agent.monitor.ap.connect(self.coverage.analysis_export)
            
        logging.info(f"{self.get_name()}: Connected environment components")
        
    async def run_phase(self):
        """
        Run phase - main execution phase
        
        Can be used for environment-level runtime activities like
        timeout monitoring or dynamic reconfiguration.
        """
        logging.info(f"{self.get_name()}: Environment running")
