"""
file: sample_agent.py
brief: Sample PyUVM agent for documentation testing

class SampleAgent

brief: A PyUVM agent demonstrating typical verification structure

::

    Module       : sample_design
    Design File  : sample_design.v
    Project      : test_project

This agent contains a driver, monitor, and sequencer for managing
transactions in the PyUVM verification environment.
"""

from pyuvm import *
import cocotb
from cocotb.triggers import RisingEdge, Timer

class SampleAgent(uvm_agent):
    """
    Sample UVM Agent
    
    This agent manages the driver, monitor, and sequencer components
    for transaction-level verification.
    
    Attributes:
        driver: Transaction driver component
        monitor: Transaction monitor component  
        sequencer: Sequence controller component
        cfg: Agent configuration object
    """
    
    def __init__(self, name, parent):
        """
        Initialize the agent
        
        Args:
            name (str): Agent instance name
            parent: Parent component in UVM hierarchy
        """
        super().__init__(name, parent)
        self.driver = None
        self.monitor = None
        self.sequencer = None
        self.cfg = None
        
    def build_phase(self):
        """
        Build phase - creates agent sub-components
        
        Creates driver, monitor, and sequencer based on configuration.
        The driver and sequencer are only created if the agent is active.
        """
        super().build_phase()
        
        # Get configuration
        self.cfg = ConfigDB().get(self, "", "cfg")
        
        # Always create monitor
        self.monitor = SampleMonitor("monitor", self)
        
        # Create driver and sequencer for active agents
        if self.cfg.is_active:
            self.driver = SampleDriver("driver", self)
            self.sequencer = SampleSequencer("sequencer", self)
            
    def connect_phase(self):
        """
        Connect phase - connects driver to sequencer
        
        Establishes the connection between the sequencer's sequence
        item port and the driver's sequence item export.
        """
        super().connect_phase()
        
        if self.cfg.is_active:
            self.driver.seq_item_port = self.sequencer.seq_item_export
