/**
 * File: sample_seq_lib.sv
 * Brief: Sequence library for sample design verification
 * 
 * Package: sample_seq_lib_pkg
 * 
 * Brief: Collection of reusable sequences for stimulus generation
 * 
 * Module       : sample_design
 * Design File  : sample_design.v
 * Project      : test_project
 * 
 * This library contains various sequence classes for generating
 * different types of stimulus patterns including random, directed,
 * boundary, and corner case scenarios.
 */

`ifndef SAMPLE_SEQ_LIB_DEF
`define SAMPLE_SEQ_LIB_DEF

//-----------------------------------------------------------------------------
// Class: sample_base_seq
// Brief: Base sequence class with common functionality
//-----------------------------------------------------------------------------
class sample_base_seq extends uvm_sequence#(sample_transaction);
  
  `uvm_object_utils(sample_base_seq)
  
  // Configuration
  int num_transactions = 10;
  sample_cfg cfg;
  
  /**
   * Constructor: new
   * Creates a new base sequence instance
   */
  function new(string name = "sample_base_seq");
    super.new(name);
  endfunction
  
  /**
   * Task: pre_body
   * Pre-sequence body tasks - get configuration
   */
  virtual task pre_body();
    if (!uvm_config_db#(sample_cfg)::get(null, "", "cfg", cfg)) begin
      `uvm_warning(get_type_name(), "Failed to get configuration")
    end
  endtask
  
  /**
   * Task: body
   * Default body - override in derived classes
   */
  virtual task body();
    `uvm_info(get_type_name(), "Base sequence body", UVM_LOW)
  endtask
  
endclass

//-----------------------------------------------------------------------------
// Class: sample_random_seq
// Brief: Fully randomized transaction sequence
//
// Description:
//   Generates completely randomized transactions with all fields
//   randomized within their constraints. Useful for general
//   functional coverage and finding unexpected corner cases.
//-----------------------------------------------------------------------------
class sample_random_seq extends sample_base_seq;
  
  `uvm_object_utils(sample_random_seq)
  
  /**
   * Constructor: new
   */
  function new(string name = "sample_random_seq");
    super.new(name);
  endfunction
  
  /**
   * Task: body
   * Generate random transactions
   */
  virtual task body();
    sample_transaction txn;
    
    repeat(num_transactions) begin
      txn = sample_transaction::type_id::create("txn");
      start_item(txn);
      
      if (!txn.randomize()) begin
        `uvm_error(get_type_name(), "Randomization failed")
      end
      
      `uvm_info(get_type_name(), 
                $sformatf("Random txn: data=%0h, addr=%0h", 
                         txn.data, txn.addr), UVM_HIGH)
      
      finish_item(txn);
    end
  endtask
  
endclass

//-----------------------------------------------------------------------------
// Class: sample_directed_seq
// Brief: Directed test sequence with specific patterns
//
// Description:
//   Generates predetermined data patterns for directed testing.
//   Includes walking ones, walking zeros, and alternating patterns
//   to test specific data path scenarios.
//-----------------------------------------------------------------------------
class sample_directed_seq extends sample_base_seq;
  
  `uvm_object_utils(sample_directed_seq)
  
  // Pattern types
  typedef enum {WALKING_ONES, WALKING_ZEROS, ALTERNATING, INCREMENTAL} pattern_t;
  rand pattern_t pattern_type;
  
  /**
   * Constructor: new
   */
  function new(string name = "sample_directed_seq");
    super.new(name);
  endfunction
  
  /**
   * Task: body
   * Generate directed pattern transactions
   */
  virtual task body();
    sample_transaction txn;
    bit [31:0] pattern;
    
    case(pattern_type)
      WALKING_ONES: begin
        for (int i = 0; i < 32; i++) begin
          txn = sample_transaction::type_id::create("txn");
          start_item(txn);
          txn.data = (1 << i);
          txn.addr = i;
          finish_item(txn);
        end
      end
      
      WALKING_ZEROS: begin
        for (int i = 0; i < 32; i++) begin
          txn = sample_transaction::type_id::create("txn");
          start_item(txn);
          txn.data = ~(1 << i);
          txn.addr = i;
          finish_item(txn);
        end
      end
      
      ALTERNATING: begin
        for (int i = 0; i < num_transactions; i++) begin
          txn = sample_transaction::type_id::create("txn");
          start_item(txn);
          txn.data = (i % 2) ? 32'hAAAA_AAAA : 32'h5555_5555;
          txn.addr = i;
          finish_item(txn);
        end
      end
      
      INCREMENTAL: begin
        for (int i = 0; i < num_transactions; i++) begin
          txn = sample_transaction::type_id::create("txn");
          start_item(txn);
          txn.data = i;
          txn.addr = i;
          finish_item(txn);
        end
      end
    endcase
  endtask
  
endclass

//-----------------------------------------------------------------------------
// Class: sample_boundary_seq
// Brief: Boundary value testing sequence
//
// Description:
//   Tests minimum and maximum values for all transaction fields.
//   Critical for validating data path width and overflow handling.
//
// Test Coverage:
//   - All zeros (minimum values)
//   - All ones (maximum values)
//   - Single bit set at MSB and LSB
//   - Alternating boundary patterns
//-----------------------------------------------------------------------------
class sample_boundary_seq extends sample_base_seq;
  
  `uvm_object_utils(sample_boundary_seq)
  
  /**
   * Constructor: new
   */
  function new(string name = "sample_boundary_seq");
    super.new(name);
  endfunction
  
  /**
   * Task: body
   * Generate boundary value transactions
   */
  virtual task body();
    sample_transaction txn;
    
    // Test 1: All zeros
    txn = sample_transaction::type_id::create("txn");
    start_item(txn);
    txn.data = 32'h0000_0000;
    txn.addr = 32'h0000_0000;
    `uvm_info(get_type_name(), "Testing all zeros", UVM_MEDIUM)
    finish_item(txn);
    
    // Test 2: All ones
    txn = sample_transaction::type_id::create("txn");
    start_item(txn);
    txn.data = 32'hFFFF_FFFF;
    txn.addr = 32'hFFFF_FFFF;
    `uvm_info(get_type_name(), "Testing all ones", UVM_MEDIUM)
    finish_item(txn);
    
    // Test 3: MSB set
    txn = sample_transaction::type_id::create("txn");
    start_item(txn);
    txn.data = 32'h8000_0000;
    txn.addr = 32'h8000_0000;
    `uvm_info(get_type_name(), "Testing MSB set", UVM_MEDIUM)
    finish_item(txn);
    
    // Test 4: LSB set
    txn = sample_transaction::type_id::create("txn");
    start_item(txn);
    txn.data = 32'h0000_0001;
    txn.addr = 32'h0000_0001;
    `uvm_info(get_type_name(), "Testing LSB set", UVM_MEDIUM)
    finish_item(txn);
  endtask
  
endclass

//-----------------------------------------------------------------------------
// Class: sample_burst_seq
// Brief: Back-to-back burst transaction sequence
//
// Description:
//   Generates consecutive transactions without idle cycles to test
//   maximum throughput and pipeline behavior. Validates DUT's ability
//   to handle sustained traffic.
//
// Parameters:
//   burst_length - Number of consecutive transactions per burst
//   num_bursts   - Total number of bursts to generate
//-----------------------------------------------------------------------------
class sample_burst_seq extends sample_base_seq;
  
  `uvm_object_utils(sample_burst_seq)
  
  rand int burst_length;
  rand int num_bursts;
  
  // Constraints
  constraint burst_length_c { burst_length inside {[4:16]}; }
  constraint num_bursts_c { num_bursts inside {[2:8]}; }
  
  /**
   * Constructor: new
   */
  function new(string name = "sample_burst_seq");
    super.new(name);
  endfunction
  
  /**
   * Task: body
   * Generate burst transactions
   */
  virtual task body();
    sample_transaction txn;
    
    for (int burst = 0; burst < num_bursts; burst++) begin
      `uvm_info(get_type_name(), 
                $sformatf("Starting burst %0d of length %0d", 
                         burst, burst_length), UVM_MEDIUM)
      
      for (int i = 0; i < burst_length; i++) begin
        txn = sample_transaction::type_id::create("txn");
        start_item(txn);
        
        if (!txn.randomize()) begin
          `uvm_error(get_type_name(), "Randomization failed")
        end
        
        txn.addr = (burst * burst_length) + i;
        finish_item(txn);
      end
      
      // Small gap between bursts
      #10ns;
    end
  endtask
  
endclass

//-----------------------------------------------------------------------------
// Class: sample_error_injection_seq
// Brief: Error injection sequence for negative testing
//
// Description:
//   Intentionally generates invalid transactions to verify error
//   handling and protocol violation detection. Tests DUT's robustness
//   against malformed inputs.
//
// Error Types:
//   - Invalid addresses
//   - Protocol violations
//   - Out-of-range data values
//   - Timing violations
//-----------------------------------------------------------------------------
class sample_error_injection_seq extends sample_base_seq;
  
  `uvm_object_utils(sample_error_injection_seq)
  
  typedef enum {INVALID_ADDR, BAD_PARITY, OUT_OF_RANGE} error_type_t;
  rand error_type_t error_type;
  
  /**
   * Constructor: new
   */
  function new(string name = "sample_error_injection_seq");
    super.new(name);
  endfunction
  
  /**
   * Task: body
   * Generate error transactions
   */
  virtual task body();
    sample_transaction txn;
    
    repeat(num_transactions) begin
      txn = sample_transaction::type_id::create("txn");
      start_item(txn);
      
      case(error_type)
        INVALID_ADDR: begin
          txn.randomize();
          txn.addr = 32'hDEAD_BEEF;  // Known invalid address
          `uvm_info(get_type_name(), "Injecting invalid address", UVM_MEDIUM)
        end
        
        BAD_PARITY: begin
          txn.randomize();
          txn.parity = ~txn.calculate_parity();  // Wrong parity
          `uvm_info(get_type_name(), "Injecting bad parity", UVM_MEDIUM)
        end
        
        OUT_OF_RANGE: begin
          txn.data = 32'hFFFF_FFFF + 1;  // Overflow
          `uvm_info(get_type_name(), "Injecting out-of-range data", UVM_MEDIUM)
        end
      endcase
      
      finish_item(txn);
    end
  endtask
  
endclass

`endif // SAMPLE_SEQ_LIB_DEF
