/**
 * File: sample_agent.sv
 * Brief: Sample UVM agent for documentation testing
 * 
 * Class: sample_agent
 * 
 * Brief: A simple UVM agent that demonstrates typical UVM structure
 * 
 * Module       : sample_design
 * Design File  : sample_design.v
 * Project      : test_project
 * 
 * This agent contains a driver, monitor, and sequencer for managing
 * transactions in the verification environment.
 */

`ifndef SAMPLE_AGENT_DEF
`define SAMPLE_AGENT_DEF

class sample_agent extends uvm_agent;
  
  `uvm_component_utils(sample_agent)
  
  // Agent components
  sample_driver    driver;
  sample_monitor   monitor;
  sample_sequencer sequencer;
  
  // Configuration
  sample_agent_cfg cfg;
  
  /**
   * Constructor: new
   * Creates a new instance of the agent
   * 
   * Parameters:
   *   name - Instance name
   *   parent - Parent component
   */
  function new(string name = "sample_agent", uvm_component parent = null);
    super.new(name, parent);
  endfunction
  
  /**
   * Function: build_phase
   * Builds the agent components
   * 
   * Parameters:
   *   phase - Build phase object
   */
  virtual function void build_phase(uvm_phase phase);
    super.build_phase(phase);
    
    if (!uvm_config_db#(sample_agent_cfg)::get(this, "", "cfg", cfg)) begin
      `uvm_fatal(get_type_name(), "Failed to get agent configuration")
    end
    
    monitor = sample_monitor::type_id::create("monitor", this);
    
    if (cfg.is_active == UVM_ACTIVE) begin
      driver = sample_driver::type_id::create("driver", this);
      sequencer = sample_sequencer::type_id::create("sequencer", this);
    end
  endfunction
  
  /**
   * Function: connect_phase
   * Connects driver to sequencer
   * 
   * Parameters:
   *   phase - Connect phase object
   */
  virtual function void connect_phase(uvm_phase phase);
    super.connect_phase(phase);
    
    if (cfg.is_active == UVM_ACTIVE) begin
      driver.seq_item_port.connect(sequencer.seq_item_export);
    end
  endfunction
  
  /**
   * Task: run_phase
   * Main run phase for the agent
   * 
   * Parameters:
   *   phase - Run phase object
   */
  virtual task run_phase(uvm_phase phase);
    `uvm_info(get_type_name(), "Sample agent running", UVM_LOW)
  endtask
  
endclass

`endif // SAMPLE_AGENT_DEF
