"""
Enhanced SystemVerilog to RST Parser with Collapsible Source Code
Parses SystemVerilog files and generates Sphinx RST documentation files.

Comment Style Support:
- Block comments (/* ... */) for functions, tasks, and classes
- Single-line comments (//) before variables for descriptions
- Inline comments (//< ) on same line as code

Advanced Features:
- Collapsible source code blocks for functions and tasks
"""
import os
import re
import argparse
from pathlib import Path

class SVParser:
   """Parser for SystemVerilog files to extract documentation."""
   def __init__(self):
       # Regex patterns for SystemVerilog constructs
       # Updated module pattern to handle both modules with ports and without ports (testbenches)
       # Matches: module name(...); or module name;
       self.module_pattern = re.compile(
           r'^\s*module\s+(\w+)\s*(?:#\s*\(.*?\))?\s*(?:\((.*?)\))?\s*;',
           re.MULTILINE | re.DOTALL
       )
       # Updated class pattern to handle parameterized classes across multiple lines
       self.class_pattern = re.compile(
           r'^\s*class\s+(\w+)\s*(?:#[^;]*)?(?:\s+extends\s+(\w+))?[^;]*;',
           re.MULTILINE | re.DOTALL
       )
       # Updated function pattern to handle ClassName::method_name syntax and exclude extern
       self.function_pattern = re.compile(
           r'^\s*(?!extern\s)(?:virtual\s+)?(?:static\s+)?function\s+(?:\w+\s+)?(?:\w+::)?(\w+)\s*\(',
           re.MULTILINE
       )
       # Updated task pattern to handle ClassName::method_name syntax and exclude extern  
       self.task_pattern = re.compile(
           r'^\s*(?!extern\s)(?:virtual\s+)?(?:static\s+)?task\s+(?:\w+::)?(\w+)\s*\(',
           re.MULTILINE
       )
       self.interface_pattern = re.compile(
           r'^\s*interface\s+(\w+)\s*(?:#\s*\(.*?\))?\s*\((.*?)\);',
           re.MULTILINE | re.DOTALL
       )
       # Package pattern - matches: package name;
       self.package_pattern = re.compile(
           r'^\s*package\s+(\w+)\s*;',
           re.MULTILINE
       )
       # Variable declaration pattern
       self.variable_pattern = re.compile(
           r'^\s*(?:local\s+|protected\s+|static\s+)?(?:const\s+)?'
           r'(?:rand\s+|randc\s+)?'
           r'(\w+(?:\s*\[.*?\])?)\s+(\w+)'
           r'(?:\s*=\s*[^;]+)?;',
           re.MULTILINE
       )
   def extract_block_comment(self, content, position):
       """Extract block comment (/* ... */) immediately before a construct."""
       before_content = content[:position]
       block_comment_pattern = re.compile(r'/\*(.*?)\*/', re.DOTALL)
       matches = list(block_comment_pattern.finditer(before_content))
       if matches:
           last_match = matches[-1]
           if position - last_match.end() < 200:
               comment_text = last_match.group(1)
               lines = comment_text.split('\n')
               cleaned_lines = []
               for line in lines:
                   line = line.strip()
                   line = re.sub(r'^\*+\s*', '', line)
                   if line:
                       cleaned_lines.append(line)
               return '\n\n'.join(cleaned_lines)
       return ""
   
   def extract_function_description(self, content, func_start, func_end):
       """Extract description from block comment inside or before function body."""
       # First try to find comment before the function
       before_desc = self.extract_block_comment(content, func_start)
       if before_desc and before_desc != "":
           return before_desc
       
       # If not found before, look inside the function body for the first block comment
       # This handles cases like: function name(); /* Brief: description */ ...
       func_content = content[func_start:func_end]
       block_comment_pattern = re.compile(r'/\*\s*(.*?)\s*\*/', re.DOTALL)
       match = block_comment_pattern.search(func_content)
       if match:
           comment_text = match.group(1)
           lines = comment_text.split('\n')
           cleaned_lines = []
           for line in lines:
               line = line.strip()
               line = re.sub(r'^\*+\s*', '', line)
               # Extract "Brief: " descriptions
               if 'Brief:' in line or 'brief:' in line:
                   line = re.sub(r'(?i)^\s*Brief:\s*', '', line)
               if line:
                   cleaned_lines.append(line)
           return '\n\n'.join(cleaned_lines)
       return "No description available."
   def extract_line_comment(self, content, position):
       """Extract single-line comments (//) immediately before a construct."""
       before_content = content[:position]
       lines = before_content.split('\n')
       comment_lines = []
       for line in reversed(lines[-10:]):
           stripped = line.strip()
           if stripped.startswith('//') and not stripped.startswith('//<'):
               comment_text = stripped[2:].strip()
               comment_lines.insert(0, comment_text)
           elif not stripped:
               continue
           else:
               break
       return ' '.join(comment_lines) if comment_lines else ""
   def extract_inline_comment(self, line):
       """Extract inline comment (//< ) from the same line."""
       if '//<' in line:
           parts = line.split('//<', 1)
           return parts[1].strip()
       return ""
   def parse_variables(self, content, start_pos, end_pos):
       """Parse variable declarations within a code block."""
       block_content = content[start_pos:end_pos]
       
       # Remove typedef blocks to avoid parsing struct members as class variables
       # Match typedef struct { ... } name;
       typedef_pattern = re.compile(r'typedef\s+(?:struct|enum)\s*\{[^}]*\}[^;]*;', re.DOTALL)
       clean_content = typedef_pattern.sub('', block_content)
       
       variables = []
       for match in self.variable_pattern.finditer(clean_content):
           var_type = match.group(1).strip()
           var_name = match.group(2).strip()
           
           # Skip if it looks like a macro or inside a comment block
           line_start = clean_content.rfind('\n', 0, match.start()) + 1
           line_end = clean_content.find('\n', match.end())
           if line_end == -1:
               line_end = len(clean_content)
           full_line = clean_content[line_start:line_end]
           
           # Skip lines that start with backtick (UVM macros)
           if full_line.strip().startswith('`'):
               continue
               
           # Skip lines inside comments
           if '/*' in clean_content[:match.start()] and '*/' in clean_content[match.end():]:
               # Could be inside a block comment
               last_open = clean_content[:match.start()].rfind('/*')
               last_close = clean_content[:match.start()].rfind('*/')
               if last_open > last_close:
                   continue  # Inside a comment block
           
           inline_comment = self.extract_inline_comment(full_line)
           preceding_comment = self.extract_line_comment(
               content[:start_pos + match.start()],
               len(content[:start_pos + match.start()])
           )
           description = inline_comment if inline_comment else preceding_comment
           if not description:
               description = "No description"
           variables.append({
               'type': var_type,
               'name': var_name,
               'description': description
           })
       
       # Debug: print first few variables for large class
       if len(block_content) > 5000 and variables:
           print(f"  [DEBUG] First 5 variables: {[(v['type'], v['name']) for v in variables[:5]]}")
       
       return variables
       return variables
   def find_block_boundaries(self, content, start_pos):
       """Find the start and end positions of a code block."""
       brace_count = 0
       in_block = False
       block_start = -1
       for i in range(start_pos, len(content)):
           if content[i] == '{':
               if not in_block:
                   block_start = i
                   in_block = True
               brace_count += 1
           elif content[i] == '}':
               brace_count -= 1
               if brace_count == 0 and in_block:
                   return block_start, i + 1
       return -1, -1
   def extract_signature(self, content, match_start):
       """
       Generic method to extract any SystemVerilog construct signature (declaration line).
       Works for classes, modules, interfaces, functions, tasks, etc.
       """
       # Find the signature start - look for the beginning of the line
       line_start = content.rfind('\n', 0, match_start)
       if line_start == -1:
           line_start = 0
       else:
           line_start += 1  # Skip the newline itself
       
       # Find the semicolon that ends the signature
       semicolon_pos = content.find(';', match_start)
       if semicolon_pos != -1:
           signature = content[line_start:semicolon_pos + 1].strip()
           # Clean up excessive whitespace while preserving structure
           signature = ' '.join(signature.split())
           return signature
       return ""
   
   def extract_function_body(self, content, match_start, match_end):
       """Extract the complete function/task body including signature."""
       # Find the function signature start - look for the beginning of the line
       line_start = content.rfind('\n', 0, match_start)
       if line_start == -1:
           line_start = 0
       else:
           line_start += 1  # Skip the newline itself
       
       # Find the endfunction/endtask
       end_pattern = re.compile(r'\bend(?:function|task)\b', re.IGNORECASE)
       end_match = end_pattern.search(content, match_end)
       if end_match:
           # Also check for optional label after endfunction/endtask
           label_end = end_match.end()
           # Look for optional ": label_name"
           label_pattern = re.compile(r'\s*:\s*\w+', re.IGNORECASE)
           label_match = label_pattern.match(content, label_end)
           if label_match:
               body_end = label_match.end()
           else:
               body_end = label_end
           # Get complete source
           source = content[line_start:body_end].strip()
           return source
       return ""
   
   def extract_module_body(self, content, match_start, match_end):
       """Extract the complete module body including declaration."""
       # Find the module declaration start - look for the beginning of the line
       line_start = content.rfind('\n', 0, match_start)
       if line_start == -1:
           line_start = 0
       else:
           line_start += 1  # Skip the newline itself
       
       # Find the endmodule
       end_pattern = re.compile(r'\bendmodule\b', re.IGNORECASE)
       end_match = end_pattern.search(content, match_end)
       if end_match:
           # Also check for optional label after endmodule
           label_end = end_match.end()
           # Look for optional ": label_name"
           label_pattern = re.compile(r'\s*:\s*\w+', re.IGNORECASE)
           label_match = label_pattern.match(content, label_end)
           if label_match:
               body_end = label_match.end()
           else:
               body_end = label_end
           # Get complete source
           source = content[line_start:body_end].strip()
           return source
       return ""
   
   def extract_package_body(self, content, match_start, match_end):
       """Extract the complete package body including declaration."""
       # Find the package declaration start - look for the beginning of the line
       line_start = content.rfind('\n', 0, match_start)
       if line_start == -1:
           line_start = 0
       else:
           line_start += 1  # Skip the newline itself
       
       # Find the endpackage
       end_pattern = re.compile(r'\bendpackage\b', re.IGNORECASE)
       end_match = end_pattern.search(content, match_end)
       if end_match:
           # Also check for optional label after endpackage
           label_end = end_match.end()
           # Look for optional ": label_name"
           label_pattern = re.compile(r'\s*:\s*\w+', re.IGNORECASE)
           label_match = label_pattern.match(content, label_end)
           if label_match:
               body_end = label_match.end()
           else:
               body_end = label_end
           # Get complete source
           source = content[line_start:body_end].strip()
           return source
       return ""
   
   def extract_interface_body(self, content, match_start, match_end):
       """Extract the complete interface body including declaration."""
       # Find the interface declaration start - look for the beginning of the line
       line_start = content.rfind('\n', 0, match_start)
       if line_start == -1:
           line_start = 0
       else:
           line_start += 1  # Skip the newline itself
       
       # Find the endinterface
       end_pattern = re.compile(r'\bendinterface\b', re.IGNORECASE)
       end_match = end_pattern.search(content, match_end)
       if end_match:
           # Also check for optional label after endinterface
           label_end = end_match.end()
           # Look for optional ": label_name"
           label_pattern = re.compile(r'\s*:\s*\w+', re.IGNORECASE)
           label_match = label_pattern.match(content, label_end)
           if label_match:
               body_end = label_match.end()
           else:
               body_end = label_end
           # Get complete source
           source = content[line_start:body_end].strip()
           return source
       return ""
   def parse_file(self, filepath):
       """Parse a single SystemVerilog file."""
       with open(filepath, 'r', encoding='utf-8', errors='ignore') as f:
           content = f.read()
       results = {
           'modules': [],
           'packages': [],
           'classes': [],
           'interfaces': [],
           'functions': [],
           'tasks': []
       }
       # Find modules
       for match in self.module_pattern.finditer(content):
           module_name = match.group(1)
           ports = match.group(2).strip() if match.group(2) else ""
           # Extract signature
           signature = self.extract_signature(content, match.start())
           comment = self.extract_block_comment(content, match.start())
           
           # Note: We no longer extract module source code individually since
           # the full file source is displayed at the top of each document
           
           # Find the module body - SystemVerilog modules use 'endmodule', not braces
           # Look for endmodule keyword to extract variables
           endmodule_pattern = re.compile(r'\bendmodule\b', re.IGNORECASE)
           endmodule_match = endmodule_pattern.search(content, match.end())
           if endmodule_match:
               module_body_end = endmodule_match.start()
               variables = self.parse_variables(content, match.end(), module_body_end)
           else:
               variables = []
           
           results['modules'].append({
               'name': module_name,
               'signature': signature,
               'ports': ports,
               'description': comment,
               'variables': variables
           })
       
       # Find packages
       for match in self.package_pattern.finditer(content):
           package_name = match.group(1)
           comment = self.extract_block_comment(content, match.start())
           
           # Note: We don't extract package source code individually since
           # the full file source is displayed at the top of each document
           
           results['packages'].append({
               'name': package_name,
               'description': comment
           })
       
       # Find classes
       for match in self.class_pattern.finditer(content):
           class_name = match.group(1)
           parent = match.group(2) if match.group(2) else None
           # Extract class signature
           signature = self.extract_signature(content, match.start())
           comment = self.extract_block_comment(content, match.start())
           
           # Find the class body - SystemVerilog classes use 'endclass', not braces
           # Look for endclass keyword
           endclass_pattern = re.compile(r'\bendclass\b', re.IGNORECASE)
           endclass_match = endclass_pattern.search(content, match.end())
           if endclass_match:
               class_body_end = endclass_match.start()
               variables = self.parse_variables(content, match.end(), class_body_end)
           else:
               variables = []
           
           results['classes'].append({
               'name': class_name,
               'parent': parent,
               'signature': signature,
               'description': comment,
               'variables': variables
           })
       # Find interfaces
       for match in self.interface_pattern.finditer(content):
           interface_name = match.group(1)
           ports = match.group(2).strip() if match.group(2) else ""
           # Extract signature
           signature = self.extract_signature(content, match.start())
           comment = self.extract_block_comment(content, match.start())
           
           # Note: We no longer extract interface source code individually since
           # the full file source is displayed at the top of each document
           
           results['interfaces'].append({
               'name': interface_name,
               'signature': signature,
               'ports': ports,
               'description': comment,
               'variables': []
           })
       # Find functions
       for match in self.function_pattern.finditer(content):
           func_name = match.group(1)
           # Extract signature (first line only)
           signature = self.extract_signature(content, match.start())
           # Extract full source code
           source_code = self.extract_function_body(content, match.start(), match.end())
           # Extract description from inside the function body
           func_end_pos = match.start() + len(source_code) if source_code else match.end()
           comment = self.extract_function_description(content, match.start(), func_end_pos)
           
           block_start, block_end = self.find_block_boundaries(content, match.end())
           variables = []
           if block_start != -1:
               variables = self.parse_variables(content, block_start, block_end)
           
           results['functions'].append({
               'name': func_name,
               'signature': signature,
               'description': comment,
               'variables': variables,
               'source': source_code,
               'position': match.start()  # Add position for multi-class files
           })
       # Find tasks
       for match in self.task_pattern.finditer(content):
           task_name = match.group(1)
           # Extract signature (first line only)
           signature = self.extract_signature(content, match.start())
           # Extract full source code
           source_code = self.extract_function_body(content, match.start(), match.end())
           # Extract description from inside the task body
           task_end_pos = match.start() + len(source_code) if source_code else match.end()
           comment = self.extract_function_description(content, match.start(), task_end_pos)
           
           block_start, block_end = self.find_block_boundaries(content, match.end())
           variables = []
           if block_start != -1:
               variables = self.parse_variables(content, block_start, block_end)
           
           results['tasks'].append({
               'name': task_name,
               'signature': signature,
               'description': comment,
               'variables': variables,
               'source': source_code,
               'position': match.start()  # Add position for multi-class files
           })
       return results

class RSTGenerator:
   """Generate RST files from parsed SystemVerilog data."""
   def __init__(self, output_dir):
       self.output_dir = Path(output_dir)
       self.output_dir.mkdir(parents=True, exist_ok=True)
       self.index_entries = []
   def write_variables(self, f, variables, indent=""):
       """Write variables section as a table."""
       if variables:
           f.write(f"{indent}**Variables:**\n\n")
           
           # Write table without extra indentation - RST tables should be at document level
           f.write(f"{indent}.. list-table::\n")
           f.write(f"{indent}   :header-rows: 1\n")
           f.write(f"{indent}   :widths: 25 25 50\n\n")
           f.write(f"{indent}   * - Type\n")
           f.write(f"{indent}     - Variable Name\n")
           f.write(f"{indent}     - Description\n")
           
           # Write table rows
           for var in variables:
               f.write(f"{indent}   * - ``{var['type']}``\n")
               f.write(f"{indent}     - ``{var['name']}``\n")
               f.write(f"{indent}     - {var['description']}\n")
           f.write("\n")
   def parse_interface_ports(self, ports_string):
       """Parse interface ports and categorize into input/output with details."""
       input_ports = []
       output_ports = []
       
       if not ports_string:
           return input_ports, output_ports
       
       # Split by commas, but be careful with array dimensions
       lines = ports_string.split('\n')
       for line in lines:
           line = line.strip()
           if not line or line.startswith('//') or line.startswith('/*'):
               continue
           
           # Extract inline comment (description)
           description = ""
           if '//<' in line:
               parts = line.split('//<', 1)
               line = parts[0].strip()
               description = parts[1].strip()
           
           # Remove trailing comma
           line = line.rstrip(',').strip()
           
           # Parse port: direction type [width] name
           # Pattern: (input|output) type [optional_width] name
           # Updated to handle "input logic [width] name" or "input logic name"
           port_pattern = re.match(
               r'(input|output)\s+(\w+)\s*(?:\[(.*?)\])?\s+(\w+)',
               line
           )
           
           if port_pattern:
               direction = port_pattern.group(1).strip()
               port_type = port_pattern.group(2).strip()
               width = port_pattern.group(3).strip() if port_pattern.group(3) else "1"
               name = port_pattern.group(4).strip()
               
               port_info = {
                   'type': port_type,
                   'width': width if width != "1" else "1 bit",
                   'name': name,
                   'description': description
               }
               
               if direction == 'input':
                   input_ports.append(port_info)
               else:
                   output_ports.append(port_info)
       
       return input_ports, output_ports
   
   def write_port_table(self, f, ports, title):
       """Write a formatted port table."""
       if not ports:
           return
       
       f.write(f"**{title}:**\n\n")
       f.write(".. list-table::\n")
       f.write("   :header-rows: 1\n")
       f.write("   :widths: 15 20 20 45\n\n")
       f.write("   * - Type\n")
       f.write("     - Width\n")
       f.write("     - Name\n")
       f.write("     - Description\n")
       
       for port in ports:
           f.write(f"   * - ``{port['type']}``\n")
           f.write(f"     - ``{port['width']}``\n")
           f.write(f"     - ``{port['name']}``\n")
           f.write(f"     - {port['description']}\n")
       
       f.write("\n")
   
   def write_collapsible_source(self, f, source_code, indent=""):
       """Write collapsible source code block using HTML."""
       if source_code:
           f.write(f"{indent}.. raw:: html\n\n")
           f.write(f"{indent}   <details>\n")
           f.write(f"{indent}   <summary><b>📄 View Source Code</b></summary>\n\n")
           f.write(f"{indent}.. code-block:: systemverilog\n")
           f.write(f"{indent}   :linenos:\n\n")
           for line in source_code.split('\n'):
               f.write(f"{indent}   {line}\n")
           f.write(f"\n")
           f.write(f"{indent}.. raw:: html\n\n")
           f.write(f"{indent}   </details>\n\n")
   def generate_rst_for_file(self, filepath, parsed_data):
       """Generate RST file for a single SystemVerilog file."""
       filename = Path(filepath).stem
       rst_filename = self.output_dir / f"{filename}.rst"
       
       # Read the complete source file content
       with open(filepath, 'r', encoding='utf-8', errors='ignore') as source_file:
           full_source = source_file.read()
       
       # Check if this is a multi-class file (like seq_lib files)
       is_multi_class = len(parsed_data['classes']) > 1
       # Check if this is specifically a sequence library file
       is_seq_lib = 'seq_lib' in filename.lower()
       
       with open(rst_filename, 'w', encoding='utf-8') as f:
           # Write title - just use the filename without "Documentation"
           title = f"{filename}"
           f.write(f"{title}\n")
           f.write("=" * len(title) + "\n\n")
           
           # Add tocdepth metadata (hidden field list for TOC depth control)
           # This tells Sphinx to include up to 5 levels of headings in the TOC
           # Field lists at document top are metadata and won't be rendered
           if is_multi_class:
               f.write(".. meta::\n")
               f.write("   :tocdepth: 5\n\n")
           
           f.write(f"Source file: ``{Path(filepath).name}``\n\n")
           
           # Add collapsible full file source code
           f.write(".. raw:: html\n\n")
           f.write("   <details>\n")
           f.write("   <summary><b>📂 View Complete File Source Code</b></summary>\n\n")
           f.write(".. code-block:: systemverilog\n")
           f.write("   :linenos:\n\n")
           for line in full_source.split('\n'):
               f.write(f"   {line}\n")
           f.write("\n")
           f.write(".. raw:: html\n\n")
           f.write("   </details>\n\n")
           f.write("----\n\n")  # Add horizontal separator
           
           # Write modules
           if parsed_data['modules']:
               f.write("Modules\n")
               f.write("-" * 7 + "\n\n")
               for module in parsed_data['modules']:
                   f.write(f"**{module['name']}**\n\n")
                   # Add signature in a highlighted code block (like Python docs)
                   if module.get('signature'):
                       f.write(".. code-block:: systemverilog\n\n")
                       f.write(f"   {module['signature']}\n\n")
                   f.write(f"{module['description']}\n\n")
                   if module['ports']:
                       f.write(f"   **Ports:**\n\n")
                       f.write(f"   .. code-block:: verilog\n\n")
                       for line in module['ports'].split('\n'):
                           f.write(f"      {line}\n")
                       f.write("\n")
                   self.write_variables(f, module['variables'])
                   # Note: Module source code not added here since full file source is at the top
           
           # Write packages
           if parsed_data['packages']:
               f.write("Packages\n")
               f.write("-" * 8 + "\n\n")
               for pkg in parsed_data['packages']:
                   f.write(f"**package {pkg['name']}**\n\n")
                   f.write(f"{pkg['description']}\n\n")
                   # Note: Package source code not added here since full file source is at the top
           
           # Write classes - different format for multi-class files
           if parsed_data['classes']:
               if is_multi_class:
                   # Multi-class file: Create section header for "Classes" or "Sequences"
                   # and subsections for each class
                   section_name = "Sequences" if is_seq_lib else "Classes"
                   f.write(f"{section_name}\n")
                   f.write("-" * len(section_name) + "\n\n")
                   
                   for i, cls in enumerate(parsed_data['classes']):
                       # Each class gets its own section (will appear in TOC)
                       class_title = f"{cls['name']}"
                       f.write(f"{class_title}\n")
                       f.write("^" * len(class_title) + "\n\n")
                       
                       # Add class signature in a highlighted code block (like Python docs)
                       if cls.get('signature'):
                           f.write(".. code-block:: systemverilog\n\n")
                           f.write(f"   {cls['signature']}\n\n")
                       
                       # For seq_lib files, add collapsible source code for each sequence RIGHT AFTER the header
                       if is_seq_lib:
                           # Extract the source code for this specific class
                           class_source = self._extract_class_source(filepath, cls['name'])
                           if class_source:
                               self.write_collapsible_source(f, class_source)
                       
                       if cls['parent']:
                           f.write(f"*Extends:* ``{cls['parent']}``\n\n")
                       f.write(f"{cls['description']}\n\n")
                       
                       self.write_variables(f, cls['variables'])
                       
                       # Write functions for this class
                       class_functions = [func for func in parsed_data['functions'] 
                                        if self._belongs_to_class(func, cls['name'], filepath)]
                       if class_functions:
                           f.write("Functions\n")
                           f.write('"' * 9 + "\n\n")
                           for j, func in enumerate(class_functions):
                               # Add anchor for cross-referencing with class name to ensure uniqueness
                               f.write(f".. _{cls['name']}-{func['name']}:\n\n")
                               # Use subsection header for function names
                               func_title = f"{func['name']}"
                               f.write(f"{func_title}\n")
                               f.write("~" * len(func_title) + "\n\n")
                               # Add signature in a highlighted code block (like Python docs)
                               if func.get('signature'):
                                   f.write(".. code-block:: systemverilog\n\n")
                                   f.write(f"   {func['signature']}\n\n")
                               f.write(f"{func['description']}\n\n")
                               self.write_variables(f, func['variables'])
                               self.write_collapsible_source(f, func.get('source', ''))
                               # Add divider after each function except the last one
                               if j < len(class_functions) - 1:
                                   f.write("----\n\n")
                       
                       # Write tasks for this class
                       class_tasks = [task for task in parsed_data['tasks']
                                    if self._belongs_to_class(task, cls['name'], filepath)]
                       if class_tasks:
                           # Add divider before Tasks section if there were functions above
                           if class_functions:
                               f.write("----\n\n")
                           f.write("Tasks\n")
                           f.write('"' * 5 + "\n\n")
                           for j, task in enumerate(class_tasks):
                               # Add anchor for cross-referencing with class name to ensure uniqueness
                               f.write(f".. _{cls['name']}-{task['name']}:\n\n")
                               # Use subsection header for task names
                               task_title = f"{task['name']}"
                               f.write(f"{task_title}\n")
                               f.write("~" * len(task_title) + "\n\n")
                               # Add signature in a highlighted code block (like Python docs)
                               if task.get('signature'):
                                   f.write(".. code-block:: systemverilog\n\n")
                                   f.write(f"   {task['signature']}\n\n")
                               f.write(f"{task['description']}\n\n")
                               self.write_variables(f, task['variables'])
                               self.write_collapsible_source(f, task.get('source', ''))
                               # Add divider after each task except the last one
                               if j < len(class_tasks) - 1:
                                   f.write("----\n\n")
                       
                       # Add spacing and divider between sequences/classes (except after the last one)
                       if i < len(parsed_data['classes']) - 1:
                           f.write("\n----\n\n")
               else:
                   # Single class file: Keep original format
                   f.write("Classes\n")
                   f.write("-" * 7 + "\n\n")
                   for cls in parsed_data['classes']:
                       f.write(f"**{cls['name']}**\n\n")
                       # Add class signature in a highlighted code block (like Python docs)
                       if cls.get('signature'):
                           f.write(".. code-block:: systemverilog\n\n")
                           f.write(f"   {cls['signature']}\n\n")
                       if cls['parent']:
                           f.write(f"*Extends:* ``{cls['parent']}``\n\n")
                       f.write(f"{cls['description']}\n\n")
                       
                       self.write_variables(f, cls['variables'])
           # Write interfaces
           if parsed_data['interfaces']:
               f.write("Interfaces\n")
               f.write("-" * 10 + "\n\n")
               for iface in parsed_data['interfaces']:
                   f.write(f"**{iface['name']}**\n\n")
                   # Add interface signature in a highlighted code block (like Python docs)
                   if iface.get('signature'):
                       f.write(".. code-block:: systemverilog\n\n")
                       f.write(f"   {iface['signature']}\n\n")
                   f.write(f"{iface['description']}\n\n")
                   
                   # Parse and display ports in separate tables for input/output
                   if iface['ports']:
                       input_ports, output_ports = self.parse_interface_ports(iface['ports'])
                       self.write_port_table(f, input_ports, "Input Ports")
                       self.write_port_table(f, output_ports, "Output Ports")
                   
                   self.write_variables(f, iface['variables'])
                   # Note: Interface source code not added here since full file source is at the top
           
           # Write functions with collapsible source (only for single-class files)
           if parsed_data['functions'] and not is_multi_class:
               f.write("Functions\n")
               f.write("-" * 9 + "\n\n")
               for i, func in enumerate(parsed_data['functions']):
                   # Add anchor for cross-referencing with file-specific prefix
                   f.write(f".. _{filename}-{func['name']}:\n\n")
                   # Use subsection header (~~~) so it appears in sidebar TOC
                   func_title = f"{func['name']}"
                   f.write(f"{func_title}\n")
                   f.write("~" * len(func_title) + "\n\n")
                   # Add signature in a highlighted code block (like Python docs)
                   if func.get('signature'):
                       f.write(".. code-block:: systemverilog\n\n")
                       f.write(f"   {func['signature']}\n\n")
                   f.write(f"{func['description']}\n\n")
                   self.write_variables(f, func['variables'])
                   self.write_collapsible_source(f, func.get('source', ''))
                   # Add divider after each function except the last one
                   if i < len(parsed_data['functions']) - 1:
                       f.write("----\n\n")
           
           # Write tasks with collapsible source (only for single-class files)
           if parsed_data['tasks'] and not is_multi_class:
               # Add divider before Tasks section if there were functions above
               if parsed_data['functions']:
                   f.write("----\n\n")
               f.write("Tasks\n")
               f.write("-" * 5 + "\n\n")
               for i, task in enumerate(parsed_data['tasks']):
                   # Add anchor for cross-referencing with file-specific prefix
                   f.write(f".. _{filename}-{task['name']}:\n\n")
                   # Use subsection header (~~~) so it appears in sidebar TOC
                   task_title = f"{task['name']}"
                   f.write(f"{task_title}\n")
                   f.write("~" * len(task_title) + "\n\n")
                   # Add signature in a highlighted code block (like Python docs)
                   if task.get('signature'):
                       f.write(".. code-block:: systemverilog\n\n")
                       f.write(f"   {task['signature']}\n\n")
                   f.write(f"{task['description']}\n\n")
                   self.write_variables(f, task['variables'])
                   self.write_collapsible_source(f, task.get('source', ''))
                   # Add divider after each task except the last one
                   if i < len(parsed_data['tasks']) - 1:
                       f.write("----\n\n")
       self.index_entries.append(filename)
       print(f"Generated: {rst_filename}")
   
   def _extract_class_source(self, filepath, class_name):
       """Extract source code for a specific class from the file."""
       try:
           with open(filepath, 'r', encoding='utf-8', errors='ignore') as f:
               content = f.read()
           
           # Find class declaration
           class_pattern = re.compile(rf'^\s*class\s+{class_name}\b', re.MULTILINE)
           class_match = class_pattern.search(content)
           if not class_match:
               return ""
           
           # Find the endclass for this specific class
           class_start = class_match.start()
           
           # Find line start
           line_start = content.rfind('\n', 0, class_start)
           if line_start == -1:
               line_start = 0
           else:
               line_start += 1
           
           # Find the endclass
           endclass_pattern = re.compile(r'\bendclass\b', re.MULTILINE | re.IGNORECASE)
           endclass_match = endclass_pattern.search(content, class_match.end())
           if endclass_match:
               # Check for optional label after endclass
               label_end = endclass_match.end()
               label_pattern = re.compile(r'\s*:\s*\w+', re.IGNORECASE)
               label_match = label_pattern.match(content, label_end)
               if label_match:
                   body_end = label_match.end()
               else:
                   body_end = label_end
               
               # Extract the complete class source
               source = content[line_start:body_end].strip()
               return source
           
           return ""
       except Exception as e:
           print(f"Error extracting class {class_name} source: {e}")
           return ""
   
   def _belongs_to_class(self, func_or_task, class_name, filepath):
       """
       Determine if a function/task belongs to a specific class using position.
       """
       try:
           with open(filepath, 'r', encoding='utf-8', errors='ignore') as f:
               content = f.read()
           
           # Find class declaration
           class_pattern = re.compile(rf'\bclass\s+{class_name}\b', re.MULTILINE)
           class_match = class_pattern.search(content)
           if not class_match:
               return False
           
           # Find the endclass for this specific class
           endclass_pattern = re.compile(rf'\bendclass\s*(?::\s*{class_name})?\b', re.MULTILINE | re.IGNORECASE)
           endclass_match = endclass_pattern.search(content, class_match.end())
           if not endclass_match:
               return False
           
           # Check if function/task position is between class and endclass
           func_pos = func_or_task.get('position', -1)
           if func_pos == -1:
               return False
               
           class_pos = class_match.start()
           endclass_pos = endclass_match.start()
           
           return class_pos < func_pos < endclass_pos
       except Exception as e:
           return False
   
   def generate_index(self, section_title="SystemVerilog Modules"):
       """Generate index.rst with all parsed files."""
       # Determine index filename based on section title
       if "testcase" in section_title.lower():
           index_filename = "testcases.rst"
           caption = "Test Cases:"
       elif "environment" in section_title.lower():
           index_filename = "environment.rst"
           caption = "UVM Environment Components:"
       else:
           index_filename = "modules_index.rst"
           caption = "Modules:"
       
       index_file = self.output_dir.parent / index_filename
       
       # Extract module name from testbench file (remove _tb suffix)
       module_name = section_title  # Use provided section title
       
       with open(index_file, 'w', encoding='utf-8') as f:
           f.write(f"{module_name}\n")
           f.write("=" * len(module_name) + "\n\n")
           f.write(".. toctree::\n")
           f.write("   :maxdepth: 7\n")  # Increased to 7 to show individual function/task names in sidebar TOC
           f.write(f"   :caption: {caption}\n\n")
           for entry in sorted(self.index_entries):
               # Write relative path from source dir
               rel_path = Path(self.output_dir.name) / entry
               f.write(f"   {rel_path}\n")
       print(f"\nGenerated index: {index_file}")

def main():
   parser = argparse.ArgumentParser(
       description='Parse SystemVerilog files and generate Sphinx RST documentation with collapsible source'
   )
   parser.add_argument(
       'input_dir',
       help='Directory containing SystemVerilog files'
   )
   parser.add_argument(
       '-o', '--output',
       default='./rst_output',
       help='Output directory for RST files (default: ./rst_output)'
   )
   parser.add_argument(
       '-t', '--title',
       default='SystemVerilog Modules',
       help='Section title for the generated index (default: SystemVerilog Modules)'
   )
   args = parser.parse_args()
   input_path = Path(args.input_dir)
   if not input_path.exists():
       print(f"Error: Directory {input_path} does not exist")
       return
   # Find all .sv and .svh files
   sv_files = list(input_path.rglob('*.sv')) + list(input_path.rglob('*.svh'))
   if not sv_files:
       print(f"No SystemVerilog files found in {input_path}")
       return
   print(f"Found {len(sv_files)} SystemVerilog files\n")
   sv_parser = SVParser()
   rst_gen = RSTGenerator(args.output)
   for sv_file in sv_files:
       print(f"Parsing: {sv_file}")
       parsed_data = sv_parser.parse_file(sv_file)
       rst_gen.generate_rst_for_file(sv_file, parsed_data)
   rst_gen.generate_index(args.title)
   print("\nDone! RST files generated successfully.")

if __name__ == '__main__':
   main()