"""
Documentation generator module - handles Sphinx project setup and building.
"""
import os
import subprocess
from pathlib import Path
from typing import Optional, List


class DocGenerator:
    """Main documentation generator class."""
    
    def __init__(
        self,
        output_dir: str = "./docs",
        project_name: str = "SystemVerilog Documentation",
        author: str = "Your Name",
        version: str = "1.0",
        theme: str = "sphinx_rtd_theme"
    ):
        """
        Initialize documentation generator.
        
        Args:
            output_dir: Output directory for documentation
            project_name: Project name for documentation
            author: Author name
            version: Project version
            theme: Sphinx HTML theme
        """
        self.output_dir = Path(output_dir)
        self.source_dir = self.output_dir / "source"
        self.build_dir = self.output_dir / "build"
        self.project_name = project_name
        self.author = author
        self.version = version
        self.theme = theme
        
    def initialize_sphinx_dirs(self):
        """Create Sphinx directory structure."""
        self.source_dir.mkdir(parents=True, exist_ok=True)
        self.build_dir.mkdir(parents=True, exist_ok=True)
        (self.source_dir / "_static").mkdir(exist_ok=True)
        (self.source_dir / "_templates").mkdir(exist_ok=True)
        print(f"✓ Created Sphinx directories in {self.output_dir}")
        
    def generate_conf_py(self, sv_env_dir: Optional[str] = None, sv_tests_dir: Optional[str] = None, 
                         py_env_dir: Optional[str] = None, py_tests_dir: Optional[str] = None):
        """Generate conf.py configuration file."""
        conf_content = f"""# Configuration file for the Sphinx documentation builder.

import os
import sys

# Project information
project = '{self.project_name}'
copyright = '2025, {self.author}'
author = '{self.author}'
version = '{self.version}'
release = '{self.version}'

# General configuration
extensions = [
    'sphinx.ext.autodoc',       # Auto-generate from docstrings
    'sphinx.ext.viewcode',      # Add links to source code
    'sphinx.ext.napoleon',      # Google/NumPy style docstrings
    'sphinx_verilog_domain',
]

templates_path = ['_templates']
exclude_patterns = []

# HTML output options
html_theme = '{self.theme}'
html_static_path = ['_static']

# Theme customization options
html_theme_options = {{
    # Navigation and Display
    'navigation_depth': -1,              # Show all levels (-1 = unlimited)
    'collapse_navigation': False,        # Keep navigation expanded
    'sticky_navigation': True,           # Sidebar scrolls with content
    'includehidden': True,               # Include hidden toctrees
    'titles_only': False,                # Show subheadings in TOC
    'prev_next_buttons_location': 'both',  # Show prev/next at top and bottom
}}

# Expand content width to use more screen space
html_css_files = ['custom.css']

# Show TOC entries for objects (functions, classes, etc.)
toc_object_entries = True

# Syntax highlighting style
pygments_style = 'sphinx'  # Options: sphinx, monokai, solarized-dark, etc.

# SystemVerilog source directories
verilog_src_dirs = ["""
        
        if sv_env_dir:
            conf_content += f"'{sv_env_dir}', "
        if sv_tests_dir:
            conf_content += f"'{sv_tests_dir}', "
            
        conf_content += """]

# Python module paths for autodoc
# Set environment variable to indicate we're building documentation
os.environ['SPHINX_BUILD'] = '1'

"""
        
        # Calculate the correct Python path based on provided directories
        if py_env_dir or py_tests_dir:
            # Use the first available Python directory to calculate the parent path
            python_dir = py_env_dir if py_env_dir else py_tests_dir
            # Convert to absolute path and get parent directory (where the package is)
            python_path = Path(python_dir).resolve().parent
            # Calculate relative path from docs/source to the Python package parent
            source_path = Path(self.source_dir).resolve()
            try:
                rel_path = os.path.relpath(python_path, source_path)
                conf_content += f"\n# Add Python package directory to path\n"
                conf_content += f"sys.path.insert(0, os.path.abspath('{rel_path}'))\n"
            except ValueError:
                # If relative path calculation fails (different drives on Windows), use absolute path
                conf_content += f"\n# Add Python package directory to path (absolute)\n"
                conf_content += f"sys.path.insert(0, r'{python_path}')\n"
        else:
            # Fallback for backward compatibility
            conf_content += "\nsys.path.insert(0, os.path.abspath('../../..'))\n"
            
        conf_content += """
# Napoleon settings for Google/NumPy style docstrings
napoleon_google_docstring = True
napoleon_numpy_docstring = True
napoleon_include_init_with_doc = True
napoleon_include_private_with_doc = False
napoleon_include_special_with_doc = True
napoleon_use_admonition_for_examples = True
napoleon_use_admonition_for_notes = True
napoleon_use_admonition_for_references = True
napoleon_use_ivar = False
napoleon_use_param = True
napoleon_use_rtype = True

# Autodoc settings
autodoc_default_options = {
    'members': True,
    'member-order': 'bysource',
    'special-members': '__init__',
    'undoc-members': True,
    'show-inheritance': True,
}
"""
        
        conf_file = self.source_dir / "conf.py"
        conf_file.write_text(conf_content)
        print(f"✓ Generated {conf_file}")
        
    def generate_custom_css(self):
        """Generate custom CSS for wider content."""
        css_content = """/* Expand content width for better use of screen space */
.wy-nav-content {
    max-width: 99% !important;  /* Default is 800px, use 99% of screen */
}

/* Optional: Adjust sidebar width if needed */
.wy-nav-side {
    width: 300px;  /* Default is 300px */
}
"""
        css_file = self.source_dir / "_static" / "custom.css"
        css_file.write_text(css_content)
        print(f"✓ Generated custom CSS")
        
    def generate_index_rst(self, include_python: bool = False):
        """Generate index.rst file."""
        index_content = f"""{self.project_name}
{'=' * len(self.project_name)}

Welcome to the {self.project_name} UVM testbench documentation.

.. toctree::
   :maxdepth: 7
   :caption: Contents:

   environment
   testcases
"""
        
        if include_python:
            index_content += """   python_environment
   python_testcases
"""
        
        index_content += """
Indices and tables
==================

* :ref:`genindex`
* :ref:`modindex`
* :ref:`search`
"""
        
        index_file = self.source_dir / "index.rst"
        index_file.write_text(index_content)
        print(f"✓ Generated {index_file}")
        
    def build_html(self):
        """Build HTML documentation using sphinx-build."""
        print("\n📚 Building HTML documentation...")
        try:
            result = subprocess.run(
                ["sphinx-build", "-b", "html", str(self.source_dir), str(self.build_dir / "html")],
                capture_output=True,
                text=True
            )
            
            if result.returncode == 0:
                print("✓ HTML documentation built successfully")
                print(f"\n📖 Documentation available at: {self.build_dir / 'html' / 'index.html'}")
                return True
            else:
                print(f"✗ Build failed:\n{result.stderr}")
                return False
        except FileNotFoundError:
            print("✗ sphinx-build not found. Please install Sphinx: pip install sphinx")
            return False
            
    def open_in_browser(self):
        """Open generated documentation in web browser."""
        index_html = self.build_dir / "html" / "index.html"
        if index_html.exists():
            import webbrowser
            webbrowser.open(f"file://{index_html.absolute()}")
            print(f"✓ Opened documentation in browser")
        else:
            print(f"✗ Documentation not found at {index_html}")


# Re-export RSTGenerator from parser module for backwards compatibility
from .parser import RSTGenerator

__all__ = ["DocGenerator", "RSTGenerator"]
