#!/usr/bin/env python3
"""
Command-line interface for sv-sphinx-doc.
"""
import argparse
import sys
from pathlib import Path
from .parser import SVParser, RSTGenerator
from .generator import DocGenerator
from . import __version__


def detect_language(directory: str) -> str:
    """
    Detect the language(s) in a directory based on file extensions.
    
    Args:
        directory: Path to the directory to check
        
    Returns:
        'systemverilog', 'python', 'mixed', or 'unknown'
    """
    dir_path = Path(directory)
    if not dir_path.exists():
        print(f"⚠️  Warning: Directory {directory} does not exist")
        return 'unknown'
    
    # Check for SystemVerilog files
    has_sv = bool(list(dir_path.rglob('*.sv')) or list(dir_path.rglob('*.svh')))
    
    # Check for Python files
    has_py = bool(list(dir_path.rglob('*.py')))
    
    if has_sv and has_py:
        return 'mixed'
    elif has_sv:
        return 'systemverilog'
    elif has_py:
        return 'python'
    else:
        print(f"⚠️  Warning: No .sv, .svh, or .py files found in {directory}")
        return 'unknown'


def main():
    """Main entry point for CLI."""
    parser = argparse.ArgumentParser(
        description='Generate Sphinx documentation for SystemVerilog and Python UVM testbenches',
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  # Auto-detect language from file extensions
  sphinx-doc --env ./env --test ./tests

  # Multiple environment/test directories
  sphinx-doc -e ./env -e ./pyenv -t ./tests -t ./pytests

  # Generate RST only without building HTML
  sphinx-doc --env ./env --test ./tests --no-build

  # Build and open in browser automatically
  sphinx-doc --env ./env --test ./tests --view
        """
    )
    
    # Version flag
    parser.add_argument(
        '--version', '-V',
        action='version',
        version=f'sphinx-doc {__version__}',
        help='Show package version and exit'
    )
    
    # Language-agnostic arguments (auto-detect from file extensions)
    parser.add_argument(
        '--env', '-e',
        action='append',
        help='Path to environment directory (auto-detects .sv/.svh or .py files). Can be used multiple times.',
        type=str,
        dest='env_dirs'
    )
    parser.add_argument(
        '--test', '-t',
        action='append',
        help='Path to tests directory (auto-detects .sv/.svh or .py files). Can be used multiple times.',
        type=str,
        dest='test_dirs'
    )
    
    # Output and configuration
    parser.add_argument(
        '--output', '-o',
        default='./docs',
        help='Output directory for documentation (default: ./docs)',
        type=str
    )
    parser.add_argument(
        '--project-name',
        help='Project name for documentation (default: auto-detected from files)',
        type=str
    )
    parser.add_argument(
        '--author',
        default='Your Name',
        help='Author name (default: Your Name)',
        type=str
    )
    parser.add_argument(
        '--revision',
        default='1.0.0',
        help='Documentation revision/version (default: 1.0.0)',
        type=str
    )
    parser.add_argument(
        '--theme',
        default='sphinx_rtd_theme',
        help='Sphinx HTML theme (default: sphinx_rtd_theme)',
        type=str
    )
    
    # Action flags
    parser.add_argument(
        '--no-build',
        action='store_true',
        help='Skip HTML build (only generate RST files)'
    )
    parser.add_argument(
        '--view',
        action='store_true',
        help='Open documentation in browser after building'
    )
    
    args = parser.parse_args()
    
    # Auto-detect language from file extensions
    sv_env_dirs = []
    sv_test_dirs = []
    py_env_dirs = []
    py_test_dirs = []
    
    # Process auto-detect directories
    if args.env_dirs:
        for env_dir in args.env_dirs:
            lang = detect_language(env_dir)
            if lang == 'systemverilog':
                sv_env_dirs.append(env_dir)
            elif lang == 'python':
                py_env_dirs.append(env_dir)
            elif lang == 'mixed':
                sv_env_dirs.append(env_dir)
                py_env_dirs.append(env_dir)
    
    if args.test_dirs:
        for test_dir in args.test_dirs:
            lang = detect_language(test_dir)
            if lang == 'systemverilog':
                sv_test_dirs.append(test_dir)
            elif lang == 'python':
                py_test_dirs.append(test_dir)
            elif lang == 'mixed':
                sv_test_dirs.append(test_dir)
                py_test_dirs.append(test_dir)
    
    # Validate inputs
    if not sv_env_dirs and not sv_test_dirs and not py_env_dirs and not py_test_dirs:
        parser.error("At least one of --env or --test must be specified")
    
    # Auto-detect project name if not provided
    project_name = args.project_name
    if not project_name and sv_env_dirs:
        sv_env_path = Path(sv_env_dirs[0])
        # Look for testbench file
        tb_files = list(sv_env_path.glob("*_tb.sv"))
        if tb_files:
            project_name = tb_files[0].stem.replace('_tb', '') + " Documentation"
        else:
            project_name = "SystemVerilog Documentation"
    elif not project_name:
        project_name = "Documentation"
    
    print("=" * 60)
    print("Sphinx Doc - Documentation Generator")
    print("=" * 60)
    print(f"Project: {project_name}")
    print(f"Output: {args.output}")
    print("=" * 60)
    
    # Initialize documentation generator
    doc_gen = DocGenerator(
        output_dir=args.output,
        project_name=project_name,
        author=args.author,
        version=args.revision,
        theme=args.theme
    )
    
    # Step 1: Initialize Sphinx directories
    print("\n📁 Initializing Sphinx project...")
    doc_gen.initialize_sphinx_dirs()
    
    # Step 2: Generate configuration files
    print("\n⚙️  Generating configuration files...")
    # Pass first SV and Python directories for config (if any)
    first_sv_env = sv_env_dirs[0] if sv_env_dirs else None
    first_sv_test = sv_test_dirs[0] if sv_test_dirs else None
    first_py_env = py_env_dirs[0] if py_env_dirs else None
    first_py_test = py_test_dirs[0] if py_test_dirs else None
    doc_gen.generate_conf_py(first_sv_env, first_sv_test, first_py_env, first_py_test)
    doc_gen.generate_custom_css()
    
    # Step 3: Generate index
    has_python = bool(py_env_dirs or py_test_dirs)
    doc_gen.generate_index_rst(include_python=has_python)
    
    # Step 4: Parse SystemVerilog files (track processed files to avoid duplicates)
    processed_sv_files = set()
    
    if sv_env_dirs:
        print(f"\n📄 Parsing SystemVerilog environment files...")
        for sv_env in sv_env_dirs:
            print(f"   Processing: {sv_env}")
            parse_sv_directory(sv_env, doc_gen.source_dir / "sv_env", "Environment", processed_sv_files)
    
    if sv_test_dirs:
        print(f"\n📄 Parsing SystemVerilog test files...")
        for sv_test in sv_test_dirs:
            print(f"   Processing: {sv_test}")
            parse_sv_directory(sv_test, doc_gen.source_dir / "sv_tests", "Testcases", processed_sv_files)
    
    # Step 5: Parse Python files
    if py_env_dirs or py_test_dirs:
        print("\n🐍 Generating Python API documentation...")
        generate_python_docs(
            doc_gen.source_dir,
            py_env_dirs,
            py_test_dirs
        )
    
    # Step 6: Build HTML automatically (unless --no-build is specified)
    if not args.no_build:
        print("\n" + "="*60)
        print("🔨 Building HTML documentation...")
        print("="*60)
        success = doc_gen.build_html()
        
        if success:
            # Step 7: Open in browser if requested
            if args.view:
                doc_gen.open_in_browser()
            else:
                print(f"\n✓ Documentation built successfully!")
                print(f"\n📂 HTML output: {doc_gen.build_dir / 'html' / 'index.html'}")
                print(f"\nTo view:")
                print(f"  xdg-open {doc_gen.build_dir / 'html' / 'index.html'}")
                print(f"  or open file://{doc_gen.build_dir.absolute() / 'html' / 'index.html'}")
        else:
            print(f"\n⚠️  HTML build encountered issues")
            print(f"\nYou can manually build with:")
            print(f"  cd {args.output}")
            print(f"  sphinx-build -b html source build/html")
    else:
        print(f"\n✓ RST files generated successfully!")
        print(f"\nTo build HTML documentation later, run:")
        print(f"  cd {args.output}")
        print(f"  sphinx-build -b html source build/html")
    
    print("\n" + "=" * 60)
    print("✨ Done!")
    print("=" * 60)


def parse_sv_directory(input_dir: str, output_dir: Path, section_title: str, processed_files: set = None):
    """Parse SystemVerilog directory and generate RST files, avoiding duplicates."""
    if processed_files is None:
        processed_files = set()
        
    input_path = Path(input_dir)
    if not input_path.exists():
        print(f"⚠️  Warning: Directory {input_path} does not exist")
        return
    
    # Find all .sv and .svh files
    sv_files = list(input_path.rglob('*.sv')) + list(input_path.rglob('*.svh'))
    if not sv_files:
        print(f"⚠️  Warning: No SystemVerilog files found in {input_path}")
        return
    
    # Filter out already processed files (by absolute path)
    new_sv_files = []
    skipped_count = 0
    for sv_file in sv_files:
        abs_path = str(sv_file.absolute())
        if abs_path not in processed_files:
            new_sv_files.append(sv_file)
            processed_files.add(abs_path)
        else:
            skipped_count += 1
    
    if not new_sv_files:
        print(f"   All {skipped_count} SystemVerilog file(s) already processed, skipping...")
        return
    
    if skipped_count > 0:
        print(f"   Found {len(new_sv_files)} new SystemVerilog files ({skipped_count} already processed)")
    else:
        print(f"   Found {len(new_sv_files)} SystemVerilog files")
    
    sv_parser = SVParser()
    rst_gen = RSTGenerator(str(output_dir))
    
    for sv_file in new_sv_files:
        print(f"   Parsing: {sv_file.name}")
        parsed_data = sv_parser.parse_file(sv_file)
        rst_gen.generate_rst_for_file(sv_file, parsed_data)
    
    rst_gen.generate_index(section_title)
    print(f"✓ Generated {len(new_sv_files)} RST files in {output_dir}")


def generate_python_docs(source_dir: Path, py_env_dirs: list, py_test_dirs: list):
    """Generate Python API documentation using sphinx-apidoc."""
    import subprocess
    import re
    
    # Generate Python environment docs
    if py_env_dirs:
        py_env_api_dir = source_dir / "py_env"
        py_env_api_dir.mkdir(exist_ok=True)
        for py_env_dir in py_env_dirs:
            env_path = Path(py_env_dir)
            if env_path.exists():
                print(f"   Parsing Python environment: {py_env_dir}")
                subprocess.run([
                    "sphinx-apidoc", "-f", "-o", str(py_env_api_dir),
                    str(env_path), "--separate", "--module-first"
                ], check=False)
        # Remove " module" suffix from titles
        clean_module_titles(py_env_api_dir)
    
    # Generate Python tests docs (separate directory)
    if py_test_dirs:
        py_tests_api_dir = source_dir / "py_tests"
        py_tests_api_dir.mkdir(exist_ok=True)
        for py_tests_dir in py_test_dirs:
            tests_path = Path(py_tests_dir)
            if tests_path.exists():
                print(f"   Parsing Python tests: {py_tests_dir}")
                subprocess.run([
                    "sphinx-apidoc", "-f", "-o", str(py_tests_api_dir),
                    str(tests_path), "--separate", "--module-first"
                ], check=False)
        # Remove " module" suffix from titles
        clean_module_titles(py_tests_api_dir)
    
    # Generate index files for Python docs
    generate_python_index_files(source_dir, py_env_dirs, py_test_dirs)
    print(f"✓ Generated Python API documentation")


def clean_module_titles(api_dir: Path):
    """Remove ' module' and ' package' suffix from RST file titles."""
    import re
    for rst_file in api_dir.glob("*.rst"):
        content = rst_file.read_text()
        # Remove " module" and " package" from titles
        content = re.sub(r'^(.+) (module|package)(\n[=]+)$', r'\1\3', content, flags=re.MULTILINE)
        rst_file.write_text(content)


def generate_python_index_files(source_dir: Path, py_env_dirs: list, py_test_dirs: list):
    """Generate Python environment and testcases index RST files."""
    # Python Environment index
    if py_env_dirs:
        py_env_rst = source_dir / "python_environment.rst"
        env_content = """Python Environment
==================

PyUVM verification environment components.

.. toctree::
   :maxdepth: 4
   :caption: Environment Components:

"""
        
        # Find generated Python API files for environment
        py_env_api_dir = source_dir / "py_env"
        if py_env_api_dir.exists():
            py_files = sorted(py_env_api_dir.glob("*.rst"))
            # Get package names to exclude
            package_names = [Path(py_env_dir).name for py_env_dir in py_env_dirs]
            for py_file in py_files:
                # Skip modules.rst and the package-level index files
                if "modules" not in py_file.stem and py_file.stem not in package_names:
                    env_content += f"   py_env/{py_file.stem}\n"
        
        py_env_rst.write_text(env_content)
    
    # Python Testcases index
    if py_test_dirs:
        py_tests_rst = source_dir / "python_testcases.rst"
        tests_content = """Python Testcases
================

PyUVM test cases and sequences.

.. toctree::
   :maxdepth: 4
   :caption: Test Components:

"""
        
        # Find generated Python API files for tests
        py_tests_api_dir = source_dir / "py_tests"
        if py_tests_api_dir.exists():
            py_files = sorted(py_tests_api_dir.glob("*.rst"))
            # Get package names to exclude
            package_names = [Path(py_tests_dir).name for py_tests_dir in py_test_dirs]
            for py_file in py_files:
                # Skip modules.rst and the package-level index files
                if "modules" not in py_file.stem and py_file.stem not in package_names:
                    tests_content += f"   py_tests/{py_file.stem}\n"
        
        py_tests_rst.write_text(tests_content)


if __name__ == '__main__':
    main()
